<?php
require_once '../config/db_connect.php';

// Turn on all error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

echo "<h1>Advanced Thumbnail Fix Utility</h1>";

// 1. Check available thumbnail files
$thumbnail_dir = __DIR__ . '/../uploads/thumbnails/';
$available_thumbnails = [];

if (is_dir($thumbnail_dir)) {
    echo "<h2>Existing Thumbnails</h2>";
    echo "<table border='1' cellpadding='5'>";
    echo "<tr><th>Filename</th><th>Size</th><th>Last Modified</th></tr>";
    
    $files = scandir($thumbnail_dir);
    foreach ($files as $file) {
        if ($file != '.' && $file != '..' && is_file($thumbnail_dir . $file)) {
            $available_thumbnails[] = $file;
            $filesize = filesize($thumbnail_dir . $file);
            $last_modified = date("Y-m-d H:i:s", filemtime($thumbnail_dir . $file));
            
            echo "<tr>";
            echo "<td>{$file}</td>";
            echo "<td>" . number_format($filesize) . " bytes</td>";
            echo "<td>{$last_modified}</td>";
            echo "</tr>";
        }
    }
    
    echo "</table>";
} else {
    echo "<div style='color:red'>Thumbnail directory does not exist at {$thumbnail_dir}</div>";
    // Create the directory
    mkdir($thumbnail_dir, 0777, true);
    echo "<div style='color:green'>Created thumbnail directory.</div>";
}

// 2. Check video records in database
echo "<h2>Video Records in Database</h2>";
$videos_query = "SELECT video_id, title, thumbnail, video_url FROM videos ORDER BY video_id DESC";
$videos_result = $conn->query($videos_query);

if ($videos_result && $videos_result->num_rows > 0) {
    echo "<table border='1' cellpadding='5'>";
    echo "<tr><th>ID</th><th>Title</th><th>Current Thumbnail</th><th>Video URL</th><th>Status</th><th>Preview</th></tr>";
    
    $videos_to_fix = [];
    while ($video = $videos_result->fetch_assoc()) {
        echo "<tr>";
        echo "<td>{$video['video_id']}</td>";
        echo "<td>" . htmlspecialchars($video['title']) . "</td>";
        echo "<td>" . ($video['thumbnail'] ? htmlspecialchars($video['thumbnail']) : "<span style='color:red'>NULL</span>") . "</td>";
        echo "<td>" . htmlspecialchars($video['video_url']) . "</td>";
        
        $thumbnail_path = '';
        $status = '';
        
        // Check if we might have a matching thumbnail
        $video_file = basename($video['video_url']);
        $video_id = $video['video_id'];
        
        // Try to find a matching thumbnail based on naming patterns
        $possible_match = null;
        foreach ($available_thumbnails as $thumbnail) {
            // Check if thumbnail name contains the video ID or part of the video file name
            $video_id_str = strval($video_id);
            $video_basename = pathinfo($video_file, PATHINFO_FILENAME);
            
            if (strpos($thumbnail, $video_id_str) !== false || 
                strpos($thumbnail, $video_basename) !== false ||
                strpos($thumbnail, substr($video_basename, -10)) !== false) {
                $possible_match = $thumbnail;
                break;
            }
            
            // Try to match based on upload timestamps - assume thumbnails uploaded same time as videos
            // This logic matches thumbnails to videos if they were uploaded around the same time
            if (preg_match('/thumb_(.+)\.[a-z]+/i', $thumbnail, $thumb_matches) && 
                preg_match('/video_(.+)\.[a-z]+/i', $video_file, $video_matches)) {
                if (abs(hexdec(substr($thumb_matches[1], 0, 8)) - hexdec(substr($video_matches[1], 0, 8))) < 300) {
                    // Timestamps are close, likely the same upload
                    $possible_match = $thumbnail;
                    break;
                }
            }
        }
        
        if ($possible_match) {
            $thumbnail_path = 'uploads/thumbnails/' . $possible_match;
            $status = "<span style='color:green'>Found matching thumbnail</span>";
            $videos_to_fix[] = [
                'video_id' => $video_id,
                'thumbnail' => $thumbnail_path
            ];
        } elseif (empty($video['thumbnail']) || strpos($video['thumbnail'], '/assets/') === 0) {
            $status = "<span style='color:orange'>No matching thumbnail found</span>";
        } elseif (file_exists(__DIR__ . '/../' . $video['thumbnail'])) {
            $status = "<span style='color:blue'>Existing thumbnail is valid</span>";
        } else {
            $status = "<span style='color:red'>Invalid thumbnail path</span>";
            $videos_to_fix[] = [
                'video_id' => $video_id,
                'thumbnail' => null  // Will set to NULL
            ];
        }
        
        echo "<td>{$status}</td>";
        
        // Preview image
        echo "<td>";
        if (!empty($thumbnail_path) && file_exists(__DIR__ . '/../' . $thumbnail_path)) {
            echo "<img src='../{$thumbnail_path}' width='100' alt='Thumbnail preview'>";
        } elseif (!empty($video['thumbnail']) && file_exists(__DIR__ . '/../' . $video['thumbnail'])) {
            echo "<img src='../{$video['thumbnail']}' width='100' alt='Thumbnail preview'>";
        } else {
            echo "No preview available";
        }
        echo "</td>";
        
        echo "</tr>";
    }
    echo "</table>";

    // 3. Fix thumbnails based on our findings
    if (count($videos_to_fix) > 0) {
        echo "<h2>Fix Thumbnails</h2>";
        echo "<p>Found " . count($videos_to_fix) . " videos that need thumbnail fixes.</p>";
        
        if (isset($_POST['apply_fixes'])) {
            $success_count = 0;
            $error_messages = [];
            
            foreach ($videos_to_fix as $fix) {
                $stmt = $conn->prepare("UPDATE videos SET thumbnail = ? WHERE video_id = ?");
                $stmt->bind_param("si", $fix['thumbnail'], $fix['video_id']);
                
                if ($stmt->execute()) {
                    $success_count++;
                } else {
                    $error_messages[] = "Failed to update video ID {$fix['video_id']}: " . $stmt->error;
                }
            }
            
            if ($success_count > 0) {
                echo "<div style='color:green; margin: 10px 0; padding: 10px; border: 1px solid green; background: #efe;'>";
                echo "<strong>Success!</strong> Updated {$success_count} videos with correct thumbnail paths.";
                echo "</div>";
            }
            
            if (!empty($error_messages)) {
                echo "<div style='color:red; margin: 10px 0; padding: 10px; border: 1px solid red; background: #fee;'>";
                echo "<strong>Errors:</strong><ul>";
                foreach ($error_messages as $error) {
                    echo "<li>{$error}</li>";
                }
                echo "</ul></div>";
            }
            
            echo "<p><a href='advanced_thumbnail_fix.php' class='btn btn-primary'>Refresh Page</a></p>";
            
        } else {
            echo "<form method='post' style='margin: 20px 0;'>";
            echo "<button type='submit' name='apply_fixes' style='padding: 10px 20px; background-color: #007bff; color: white; border: none; border-radius: 5px; cursor: pointer;'>Apply Fixes</button>";
            echo "</form>";
            
            echo "<h3>Proposed Changes:</h3>";
            echo "<table border='1' cellpadding='5'>";
            echo "<tr><th>Video ID</th><th>New Thumbnail Path</th></tr>";
            
            foreach ($videos_to_fix as $fix) {
                echo "<tr>";
                echo "<td>{$fix['video_id']}</td>";
                echo "<td>" . ($fix['thumbnail'] ?? '<em>NULL</em>') . "</td>";
                echo "</tr>";
            }
            
            echo "</table>";
        }
    } else {
        echo "<p>No thumbnails need to be fixed.</p>";
    }
    
} else {
    echo "<p>No videos found in the database.</p>";
}

// Let's validate our paths
echo "<h2>Path Tests</h2>";

$test_paths = [
    "../uploads/thumbnails/",
    __DIR__ . "/../uploads/thumbnails/"
];

foreach ($test_paths as $path) {
    echo "<p>";
    echo "Path: " . htmlspecialchars($path) . "<br>";
    echo "Exists: " . (is_dir($path) ? "Yes" : "No") . "<br>";
    echo "Readable: " . (is_readable($path) ? "Yes" : "No") . "<br>";
    echo "Writable: " . (is_writable($path) ? "Yes" : "No");
    echo "</p>";
}

// Test image rendering
echo "<h2>Image Render Test</h2>";
if (!empty($available_thumbnails)) {
    $test_image = $available_thumbnails[0];
    $test_paths = [
        "../uploads/thumbnails/{$test_image}",
        "/uploads/thumbnails/{$test_image}",
        "uploads/thumbnails/{$test_image}"
    ];
    
    echo "<div style='display: flex; flex-wrap: wrap;'>";
    foreach ($test_paths as $index => $path) {
        echo "<div style='margin: 20px; text-align: center;'>";
        echo "<p><strong>Path {$index + 1}:</strong><br>" . htmlspecialchars($path) . "</p>";
        echo "<img src='{$path}' alt='Test image' style='max-width: 200px; border: 1px solid #ccc;'><br>";
        echo "File " . (file_exists(__DIR__ . '/' . $path) ? "<span style='color:green'>exists</span>" : "<span style='color:red'>does not exist</span>");
        echo "</div>";
    }
    echo "</div>";
}
?>
<style>
    body { font-family: Arial, sans-serif; max-width: 1200px; margin: 0 auto; padding: 20px; }
    h1, h2, h3 { color: #333; }
    table { border-collapse: collapse; width: 100%; margin-bottom: 20px; }
    th { background-color: #f5f5f5; }
    th, td { padding: 8px; border: 1px solid #ddd; }
    tr:nth-child(even) { background-color: #f9f9f9; }
</style>
