<?php
/**
 * AJAX Handler for Getting Course Videos
 * Returns JSON response with videos filtered by course_id
 */

// Start session and include required files
require_once '../../includes/session_start.php';
require_once '../../config/db_connect.php';

// Check if user is logged in and is a teacher
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_role']) || $_SESSION['user_role'] !== 'teacher') {
    echo json_encode([
        'success' => false,
        'message' => 'Unauthorized access'
    ]);
    exit;
}

// Get current user ID
$userId = $_SESSION['user_id'];

// Get course ID from POST data (to match our JavaScript)
$courseId = isset($_POST['course_id']) ? intval($_POST['course_id']) : 0;

try {
    // Check if the course belongs to the teacher if a specific course is selected
    if ($courseId > 0) {
        $stmt = $conn->prepare("SELECT course_id FROM courses WHERE course_id = ? AND teacher_id = ?");
        $stmt->bind_param("ii", $courseId, $userId);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows === 0) {
            echo json_encode([
                'success' => false,
                'message' => 'Course not found or you do not have permission to view it'
            ]);
            exit;
        }
    }
    
    // Check database connection first
    if (!$conn) {
        echo json_encode([
            'success' => false,
            'message' => 'Database connection failed'
        ]);
        exit;
    }

    // Get videos for the course or all videos if course_id is 0
    $videos = [];
    
    if ($courseId > 0) {
        // Get videos for a specific course (using correct query structure from videos.php)
        $sql = "SELECT v.video_id, v.title 
                FROM videos v 
                LEFT JOIN courses c ON v.course_id = c.course_id 
                WHERE c.teacher_id = ? AND v.course_id = ? AND v.is_published = 1 
                ORDER BY v.video_order ASC, v.created_at DESC";
        $stmt = $conn->prepare($sql);
        
        if (!$stmt) {
            echo json_encode([
                'success' => false,
                'message' => 'SQL prepare failed: ' . $conn->error
            ]);
            exit;
        }
        
        $stmt->bind_param("ii", $userId, $courseId);
    } else {
        // Get all videos (using correct query structure from videos.php)
        $sql = "SELECT v.video_id, v.title 
                FROM videos v 
                LEFT JOIN courses c ON v.course_id = c.course_id 
                WHERE c.teacher_id = ? AND v.is_published = 1 
                ORDER BY v.video_order ASC, v.created_at DESC";
        $stmt = $conn->prepare($sql);
        
        if (!$stmt) {
            echo json_encode([
                'success' => false,
                'message' => 'SQL prepare failed: ' . $conn->error
            ]);
            exit;
        }
        
        $stmt->bind_param("i", $userId);
    }
    
    if (!$stmt->execute()) {
        echo json_encode([
            'success' => false,
            'message' => 'SQL execute failed: ' . $stmt->error
        ]);
        exit;
    }
    $result = $stmt->get_result();
    
    while ($row = $result->fetch_assoc()) {
        // For dropdown, we only need video_id and title
        $videos[] = [
            'video_id' => $row['video_id'],
            'title' => $row['title']
        ];
    }
    
    // Return success response with data (format for JavaScript)
    echo json_encode([
        'success' => true,
        'videos' => $videos,
        'course_id' => $courseId
    ]);
    
} catch (Exception $e) {
    // Return error response
    echo json_encode([
        'success' => false,
        'message' => 'Error loading videos: ' . $e->getMessage()
    ]);
}
