<?php
/**
 * AJAX Handler for Updating Video Status
 * Updates the is_published status of a video and returns JSON response
 */

// Start session and include required files
session_start();
require_once '../../config/database.php';
require_once '../../auth/functions.php';

// Check if user is logged in and is a teacher
if (!isLoggedIn() || !hasRole('teacher')) {
    echo json_encode([
        'success' => false,
        'message' => 'Unauthorized access'
    ]);
    exit;
}

// Check if request is POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode([
        'success' => false,
        'message' => 'Invalid request method'
    ]);
    exit;
}

// Get current user ID
$userId = $_SESSION['user_id'];

// Get POST data
$videoId = isset($_POST['video_id']) ? intval($_POST['video_id']) : 0;
$isPublished = isset($_POST['is_published']) ? intval($_POST['is_published']) : 0;

// Validate input
if ($videoId <= 0) {
    echo json_encode([
        'success' => false,
        'message' => 'Invalid video ID'
    ]);
    exit;
}

try {
    // Check if video belongs to the teacher
    $stmt = $conn->prepare("SELECT video_id, status FROM videos WHERE video_id = ? AND teacher_id = ?");
    $stmt->bind_param("ii", $videoId, $userId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 0) {
        echo json_encode([
            'success' => false,
            'message' => 'Video not found or you do not have permission to update it'
        ]);
        exit;
    }
    
    $video = $result->fetch_assoc();
    
    // Check if video is approved by admin (if it has pending status, it can't be published)
    if ($isPublished && $video['status'] === 'pending') {
        echo json_encode([
            'success' => false,
            'message' => 'This video is pending admin approval and cannot be published yet'
        ]);
        exit;
    }
    
    // Update video publish status
    $stmt = $conn->prepare("UPDATE videos SET is_published = ? WHERE video_id = ? AND teacher_id = ?");
    $stmt->bind_param("iii", $isPublished, $videoId, $userId);
    $stmt->execute();
    
    if ($stmt->affected_rows > 0) {
        echo json_encode([
            'success' => true,
            'message' => $isPublished ? 'Video published successfully' : 'Video unpublished successfully',
            'data' => [
                'video_id' => $videoId,
                'is_published' => $isPublished
            ]
        ]);
    } else {
        echo json_encode([
            'success' => false,
            'message' => 'No changes made to video status'
        ]);
    }
    
} catch (Exception $e) {
    // Return error response
    echo json_encode([
        'success' => false,
        'message' => 'Error updating video status: ' . $e->getMessage()
    ]);
}
