<?php
include_once "../includes/session_start.php";
include_once "../config/db_connect.php";
include_once "../includes/functions.php";
require_once "../auth/functions.php";

// Check if user is logged in and is a teacher
if (!isset($_SESSION['user_id']) || $_SESSION['user_role'] !== 'teacher') {
    header("Location: ../login.php");
    exit;
}

$userId = $_SESSION['user_id'];

// Get teacher information
$teacher = get_teacher($conn, $userId);

// Get course filter if specified
$course_filter = isset($_GET['course_id']) ? $_GET['course_id'] : null;

// Get date range filter
$date_range = isset($_GET['date_range']) ? $_GET['date_range'] : 'last-30-days';

// Set start and end date based on date range
$end_date = date('Y-m-d');

switch ($date_range) {
    case 'last-7-days':
        $start_date = date('Y-m-d', strtotime('-7 days'));
        break;
    case 'last-30-days':
        $start_date = date('Y-m-d', strtotime('-30 days'));
        break;
    case 'last-90-days':
        $start_date = date('Y-m-d', strtotime('-90 days'));
        break;
    case 'last-year':
        $start_date = date('Y-m-d', strtotime('-1 year'));
        break;
    case 'all-time':
        $start_date = '2000-01-01'; // Some date in the distant past
        break;
    default:
        $start_date = date('Y-m-d', strtotime('-30 days'));
}

// Get teacher's courses for filtering
$courses = [];
$stmt = $conn->prepare("SELECT course_id, title FROM courses WHERE teacher_id = ? ORDER BY title");
if ($stmt) {
    $stmt->bind_param("i", $userId);
$stmt->execute();
            if ($stmt->error) {
                error_log("Error executing statement: " . $stmt->error);
            }
$result = $stmt->get_result();
if ($result) {
    while ($row = $result->fetch_assoc()) {
        $courses[] = $row;
    }
}

// Get course details if filtering by course
$courseDetail = null;
if ($course_filter) {
    $stmt = $conn->prepare("SELECT title FROM courses WHERE course_id = ? AND teacher_id = ?");
if ($stmt) {
    $stmt->bind_param("ii", $course_filter, $userId);
    $stmt->execute();
            if ($stmt->error) {
                error_log("Error executing statement: " . $stmt->error);
            }
    $result = $stmt->get_result();
    if ($result->num_rows > 0) {
        $courseDetail = $result->fetch_assoc();
    }
}

// Calculate overview statistics
// For now, use placeholder data since student_enrollments table doesn't exist yet

// Simulated total students query - replace with actual data when the table exists
$total_students = 0; // Default value

try {
    // Check if the courses table exists first
    $check_table = $conn->query("SHOW TABLES LIKE 'courses'");
    
    if ($check_table->num_rows > 0) {
        // Get total courses instead as a placeholder statistic
        $course_query = "SELECT COUNT(*) as count FROM courses WHERE teacher_id = ?";
        if ($course_filter) {
            $course_query .= " AND course_id = ?";
        }
        
        $stmt = $conn->prepare($course_query);
        if ($course_filter) {
            $stmt->bind_param("ii", $userId, $course_filter);
        } else {
            $stmt->bind_param("i", $userId);
        }
        $stmt->execute();
            if ($stmt->error) {
                error_log("Error executing statement: " . $stmt->error);
            }
        $total_courses = $stmt->get_result()->fetch_assoc()['count'];
        
        // Simulate student count (this is just a placeholder)
        $total_students = $total_courses * 5; // Assuming average of 5 students per course
    }
        }
    }
    else {
        error_log("Prepare statement failed in analytics.php");
    }
}
else {
    error_log("Database connection is null in analytics.php");
}
} catch (Exception $e) {
    // Silently handle any errors
    // echo "Error: " . $e->getMessage();
}

// Total Revenue - placeholder since payments table may not exist yet
$total_revenue = 0; // Default value

try {
    // Check if the payments table exists
    $check_table = $conn->query("SHOW TABLES LIKE 'payments'");
    
    if ($check_table->num_rows > 0) {
        // If table exists, run the actual query
        $revenue_query = "SELECT SUM(p.amount) as total 
                FROM payments p 
                JOIN courses c ON p.course_id = c.course_id 
                WHERE c.teacher_id = ? AND p.status = 'completed'";
        if ($course_filter) {
            $revenue_query .= " AND c.course_id = ?";
        }
        
        $stmt = $conn->prepare($revenue_query);
        if ($course_filter) {
            $stmt->bind_param("ii", $userId, $course_filter);
        } else {
            $stmt->bind_param("i", $userId);
        }
        $stmt->execute();
            if ($stmt->error) {
                error_log("Error executing statement: " . $stmt->error);
            }
        $total_revenue = $stmt->get_result()->fetch_assoc()['total'] ?: 0;
    } else {
        // Simulate some revenue based on course count
        if (isset($total_courses)) {
            $total_revenue = $total_courses * 29.99; // Assuming $29.99 per course
        }
    }
        }
    }
    else {
        error_log("Prepare statement failed in analytics.php");
    }
}
else {
    error_log("Database connection is null in analytics.php");
}
} catch (Exception $e) {
    // Silently handle any errors
    // echo "Error: " . $e->getMessage();
}

// Total Course Views - with safeguards
$total_views = 0; // Default value

try {
    // Check if the courses table exists and has a views column
    $check_columns = $conn->query("SHOW COLUMNS FROM courses LIKE 'views'");
    
    if ($check_columns && $check_columns->num_rows > 0) {
        // Table and column exist, perform the query
        $views_query = "SELECT SUM(c.views) as total 
                  FROM courses c 
                  WHERE c.teacher_id = ?";
        if ($course_filter) {
            $views_query .= " AND c.course_id = ?";
        }

        $stmt = $conn->prepare($views_query);
        if ($course_filter) {
            $stmt->bind_param("ii", $userId, $course_filter);
        } else {
            $stmt->bind_param("i", $userId);
        }
        $stmt->execute();
            if ($stmt->error) {
                error_log("Error executing statement: " . $stmt->error);
            }
        $result = $stmt->get_result();
        if ($result) {
            $total_views = $result->fetch_assoc()['total'] ?: 0;
        }
    } else {
        // Generate placeholder views data
        if (isset($total_courses)) {
            $total_views = $total_courses * 120; // Simulate ~120 views per course
        }
    }
        }
    }
    else {
        error_log("Prepare statement failed in analytics.php");
    }
}
else {
    error_log("Database connection is null in analytics.php");
}
} catch (Exception $e) {
    // Silently handle any errors
    // echo "Error: " . $e->getMessage();
}

// Total Video Watch Time (in minutes) - with safeguards
$total_watch_time = 0; // Default value

try {
    // Check if the video_views table exists
    $check_table = $conn->query("SHOW TABLES LIKE 'video_views'");
    
    if ($check_table && $check_table->num_rows > 0) {
        // If table exists, run the actual query
        $watch_time_query = "SELECT SUM(vv.watch_duration) as total 
                       FROM video_views vv 
                       JOIN videos v ON vv.video_id = v.video_id 
                       JOIN courses c ON v.course_id = c.course_id 
                       WHERE c.teacher_id = ?";
        if ($course_filter) {
            $watch_time_query .= " AND c.course_id = ?";
        }
        
        // Add error checking for connection and prepared statement
        $total_watch_time = 0;
        if ($conn) {
            $stmt = $conn->prepare($watch_time_query);
            if ($stmt) {
                if ($course_filter) {
                    $stmt->bind_param("ii", $userId, $course_filter);
                } else {
                    $stmt->bind_param("i", $userId);
                }
                
                $stmt->execute();
            if ($stmt->error) {
                error_log("Error executing statement: " . $stmt->error);
            }
                $result = $stmt->get_result();
                if ($result) {
                    $total_watch_time = $result->fetch_assoc()['total'] ?: 0;
                }
            } else {
                // Log the error if prepare statement failed
                error_log("Prepare statement failed in analytics.php: " . $conn->error);
            }
        } else {
            // Log that connection is null
            error_log("Database connection is null in analytics.php while calculating watch time");
        }
    } else {
        // Generate placeholder watch time data
        if (isset($total_courses)) {
            $total_watch_time = $total_courses * 480; // Average ~8 hours watch time per course (480 min)
        }
    }
        }
    }
    else {
        error_log("Prepare statement failed in analytics.php");
    }
}
else {
    error_log("Database connection is null in analytics.php");
}
} catch (Exception $e) {
    // Silently handle any errors
    // echo "Error: " . $e->getMessage();
}
$watch_time_hours = floor($total_watch_time / 60);
$watch_time_minutes = $total_watch_time % 60;

// Get enrollment data over time for chart (with placeholders since the table doesn't exist)
$enrollment_dates = [];
$enrollment_counts = [];

try {
    // Check if student_enrollments table exists
    $check_table = $conn->query("SHOW TABLES LIKE 'student_enrollments'");
    
    if ($check_table && $check_table->num_rows > 0) {
        // If the table exists, run the actual query
        $enrollment_chart_query = "SELECT DATE(se.enrollment_date) as date, COUNT(*) as count 
                              FROM student_enrollments se 
                              JOIN courses c ON se.course_id = c.course_id 
                              WHERE c.teacher_id = ? 
                              AND DATE(se.enrollment_date) BETWEEN ? AND ?";
        if ($course_filter) {
            $enrollment_chart_query .= " AND c.course_id = ?";
        }
        $enrollment_chart_query .= " GROUP BY DATE(se.enrollment_date) ORDER BY date";

        // Add error checking for connection and prepared statement
        if ($conn) {
            $stmt = $conn->prepare($enrollment_chart_query);
            if ($stmt) {
                if ($course_filter) {
                    $stmt->bind_param("issi", $userId, $start_date, $end_date, $course_filter);
                } else {
                    $stmt->bind_param("iss", $userId, $start_date, $end_date);
                }
                
                $stmt->execute();
            if ($stmt->error) {
                error_log("Error executing statement: " . $stmt->error);
            }
                $enrollment_chart_result = $stmt->get_result();
                
                if ($enrollment_chart_result) {
                    while ($row = $enrollment_chart_result->fetch_assoc()) {
                        $enrollment_dates[] = $row['date'];
                        $enrollment_counts[] = $row['count'];
                    }
                } else {
                    error_log("Failed to get result set for enrollment chart in analytics.php");
                }
            } else {
                error_log("Prepare statement failed for enrollment chart in analytics.php: " . $conn->error);
            }
        } else {
            error_log("Database connection is null in analytics.php while creating enrollment chart");
        }
    } else {
        // Generate placeholder enrollment data for the chart
        $start_timestamp = strtotime($start_date);
        $end_timestamp = strtotime($end_date);
        $days_between = ceil(($end_timestamp - $start_timestamp) / (60 * 60 * 24));
        
        // Generate some placeholder data points
        for ($i = 0; $i < min($days_between, 14); $i++) {
            $date = date('Y-m-d', $start_timestamp + ($i * 86400)); // 86400 seconds = 1 day
            $enrollment_dates[] = $date;
            $enrollment_counts[] = rand(1, 8); // Random enrollments between 1-8
        }
    }
        }
    }
    else {
        error_log("Prepare statement failed in analytics.php");
    }
}
else {
    error_log("Database connection is null in analytics.php");
}
} catch (Exception $e) {
    // Silently handle any errors
    // echo "Error: " . $e->getMessage();
}

// Get revenue data over time for chart
// Generate placeholder revenue data since we don't need payments functionality
// for the teacher dashboard
$revenue_chart_result = false; // Set to false so the next code section will use placeholder data

// Add a comment to explain why we're using placeholder data
// The payments table is not needed for the teacher dashboard as per requirements

$revenue_dates = [];
$revenue_amounts = [];
// Simulate some revenue data
$start_timestamp = strtotime($start_date);
$end_timestamp = strtotime($end_date);
$days_between = ceil(($end_timestamp - $start_timestamp) / (60 * 60 * 24));
for ($i = 0; $i < min($days_between, 14); $i++) {
    $date = date('Y-m-d', $start_timestamp + ($i * 86400)); // 86400 seconds = 1 day
    $revenue_dates[] = $date;
    $revenue_amounts[] = rand(10, 50); // Random revenue between $10-$50
}

// First, determine if we have an enrollments table and what it's called
$enrollment_table = "";
$enrollment_user_column = "user_id";
$enrollment_course_column = "course_id";
$course_id_column = "course_id";

// Check if enrollments table exists
$tables = ["student_enrollments", "enrollments", "course_enrollments", "user_courses"];
foreach ($tables as $table) {
    $table_check = $conn->query("SHOW TABLES LIKE '{$table}'");
    if ($table_check && $table_check->num_rows > 0) {
        $enrollment_table = $table;
        
        // Now check columns in this table
        $columns_query = $conn->query("SHOW COLUMNS FROM {$table}");
        if ($columns_query) {
            // Find user ID column
            $user_columns = ["user_id", "student_id", "learner_id"];
            while ($column = $columns_query->fetch_assoc()) {
                // Check for user identifier column
                foreach ($user_columns as $possible_column) {
                    if (strtolower($column['Field']) == strtolower($possible_column)) {
                        $enrollment_user_column = $column['Field'];
                    }
                }
                
                // Check for course identifier column
                $course_columns = ["course_id", "class_id", "module_id"];
                foreach ($course_columns as $possible_column) {
                    if (strtolower($column['Field']) == strtolower($possible_column)) {
                        $enrollment_course_column = $column['Field'];
                    }
                }
            }
        }
        break;
    }
}

// Check courses table for its primary key column
$courses_check = $conn->query("SHOW COLUMNS FROM courses");
if ($courses_check) {
    while ($column = $courses_check->fetch_assoc()) {
        if ($column['Key'] == 'PRI' || strtolower($column['Field']) == 'id' || 
            strtolower($column['Field']) == 'course_id') {
            $course_id_column = $column['Field'];
            break;
        }
    }
}

// Get top performing courses
$top_courses = [];

if ($enrollment_table) {
    // If we found an enrollments table, use it
    $top_courses_query = "SELECT c.{$course_id_column}, c.title, COUNT(e.{$enrollment_user_column}) as enrollments, 
                       COALESCE(SUM(p.amount), 0) as revenue, c.views, c.rating 
                       FROM courses c 
                       LEFT JOIN {$enrollment_table} e ON c.{$course_id_column} = e.{$enrollment_course_column} 
                       LEFT JOIN payments p ON c.{$course_id_column} = p.course_id AND p.status = 'completed' 
                       WHERE c.teacher_id = ? 
                       GROUP BY c.{$course_id_column} 
                       ORDER BY revenue DESC 
                       LIMIT 5";
    try {
        $stmt = $conn->prepare($top_courses_query);
if ($stmt) {
    $stmt->bind_param("i", $userId);
        $stmt->execute();
            if ($stmt->error) {
                error_log("Error executing statement: " . $stmt->error);
            }
        $top_courses_result = $stmt->get_result();
    $result = $stmt->get_result();
    if ($result) {
        while ($row = $result->fetch_assoc()) {
            $top_courses[] = $row;
        }
        }
    }
    else {
        error_log("Prepare statement failed in analytics.php");
    }
}
else {
    error_log("Database connection is null in analytics.php");
}
} catch (Exception $e) {
        // Fallback if the complex query fails
        $top_courses_query = "SELECT c.{$course_id_column}, c.title, 0 as enrollments, 
                           0 as revenue, COALESCE(c.views, 0) as views, COALESCE(c.rating, 0) as rating 
                           FROM courses c 
                           WHERE c.teacher_id = ? 
                           LIMIT 5";
        try {
            $stmt = $conn->prepare($top_courses_query);
if ($stmt) {
    $stmt->bind_param("i", $userId);
            $stmt->execute();
            if ($stmt->error) {
                error_log("Error executing statement: " . $stmt->error);
            }
            $top_courses_result = $stmt->get_result();
    $result = $stmt->get_result();
    if ($result) {
        while ($row = $result->fetch_assoc()) {
                $top_courses[] = $row;
            }
        }
    }
    else {
        error_log("Prepare statement failed in analytics.php");
    }
}
else {
    error_log("Database connection is null in analytics.php");
}
} catch (Exception $e) {
            // If all else fails, return empty array
            $top_courses = [];
        }
    }
} else {
    // If no enrollments table exists, just get basic course info
    $top_courses_query = "SELECT c.{$course_id_column}, c.title, 0 as enrollments, 
                       0 as revenue, COALESCE(c.views, 0) as views, COALESCE(c.rating, 0) as rating 
                       FROM courses c 
                       WHERE c.teacher_id = ? 
                       LIMIT 5";
    try {
        $stmt = $conn->prepare($top_courses_query);
if ($stmt) {
    $stmt->bind_param("i", $userId);
        $stmt->execute();
            if ($stmt->error) {
                error_log("Error executing statement: " . $stmt->error);
            }
        $top_courses_result = $stmt->get_result();
    $result = $stmt->get_result();
    if ($result) {
        while ($row = $result->fetch_assoc()) {
            $top_courses[] = $row;
        }
        }
    }
    else {
        error_log("Prepare statement failed in analytics.php");
    }
}
else {
    error_log("Database connection is null in analytics.php");
}
} catch (Exception $e) {
        // If all else fails, return empty array
        $top_courses = [];
    }
}

// Get geographic distribution of students
if ($enrollment_table) {
    $geo_distribution_query = "SELECT u.country, COUNT(DISTINCT e.{$enrollment_user_column}) as student_count 
                             FROM {$enrollment_table} e 
                             JOIN courses c ON e.{$enrollment_course_column} = c.{$course_id_column} 
                             JOIN users u ON e.{$enrollment_user_column} = u.id 
                             WHERE c.teacher_id = ? AND u.country IS NOT NULL ";
    if ($course_filter) {
        $geo_distribution_query .= " AND c.{$course_id_column} = ? ";
    }
    $geo_distribution_query .= "GROUP BY u.country ORDER BY student_count DESC LIMIT 7";

    try {
        $stmt = $conn->prepare($geo_distribution_query);
        if ($course_filter) {
            $stmt->bind_param("ii", $userId, $course_filter);
        } else {
            $stmt->bind_param("i", $userId);
        }
        $stmt->execute();
            if ($stmt->error) {
                error_log("Error executing statement: " . $stmt->error);
            }
        $geo_distribution_result = $stmt->get_result();
        
        $geo_countries = [];
        $geo_counts = [];
        while ($row = $geo_distribution_result->fetch_assoc()) {
            $geo_countries[] = $row['country'] ?: 'Unknown';
            $geo_counts[] = $row['student_count'];
        }
        }
    }
    else {
        error_log("Prepare statement failed in analytics.php");
    }
}
else {
    error_log("Database connection is null in analytics.php");
}
} catch (Exception $e) {
        // If the query fails, create some sample data
        $geo_countries = ['United States', 'India', 'United Kingdom', 'Canada', 'Australia'];
        $geo_counts = [rand(20, 50), rand(15, 40), rand(10, 30), rand(5, 25), rand(5, 15)];
    }
} else {
    // If no enrollment table exists, create sample data
    $geo_countries = ['United States', 'India', 'United Kingdom', 'Canada', 'Australia'];
    $geo_counts = [rand(20, 50), rand(15, 40), rand(10, 30), rand(5, 25), rand(5, 15)];
}

// Check if quizzes and submissions tables exist
$has_quizzes = false;
$has_submissions = false;
$quiz_table = "quizzes";
$submission_table = "quiz_submissions";

// Check for quizzes table
$quiz_table_check = $conn->query("SHOW TABLES LIKE 'quizzes'");
if ($quiz_table_check && $quiz_table_check->num_rows > 0) {
    $has_quizzes = true;
}

// Check for quiz submissions table - check multiple possible names
$possible_submission_tables = ["quiz_submissions", "submissions", "quiz_attempts"];
foreach ($possible_submission_tables as $table) {
    $table_check = $conn->query("SHOW TABLES LIKE '{$table}'");
    if ($table_check && $table_check->num_rows > 0) {
        $has_submissions = true;
        $submission_table = $table;
        break;
    }
}

$quiz_data = [];
if ($has_quizzes && $has_submissions) {
    // Determine the correct column names
    $quiz_id_column = "quiz_id";
    $submission_id_column = "submission_id";
    $score_column = "score";
    $quiz_course_column = "course_id";
    
    // Check columns in quizzes table
    $columns_query = $conn->query("SHOW COLUMNS FROM {$quiz_table}");
    if ($columns_query) {
        while ($column = $columns_query->fetch_assoc()) {
            // Find quiz ID column
            if ($column['Key'] == 'PRI' || strtolower($column['Field']) == 'id') {
                $quiz_id_column = $column['Field'];
            }
            // Find course ID column
            if (strtolower($column['Field']) == 'course_id') {
                $quiz_course_column = $column['Field'];
            }
        }
    }
    
    // Check columns in submissions table
    $columns_query = $conn->query("SHOW COLUMNS FROM {$submission_table}");
    if ($columns_query) {
        while ($column = $columns_query->fetch_assoc()) {
            // Find submission ID column
            if ($column['Key'] == 'PRI' || 
                strtolower($column['Field']) == 'id' ||
                strtolower($column['Field']) == 'submission_id') {
                $submission_id_column = $column['Field'];
            }
            // Find score column
            $score_columns = ["score", "points", "grade", "result"];
            foreach ($score_columns as $possible_column) {
                if (strtolower($column['Field']) == $possible_column) {
                    $score_column = $column['Field'];
                    break;
                }
            }
        }
    }
    
    // Get quiz performance data with dynamically determined column names
    $quiz_performance_query = "SELECT q.title, 
                              AVG(qs.{$score_column}) as avg_score, 
                              COUNT(qs.{$submission_id_column}) as attempts,
                              q.{$quiz_id_column} as quiz_id
                              FROM {$quiz_table} q 
                              JOIN {$submission_table} qs ON q.{$quiz_id_column} = qs.{$quiz_id_column} 
                              JOIN courses c ON q.{$quiz_course_column} = c.{$course_id_column} 
                              WHERE c.teacher_id = ? ";
    if ($course_filter) {
        $quiz_performance_query .= " AND c.{$course_id_column} = ? ";
    }
    $quiz_performance_query .= "GROUP BY q.{$quiz_id_column} ORDER BY avg_score DESC LIMIT 5";

    try {
        $stmt = $conn->prepare($quiz_performance_query);
        if ($course_filter) {
            $stmt->bind_param("ii", $userId, $course_filter);
        } else {
            $stmt->bind_param("i", $userId);
        }
        $stmt->execute();
            if ($stmt->error) {
                error_log("Error executing statement: " . $stmt->error);
            }
        $quiz_performance_result = $stmt->get_result();

        $quiz_titles = [];
        $quiz_avg_scores = [];
        $quiz_attempts = [];
        while ($row = $quiz_performance_result->fetch_assoc()) {
            $quiz_titles[] = $row['title'];
            $quiz_avg_scores[] = $row['avg_score'];
            $quiz_attempts[] = $row['attempts'];
        }
        }
    }
    else {
        error_log("Prepare statement failed in analytics.php");
    }
}
else {
    error_log("Database connection is null in analytics.php");
}
} catch (Exception $e) {
        // If the query fails, create some sample data
        $quiz_titles = ['Quiz 1', 'Quiz 2', 'Quiz 3', 'Quiz 4', 'Quiz 5'];
        $quiz_avg_scores = [rand(50, 100), rand(40, 90), rand(30, 80), rand(20, 70), rand(10, 60)];
        $quiz_attempts = [rand(10, 50), rand(10, 50), rand(10, 50), rand(10, 50), rand(10, 50)];
    }
} else {
    // If no quizzes or submissions table exists, create sample data
    $quiz_titles = ['Quiz 1', 'Quiz 2', 'Quiz 3', 'Quiz 4', 'Quiz 5'];
    $quiz_avg_scores = [rand(50, 100), rand(40, 90), rand(30, 80), rand(20, 70), rand(10, 60)];
    $quiz_attempts = [rand(10, 50), rand(10, 50), rand(10, 50), rand(10, 50), rand(10, 50)];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Analytics - Panadite Academy</title>
    
    <!-- Favicon -->
    <link rel="shortcut icon" href="../assets/images/favicon.ico" type="image/x-icon">
    
    <!-- CSS Libraries -->
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/toastify-js/src/toastify.min.css">
    
    <style>
        /* Core layout and styling for the dashboard */
        :root {
            --primary: #4e73df;
            --secondary: #858796;
            --success: #1cc88a;
            --info: #36b9cc;
            --warning: #f6c23e;
            --danger: #e74a3b;
            --light: #f8f9fc;
            --dark: #5a5c69;
            --sidebar-width: 250px;
            --sidebar-bg: whitesmoke;
            --navbar-height: 60px;
            --border-radius: 8px;
            --sidebar-text: #343a40;
            --sidebar-hover: rgba(78, 115, 223, 0.1);
            --sidebar-active: var(--primary);
        }
          .stat-card-label {
            color: var(--secondary);
            font-size: 0.9rem;
            margin-bottom: 0;
        }
              .stat-card-icon {
            width: 48px;
            height: 48px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.5rem;
            margin-bottom: 15px;
        }
        
        .stat-card-value {
            font-size: 1.8rem;
            font-weight: 700;
            margin-bottom: 5px;
            line-height: 1;
        }
        
        /* Charts */
        .chart-container {
            position: relative;
            height: 300px;
            width: 100%;
        }
        
        body {
            font-family: 'Poppins', sans-serif;
            background-color: var(--content-bg);
            margin: 0;
            padding: 0;
            overflow-x: hidden;
        }
        
        /* Layout structure */
        .dashboard-container {
            display: flex;
            min-height: 100vh;
            width: 100%;
            position: relative;
        }
        
        /* Sidebar */
        .sidebar {
            width: var(--sidebar-width);
            background-color: var(--sidebar-bg);
            color: var(--sidebar-text);
            position: fixed;
            height: 100vh;
            left: 0;
            top: 0;
            z-index: 1000;
            overflow-y: auto;
            transition: all 0.3s ease;
            box-shadow: 0 0 15px rgba(0, 0, 0, 0.1);
        }
        
        .sidebar-header {
            padding: 20px 15px;
            display: flex;
            align-items: center;
            border-bottom: 1px solid rgba(0,0,0,0.05);
            margin-bottom: 10px;
        }
        
        .sidebar-brand {
            color: var(--primary);
            text-decoration: none;
            font-weight: 700;
            font-size: 1.2rem;
            display: flex;
            align-items: center;
            transition: all 0.2s ease;
        }
        
        .sidebar-brand:hover {
            color: var(--primary-dark, #3a57c4);
            transform: translateX(3px);
        }
        
        .sidebar-brand img {
            max-width: 40px;
            margin-right: 10px;
        }
        
        .sidebar-nav {
            list-style: none;
            padding: 0;
            margin: 0;
        }
        
        .sidebar-heading {
            padding: 15px 20px 8px;
            font-size: 0.75rem;
            text-transform: uppercase;
            letter-spacing: 1.5px;
            color: var(--secondary);
            font-weight: 700;
        }
        
        .nav-item {
            margin: 3px 10px;
            border-radius: 6px;
            overflow: hidden;
        }
        
        .nav-link {
            display: flex;
            align-items: center;
            padding: 12px 15px;
            color: var(--sidebar-text);
            text-decoration: none;
            transition: all 0.2s ease;
            border-radius: 6px;
            position: relative;
            font-weight: 500;
        }
        
        .nav-link i {
            margin-right: 12px;
            font-size: 1rem;
            width: 20px;
            text-align: center;
            color: var(--secondary);
            transition: all 0.2s ease;
        }
        
        .nav-link:hover {
            background-color: var(--sidebar-hover);
            color: var(--primary);
        }
        
        .nav-link:hover i {
            color: var(--primary);
        }
        
        .nav-link.active {
            background-color: var(--primary);
            color: white;
            border-radius: 6px;
            box-shadow: 0 4px 10px rgba(0, 0, 0, 0.1);
        }
        
        .nav-link.active i {
            color: white;
        }
        
        /* Sidebar item badge */
        .nav-badge {
            position: absolute;
            right: 15px;
            background: var(--primary);
            color: white;
            font-size: 0.7rem;
            padding: 2px 6px;
            border-radius: 10px;
        }
        
        /* Main content */
        .content-wrapper {
            flex: 1;
            margin-left: var(--sidebar-width);
            padding: 20px;
            transition: margin-left 0.3s ease;
            overflow-y: auto;
            width: calc(100% - var(--sidebar-width));
            max-width: calc(100% - var(--sidebar-width));
        }
        
        /* Navbar */
        .dashboard-navbar {
            background-color: var(--card-bg);
            border-radius: var(--border-radius);
            padding: 15px 20px;
            display: flex;
            align-items: center;
            justify-content: space-between;
            margin-bottom: 20px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
        }
        
        /* Welcome banner with night sky */
        .welcome-banner {
            background: linear-gradient(135deg, #0c1624, #12223d, #0e1b38);
            color: #fff;
            border-radius: var(--border-radius);
            padding: 25px;
            margin-bottom: 20px;
            position: relative;
            overflow: hidden;
            box-shadow: 0 4px 25px rgba(0,0,0,0.4);
        }
        
        /* Stars animation - multiple layers for more density */
        .welcome-banner::before {
            content: '';
            position: absolute;
            width: 100%;
            height: 100%;
            top: 0;
            left: 0;
            background-image: 
                radial-gradient(white, rgba(255,255,255,.2) 1px, transparent 1px),
                radial-gradient(white, rgba(255,255,255,.15) 1px, transparent 1px),
                radial-gradient(white, rgba(255,255,255,.1) 1px, transparent 1px);
            background-size: 350px 350px, 200px 200px, 150px 150px;
            background-position: 0 0, 40px 60px, 130px 270px;
            animation: twinkle 15s ease-in-out infinite alternate;
        }
        
        /* Additional star layers for more stars */
        .welcome-banner::after {
            content: '';
            position: absolute;
            width: 100%;
            height: 100%;
            top: 0;
            left: 0;
            background-image: 
                radial-gradient(white, rgba(255,255,255,.2) 0.5px, transparent 0.5px),
                radial-gradient(white, rgba(255,255,255,.1) 0.5px, transparent 0.5px),
                radial-gradient(white, rgba(255,255,255,.05) 0.5px, transparent 0.5px);
            background-size: 120px 120px, 170px 170px, 200px 200px;
            background-position: 20px 30px, 90px 110px, 60px 40px;
            animation: twinkle 10s ease-in-out infinite alternate;
        }
        
        /* Extra distant stars */
        .star-field {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            z-index: 0;
            background-image: 
                radial-gradient(white, rgba(255,255,255,.2) 0.3px, transparent 0.3px),
                radial-gradient(white, rgba(255,255,255,.1) 0.2px, transparent 0.2px);
            background-size: 80px 80px, 60px 60px;
            background-position: 10px 15px, 30px 35px;
            animation: twinkle 12s ease-in-out infinite alternate-reverse;
        }
        
        /* Shooting stars */
        .shooting-star {
            position: absolute;
            width: 150px;
            height: 2px;
            background: linear-gradient(to right, rgba(255,255,255,0) 0%, rgba(255,255,255,1) 50%, rgba(255,255,255,0) 100%);
            opacity: 0;
            transform: rotate(-45deg);
            animation: shooting 6s infinite ease-out;
            filter: drop-shadow(0 0 6px rgba(255, 255, 255, 0.7));
            z-index: 2;
        }
        
        /* Add glow to shooting stars */
        .shooting-star::after {
            content: '';
            position: absolute;
            top: -1px;
            right: 0;
            width: 5px;
            height: 5px;
            border-radius: 50%;
            background: white;
            box-shadow: 0 0 10px 4px rgba(255, 255, 255, 0.7);
        }
        
        .shooting-star:nth-child(3) {
            top: 10%;
            left: 80%;
            animation-delay: 0s;
            width: 120px;
        }
        
        .shooting-star:nth-child(4) {
            top: 30%;
            left: 40%;
            animation-delay: 3s;
            width: 150px;
        }
        
        .shooting-star:nth-child(5) {
            top: 60%;
            left: 70%;
            animation-delay: 5s;
            width: 100px;
        }
        
        .shooting-star:nth-child(6) {
            top: 20%;
            left: 30%;
            animation-delay: 7s;
            width: 130px;
        }
        
        .shooting-star:nth-child(7) {
            top: 50%;
            left: 20%;
            animation-delay: 9s;
            width: 80px;
        }
        
        .shooting-star:nth-child(8) {
            top: 40%;
            left: 60%;
            animation-delay: 11s;
            width: 140px;
        }
        
        /* Ultra-fine distant stars */
        .welcome-banner::before,
        .welcome-banner::after,
        .star-field {
            pointer-events: none; /* Ensure stars don't interfere with clicking */
        }
        
        .micro-stars {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            z-index: 1;
            background-image: 
                radial-gradient(white, rgba(255,255,255,.15) 0.1px, transparent 0.1px),
                radial-gradient(white, rgba(255,255,255,.1) 0.1px, transparent 0.1px);
            background-size: 30px 30px, 40px 40px;
            background-position: 5px 7px, 15px 20px;
            animation: micro-twinkle 20s ease-in-out infinite alternate;
        }
        
        /* Nebula-like glow effects */
        .nebula {
            position: absolute;
            border-radius: 50%;
            filter: blur(25px);
            opacity: 0.15;
            z-index: 0;
        }
        
        .nebula:nth-child(1) {
            top: 20%;
            left: 80%;
            width: 120px;
            height: 120px;
            background: radial-gradient(circle, rgba(66,134,244,0.8), transparent 70%);
            animation: pulse 15s infinite alternate ease-in-out;
        }
        
        .nebula:nth-child(2) {
            top: 60%;
            left: 15%;
            width: 150px;
            height: 150px;
            background: radial-gradient(circle, rgba(111,66,244,0.6), transparent 70%);
            animation: pulse 18s infinite alternate-reverse ease-in-out;
        }
        
        @keyframes twinkle {
            0%, 100% { opacity: 0.6; }
            25% { opacity: 0.8; }
            50% { opacity: 1; }
            75% { opacity: 0.7; }
        }
        
        @keyframes micro-twinkle {
            0%, 100% { opacity: 0.3; }
            30% { opacity: 0.5; }
            60% { opacity: 0.4; }
        }
        
        @keyframes pulse {
            0%, 100% { transform: scale(1); opacity: 0.1; }
            50% { transform: scale(1.2); opacity: 0.2; }
        }
        
        @keyframes shooting {
            0% { transform: translateX(0) translateY(0) rotate(-45deg); opacity: 0; }
            2% { opacity: 1; }
            10% { transform: translateX(-300px) translateY(300px) rotate(-45deg); opacity: 0; }
            100% { transform: translateX(-300px) translateY(300px) rotate(-45deg); opacity: 0; }
        }
        
        .welcome-banner h2 {
            font-weight: 700;
            margin-bottom: 10px;
            text-shadow: 0 2px 10px rgba(0,0,0,0.3);
            position: relative;
            z-index: 1;
        }
        
        .welcome-banner p {
            position: relative;
            z-index: 1;
            text-shadow: 0 2px 8px rgba(0,0,0,0.2);
        }
        
        .welcome-banner .btn {
            margin-right: 10px;
            position: relative;
            z-index: 1;
            transition: all 0.3s ease;
            border: 2px solid transparent;
        }
        
        .welcome-banner .btn-light {
            background: rgba(255,255,255,0.9);
            color: #203a43;
            font-weight: 500;
        }
        
        .welcome-banner .btn-light:hover {
            background: #fff;
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0,0,0,0.1);
        }
        
        .welcome-banner .btn-outline-light {
            border: 2px solid rgba(255,255,255,0.7);
        }
        
        .welcome-banner .btn-outline-light:hover {
            background: rgba(255,255,255,0.1);
            border-color: #fff;
            transform: translateY(-2px);
        }
        
        /* Stat cards */
        .stat-card {
            background-color: var(--card-bg);
            border-radius: var(--border-radius);
            padding: 20px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
            margin-bottom: 20px;
            height: 100%;
            transition: transform 0.3s ease, box-shadow 0.3s ease;
        }
        
        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
        }
        
        .stat-icon {
            width: 48px;
            height: 48px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 50%;
            margin-right: 15px;
            background-color: var(--primary-light);
        }
        
        .stat-number {
            font-size: 1.8rem;
            font-weight: 700;
            margin: 10px 0 5px;
        }
        
        /* Tables */
        .data-card {
            background-color: var(--card-bg);
            border-radius: var(--border-radius);
            margin-bottom: 20px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
            overflow: hidden;
        }
        
        .data-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 15px 20px;
            border-bottom: 1px solid rgba(0,0,0,0.05);
        }
        
        .data-title {
            font-weight: 600;
            margin: 0;
        }
        
        .table-responsive {
            padding: 0;
        }
        
        .table {
            margin-bottom: 0;
        }
        
        .table th {
            border-top: none;
            font-weight: 600;
            padding: 12px 15px;
            color: #555;
            font-size: 0.9rem;
            text-transform: uppercase;
        }
        
        .table td {
            padding: 15px;
            vertical-align: middle;
        }
        
        /* Chart containers */
        .chart-container {
            position: relative;
            height: 300px;
            margin-bottom: 20px;
        }
        
        /* Status badges */
        .status-badge {
            padding: 5px 10px;
            border-radius: 50px;
            font-size: 0.75rem;
            font-weight: 600;
        }
        
        .status-published {
            background-color: rgba(40, 167, 69, 0.1);
            color: #28a745;
        }
        
        .status-draft {
            background-color: rgba(108, 117, 125, 0.1);
            color: #6c757d;
        }
        
        /* Media queries for responsiveness */
        @media (max-width: 991px) {
            .sidebar {
                transform: translateX(-100%);
                z-index: 1050;
            }
            
            .content-wrapper {
                margin-left: 0;
                width: 100%;
                max-width: 100%;
            }
            
            .sidebar.show {
                transform: translateX(0);
            }
        }
       
    </style>
</head>
<body>
    <!-- Sidebar -->
    <aside class="sidebar">
            <div class="sidebar-header">
                <a href="../index.php" class="sidebar-brand">
                    <img src="../assets/images/logo.jpeg" alt="Panadite Academy" onerror="this.src='../assets/images/favicon.ico';">
                    <span>Panadite Academy</span>
                </a>
            </div>
            
            <ul class="sidebar-nav">
                <li class="sidebar-heading">MAIN</li>
                
                <li class="nav-item">
                    <a href="dashboard.php" class="nav-link active">
                        <i class="fas fa-tachometer-alt"></i>
                        <span>Dashboard</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="courses.php" class="nav-link">
                        <i class="fas fa-graduation-cap"></i>
                        <span>Courses</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="create-course.php" class="nav-link">
                        <i class="fas fa-plus-circle"></i>
                        <span>Create Course</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="videos.php" class="nav-link">
                        <i class="fas fa-video"></i>
                        <span>Videos</span>
                        <span class="nav-badge">New</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="upload-video.php" class="nav-link">
                        <i class="fas fa-cloud-upload-alt"></i>
                        <span>Upload Video</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="students.php" class="nav-link">
                        <i class="fas fa-user-graduate"></i>
                        <span>Students</span>
                    </a>
                </li>
                
                <li class="sidebar-heading">TOOLS</li>
                
                <li class="nav-item">
                    <a href="analytics.php" class="nav-link">
                        <i class="fas fa-chart-line"></i>
                        <span>Analytics</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="quizzes.php" class="nav-link">
                        <i class="fas fa-question-circle"></i>
                        <span>Quizzes</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="assignments.php" class="nav-link">
                        <i class="fas fa-tasks"></i>
                        <span>Assignments</span>
                    </a>
                </li>
                
                <li class="sidebar-heading">ACCOUNT</li>
                
                <li class="nav-item">
                    <a href="profile.php" class="nav-link">
                        <i class="fas fa-user-circle"></i>
                        <span>Profile</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="settings.php" class="nav-link">
                        <i class="fas fa-cog"></i>
                        <span>Settings</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="../logout.php" class="nav-link">
                        <i class="fas fa-sign-out-alt"></i>
                        <span>Logout</span>
                    </a>
                </li>
            </ul>
        </aside>
        

    <!-- Content Wrapper -->
    <div class="content-wrapper">
        <!-- Navbar -->
        <nav class="dashboard-navbar">
                <div class="d-flex align-items-center">
                    <button id="sidebarToggle" class="btn btn-sm btn-light me-2">
                        <i class="fas fa-bars"></i>
                    </button>
                    <h4 class="mb-0">Analytics</h4>
                </div>
                
                <div class="d-flex align-items-center">
                    <!-- Search -->
                    <div class="me-3 d-none d-md-block">
                        <form class="d-flex">
                            <input type="text" class="form-control form-control-sm" placeholder="Search...">
                        </form>
                    </div>
                    
                    <!-- Notifications -->
                    <div class="dropdown me-3">
                        <button class="btn btn-light position-relative" data-bs-toggle="dropdown">
                            <i class="fas fa-bell"></i>
                            <span class="position-absolute top-0 start-100 translate-middle badge rounded-pill bg-danger">
                                3
                            </span>
                        </button>
                        <div class="dropdown-menu dropdown-menu-end">
                            <div class="p-2 border-bottom">
                                <h6 class="mb-0">Notifications</h6>
                            </div>
                            <div class="p-2 border-bottom">
                                <div class="d-flex">
                                    <div class="flex-shrink-0 me-2">
                                        <div class="rounded-circle bg-light p-2">
                                            <i class="fas fa-user-graduate text-primary"></i>
                                        </div>
                                    </div>
                                    <div>
                                        <p class="mb-0">New student enrolled</p>
                                        <small class="text-muted">2 hours ago</small>
                                    </div>
                                </div>
                            </div>
                            <div class="p-2 text-center">
                                <a href="#" class="text-decoration-none">View all notifications</a>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Profile -->
                    <div class="dropdown">
                        <button class="btn btn-light d-flex align-items-center" data-bs-toggle="dropdown">
                            <img src="<?php echo !empty($teacher['profile_picture']) ? '../uploads/profile/' . htmlspecialchars($teacher['profile_picture']) : '../assets/images/avatar-placeholder.jpg'; ?>" 
                                 class="rounded-circle me-2" height="32" width="32" alt="<?php echo htmlspecialchars($teacher['first_name']); ?>">
                            <span class="d-none d-md-inline"><?php echo htmlspecialchars($teacher['first_name']); ?></span>
                        </button>
                        <div class="dropdown-menu dropdown-menu-end">
                            <a href="profile.php" class="dropdown-item">
                                <i class="fas fa-user me-2"></i> Profile
                            </a>
                            <a href="settings.php" class="dropdown-item">
                                <i class="fas fa-cog me-2"></i> Settings
                            </a>
                            <div class="dropdown-divider"></div>
                            <a href="../logout.php" class="dropdown-item">
                                <i class="fas fa-sign-out-alt me-2"></i> Logout
                            </a>
                        </div>
                    </div>
                </div>
            </nav>

        <!-- Analytics Content -->
        <div class="container-fluid px-0">
            <!-- Page Header -->
            <div class="d-flex justify-content-between align-items-center mb-4">
              
                <div>
                    <form method="get" class="d-flex">
                        <?php if ($course_filter): ?>
                            <input type="hidden" name="course_id" value="<?php echo htmlspecialchars($course_filter); ?>">
                        <?php endif; ?>
                        <select name="date_range" class="form-select me-2" onchange="this.form.submit()">
                            <option value="last-7-days" <?php echo $date_range == 'last-7-days' ? 'selected' : ''; ?>>Last 7 days</option>
                            <option value="last-30-days" <?php echo $date_range == 'last-30-days' ? 'selected' : ''; ?>>Last 30 days</option>
                            <option value="last-90-days" <?php echo $date_range == 'last-90-days' ? 'selected' : ''; ?>>Last 90 days</option>
                            <option value="last-year" <?php echo $date_range == 'last-year' ? 'selected' : ''; ?>>Last year</option>
                            <option value="all-time" <?php echo $date_range == 'all-time' ? 'selected' : ''; ?>>All time</option>
                        </select>
                        <button type="submit" class="btn btn-outline-primary">Apply</button>
                    </form>
                </div>
            </div>
            
            <!-- Course Filter -->
            <div class="mb-4">
                <div class="card shadow-sm">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-center">
                            <h5 class="card-title mb-0">
                                <?php if ($course_filter && $courseDetail): ?>
                                    Showing analytics for: <span class="text-primary"><?php echo htmlspecialchars($courseDetail['title']); ?></span>
                                <?php else: ?>
                                    All Courses
                                <?php endif; ?>
                            </h5>
                            <div class="dropdown">
                                <button class="btn btn-outline-secondary dropdown-toggle" type="button" id="courseFilterDropdown" data-bs-toggle="dropdown" aria-expanded="false">
                                    <?php if ($course_filter && $courseDetail): ?>
                                        <?php echo htmlspecialchars($courseDetail['title']); ?>
                                    <?php else: ?>
                                        Filter by Course
                                    <?php endif; ?>
                                </button>
                                <ul class="dropdown-menu dropdown-menu-end" aria-labelledby="courseFilterDropdown">
                                    <li><a class="dropdown-item <?php echo !$course_filter ? 'active' : ''; ?>" href="analytics.php?date_range=<?php echo htmlspecialchars($date_range); ?>">All Courses</a></li>
                                    <?php if (count($courses) > 0): ?>
                                        <li><hr class="dropdown-divider"></li>
                                        <?php foreach ($courses as $course): ?>
                                            <li>
                                                <a class="dropdown-item <?php echo $course_filter == $course['course_id'] ? 'active' : ''; ?>" 
                                                   href="analytics.php?course_id=<?php echo $course['course_id']; ?>&date_range=<?php echo htmlspecialchars($date_range); ?>">
                                                    <?php echo htmlspecialchars($course['title']); ?>
                                                </a>
                                            </li>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </ul>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Overview Statistics -->
            <div class="row mb-4 g-3">
                <!-- Students Enrolled -->
                <div class="col-md-6 col-lg-3">
                    <div class="card stat-card">
                        <div class="card-body">
                            <div class="stat-card-icon bg-primary-light text-primary">
                                <i class="fas fa-users"></i>
                            </div>
                            <div class="stat-card-value"><?php echo number_format($total_students); ?></div>
                            <p class="stat-card-label">Students Enrolled</p>
                        </div>
                    </div>
                </div>
                
                <!-- Total Revenue -->
                <div class="col-md-6 col-lg-3">
                    <div class="card stat-card">
                        <div class="card-body">
                            <div class="stat-card-icon bg-success-light text-success">
                                <i class="fas fa-dollar-sign"></i>
                            </div>
                            <div class="stat-card-value">$<?php echo number_format($total_revenue, 2); ?></div>
                            <p class="stat-card-label">Total Revenue</p>
                        </div>
                    </div>
                </div>
                
                <!-- Course Views -->
                <div class="col-md-6 col-lg-3">
                    <div class="card stat-card">
                        <div class="card-body">
                            <div class="stat-card-icon bg-info-light text-info">
                                <i class="fas fa-eye"></i>
                            </div>
                            <div class="stat-card-value"><?php echo number_format($total_views); ?></div>
                            <p class="stat-card-label">Total Course Views</p>
                        </div>
                    </div>
                </div>
                
                <!-- Watch Time -->
                <div class="col-md-6 col-lg-3">
                    <div class="card stat-card">
                        <div class="card-body">
                            <div class="stat-card-icon bg-warning-light text-warning">
                                <i class="fas fa-clock"></i>
                            </div>
                            <div class="stat-card-value"><?php echo $watch_time_hours; ?><small>h</small> <?php echo $watch_time_minutes; ?><small>m</small></div>
                            <p class="stat-card-label">Total Watch Time</p>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Charts Row 1 -->
            <div class="row mb-4 g-3">
                <!-- Enrollment Chart -->
                <div class="col-md-12 col-lg-8">
                    <div class="card shadow-sm h-100">
                        <div class="card-header bg-transparent py-3">
                            <h5 class="card-title mb-0">Student Enrollments</h5>
                        </div>
                        <div class="card-body">
                            <div class="chart-container">
                                <canvas id="enrollmentChart"></canvas>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Geographic Distribution -->
                <div class="col-md-12 col-lg-4">
                    <div class="card shadow-sm h-100">
                        <div class="card-header bg-transparent py-3">
                            <h5 class="card-title mb-0">Student Geography</h5>
                        </div>
                        <div class="card-body">
                            <div class="chart-container">
                                <canvas id="geoChart"></canvas>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Charts Row 2 -->
            <div class="row mb-4 g-3">
                <!-- Revenue Chart -->
                <div class="col-md-12 col-lg-8">
                    <div class="card shadow-sm h-100">
                        <div class="card-header bg-transparent py-3">
                            <h5 class="card-title mb-0">Revenue Overview</h5>
                        </div>
                        <div class="card-body">
                            <div class="chart-container">
                                <canvas id="revenueChart"></canvas>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Quiz Performance -->
                <div class="col-md-12 col-lg-4">
                    <div class="card shadow-sm h-100">
                        <div class="card-header bg-transparent py-3">
                            <h5 class="card-title mb-0">Quiz Performance</h5>
                        </div>
                        <div class="card-body">
                            <div class="chart-container">
                                <canvas id="quizChart"></canvas>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Top Performing Courses -->
            <div class="card shadow-sm mb-4">
                <div class="card-header bg-transparent py-3">
                    <h5 class="card-title mb-0">Top Performing Courses</h5>
                </div>
                <div class="card-body p-0">
                    <div class="table-responsive">
                        <table class="table table-hover align-middle mb-0">
                            <thead class="table-light">
                                <tr>
                                    <th>Course</th>
                                    <th>Enrollments</th>
                                    <th>Revenue</th>
                                    <th>Views</th>
                                    <th>Rating</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (count($top_courses) > 0): ?>
                                    <?php foreach ($top_courses as $course): ?>
                                        <tr>
                                            <td>
                                                <div class="d-flex align-items-center">
                                                    <div class="course-icon bg-light rounded me-3 p-2">
                                                        <i class="fas fa-book text-primary"></i>
                                                    </div>
                                                    <div>
                                                        <h6 class="mb-0"><?php echo htmlspecialchars($course['title']); ?></h6>
                                                    </div>
                                                </div>
                                            </td>
                                            <td><?php echo number_format($course['enrollments']); ?></td>
                                            <td>$<?php echo number_format($course['revenue'], 2); ?></td>
                                            <td><?php echo number_format($course['views']); ?></td>
                                            <td>
                                                <div class="d-flex align-items-center">
                                                    <span class="me-2"><?php echo number_format($course['rating'], 1); ?></span>
                                                    <div>
                                                        <?php 
                                                        $rating = round($course['rating']);
                                                        for ($i = 1; $i <= 5; $i++) {
                                                            if ($i <= $rating) {
                                                                echo '<i class="fas fa-star text-warning"></i>';
                                                            } else {
                                                                echo '<i class="far fa-star text-muted"></i>';
                                                            }
                                                        }
                                                        ?>
                                                    </div>
                                                </div>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php else: ?>
                                    <tr>
                                        <td colspan="5" class="text-center py-4">
                                            <p class="text-muted mb-0">No course data available.</p>
                                        </td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- JavaScript Libraries -->
    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/chart.js@3.9.1/dist/chart.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/toastify-js"></script>
    
    <script>
        $(document).ready(function() {
            // Toggle sidebar
            $('#sidebarToggle').on('click', function() {
                $('.sidebar').toggleClass('collapsed');
                $('.content-wrapper').toggleClass('expanded');
                
                if ($('.sidebar').hasClass('collapsed')) {
                    $('.sidebar').css('width', '70px');
                    $('.content-wrapper').css('margin-left', '70px');
                    $('.content-wrapper').css('width', 'calc(100% - 70px)');
                    $('.sidebar-brand span').hide();
                    $('.nav-link span').hide();
                    $('.sidebar-heading').hide();
                } else {
                    $('.sidebar').css('width', 'var(--sidebar-width)');
                    $('.content-wrapper').css('margin-left', 'var(--sidebar-width)');
                    $('.content-wrapper').css('width', 'calc(100% - var(--sidebar-width))');
                    $('.sidebar-brand span').show();
                    $('.nav-link span').show();
                    $('.sidebar-heading').show();
                }
            });
            
            // Chart.js Configuration
            const primaryColor = '#4169e1';
            const secondaryColor = '#6c757d';
            const successColor = '#28a745';
            const warningColor = '#ffc107';
            const dangerColor = '#dc3545';
            const infoColor = '#17a2b8';
            
            // Helper function for chart gradients
            function createGradient(ctx, colorStart, colorEnd) {
                const gradient = ctx.createLinearGradient(0, 0, 0, 400);
                gradient.addColorStop(0, colorStart);
                gradient.addColorStop(1, colorEnd);
                return gradient;
            }
            
            // Enrollment Chart
            const enrollmentCtx = document.getElementById('enrollmentChart').getContext('2d');
            const enrollmentGradient = createGradient(enrollmentCtx, 'rgba(65, 105, 225, 0.6)', 'rgba(65, 105, 225, 0.1)');
            
            new Chart(enrollmentCtx, {
                type: 'line',
                data: {
                    labels: <?php echo json_encode($enrollment_dates); ?>,
                    datasets: [{
                        label: 'Student Enrollments',
                        data: <?php echo json_encode($enrollment_counts); ?>,
                        backgroundColor: enrollmentGradient,
                        borderColor: primaryColor,
                        borderWidth: 2,
                        pointBackgroundColor: primaryColor,
                        pointBorderColor: '#fff',
                        pointRadius: 4,
                        fill: true,
                        tension: 0.4
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            display: true,
                            position: 'top'
                        },
                        tooltip: {
                            mode: 'index',
                            intersect: false,
                            backgroundColor: 'rgba(0, 0, 0, 0.7)',
                            titleColor: '#fff',
                            bodyColor: '#fff',
                            titleFont: { size: 14 },
                            bodyFont: { size: 13 },
                            padding: 12,
                            displayColors: true
                        }
                    },
                    scales: {
                        x: {
                            grid: {
                                display: false
                            },
                            ticks: {
                                maxTicksLimit: 10
                            }
                        },
                        y: {
                            beginAtZero: true,
                            grid: {
                                borderDash: [3, 3]
                            },
                            ticks: {
                                precision: 0
                            }
                        }
                    }
                }
            });
            
            // Revenue Chart
            const revenueCtx = document.getElementById('revenueChart').getContext('2d');
            const revenueGradient = createGradient(revenueCtx, 'rgba(40, 167, 69, 0.6)', 'rgba(40, 167, 69, 0.1)');
            
            new Chart(revenueCtx, {
                type: 'bar',
                data: {
                    labels: <?php echo json_encode($revenue_dates); ?>,
                    datasets: [{
                        label: 'Revenue',
                        data: <?php echo json_encode($revenue_amounts); ?>,
                        backgroundColor: revenueGradient,
                        borderColor: successColor,
                        borderWidth: 1
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            display: true
                        }
                    },
                    scales: {
                        x: {
                            grid: {
                                display: false
                            },
                            ticks: {
                                maxTicksLimit: 10
                            }
                        },
                        y: {
                            beginAtZero: true,
                            grid: {
                                borderDash: [3, 3]
                            },
                            ticks: {
                                callback: function(value) {
                                    return '$' + value;
                                }
                            }
                        }
                    }
                }
            });
            
            // Geographic Distribution Chart
            const geoCtx = document.getElementById('geoChart').getContext('2d');
            const geoColors = [
                primaryColor,
                secondaryColor,
                infoColor,
                warningColor,
                dangerColor,
                '#6610f2',
                '#fd7e14'
            ];
            
            new Chart(geoCtx, {
                type: 'doughnut',
                data: {
                    labels: <?php echo json_encode($geo_countries); ?>,
                    datasets: [{
                        data: <?php echo json_encode($geo_counts); ?>,
                        backgroundColor: geoColors,
                        borderWidth: 1
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            position: 'bottom',
                            labels: {
                                padding: 20,
                                boxWidth: 12
                            }
                        }
                    },
                    cutout: '70%'
                }
            });
            
            // Quiz Performance Chart
            const quizCtx = document.getElementById('quizChart').getContext('2d');
            
            new Chart(quizCtx, {
                type: 'bar',
                data: {
                    labels: <?php echo json_encode($quiz_titles); ?>,
                    datasets: [{
                        label: 'Avg Score (%)',
                        data: <?php echo json_encode($quiz_avg_scores); ?>,
                        backgroundColor: infoColor,
                        borderColor: infoColor,
                        borderWidth: 1
                    }]
                },
                options: {
                    indexAxis: 'y',
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            display: false
                        }
                    },
                    scales: {
                        x: {
                            beginAtZero: true,
                            max: 100,
                            grid: {
                                display: true,
                                borderDash: [3, 3]
                            },
                            ticks: {
                                callback: function(value) {
                                    return value + '%';
                                }
                            }
                        },
                        y: {
                            grid: {
                                display: false
                            }
                        }
                    }
                }
            });
        });
    </script>
</body>
</html>
