<?php
/**
 * Apply Database Fix for File Storage Issues
 * This script fixes the VARCHAR(255) limitation and tests the results
 */

require_once 'includes/session_start.php';
require_once 'includes/db_connection.php';

// Only allow admin access
if (!isset($_SESSION['user_role']) || $_SESSION['user_role'] !== 'admin') {
    die('Access denied. Admin only.');
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Database Fix - Panadite Academy</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body>
<div class="container mt-4">
    <h1>🔧 Database Fix: File Storage Field Size</h1>
    <p class="text-muted">Fixing VARCHAR(255) → TEXT to support multiple file uploads</p>

    <?php
    try {
        echo "<div class='alert alert-info'><strong>Starting database migration...</strong></div>";
        
        // 1. Backup current schema
        echo "<h3>📋 Step 1: Schema Backup</h3>";
        $backup_query = "CREATE TABLE IF NOT EXISTS assignment_submissions_backup_" . date('Y_m_d_H_i') . " AS SELECT * FROM assignment_submissions WHERE file_url IS NOT NULL";
        
        if ($conn->query($backup_query)) {
            echo "<div class='alert alert-success'>✅ Backup created successfully</div>";
        } else {
            throw new Exception("Backup failed: " . $conn->error);
        }
        
        // 2. Apply the field size fix
        echo "<h3>🔧 Step 2: Applying Field Size Fix</h3>";
        $alter_query = "ALTER TABLE assignment_submissions MODIFY COLUMN file_url TEXT";
        
        if ($conn->query($alter_query)) {
            echo "<div class='alert alert-success'>✅ Field size increased: VARCHAR(255) → TEXT</div>";
        } else {
            throw new Exception("Schema update failed: " . $conn->error);
        }
        
        // 3. Verify the changes
        echo "<h3>🔍 Step 3: Verification</h3>";
        $verify_query = "DESCRIBE assignment_submissions";
        $result = $conn->query($verify_query);
        
        echo "<table class='table table-sm'>";
        echo "<thead><tr><th>Field</th><th>New Type</th><th>Status</th></tr></thead>";
        echo "<tbody>";
        
        while ($row = $result->fetch_assoc()) {
            if ($row['Field'] === 'file_url') {
                $status = (strtolower($row['Type']) === 'text') ? 
                    "<span class='badge bg-success'>✅ FIXED</span>" : 
                    "<span class='badge bg-danger'>❌ NOT FIXED</span>";
                echo "<tr>";
                echo "<td><strong>" . $row['Field'] . "</strong></td>";
                echo "<td>" . $row['Type'] . "</td>";
                echo "<td>" . $status . "</td>";
                echo "</tr>";
            }
        }
        echo "</tbody></table>";
        
        // 4. Test with actual data
        echo "<h3>🧪 Step 4: Testing Current Data</h3>";
        $test_query = "SELECT 
            COUNT(*) as total_files,
            COUNT(CASE WHEN CHAR_LENGTH(file_url) >= 255 THEN 1 END) as previously_truncated,
            MAX(CHAR_LENGTH(file_url)) as max_length
            FROM assignment_submissions 
            WHERE file_url IS NOT NULL AND file_url != ''";
        
        $result = $conn->query($test_query);
        $stats = $result->fetch_assoc();
        
        echo "<div class='row'>";
        echo "<div class='col-md-4'>";
        echo "<div class='card'>";
        echo "<div class='card-body text-center'>";
        echo "<h5>Total Files</h5>";
        echo "<h2 class='text-primary'>" . $stats['total_files'] . "</h2>";
        echo "</div></div></div>";
        
        echo "<div class='col-md-4'>";
        echo "<div class='card'>";
        echo "<div class='card-body text-center'>";
        echo "<h5>Previously Truncated</h5>";
        echo "<h2 class='text-warning'>" . $stats['previously_truncated'] . "</h2>";
        echo "</div></div></div>";
        
        echo "<div class='col-md-4'>";
        echo "<div class='card'>";
        echo "<div class='card-body text-center'>";
        echo "<h5>Max Field Length</h5>";
        echo "<h2 class='text-info'>" . $stats['max_length'] . "</h2>";
        echo "</div></div></div>";
        echo "</div>";
        
        echo "<div class='alert alert-success mt-4'>";
        echo "<h4>🎉 Migration Completed Successfully!</h4>";
        echo "<ul>";
        echo "<li>✅ File field size increased to TEXT (65,535 characters)</li>";
        echo "<li>✅ Existing data preserved</li>";
        echo "<li>✅ New submissions can now store unlimited files</li>";
        echo "</ul>";
        echo "</div>";
        
    } catch (Exception $e) {
        echo "<div class='alert alert-danger'>";
        echo "<strong>❌ Migration Failed:</strong> " . $e->getMessage();
        echo "</div>";
    }
    ?>
    
    <div class="mt-4">
        <a href="teacher/view-submissions.php?id=41" class="btn btn-primary">Test View Submissions</a>
        <a href="debug_file_submissions.php" class="btn btn-secondary">Run Debug Again</a>
        <a href="teacher/assignments.php" class="btn btn-outline-primary">Back to Assignments</a>
    </div>
</div>
</body>
</html>
