<?php
// Set current page for active menu highlighting
$current_page = 'assignments.php';

// Include the centralized teacher layout
require_once __DIR__ . '/components/teacher_layout.php';

// Error handling and reporting during development
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Fatal error handler to capture and display errors
function assignments_shutdown_handler() {
    $error = error_get_last();
    if ($error !== NULL && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
        error_log('Fatal error in assignments.php: ' . print_r($error, true));
    }
}
register_shutdown_function('assignments_shutdown_handler');

/**
 * Get teacher data with robust fallback approach
 */
function get_teacher($conn, $teacher_id) {
    // Default teacher data in case of failure
    $default_teacher = [
        'user_id' => $teacher_id,
        'first_name' => 'Academy',
        'last_name' => 'Teacher',
        'email' => 'teacher@panadite.com',
        'bio' => 'Teacher profile is currently unavailable.',
        'profile_picture' => '../assets/img/default-teacher.png',
    ];
    
    // Check if we have a valid connection
    if (!$conn || $conn->connect_error) {
        error_log("Database connection failed in get_teacher");
        return $default_teacher;
    }
    
    try {
        // Check if the user table exists
        $table_check = $conn->query("SHOW TABLES LIKE 'users'");
        if (!$table_check || $table_check->num_rows == 0) {
            error_log("Users table does not exist");
            return $default_teacher;
        }
        
        // Prepare statement with left join to teacher_profiles
        $stmt = $conn->prepare("SELECT u.user_id, u.first_name, u.last_name, u.email, 
                              tp.bio, COALESCE(tp.profile_image, '../assets/img/default-teacher.png') as profile_picture
                              FROM users u 
                              LEFT JOIN teacher_profiles tp ON u.user_id = tp.teacher_id 
                              WHERE u.user_id = ? AND u.user_role = 'teacher'");
        
        if (!$stmt) {
            error_log("Failed to prepare statement in get_teacher: " . $conn->error);
            return $default_teacher;
        }
        
        $stmt->bind_param("i", $teacher_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result && $result->num_rows > 0) {
            return $result->fetch_assoc();
        } else {
            error_log("No teacher found with ID $teacher_id");
            return $default_teacher;
        }
    } catch (Exception $e) {
        error_log("Exception in get_teacher: " . $e->getMessage());
        return $default_teacher;
    }
}

$message = '';
$userId = $_SESSION['user_id'];
$course_filter = isset($_GET['course_id']) ? $_GET['course_id'] : null;

// Handle assignment deletion
if (isset($_GET['delete']) && is_numeric($_GET['delete'])) {
    $assignment_id = (int)$_GET['delete'];
    
    if ($conn) {
        try {
            // First verify the assignment belongs to this teacher
            $verify_stmt = $conn->prepare("SELECT assignment_id, title FROM assignments WHERE assignment_id = ? AND teacher_id = ?");
            $verify_stmt->bind_param("ii", $assignment_id, $userId);
            $verify_stmt->execute();
            $verify_result = $verify_stmt->get_result();
            
            if ($verify_result && $verify_result->num_rows > 0) {
                $assignment = $verify_result->fetch_assoc();
                
                // Check if assignment has submissions
                $submissions_stmt = $conn->prepare("SELECT COUNT(*) as count FROM assignment_submissions WHERE assignment_id = ?");
                $submissions_stmt->bind_param("i", $assignment_id);
                $submissions_stmt->execute();
                $submissions_result = $submissions_stmt->get_result();
                $submission_count = 0;
                
                if ($submissions_result) {
                    $submission_data = $submissions_result->fetch_assoc();
                    $submission_count = $submission_data['count'];
                }
                
                // Debug: Log the actual counts for troubleshooting
                error_log("Assignment ID: $assignment_id, Submission count in deletion: $submission_count, Force delete: " . ($force_delete ? 'true' : 'false'));
                
                // Check if force delete is requested (for assignments with submissions)
                $force_delete = isset($_GET['force']) && $_GET['force'] == '1';
                
                // For testing purposes, always allow deletion with cascade if there are submissions
                if ($submission_count > 0) {
                    // Cascade delete - remove all related data
                    $conn->begin_transaction();
                    try {
                        // Delete assignment submissions (includes grades/marks)
                        $delete_submissions = $conn->prepare("DELETE FROM assignment_submissions WHERE assignment_id = ?");
                        $delete_submissions->bind_param("i", $assignment_id);
                        $delete_submissions->execute();
                        
                        // Delete any assignment files if they exist
                        $delete_files = $conn->prepare("DELETE FROM assignment_files WHERE assignment_id = ?");
                        if ($delete_files) {
                            $delete_files->bind_param("i", $assignment_id);
                            $delete_files->execute();
                        }
                        
                        // Delete the assignment itself
                        $delete_assignment = $conn->prepare("DELETE FROM assignments WHERE assignment_id = ? AND teacher_id = ?");
                        $delete_assignment->bind_param("ii", $assignment_id, $userId);
                        $delete_assignment->execute();
                        
                        $conn->commit();
                        $message = '<div class="alert alert-success">Assignment "' . htmlspecialchars($assignment['title']) . '" and all ' . $submission_count . ' student submissions have been permanently deleted.</div>';
                        
                        // Redirect to remove the delete parameter from URL
                        echo '<script>
                            setTimeout(function() {
                                window.location.href = "assignments.php' . ($course_filter ? '?course_id=' . $course_filter : '') . '";
                            }, 3000);
                        </script>';
                    } catch (Exception $e) {
                        $conn->rollback();
                        error_log("Error in cascade delete: " . $e->getMessage());
                        $message = '<div class="alert alert-danger">Error deleting assignment and submissions. Please try again.</div>';
                    }
                } else {
                    // Safe to delete - no submissions
                    $delete_stmt = $conn->prepare("DELETE FROM assignments WHERE assignment_id = ? AND teacher_id = ?");
                    $delete_stmt->bind_param("ii", $assignment_id, $userId);
                    
                    if ($delete_stmt->execute()) {
                        $message = '<div class="alert alert-success">Assignment "' . htmlspecialchars($assignment['title']) . '" has been deleted successfully.</div>';
                        
                        // Redirect to remove the delete parameter from URL
                        echo '<script>
                            setTimeout(function() {
                                window.location.href = "assignments.php' . ($course_filter ? '?course_id=' . $course_filter : '') . '";
                            }, 2000);
                        </script>';
                    } else {
                        $message = '<div class="alert alert-danger">Error deleting assignment. Please try again.</div>';
                    }
                }
            } else {
                $message = '<div class="alert alert-danger">Assignment not found or you don\'t have permission to delete it.</div>';
            }
        } catch (Exception $e) {
            error_log("Error deleting assignment: " . $e->getMessage());
            $message = '<div class="alert alert-danger">Error deleting assignment. Please try again.</div>';
        }
    } else {
        $message = '<div class="alert alert-danger">Database connection error. Please try again later.</div>';
    }
}

// Check database connection
if (!$conn) {
    error_log("Database connection is null in assignments.php");
    $message = '<div class="alert alert-danger">Database connection error. Please try again later.</div>';
}

// Get teacher information with fallback for database errors
if ($conn) {
    $teacher = get_teacher($conn, $userId);
} else {
    // Provide minimal teacher info if connection failed
    $teacher = [
        'user_id' => $userId,
        'first_name' => 'User',
        'last_name' => '',
        'profile_picture' => ''
    ];
}

// Get teacher's courses for filtering
$courses = [];
if ($conn) {
    try {
        $stmt = $conn->prepare("SELECT course_id, title FROM courses WHERE teacher_id = ? ORDER BY title");
        $stmt->bind_param("i", $userId);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($result) {
            while ($row = $result->fetch_assoc()) {
                $courses[] = $row;
            }
        }
    } catch (Exception $e) {
        error_log("Error retrieving courses in assignments.php: " . $e->getMessage());
    }
}

// Initialize assignments array
$assignments = [];

// Only proceed with database operations if connection is available
if ($conn) {
    try {
        // Check if the assignments table exists
        $check_table = $conn->query("SHOW TABLES LIKE 'assignments'");
        
        if ($check_table && $check_table->num_rows > 0) {
            // Check if the assignment_submissions table exists and has score column
            $has_submissions = false;
            $check_submissions = $conn->query("SHOW TABLES LIKE 'assignment_submissions'");
            if ($check_submissions && $check_submissions->num_rows > 0) {
                // Check if score column exists
                $check_score = $conn->query("SHOW COLUMNS FROM assignment_submissions LIKE 'score'");
                $has_submissions = ($check_score && $check_score->num_rows > 0);
            }
            
            // Build query based on available tables and columns
            if ($has_submissions) {
                // Full query with submissions and scores
                $query = "SELECT a.*, c.title as course_title, 
                        COUNT(DISTINCT s.student_id) as submission_count,
                        ROUND(AVG(s.score), 1) as avg_score
                        FROM assignments a 
                        JOIN courses c ON a.course_id = c.course_id 
                        LEFT JOIN assignment_submissions s ON a.assignment_id = s.assignment_id 
                        WHERE a.teacher_id = ?";
            } else {
                // Simplified query without submissions data
                $query = "SELECT a.*, c.title as course_title, 
                        0 as submission_count,
                        0 as avg_score
                        FROM assignments a 
                        JOIN courses c ON a.course_id = c.course_id 
                        WHERE a.teacher_id = ?";
            }
            
            $queryParams = [$userId];
            if ($course_filter) {
                $query .= " AND a.course_id = ?";
                $queryParams[] = $course_filter;
            }
            
            $query .= " GROUP BY a.assignment_id ORDER BY a.due_date DESC";
            
            // Prepare and execute the query
            $stmt = $conn->prepare($query);
            if ($stmt) {
                $paramTypes = str_repeat("i", count($queryParams));
                $stmt->bind_param($paramTypes, ...$queryParams);
                $stmt->execute();
                $result = $stmt->get_result();
                
                // Fetch assignments
                if ($result) {
                    while ($row = $result->fetch_assoc()) {
                        $assignments[] = $row;
                    }
                } else {
                    error_log("Error getting result set in assignments.php");
                    $message = '<div class="alert alert-warning">Could not retrieve assignments data. Please try again later.</div>';
                }
            } else {
                error_log("Error preparing statement in assignments.php: " . $conn->error);
                $message = '<div class="alert alert-warning">Could not process assignments query. Please try again later.</div>';
            }
        } else {
            // Assignments table doesn't exist, show a message
            $message = '<div class="alert alert-info">The assignments feature is currently being set up. Check back soon!</div>';
        }
    } catch (Exception $e) {
        // Log the specific exception but show a user-friendly message
        error_log("Exception in assignments.php: " . $e->getMessage());
        $message = '<div class="alert alert-info">We\'re currently updating the assignments system. Please check back later.</div>';
    }
} else {
    // Connection is null, show error message
    $message = '<div class="alert alert-danger">Database connection error. Please try again later.</div>';
}

// The assignments array is already declared and populated above

// Get course details if filtering by course
$courseDetail = null;
if ($course_filter && $conn) {
    try {
        $stmt = $conn->prepare("SELECT title FROM courses WHERE course_id = ? AND teacher_id = ?");
        if ($stmt) {
            $stmt->bind_param("ii", $course_filter, $userId);
            $stmt->execute();
            $result = $stmt->get_result();
            if ($result && $result->num_rows > 0) {
                $courseDetail = $result->fetch_assoc();
            }
        } else {
            error_log("Error preparing statement for course detail in assignments.php: " . $conn->error);
        }
    } catch (Exception $e) {
        error_log("Error retrieving course details in assignments.php: " . $e->getMessage());
    }
}
?>
<!-- Page-specific CSS for Assignments -->
<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/toastify-js/src/toastify.min.css">
<!-- DataTables -->
<link rel="stylesheet" href="https://cdn.datatables.net/1.13.6/css/dataTables.bootstrap5.min.css">
<link rel="stylesheet" href="https://cdn.datatables.net/responsive/2.4.1/css/responsive.bootstrap5.min.css">

<style>
        



        :root {
            --primary: #4e73df;
            --secondary: #858796;
            --success: #1cc88a;
            --info: #36b9cc;
            --warning: #f6c23e;
            --danger: #e74a3b;
            --light: #f8f9fc;
            --dark: #5a5c69;
            --sidebar-width: 250px;
            --sidebar-bg: whitesmoke;
            --navbar-height: 60px;
            --border-radius: 8px;
            --sidebar-text: #343a40;
            --sidebar-hover: rgba(78, 115, 223, 0.1);
            --sidebar-active: var(--primary);
        }
        
        body {
            font-family: 'Poppins', sans-serif;
            background-color: #f5f7fb;
            color: #333;
            display: flex;
            min-height: 100vh;
            margin: 0;
        }
        
        /* Dashboard container */
        .dashboard-container {
            display: flex;
            width: 100%;
        }
        
        /* Sidebar styles */
        .sidebar {
            width: var(--sidebar-width);
            background-color: whitesmoke;
            height: 100vh;
            position: fixed;
            left: 0;
            top: 0;
            overflow-y: auto;
            transition: all 0.3s ease;
            z-index: 1000;
            box-shadow: 0 0 15px rgba(0, 0, 0, 0.05);
        }
        
        /* Sidebar header */
        .sidebar-header {
            padding: 20px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-bottom: 1px solid rgba(0, 0, 0, 0.05);
        }
        
        .sidebar-brand {
            color: var(--primary);
            text-decoration: none;
            font-weight: 700;
            font-size: 1.2rem;
            display: flex;
            align-items: center;
            transition: all 0.2s ease;
        }
        
        .sidebar-brand:hover {
            opacity: 0.9;
        }
        
        .sidebar-brand img {
            height: 30px;
            margin-right: 10px;
        }
        
        /* Sidebar navigation */
        .sidebar-nav {
            padding: 0;
            list-style: none;
            margin: 0;
        }
        
        .sidebar-heading {
            font-size: 0.75rem;
            font-weight: 600;
            color: #666;
            padding: 20px 20px 8px;
            margin-bottom: 0;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .nav-item {
            margin-bottom: 5px;
        }
        
        .nav-link {
            display: flex;
            align-items: center;
            padding: 12px 20px;
            color: #495057;
            text-decoration: none;
            border-radius: 5px;
            margin: 0 10px;
            transition: all 0.2s ease;
        }
        
        .nav-link i {
            margin-right: 10px;
            font-size: 1.1rem;
            width: 20px;
            text-align: center;
            color: #555;
        }
        
        .nav-link:hover {
            background-color: rgba(39, 107, 228, 0.1);
            color: var(--primary);
        }
        
        .nav-link:hover i {
            color: var(--primary);
        }
        
        .nav-link.active {
            background-color: var(--primary-light);
            color: var(--primary);
            font-weight: 500;
            box-shadow: 0 2px 5px rgba(39, 107, 228, 0.1);
        }
        
        .nav-link.active i {
            color: var(--primary);
        }
        
        .nav-badge {
            background-color: var(--primary);
            color: white;
            font-size: 0.7rem;
            padding: 2px 6px;
            border-radius: 10px;
            margin-left: 10px;
        }
        
        /* Content wrapper */
        .content-wrapper {
            margin-left: var(--sidebar-width);
            width: calc(100% - var(--sidebar-width));
            max-width: 100%;
            transition: all 0.3s ease;
            padding: 20px;
        }
        
        /* Dashboard navbar */
        .dashboard-navbar {
            background-color: #fff;
            box-shadow: 0 2px 15px rgba(0, 0, 0, 0.05);
            border-radius: 10px;
            padding: 15px 20px;
            margin-bottom: 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        /* Quiz card */
        .quiz-card {
            transition: all 0.3s ease;
            border: none;
            box-shadow: 0 2px 15px rgba(0, 0, 0, 0.05);
            border-radius: 10px;
            overflow: hidden;
            height: 100%;
            background-color: #fff;
        }
        
        .quiz-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 5px 20px rgba(0, 0, 0, 0.1);
        }
        
        .quiz-card .card-header {
            background-color: var(--primary-light);
            border-bottom: none;
            padding: 1rem;
        }
        
        .quiz-status {
            font-size: 12px;
            padding: 3px 10px;
            border-radius: 20px;
            display: inline-block;
        }
        
        .status-published {
            background-color: rgba(40, 167, 69, 0.1);
            color: var(--success);
        }
        
        .status-draft {
            background-color: rgba(108, 117, 125, 0.1);
            color: var(--secondary);
        }
        
        .quiz-meta {
            display: flex;
            justify-content: space-between;
            font-size: 13px;
            color: #6c757d;
            margin-bottom: 8px;
        }
        
        .quiz-stats {
            display: flex;
            justify-content: space-between;
            margin-top: 15px;
            border-top: 1px solid rgba(0, 0, 0, 0.05);
            padding-top: 15px;
        }
        
        .stat-item {
            text-align: center;
        }
        
        .stat-value {
            font-size: 1.2rem;
            font-weight: 600;
            color: var(--primary);
        }
        
        .stat-label {
            font-size: 0.8rem;
            color: #6c757d;
        }
        
        .table-action-btn {
            width: 32px;
            height: 32px;
            border-radius: 50%;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            transition: all 0.2s ease;
            margin-right: 5px;
            background-color: #f8f9fa;
            border: none;
        }
        
        .table-action-btn:hover {
            background-color: var(--primary-light);
        }
        
        .table-action-btn i {
            font-size: 14px;
        }
        
        .quiz-actions {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-top: 1rem;
        }

           /* Media queries for responsiveness */
        @media (max-width: 991px) {
            .sidebar {
                transform: translateX(-100%);
                z-index: 1050;
            }
            
            .content-wrapper {
                margin-left: 0;
                width: 100%;
                max-width: 100%;
            }
            
            .sidebar.show {
                transform: translateX(0);
            }
        }
        
        body {
            font-family: 'Poppins', sans-serif;
            background-color: #f5f7fb;
            color: #333;
            display: flex;
            min-height: 100vh;
            margin: 0;
        }
        
        /* Dashboard container */
        .dashboard-container {
            display: flex;
            width: 100%;
        }
        
        /* Sidebar styles */
        .sidebar {
            width: var(--sidebar-width);
            background-color: whitesmoke;
            height: 100vh;
            position: fixed;
            left: 0;
            top: 0;
            overflow-y: auto;
            transition: all 0.3s ease;
            z-index: 1000;
            box-shadow: 0 0 15px rgba(0, 0, 0, 0.05);
        }
        
        /* Sidebar header */
        .sidebar-header {
            padding: 15px 20px;
            text-align: center;
        }
        
        .sidebar-brand {
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 10px;
            color: var(--primary);
            text-decoration: none;
            font-size: 1.2rem;
            font-weight: 600;
        }
        
        .sidebar-brand img {
            width: 35px;
            margin-right: 10px;
        }
        
        /* Sidebar menu */
        .sidebar-menu {
            padding: 10px;
        }
        
        .sidebar-heading {
            font-size: 0.75rem;
            text-transform: uppercase;
            letter-spacing: 1px;
            color: #888;
            padding: 10px 20px;
            margin-top: 15px;
        }
        
        .nav-link {
            display: flex;
            align-items: center;
            padding: 12px 20px;
            color: #333;
            text-decoration: none;
            border-radius: 5px;
            margin-bottom: 5px;
            transition: all 0.3s ease;
        }
        
        .nav-link:hover, .nav-link.active {
            background-color: var(--primary-light);
            color: var(--primary);
        }
        
        .nav-link.active {
            font-weight: 500;
        }
        
        .nav-link i {
            margin-right: 10px;
            width: 20px;
            text-align: center;
            font-size: 1.1rem;
        }
        
        /* Content wrapper */
        .content-wrapper {
            flex: 1;
            margin-left: var(--sidebar-width);
            padding: 20px;
            transition: all 0.3s ease;
            width: calc(100% - var(--sidebar-width));
        }
        
        /* Navbar styles */
        .navbar {
            background-color: #fff;
            padding: 12px 20px;
            border-radius: 10px;
            box-shadow: 0 2px 15px rgba(0, 0, 0, 0.03);
            margin-bottom: 20px;
        }
        
        .navbar-brand {
            font-weight: 600;
            color: var(--primary);
        }
        
        .navbar-search {
            position: relative;
            max-width: 350px;
        }
        
        .navbar-search i {
            position: absolute;
            left: 10px;
            top: 11px;
            color: #999;
        }
        
        .navbar-search input {
            padding-left: 35px;
            border-radius: 20px;
            border: 1px solid #eee;
        }
        
        /* Dashboard content */
        .page-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }
        
        .page-title {
            margin-bottom: 0;
            font-weight: 600;
        }
        
        /* Cards */
        .card {
            border: none;
            border-radius: 10px;
            box-shadow: 0 2px 15px rgba(0, 0, 0, 0.03);
            margin-bottom: 20px;
            transition: transform 0.3s ease, box-shadow 0.3s ease;
        }
        
        .card:hover {
            transform: translateY(-5px);
            box-shadow: 0 5px 20px rgba(0, 0, 0, 0.05);
        }
        
        .stat-card {
            padding: 20px;
            display: flex;
            align-items: center;
        }
        
        .stat-card-icon {
            width: 60px;
            height: 60px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 50%;
            font-size: 1.5rem;
            margin-right: 20px;
        }
        
        .stat-card-info h5 {
            font-size: 0.9rem;
            margin-bottom: 5px;
            color: #888;
        }
        
        .stat-card-info h3 {
            font-size: 1.5rem;
            margin-bottom: 0;
            font-weight: 600;
        }
        
        .btn-panadite {
            background-color: var(--primary);
            color: white;
            border: none;
        }
        
        .btn-panadite:hover {
            background-color: #1c54c4;
            color: white;
        }
        
        /* Assignment-specific styles */
        .assignment-card {
            height: 100%;
        }
        
        .assignment-card .card-body {
            display: flex;
            flex-direction: column;
        }
        
        .assignment-stats {
            margin-top: auto;
            padding-top: 15px;
            border-top: 1px solid #eee;
        }
        
        .due-date {
            font-size: 0.85rem;
            color: #666;
        }
        
        .overdue {
            color: var(--danger);
            font-weight: 500;
        }
        
        .badge-pill {
            border-radius: 20px;
            padding: 5px 15px;
        }
        
        /* View toggle buttons */
        .view-toggle .btn {
            border-radius: 20px;
            padding: 8px 15px;
        }
        
        /* Filter styles */
        .filter-section {
            background: white;
            padding: 20px;
            border-radius: 10px;
            box-shadow: 0 2px 15px rgba(0, 0, 0, 0.03);
            margin-bottom: 20px;
        }
        
        /* Media queries */
        @media (max-width: 992px) {
            .sidebar.collapsed {
                width: 70px;
            }
            
            .sidebar.collapsed .sidebar-brand span,
            .sidebar.collapsed .nav-link span,
            .sidebar.collapsed .sidebar-heading {
                display: none;
            }
            
            .content-wrapper.expanded {
                margin-left: 70px;
                width: calc(100% - 70px);
            }
        }

           /* Media queries for responsiveness */
           @media (max-width: 991px) {
            .sidebar {
                transform: translateX(-100%);
                z-index: 1050;
            }
            
            .content-wrapper {
                margin-left: 0;
                width: 100%;
                max-width: 100%;
            }
            
            .sidebar.show {
                transform: translateX(0);
            }
        }


       
    </style>
</head>
<body>
    <!-- Main Assignment Content Container -->
                    <!-- Profile -->
               
            <!-- Page Content -->
            <div class="container-fluid">
                <!-- Page Header -->
                <div class="page-header">
           
                    <a href="create-assignment.php" class="btn btn-panadite"><i class="fas fa-plus me-2"></i>Create Assignment</a>
                </div>
                
                <!-- Alert Message -->
                <?php if(!empty($message)): ?>
                    <?= $message ?>
                <?php endif; ?>
                
                <!-- Statistics Cards -->
                <div class="row mb-4">
                    <div class="col-md-3">
                        <div class="card stat-card">
                            <div class="stat-card-icon bg-primary-light text-primary">
                                <i class="fas fa-tasks"></i>
                            </div>
                            <div class="stat-card-info">
                                <h5 >Total Assignments</h5>
                                <h3 style='text-align: center'><?= count($assignments) ?></h3>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card stat-card">
                            <div class="stat-card-icon bg-success-light text-success">
                                <i class="fas fa-check-circle"></i>
                            </div>
                            <div class="stat-card-info">
                                <h5>Submitted</h5>
                                <h3 style='text-align: center'><?php
                                    $submitted = 0;
                                    foreach ($assignments as $assignment) {
                                        $submitted += $assignment['submission_count'];
                                    }
                                    echo $submitted;
                                ?></h3>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card stat-card">
                            <div class="stat-card-icon bg-warning-light text-warning">
                                <i class="fas fa-clock"></i>
                            </div>
                            <div class="stat-card-info">
                                <h5 >Pending</h5>
                                <?php
                                    $pending = 0;
                                    foreach ($assignments as $assignment) {
                                        // Calculate pending based on due date not passed
                                        if(strtotime($assignment['due_date']) > time()) {
                                            $pending++;
                                        }
                                    }
                                ?>
                                <h3 style='text-align: center'><?= $pending ?></h3>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card stat-card">
                            <div class="stat-card-icon bg-info-light text-info">
                                <i class="fas fa-chart-line"></i>
                            </div>
                            <div class="stat-card-info">
                                <h5>Avg. Score</h5>
                                <h3 style='text-align: center'><?php
                                    $total_score = 0;
                                    $count = 0;
                                    foreach ($assignments as $assignment) {
                                        if ($assignment['avg_score'] > 0) {
                                            $total_score += $assignment['avg_score'];
                                            $count++;
                                        }
                                    }
                                    echo $count > 0 ? round($total_score / $count, 1) . '%' : 'N/A';
                                ?></h3>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Filters and View Toggle -->
                <div class="filter-section mb-4">
                    <div class="row align-items-center">
                        <div class="col-md-4">
                            <form id="courseFilterForm" method="get">
                                <div class="form-group">
                                    <label for="course_id" class="form-label">Filter by Course:</label>
                                    <select class="form-select" name="course_id" id="course_id" onchange="this.form.submit()">
                                        <option value="">All Courses</option>
                                        <?php foreach($courses as $course): ?>
                                            <option value="<?= $course['course_id'] ?>" <?= $course_filter == $course['course_id'] ? 'selected' : '' ?>>
                                                <?= htmlspecialchars($course['title']) ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                            </form>
                        </div>
                        <div class="col-md-4">
                            <div class="form-group">
                                <label for="status_filter" class="form-label">Status:</label>
                                <select class="form-select" id="status_filter">
                                    <option value="all">All</option>
                                    <option value="active">Active</option>
                                    <option value="past_due">Past Due</option>
                                    <option value="graded">Graded</option>
                                </select>
                            </div>
                        </div>
                        <div class="col-md-4 text-end">
                            <label class="form-label d-block">View:</label>
                            <div class="view-toggle btn-group" role="group">
                                <button type="button" class="btn btn-sm btn-primary active" id="gridViewBtn">
                                    <i class="fas fa-th"></i> Grid
                                </button>
                                <button type="button" class="btn btn-sm btn-outline-primary" id="listViewBtn">
                                    <i class="fas fa-list"></i> List
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Assignments Grid View -->
                <div class="row" id="assignmentsGrid">
                    <?php if(empty($assignments)): ?>
                        <div class="col-12">
                            <div class="alert alert-info">
                                <i class="fas fa-info-circle me-2"></i>No assignments found. <?php echo $course_filter ? 'Try selecting a different course or ' : '' ?><a href="create-assignment.php" class="alert-link">create a new assignment</a>.
                            </div>
                        </div>
                    <?php else: ?>
                        <?php foreach($assignments as $assignment): ?>
                            <?php 
                                // Parse JSON data safely
                                $submission_types = json_decode($assignment['submission_type'] ?? '[]', true) ?: [];
                                $allowed_formats = json_decode($assignment['allowed_formats'] ?? '[]', true) ?: [];
                                $is_past_due = strtotime($assignment['due_date']) < time();
                                $is_published = $assignment['is_published'] ?? 1;
                            ?>
                            <div class="col-md-4 mb-4 assignment-item" data-status="<?= $is_past_due ? 'past_due' : 'active' ?>">
                                <div class="card assignment-card h-100">
                                    <div class="card-body d-flex flex-column">
                                        <div class="d-flex justify-content-between align-items-start mb-2">
                                            <h5 class="card-title"><?= htmlspecialchars($assignment['title']) ?></h5>
                                            <div class="d-flex flex-column align-items-end gap-1">
                                                <span class="badge bg-<?= $is_past_due ? 'danger' : 'primary' ?> badge-pill">
                                                    <?= $is_past_due ? 'Past Due' : 'Active' ?>
                                                </span>
                                                <?php if (!$is_published): ?>
                                                    <span class="badge bg-warning text-dark">Draft</span>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                        
                                        <p class="card-text mb-2 text-muted small">
                                            <i class="fas fa-book me-1"></i> <?= htmlspecialchars($assignment['course_title']) ?>
                                        </p>
                                        
                                        <p class="card-text mb-3">
                                            <?= mb_substr(htmlspecialchars($assignment['description']), 0, 100) . (mb_strlen($assignment['description']) > 100 ? '...' : '') ?>
                                        </p>
                                        
                                        <!-- Enhanced Assignment Info -->
                                        <div class="assignment-details mb-3">
                                            <div class="row g-2 mb-2">
                                                <div class="col-6">
                                                    <small class="text-muted d-block"><i class="fas fa-star me-1"></i>Points:</small>
                                                    <span class="fw-bold"><?= $assignment['points'] ?? 0 ?></span>
                                                </div>
                                                <div class="col-6">
                                                    <small class="text-muted d-block"><i class="fas fa-certificate me-1"></i>Final Grade:</small>
                                                    <span class="badge bg-<?= ($assignment['contributes_to_final_grade'] ?? 1) ? 'success' : 'secondary' ?> text-white">
                                                        <?= ($assignment['contributes_to_final_grade'] ?? 1) ? 'Yes' : 'No' ?>
                                                    </span>
                                                </div>
                                            </div>
                                            
                                            <?php if (!empty($submission_types)): ?>
                                                <div class="mb-2">
                                                    <small class="text-muted d-block"><i class="fas fa-upload me-1"></i>Submission Types:</small>
                                                    <div class="d-flex flex-wrap gap-1">
                                                        <?php foreach(array_slice($submission_types, 0, 3) as $type): ?>
                                                            <span class="badge bg-light text-dark border"><?= ucfirst(str_replace('_', ' ', $type)) ?></span>
                                                        <?php endforeach; ?>
                                                        <?php if (count($submission_types) > 3): ?>
                                                            <span class="badge bg-light text-muted border">+<?= count($submission_types) - 3 ?> more</span>
                                                        <?php endif; ?>
                                                    </div>
                                                </div>
                                            <?php endif; ?>
                                            
                                            <?php if ($assignment['group_assignment'] ?? 0): ?>
                                                <div class="mb-2">
                                                    <small class="text-muted d-block"><i class="fas fa-users me-1"></i>Group Assignment:</small>
                                                    <span class="text-info">Max <?= $assignment['max_group_size'] ?? 1 ?> members</span>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                        
                                        <p class="due-date mb-3 <?= $is_past_due ? 'overdue' : '' ?>">
                                            <i class="far fa-calendar-alt me-1"></i> Due: <?= date('M d, Y g:i A', strtotime($assignment['due_date'])) ?>
                                        </p>
                                        
                                        <div class="assignment-stats d-flex justify-content-between mt-auto">
                                            <div>
                                                <i class="fas fa-users me-1"></i> <?= $assignment['submission_count'] ?> submissions
                                            </div>
                                            <div>
                                                <i class="fas fa-chart-line me-1"></i> Avg: <?= $assignment['avg_score'] > 0 ? $assignment['avg_score'] . '%' : 'N/A' ?>
                                            </div>
                                        </div>
                                        
                                        <div class="mt-3 d-flex gap-2">
                                            <a href="edit-assignment.php?id=<?= $assignment['assignment_id'] ?>" class="btn btn-sm btn-outline-primary">
                                                <i class="fas fa-edit"></i> Edit
                                            </a>
                                            <a href="view-submissions.php?id=<?= $assignment['assignment_id'] ?>" class="btn btn-sm btn-outline-info">
                                                <i class="fas fa-eye"></i> Submissions
                                            </a>
                                            <button class="btn btn-sm btn-outline-danger" 
                                                    onclick="confirmDelete('<?= $assignment['assignment_id'] ?>', '<?= addslashes(htmlspecialchars($assignment['title'])) ?>', <?= $assignment['submission_count'] > 0 ? 'true' : 'false' ?>)">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
                
                <!-- Assignments List View -->
                <div id="assignmentsList" class="card" style="display: none;">
                    <div class="card-body p-0">
                        <div class="table-responsive">
                            <table class="table table-hover align-middle" id="assignmentsTable">
                                <thead>
                                    <tr>
                                        <th>Assignment</th>
                                        <th>Course</th>
                                        <th>Details</th>
                                        <th>Due Date</th>
                                        <th>Status</th>
                                        <th>Submissions</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if(empty($assignments)): ?>
                                        <tr>
                                            <td colspan="7" class="text-center">No assignments found</td>
                                        </tr>
                                    <?php else: ?>
                                        <?php foreach($assignments as $assignment): ?>
                                            <?php 
                                                $submission_types = json_decode($assignment['submission_type'] ?? '[]', true) ?: [];
                                                $allowed_formats = json_decode($assignment['allowed_formats'] ?? '[]', true) ?: [];
                                                $is_past_due = strtotime($assignment['due_date']) < time();
                                                $is_published = $assignment['is_published'] ?? 1;
                                            ?>
                                            <tr class="assignment-item" data-status="<?= $is_past_due ? 'past_due' : 'active' ?>">
                                                <td>
                                                    <div class="d-flex flex-column">
                                                        <span class="fw-medium"><?= htmlspecialchars($assignment['title']) ?></span>
                                                        <small class="text-muted"><?= mb_substr(htmlspecialchars($assignment['description']), 0, 60) . (mb_strlen($assignment['description']) > 60 ? '...' : '') ?></small>
                                                    </div>
                                                </td>
                                                <td><?= htmlspecialchars($assignment['course_title']) ?></td>
                                                <td>
                                                    <div class="d-flex flex-column gap-1">
                                                        <div class="d-flex align-items-center gap-2">
                                                            <span class="badge bg-primary"><?= $assignment['points'] ?? 0 ?> pts</span>
                                                            <span class="badge bg-<?= ($assignment['contributes_to_final_grade'] ?? 1) ? 'success' : 'secondary' ?>">
                                                                <?= ($assignment['contributes_to_final_grade'] ?? 1) ? 'Final Grade' : 'Practice' ?>
                                                            </span>
                                                            <?php if ($assignment['group_assignment'] ?? 0): ?>
                                                                <span class="badge bg-info">Group</span>
                                                            <?php endif; ?>
                                                        </div>
                                                        <?php if (!empty($submission_types)): ?>
                                                            <div class="d-flex flex-wrap gap-1">
                                                                <?php foreach(array_slice($submission_types, 0, 2) as $type): ?>
                                                                    <span class="badge bg-light text-dark border" style="font-size: 0.7em;">
                                                                        <?= ucfirst(str_replace('_', ' ', $type)) ?>
                                                                    </span>
                                                                <?php endforeach; ?>
                                                                <?php if (count($submission_types) > 2): ?>
                                                                    <span class="badge bg-light text-muted border" style="font-size: 0.7em;">+<?= count($submission_types) - 2 ?></span>
                                                                <?php endif; ?>
                                                            </div>
                                                        <?php endif; ?>
                                                    </div>
                                                </td>
                                                <td class="<?= $is_past_due ? 'overdue' : '' ?>">
                                                    <div class="d-flex flex-column">
                                                        <span><?= date('M d, Y', strtotime($assignment['due_date'])) ?></span>
                                                        <small class="text-muted"><?= date('g:i A', strtotime($assignment['due_date'])) ?></small>
                                                    </div>
                                                </td>
                                                <td>
                                                    <div class="d-flex flex-column gap-1">
                                                        <span class="badge bg-<?= $is_past_due ? 'danger' : 'primary' ?>">
                                                            <?= $is_past_due ? 'Past Due' : 'Active' ?>
                                                        </span>
                                                        <?php if (!$is_published): ?>
                                                            <span class="badge bg-warning text-dark">Draft</span>
                                                        <?php endif; ?>
                                                    </div>
                                                </td>
                                                <td>
                                                    <div class="d-flex flex-column text-center">
                                                        <span class="fw-bold"><?= $assignment['submission_count'] ?></span>
                                                        <small class="text-muted">Avg: <?= $assignment['avg_score'] > 0 ? $assignment['avg_score'] . '%' : 'N/A' ?></small>
                                                    </div>
                                                </td>
                                                <td>
                                                    <div class="d-flex gap-1">
                                                        <a href="edit-assignment.php?id=<?= $assignment['assignment_id'] ?>" class="btn btn-sm btn-outline-primary" title="Edit Assignment">
                                                            <i class="fas fa-edit"></i>
                                                        </a>
                                                        <a href="view-submissions.php?id=<?= $assignment['assignment_id'] ?>" class="btn btn-sm btn-outline-info" title="View Submissions">
                                                            <i class="fas fa-eye"></i>
                                                        </a>
                                                        <button class="btn btn-sm btn-outline-danger" 
                                                                onclick="confirmDelete('<?= $assignment['assignment_id'] ?>', '<?= addslashes(htmlspecialchars($assignment['title'])) ?>', <?= $assignment['submission_count'] > 0 ? 'true' : 'false' ?>)"
                                                                title="Delete Assignment">
                                                            <i class="fas fa-trash"></i>
                                                        </button>
                                                    </div>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- JavaScript Libraries -->
    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/toastify-js"></script>
    <!-- DataTables temporarily disabled to fix warnings
    <script src="https://cdn.datatables.net/1.13.6/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.6/js/dataTables.bootstrap5.min.js"></script>
    -->
    
    <script>
        $(document).ready(function() {
            // DataTables initialization removed to fix the warnings
            // Basic table functionality will still work without DataTables
            
            // Toggle sidebar
            $('#sidebarToggle').on('click', function() {
                $('.sidebar').toggleClass('collapsed');
                $('.content-wrapper').toggleClass('expanded');
                
                if ($('.sidebar').hasClass('collapsed')) {
                    $('.sidebar').css('width', '70px');
                    $('.content-wrapper').css('margin-left', '70px');
                    $('.content-wrapper').css('width', 'calc(100% - 70px)');
                    $('.sidebar-brand span').hide();
                    $('.nav-link span').hide();
                    $('.sidebar-heading').hide();
                } else {
                    $('.sidebar').css('width', 'var(--sidebar-width)');
                    $('.content-wrapper').css('margin-left', 'var(--sidebar-width)');
                    $('.content-wrapper').css('width', 'calc(100% - var(--sidebar-width))');
                    $('.sidebar-brand span').show();
                    $('.nav-link span').show();
                    $('.sidebar-heading').show();
                }
            });
            
            // View toggle
            $('#gridViewBtn').on('click', function() {
                $(this).addClass('btn-primary').removeClass('btn-outline-primary');
                $('#listViewBtn').addClass('btn-outline-primary').removeClass('btn-primary');
                $('#assignmentsGrid').show();
                $('#assignmentsList').hide();
            });
            
            $('#listViewBtn').on('click', function() {
                $(this).addClass('btn-primary').removeClass('btn-outline-primary');
                $('#gridViewBtn').addClass('btn-outline-primary').removeClass('btn-primary');
                $('#assignmentsGrid').hide();
                $('#assignmentsList').show();
            });
            
            // Status filter
            $('#status_filter').on('change', function() {
                const status = $(this).val();
                
                if (status === 'all') {
                    $('.assignment-item').show();
                } else {
                    $('.assignment-item').hide();
                    $('.assignment-item[data-status="' + status + '"]').show();
                }
            });
        });
        
        // Delete confirmation - Simplified for testing purposes
        function confirmDelete(assignmentId, title, hasSubmissions) {
            // Always use enhanced warning and force delete for testing reliability
            const confirmMessage = `⚠️ DELETE ASSIGNMENT: "${title}"\n\n` +
                `This will permanently delete:\n` +
                `• The assignment\n` +
                `• ALL student submissions (if any)\n` +
                `• ALL grades and marks (if any)\n` +
                `• ALL related files (if any)\n\n` +
                `This action CANNOT be undone!\n\n` +
                `Are you sure you want to continue?`;
            
            if (confirm(confirmMessage)) {
                // Always use force delete to handle any edge cases
                window.location.href = `assignments.php?delete=${assignmentId}&force=1`;
            }
        }
    </script>

<?php
// Include the centralized teacher layout footer
require_once __DIR__ . '/components/teacher_layout_end.php';
?>
