<?php
// Start session if not already started
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// Include database connection
require_once '../includes/config.php';
require_once '../includes/functions.php';
require_once '../auth/functions.php';

// Check if user is logged in with teacher role
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'teacher') {
    header('Content-Type: application/json');
    echo json_encode([
        'success' => false,
        'message' => 'User not logged in or unauthorized',
    ]);
    exit;
}

// Get user ID
$userId = $_SESSION['user_id'];

// Default response structure
$response = [
    'success' => false,
    'message' => 'Invalid request',
    'data' => null
];

// Handle different AJAX actions
if (isset($_POST['action'])) {
    $action = $_POST['action'];
    
    switch ($action) {
        case 'get_month_data':
            // Get month and year from request
            $month = isset($_POST['month']) ? intval($_POST['month']) : date('n');
            $year = isset($_POST['year']) ? intval($_POST['year']) : date('Y');
            
            // Calculate first day of month and number of days
            $firstDayOfMonth = date('w', strtotime("$year-$month-01"));
            $daysInMonth = date('t', strtotime("$year-$month-01"));
            $monthName = date('F Y', strtotime("$year-$month-01"));
            
            // Get events for the month
            $events = getEventsForMonth($userId, $month, $year);
            
            // Generate calendar data
            $calendarData = [
                'month' => $month,
                'year' => $year,
                'monthName' => $monthName,
                'firstDayOfMonth' => $firstDayOfMonth,
                'daysInMonth' => $daysInMonth,
                'events' => $events,
                'today' => date('j'),
                'currentMonth' => date('n'),
                'currentYear' => date('Y')
            ];
            
            $response = [
                'success' => true,
                'message' => 'Calendar data retrieved successfully',
                'data' => $calendarData
            ];
            break;
            
        case 'filter_events':
            // Get filter type from request
            $filterType = isset($_POST['filter_type']) ? $_POST['filter_type'] : 'all';
            
            // Get events based on filter
            $events = getFilteredEvents($userId, $filterType);
            
            $response = [
                'success' => true,
                'message' => 'Events filtered successfully',
                'data' => [
                    'events' => $events,
                    'filter_type' => $filterType
                ]
            ];
            break;
            
        case 'get_event_details':
            // Get event ID from request
            if (isset($_POST['event_id'])) {
                $eventId = intval($_POST['event_id']);
                
                // Get event details
                $eventDetails = getEventDetails($userId, $eventId);
                
                if ($eventDetails) {
                    $response = [
                        'success' => true,
                        'message' => 'Event details retrieved successfully',
                        'data' => $eventDetails
                    ];
                } else {
                    $response = [
                        'success' => false,
                        'message' => 'Event not found',
                        'data' => null
                    ];
                }
            }
            break;
            
        default:
            $response = [
                'success' => false,
                'message' => 'Unknown action',
                'data' => null
            ];
    }
} 

// Helper function to get events for a specific month
function getEventsForMonth($userId, $month, $year) {
    global $conn;
    
    // Placeholder for actual database query
    // This would be replaced with an actual query to your events table
    $events = [];
    
    // Sample events for demonstration - teacher-specific events
    if ($month == 7 && $year == 2025) {
        $events = [
            [
                'id' => 1,
                'title' => 'Lecture: Calculus 101',
                'type' => 'class',
                'course_title' => 'Calculus 101',
                'event_date' => '2025-07-08',
                'start_time' => '10:00:00',
                'end_time' => '11:30:00',
                'description' => 'Lecture on differential equations'
            ],
            [
                'id' => 2,
                'title' => 'Office Hours',
                'type' => 'office_hours',
                'course_title' => 'Faculty Services',
                'event_date' => '2025-07-15',
                'start_time' => '13:00:00',
                'end_time' => '15:00:00',
                'description' => 'Open office hours for student consultations'
            ],
            [
                'id' => 3,
                'title' => 'Grade Assignments',
                'type' => 'grading',
                'course_title' => 'English Literature',
                'event_date' => '2025-07-22',
                'start_time' => '14:00:00',
                'end_time' => '16:00:00',
                'description' => 'Grade student essay submissions'
            ]
        ];
    }
    
    return $events;
}

// Helper function to get filtered events
function getFilteredEvents($userId, $filterType) {
    global $conn;
    
    // Placeholder for actual database query
    // This would be replaced with an actual query to your events table
    $events = [];
    
    // Sample events for demonstration - teacher-specific events
    $allEvents = [
        [
            'id' => 1,
            'title' => 'Lecture: Calculus 101',
            'type' => 'class',
            'course_title' => 'Calculus 101',
            'event_date' => '2025-07-08',
            'start_time' => '10:00:00',
            'end_time' => '11:30:00',
            'description' => 'Lecture on differential equations'
        ],
        [
            'id' => 2,
            'title' => 'Office Hours',
            'type' => 'office_hours',
            'course_title' => 'Faculty Services',
            'event_date' => '2025-07-15',
            'start_time' => '13:00:00',
            'end_time' => '15:00:00',
            'description' => 'Open office hours for student consultations'
        ],
        [
            'id' => 3,
            'title' => 'Grade Assignments',
            'type' => 'grading',
            'course_title' => 'English Literature',
            'event_date' => '2025-07-22',
            'start_time' => '14:00:00',
            'end_time' => '16:00:00',
            'description' => 'Grade student essay submissions'
        ]
    ];
    
    if ($filterType == 'all') {
        $events = $allEvents;
    } else {
        foreach ($allEvents as $event) {
            if ($event['type'] == $filterType) {
                $events[] = $event;
            }
        }
    }
    
    return $events;
}

// Helper function to get event details
function getEventDetails($userId, $eventId) {
    global $conn;
    
    // Placeholder for actual database query
    // This would be replaced with an actual query to your events table
    $events = [
        1 => [
            'id' => 1,
            'title' => 'Lecture: Calculus 101',
            'type' => 'class',
            'course_title' => 'Calculus 101',
            'event_date' => '2025-07-08',
            'start_time' => '10:00:00',
            'end_time' => '11:30:00',
            'description' => 'Lecture on differential equations',
            'location' => 'Room 301',
            'materials' => 'Textbook chapters 5-6',
            'students_enrolled' => '45'
        ],
        2 => [
            'id' => 2,
            'title' => 'Office Hours',
            'type' => 'office_hours',
            'course_title' => 'Faculty Services',
            'event_date' => '2025-07-15',
            'start_time' => '13:00:00',
            'end_time' => '15:00:00',
            'description' => 'Open office hours for student consultations',
            'location' => 'Faculty Office 210',
            'notes' => 'Drop-in basis, first come first served',
            'availability' => '2 hours'
        ],
        3 => [
            'id' => 3,
            'title' => 'Grade Assignments',
            'type' => 'grading',
            'course_title' => 'English Literature',
            'event_date' => '2025-07-22',
            'start_time' => '14:00:00',
            'end_time' => '16:00:00',
            'description' => 'Grade student essay submissions',
            'assignment_name' => 'Macbeth Analysis',
            'submissions_count' => '28',
            'deadline' => 'Grades due by July 24'
        ]
    ];
    
    return isset($events[$eventId]) ? $events[$eventId] : null;
}

// Return response as JSON
header('Content-Type: application/json');
echo json_encode($response);
exit;
?>
