<?php
// Start session if not already started
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// Include database connection
require_once '../includes/config.php';
require_once '../includes/functions.php';

// Check if user is logged in
if (!isset($_SESSION['user_id']) || $_SESSION['role'] != 'teacher') {
    header('Location: ../login.php');
    exit;
}

// Get current month and year
$month = isset($_GET['month']) ? intval($_GET['month']) : date('n');
$year = isset($_GET['year']) ? intval($_GET['year']) : date('Y');

// Get first day of month and number of days
$firstDayOfMonth = date('w', strtotime("$year-$month-01"));
$daysInMonth = date('t', strtotime("$year-$month-01"));
$monthName = date('F Y', strtotime("$year-$month-01"));

// Get events for the month - sample data for teacher
$userId = $_SESSION['user_id'];
// This would typically be a database query
// For now, we'll use sample data
$events = [
    [
        'id' => 1,
        'title' => 'Programming Lecture',
        'type' => 'class',
        'course_title' => 'Advanced JavaScript',
        'event_date' => '2025-07-08',
        'start_time' => '09:00:00',
        'end_time' => '11:00:00',
        'description' => 'Covering modern ES6+ features'
    ],
    [
        'id' => 2,
        'title' => 'Assignment Due',
        'type' => 'assignment',
        'course_title' => 'Web Development',
        'event_date' => '2025-07-15',
        'start_time' => '23:59:00',
        'end_time' => '23:59:00',
        'description' => 'Final project submissions due'
    ],
    [
        'id' => 3,
        'title' => 'Mid-Term Exam',
        'type' => 'exam',
        'course_title' => 'Data Structures',
        'event_date' => '2025-07-22',
        'start_time' => '13:00:00',
        'end_time' => '15:00:00',
        'description' => 'Covers chapters 1-5 of the textbook'
    ]
];
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Teacher Calendar | Panadite Academy</title>
    
    <!-- Favicon -->
    <link rel="icon" href="../assets/images/favicon.png" type="image/png">
    
    <!-- External CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://unpkg.com/aos@2.3.1/dist/aos.css" rel="stylesheet">
    
    <!-- Internal CSS -->
    <link rel="stylesheet" href="assets/css/teacher-style.css">
    <link rel="stylesheet" href="assets/css/dashboard-fixes.css">
    <link rel="stylesheet" href="../student/assets/css/calendar-fixes.css">
    <link rel="stylesheet" href="../student/assets/css/calendar-animations.css">
    <link rel="stylesheet" href="../student/assets/css/calendar-loading.css">
    
    <style>
        :root {
            --calendar-primary: #1cc88a;
            --calendar-secondary: #36b9cc;
            --calendar-accent: #f6c23e;
            --calendar-light: #f8f9fc;
            --calendar-dark: #5a5c69;
        }
        
        /* Teacher specific overrides */
        .calendar-header {
            background: linear-gradient(135deg, #1cc88a 0%, #169a6a 100%);
        }
        
        .calendar-event.class {
            background-color: #1cc88a;
        }
        
        .calendar-event.assignment {
            background-color: #36b9cc;
        }
        
        .calendar-event.exam {
            background-color: #f6c23e;
        }
        
        .filter-chip.active {
            background-color: #1cc88a;
            color: white;
        }
    </style>
</head>
<body>
    <div class="app-container">
        <?php include 'components/sidebar.php'; ?>
        
        <main>
            <?php include 'components/navbar.php'; ?>
            
            <!-- Spacer element to create distance between navbar and content -->
            <div style="height: 60px; width: 100%; clear: both;"></div>
            
            <div class="container-fluid py-4">
                <!-- Animated background elements -->
                <div class="animated-bg-element"></div>
                <div class="animated-bg-element"></div>
                <div class="animated-bg-element"></div>
                
                <!-- Calendar Header -->
                <div class="calendar-header">
                    <div style="display: flex; align-items: center; position: relative; z-index: 1;">
                        <div>
                            <i class="fas fa-calendar-alt fa-2x mb-3"></i>
                            <h1>Teacher Calendar</h1>
                            <p>Organize your classes, assignments, and exams all in one place</p>
                        </div>
                    </div>
                </div>
                
                <!-- Calendar Navigation placeholder -->
                <div style="display: none;"><!-- Legacy navigation removed --></div>
                
                <!-- Calendar Controls -->
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <!-- Month navigation with animations -->
                    <div class="month-nav">
                        <button onclick="navigateMonth('prev')"><i class="fas fa-chevron-left"></i></button>
                        <div class="month-year">July 2025</div>
                        <button onclick="navigateMonth('next')"><i class="fas fa-chevron-right"></i></button>
                    </div>
                    
                    <!-- Calendar view options -->
                    <div class="view-switcher">
                        <button class="active">Month</button>
                        <button>Week</button>
                        <button>Day</button>
                        <button>List</button>
                    </div>
                    
                    <!-- Date picker dropdown -->
                    <div class="date-picker-dropdown">
                        <div class="dropdown-toggle">
                            <i class="fas fa-search"></i>
                            <span>Jump to date</span>
                        </div>
                        <div class="dropdown-menu">
                            <input type="date" class="form-control mb-2">
                            <button class="btn-primary w-100">Go</button>
                        </div>
                    </div>
                </div>
                
                <!-- Calendar filters -->
                <div class="calendar-filters mb-4">
                    <div class="filter-chip active">
                        <span class="checkmark"></span>
                        <span>All Events</span>
                    </div>
                    <div class="filter-chip">
                        <span class="checkmark"></span>
                        <span>Classes</span>
                    </div>
                    <div class="filter-chip">
                        <span class="checkmark"></span>
                        <span>Assignments</span>
                    </div>
                    <div class="filter-chip">
                        <span class="checkmark"></span>
                        <span>Exams</span>
                    </div>
                    <div class="filter-chip">
                        <span class="checkmark"></span>
                        <span>Personal</span>
                    </div>
                </div>
                
                <div id="calendar-container">
                    <!-- Calendar Grid -->
                    <div id="calendar-main">
                        <table class="calendar-table">
                            <thead>
                                <tr>
                                    <th>Sun</th>
                                    <th>Mon</th>
                                    <th>Tue</th>
                                    <th>Wed</th>
                                    <th>Thu</th>
                                    <th>Fri</th>
                                    <th>Sat</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php
                                    $dayCount = 1;
                                    $totalCells = $firstDayOfMonth + $daysInMonth;
                                    $totalRows = ceil($totalCells / 7);
                                    
                                    for ($i = 0; $i < $totalRows; $i++) {
                                        echo "<tr>";
                                        for ($j = 0; $j < 7; $j++) {
                                            $cellNumber = $i * 7 + $j;
                                            
                                            if ($cellNumber < $firstDayOfMonth || $dayCount > $daysInMonth) {
                                                echo "<td class='empty'></td>";
                                            } else {
                                                $today = date('j') == $dayCount && date('n') == $month && date('Y') == $year ? ' today' : '';
                                                echo "<td class='day-cell{$today}'>";
                                                echo "<div class='day-number'>{$dayCount}</div>";
                                                echo "<div class='day-events'>";
                                                
                                                // Display events for this day
                                                foreach ($events as $event) {
                                                    $eventDate = date('j', strtotime($event['event_date']));
                                                    if ($eventDate == $dayCount) {
                                                        echo "<div class='calendar-event {$event['type']}' onclick=\"showEventModal('{$event['title']}', '".date('h:i A', strtotime($event['start_time']))." - ".date('h:i A', strtotime($event['end_time']))."', '{$event['course_title']}', '{$event['description']}')\">";
                                                        echo "<span class='event-title'>{$event['title']}</span>";
                                                        echo "</div>";
                                                    }
                                                }
                                                
                                                echo "</div>";
                                                echo "</td>";
                                                $dayCount++;
                                            }
                                        }
                                        echo "</tr>";
                                    }
                                ?>
                            </tbody>
                        </table>
                    </div>
                    
                    <!-- Side Panel -->
                    <div id="calendar-sidebar">
                        <!-- Upcoming Events -->
                        <div class="calendar-sidebar-section upcoming-events">
                            <h3>
                                <i class="fas fa-calendar-check"></i>
                                Upcoming Classes & Events
                            </h3>
                            
                            <div class="upcoming-events-content">
                                <?php if (count($events) > 0): ?>
                                    <?php foreach(array_slice($events, 0, 5) as $event): ?>
                                    <div class="event-item" onclick="showEventModal('<?php echo htmlspecialchars($event['title']); ?>', '<?php echo date('h:i A', strtotime($event['start_time'])); ?> - <?php echo date('h:i A', strtotime($event['end_time'])); ?>', '<?php echo htmlspecialchars($event['course_title']); ?>', '<?php echo htmlspecialchars($event['description']); ?>')">
                                        <div class="event-title">
                                            <?php if($event['type'] == 'class'): ?>
                                                <span class="event-badge recurring">Class</span>
                                            <?php elseif($event['type'] == 'assignment'): ?>
                                                <span class="event-badge important">Assignment</span>
                                            <?php elseif($event['type'] == 'exam'): ?>
                                                <span class="event-badge reminder">Exam</span>
                                            <?php endif; ?>
                                            <?php echo htmlspecialchars($event['title']); ?>
                                        </div>
                                        <div class="event-course">
                                            <i class="fas fa-bookmark"></i>
                                            <?php echo htmlspecialchars($event['course_title']); ?>
                                        </div>
                                        <div class="event-date">
                                            <i class="fas fa-calendar"></i>
                                            <?php echo date('M j, Y', strtotime($event['event_date'])); ?>
                                        </div>
                                    </div>
                                    <?php endforeach; ?>
                                <?php else: ?>
                                    <div class="no-events">
                                        <p>No upcoming events scheduled for the upcoming days</p>
                                        <button class="btn-primary">Add Event</button>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <!-- Quick Actions -->
                        <div class="calendar-sidebar-section quick-actions">
                            <h3>
                                <i class="fas fa-bolt"></i>
                                Quick Actions
                            </h3>
                            <div class="quick-action-buttons">
                                <button class="btn-action">
                                    <i class="fas fa-plus"></i>
                                    Create Event
                                </button>
                                <button class="btn-action">
                                    <i class="fas fa-clipboard-list"></i>
                                    Manage Assignments
                                </button>
                                <button class="btn-action">
                                    <i class="fas fa-file-export"></i>
                                    Export Calendar
                                </button>
                                <button class="btn-action">
                                    <i class="fas fa-sync"></i>
                                    Refresh
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </main>
    </div>
    
    <!-- Event Detail Modal -->
    <div class="calendar-modal" id="eventModal">
        <div class="calendar-modal-content">
            <div class="calendar-modal-header">
                <h4 id="eventTitle">Event Title</h4>
                <button onclick="hideEventModal()" style="background: none; border: none; cursor: pointer;">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <div class="calendar-modal-body">
                <div class="mb-3">
                    <i class="fas fa-clock mr-2" style="color: var(--calendar-primary);"></i>
                    <span id="eventTime">10:00 AM - 11:30 AM</span>
                </div>
                <div class="mb-3">
                    <i class="fas fa-map-marker-alt mr-2" style="color: var(--calendar-primary);"></i>
                    <span id="eventLocation">Room 101</span>
                </div>
                <div class="mb-3">
                    <i class="fas fa-align-left mr-2" style="color: var(--calendar-primary);"></i>
                    <p id="eventDescription" style="margin-top: 5px;">Event description goes here...</p>
                </div>
            </div>
            <div class="calendar-modal-footer">
                <button class="btn-outline" onclick="hideEventModal()">Close</button>
                <button class="btn-primary">Edit Event</button>
            </div>
        </div>
    </div>

    <!-- External JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://unpkg.com/aos@2.3.1/dist/aos.js"></script>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="../assets/js/calendar-functions.js"></script>
    
    <!-- AOS Init -->
    <script>
        AOS.init();
    </script>
    
    <!-- Calendar Functionality -->
    <script>
        // Current calendar state
        let currentState = {
            month: <?php echo date('n'); ?>,
            year: <?php echo date('Y'); ?>,
            view: 'month',
            filter: 'all'
        };
        
        // Initialize interactive elements
        document.addEventListener('DOMContentLoaded', function() {
            // Apply animations to calendar elements
            applyAnimations();
            
            // Initialize filters with AJAX support
            initializeFilters();
            
            // Initialize view switcher
            initializeViewSwitcher();
            
            // Initialize date picker dropdown
            initializeDatePicker();
            
            // Apply layout fixes
            setTimeout(applyLayoutFixes, 100);
            
            // Initialize the current month view
            updateCalendarDisplay();
        });
        
        // Apply animations to calendar elements
        function applyAnimations() {
            const calendarEvents = document.querySelectorAll('.calendar-event');
            calendarEvents.forEach((event, index) => {
                event.style.animationDelay = (0.1 * index) + 's';
            });
        }
        
        // Initialize filters with AJAX functionality
        function initializeFilters() {
            const filterChips = document.querySelectorAll('.filter-chip');
            filterChips.forEach(chip => {
                chip.addEventListener('click', function() {
                    if (!this.classList.contains('active')) {
                        // Update UI
                        document.querySelector('.filter-chip.active').classList.remove('active');
                        this.classList.add('active');
                        
                        // Get filter type
                        const filterType = this.querySelector('span:last-child').textContent.toLowerCase();
                        currentState.filter = filterType === 'all events' ? 'all' : filterType.toLowerCase();
                        
                        // Make AJAX request to filter events
                        filterEvents(currentState.filter);
                    }
                });
            });
        }
        
        // Initialize view switcher with AJAX functionality
        function initializeViewSwitcher() {
            const viewButtons = document.querySelectorAll('.view-switcher button');
            viewButtons.forEach(button => {
                button.addEventListener('click', function() {
                    // Update UI
                    document.querySelector('.view-switcher button.active').classList.remove('active');
                    this.classList.add('active');
                    
                    // Update current view
                    currentState.view = this.textContent.toLowerCase();
                    
                    // Update calendar display
                    updateCalendarDisplay();
                });
            });
        }
        
        // Initialize date picker dropdown
        function initializeDatePicker() {
            const datePickerToggle = document.querySelector('.date-picker-dropdown .dropdown-toggle');
            const datePickerMenu = document.querySelector('.date-picker-dropdown .dropdown-menu');
            const dateInput = datePickerMenu.querySelector('input[type="date"]');
            const goButton = datePickerMenu.querySelector('button');
            
            // Toggle dropdown menu
            datePickerToggle.addEventListener('click', function() {
                datePickerMenu.classList.toggle('show');
            });
            
            // Close dropdown when clicking outside
            document.addEventListener('click', function(e) {
                if (!datePickerToggle.contains(e.target) && !datePickerMenu.contains(e.target)) {
                    datePickerMenu.classList.remove('show');
                }
            });
            
            // Handle go button click
            goButton.addEventListener('click', function() {
                if (dateInput.value) {
                    const date = new Date(dateInput.value);
                    currentState.month = date.getMonth() + 1;
                    currentState.year = date.getFullYear();
                    
                    // Update calendar display
                    updateCalendarDisplay();
                    
                    // Close dropdown
                    datePickerMenu.classList.remove('show');
                }
            });
        }
        
        // Navigation functions with AJAX
        function navigateMonth(direction) {
            // Update month based on direction
            if (direction === 'prev') {
                if (currentState.month === 1) {
                    currentState.month = 12;
                    currentState.year--;
                } else {
                    currentState.month--;
                }
            } else if (direction === 'next') {
                if (currentState.month === 12) {
                    currentState.month = 1;
                    currentState.year++;
                } else {
                    currentState.month++;
                }
            }
            
            // Update calendar display
            updateCalendarDisplay();
        }
        
        // Update calendar display via AJAX
        function updateCalendarDisplay() {
            // Show loading indicator
            const calendarContainer = document.getElementById('calendar-container');
            if (calendarContainer) {
                calendarContainer.classList.add('loading');
            }
            
            // Update month-year display
            const monthYear = document.querySelector('.month-year');
            if (monthYear) {
                const date = new Date(currentState.year, currentState.month - 1, 1);
                monthYear.textContent = date.toLocaleDateString('en-US', { month: 'long', year: 'numeric' });
            }
            
            // Make AJAX request to get calendar data
            const formData = new FormData();
            formData.append('action', 'get_month_data');
            formData.append('month', currentState.month);
            formData.append('year', currentState.year);
            
            fetch('calendar-ajax.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Update calendar grid based on view type
                    updateCalendarGrid(data.data, currentState.view);
                    
                    // Apply animations to new elements
                    setTimeout(applyAnimations, 100);
                } else {
                    console.error('Error fetching calendar data:', data.message);
                }
                
                // Hide loading indicator
                if (calendarContainer) {
                    calendarContainer.classList.remove('loading');
                }
            })
            .catch(error => {
                console.error('AJAX error:', error);
                
                // Hide loading indicator
                if (calendarContainer) {
                    calendarContainer.classList.remove('loading');
                }
            });
        }
        
        // Filter events via AJAX
        function filterEvents(filterType) {
            // Show loading indicator for events section
            const eventsContainer = document.querySelector('.upcoming-events');
            if (eventsContainer) {
                eventsContainer.classList.add('loading');
            }
            
            // Make AJAX request to filter events
            const formData = new FormData();
            formData.append('action', 'filter_events');
            formData.append('filter_type', filterType);
            
            fetch('calendar-ajax.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Update events display
                    updateEventsDisplay(data.data.events);
                } else {
                    console.error('Error filtering events:', data.message);
                }
                
                // Hide loading indicator
                if (eventsContainer) {
                    eventsContainer.classList.remove('loading');
                }
            })
            .catch(error => {
                console.error('AJAX error:', error);
                
                // Hide loading indicator
                if (eventsContainer) {
                    eventsContainer.classList.remove('loading');
                }
            });
        }
        
        // Event modal functions
        function showEventModal(title, time, location, description) {
            document.getElementById('eventTitle').textContent = title;
            document.getElementById('eventTime').textContent = time;
            document.getElementById('eventLocation').textContent = location;
            document.getElementById('eventDescription').textContent = description;
            
            const modal = document.getElementById('eventModal');
            modal.style.display = 'flex';
            
            // Add animation class
            modal.classList.add('show-modal');
            
            // Close modal when clicking outside
            modal.onclick = function(e) {
                if (e.target === modal) {
                    hideEventModal();
                }
            };
        }
        
        function hideEventModal() {
            const modal = document.getElementById('eventModal');
            modal.classList.remove('show-modal');
            setTimeout(() => {
                modal.style.display = 'none';
            }, 300); // Match transition duration in CSS
        }
        
        // Layout fixes function
        function applyLayoutFixes() {
            // Fix app container and main structure
            const appContainer = document.querySelector('.app-container');
            const sidebar = document.querySelector('.sidebar');
            const mainElement = document.querySelector('main');
            const mainContent = document.querySelector('.main-content');
            
            if (appContainer) {
                appContainer.style.display = 'flex';
                appContainer.style.width = '100vw';
                appContainer.style.maxWidth = '100vw';
                appContainer.style.overflow = 'hidden';
            }
            
            if (sidebar) {
                sidebar.style.width = '280px';
                sidebar.style.minWidth = '280px';
                sidebar.style.flexShrink = '0';
                sidebar.style.position = 'fixed';
                sidebar.style.height = '100vh';
            }
            
            if (mainElement) {
                mainElement.style.marginLeft = '280px';
                mainElement.style.width = 'calc(100vw - 280px)';
                mainElement.style.maxWidth = 'calc(100vw - 280px)';
                mainElement.style.boxSizing = 'border-box';
                mainElement.style.overflowX = 'hidden';
            }
        }
    </script>
    
    <!-- Include calendar functions -->
    <script src="calendar-functions.js"></script>
</body>
</html><?php
