<?php
/**
 * 🗄️ DATABASE STRUCTURE VERIFICATION & FIX SCRIPT
 */

session_start();
require_once __DIR__ . '/../includes/config.php';

echo "<html><head><title>🗄️ Database Structure Check</title></head><body style='font-family: monospace; padding: 20px;'>";
echo "<h1>🗄️ DATABASE STRUCTURE VERIFICATION</h1>";
echo "<hr>";

// 1. CHECK CURRENT VIDEOS TABLE STRUCTURE
echo "<h2>1. 📋 CURRENT VIDEOS TABLE STRUCTURE</h2>";
try {
    $result = $conn->query("DESCRIBE videos");
    $existing_columns = [];
    
    if ($result) {
        echo "<table border='1' style='border-collapse: collapse; margin: 10px 0; width: 100%;'>";
        echo "<tr style='background: #f0f0f0;'><th>Field</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th><th>Extra</th></tr>";
        while ($row = $result->fetch_assoc()) {
            $existing_columns[] = $row['Field'];
            echo "<tr>";
            echo "<td><strong>{$row['Field']}</strong></td>";
            echo "<td>{$row['Type']}</td>";
            echo "<td>{$row['Null']}</td>";
            echo "<td>{$row['Key']}</td>";  
            echo "<td>{$row['Default']}</td>";
            echo "<td>{$row['Extra']}</td>";
            echo "</tr>";
        }
        echo "</table>";
    }
} catch (Exception $e) {
    echo "<strong>❌ Error:</strong> " . $e->getMessage() . "<br>";
}

echo "<hr>";

// 2. CHECK REQUIRED COLUMNS FOR VIDEO URLs
echo "<h2>2. 🔍 REQUIRED COLUMNS CHECK</h2>";
$required_columns = [
    'video_url' => 'TEXT',
    'platform' => 'VARCHAR(50)', 
    'video_id_external' => 'VARCHAR(255)',
    'embed_code' => 'TEXT',
    'video_type' => 'ENUM(\'file\', \'url\')'
];

$missing_columns = [];
$sql_fixes = [];

foreach ($required_columns as $column => $type) {
    if (in_array($column, $existing_columns)) {
        echo "<strong>$column:</strong> ✅ <span style='color: green;'>EXISTS</span><br>";
    } else {
        echo "<strong>$column:</strong> ❌ <span style='color: red;'>MISSING</span> (Type: $type)<br>";
        $missing_columns[] = $column;
        $sql_fixes[] = "ALTER TABLE videos ADD COLUMN $column $type NULL";
    }
}

echo "<hr>";

// 3. GENERATE FIX SCRIPT
if (!empty($missing_columns)) {
    echo "<h2>3. 🔧 DATABASE FIX REQUIRED</h2>";
    echo "<p style='color: red; font-weight: bold;'>Missing columns detected! Here are the SQL commands to fix:</p>";
    
    echo "<div style='background: #f8f8f8; padding: 15px; border-radius: 5px; margin: 10px 0;'>";
    echo "<strong>SQL Fix Commands:</strong><br>";
    foreach ($sql_fixes as $sql) {
        echo "<code style='display: block; margin: 5px 0; padding: 5px; background: white; border: 1px solid #ddd;'>$sql;</code>";
    }
    echo "</div>";
    
    // AUTOMATIC FIX OPTION
    echo "<form method='POST' style='margin: 20px 0;'>";
    echo "<input type='hidden' name='auto_fix' value='1'>";
    echo "<button type='submit' style='padding: 10px 20px; background: #dc3545; color: white; border: none; border-radius: 3px; font-weight: bold;'>🔧 AUTO-FIX DATABASE STRUCTURE</button>";
    echo "<p style='color: #666; font-size: 12px;'>⚠️ This will automatically add missing columns to your videos table.</p>";
    echo "</form>";
    
} else {
    echo "<h2>3. ✅ DATABASE STRUCTURE OK</h2>";
    echo "<p style='color: green; font-weight: bold;'>All required columns exist! The database structure is correct.</p>";
}

// 4. PROCESS AUTO-FIX
if ($_POST && isset($_POST['auto_fix'])) {
    echo "<hr>";
    echo "<h2>4. 🔧 EXECUTING AUTO-FIX</h2>";
    
    foreach ($sql_fixes as $sql) {
        echo "<strong>Executing:</strong> <code>$sql</code><br>";
        try {
            if ($conn->query($sql)) {
                echo "✅ <span style='color: green;'>SUCCESS</span><br><br>";
            } else {
                echo "❌ <span style='color: red;'>FAILED: " . $conn->error . "</span><br><br>";
            }
        } catch (Exception $e) {
            echo "❌ <span style='color: red;'>ERROR: " . $e->getMessage() . "</span><br><br>";
        }
    }
    
    echo "<p style='background: #d4edda; padding: 10px; border-radius: 5px; color: #155724;'>";
    echo "🔄 <strong>Refresh this page</strong> to verify the fixes were applied correctly.";
    echo "</p>";
}

echo "<hr>";

// 5. TEST INSERT QUERY
echo "<h2>5. 🧪 TEST VIDEO URL INSERT</h2>";
echo "<form method='POST' style='background: #f0f0f0; padding: 15px; border-radius: 5px;'>";
echo "<h3>Test Video URL Database Insert:</h3>";
echo "<input type='hidden' name='test_insert' value='1'>";
echo "<button type='submit' style='padding: 10px 20px; background: #007bff; color: white; border: none; border-radius: 3px;'>🧪 Test Insert Video URL Record</button>";
echo "</form>";

if ($_POST && isset($_POST['test_insert'])) {
    echo "<h3>🔍 TESTING INSERT:</h3>";
    try {
        $teacher_id = $_SESSION['user_id'] ?? 999;
        $test_sql = "INSERT INTO videos (teacher_id, course_id, title, description, video_url, platform, video_id_external, video_type, is_published, created_at) 
                     VALUES (?, 1, 'Test Video URL', 'Test video from URL', 'https://youtu.be/test123', 'youtube', 'test123', 'url', 1, NOW())";
        
        $stmt = $conn->prepare($test_sql);
        $stmt->bind_param("i", $teacher_id);
        
        if ($stmt->execute()) {
            $video_id = $conn->insert_id;
            echo "✅ <strong>SUCCESS!</strong> Test video inserted with ID: $video_id<br>";
            echo "The database can handle video URL records correctly.<br>";
            
            // Clean up test record
            $conn->query("DELETE FROM videos WHERE id = $video_id");
            echo "🧹 Test record cleaned up.<br>";
        } else {
            echo "❌ <strong>INSERT FAILED:</strong> " . $stmt->error . "<br>";
        }
    } catch (Exception $e) {
        echo "❌ <strong>ERROR:</strong> " . $e->getMessage() . "<br>";
    }
}

echo "<br><br>";
echo "<a href='debug-video-upload.php' style='display: inline-block; padding: 10px 20px; background: #17a2b8; color: white; text-decoration: none; border-radius: 3px; margin-right: 10px;'>🔍 Run Full Debug</a>";
echo "<a href='upload-video.php' style='display: inline-block; padding: 10px 20px; background: #28a745; color: white; text-decoration: none; border-radius: 3px;'>← Back to Upload</a>";

echo "</body></html>";
?>
