<?php
/**
 * Centralized authentication for teacher pages
 * Handles session and role verification
 */

// Include the correct authentication functions (NOT session.php - that causes conflicts)
require_once __DIR__ . '/../../auth/functions.php';
require_once __DIR__ . '/../../config/db_connect.php';

// Track this for redirect loop prevention
$current_script = basename($_SERVER['SCRIPT_NAME']);

// Debug the authentication state
error_log('Teacher Auth Check - Script: ' . $current_script);
error_log('Teacher Auth Check - Session: ' . print_r($_SESSION, true));

// We need teacher role for all pages in this directory
try {
    // Emergency bypass for testing - add ?emergency_login=teacher to the URL
    if (isset($_GET['emergency_login']) && $_GET['emergency_login'] === 'teacher') {
        // Set up emergency session
        $_SESSION['user_id'] = 999; // Emergency ID
        $_SESSION['username'] = 'teacher';
        $_SESSION['first_name'] = 'Emergency';
        $_SESSION['last_name'] = 'Teacher';
        $_SESSION['user_role'] = 'teacher';
        $_SESSION['logged_in'] = true;
        $_SESSION['is_logged_in'] = true;
        error_log('EMERGENCY LOGIN ACTIVATED');
    }
    
    // Check if the user is logged in and has teacher role
    if (!isLoggedIn()) {
        // User is not logged in, track redirect count to prevent loops
        if (!isset($_SESSION['login_redirect_count'])) {
            $_SESSION['login_redirect_count'] = 1;
        } else {
            $_SESSION['login_redirect_count']++;
        }
        
        // If we're in a redirect loop (more than 2 redirects in a row)
        if ($_SESSION['login_redirect_count'] > 2) {
            // We might be in a redirect loop, show an error instead
            echo '<!DOCTYPE html>
            <html lang="en">
            <head>
                <meta charset="UTF-8">
                <meta name="viewport" content="width=device-width, initial-scale=1.0">
                <title>Authentication Error</title>
                <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
            </head>
            <body class="bg-light">
                <div class="container py-5">
                    <div class="row justify-content-center">
                        <div class="col-md-8 col-lg-6">
                            <div class="card shadow">
                                <div class="card-body text-center p-5">
                                    <h3 class="text-danger mb-4"><i class="fas fa-exclamation-triangle me-2"></i>Authentication Error</h3>
                                    <p>Too many redirects detected. Please try clearing your cookies and cache.</p>
                                    <div class="mt-4">
                                        <a href="../auth/login.php?clear_session=true" class="btn btn-primary">Go to Login</a>
                                        <button onclick="clearCookiesAndReload()" class="btn btn-outline-secondary ms-2">Clear Cookies & Retry</button>
                                    </div>
                                    <hr class="my-4">
                                    <div class="text-start small text-muted">
                                        <p><strong>For developers:</strong> Check session handling and authentication logic.</p>
                                        <p>Try these troubleshooting steps:</p>
                                        <ol>
                                            <li>Use incognito/private browsing mode</li>
                                            <li>Clear browser cookies and cache</li>
                                            <li>Try emergency login with teacher/teacher123</li>
                                            <li>Check server logs for session errors</li>
                                        </ol>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <script>
                function clearCookiesAndReload() {
                    document.cookie.split(";").forEach(function(c) { 
                        document.cookie = c.replace(/^ +/, "").replace(/=.*/, "=;expires=" + new Date().toUTCString() + ";path=/"); 
                    });
                    window.location.reload();
                }
                </script>
            </body>
            </html>';
            
            // Reset the counter to prevent permanent lock
            $_SESSION['login_redirect_count'] = 0;
            exit;
        }
        
        // Normal redirect to login
        $current_uri = $_SERVER['REQUEST_URI'];
        
        // Store the current page URL directly in the session
        $_SESSION['last_page'] = $current_uri;
        
        // Create the redirect URL with proper encoding
        $redirect_url = '../auth/login.php?redirect=' . urlencode($current_uri);
        
        // Don't append clear_session parameter - we want to preserve the redirect
        header("Location: {$redirect_url}");
        exit;
    }
    
    // User is logged in, check if they have the teacher role
    if (!hasPermission('teacher')) {
        // User is logged in but is not a teacher, redirect to unauthorized page  
        error_log('Teacher Auth - Access denied for user role: ' . ($_SESSION['user_role'] ?? 'unknown'));
        header("Location: ../auth/unauthorized.php?role=teacher");
        exit;
    }
    
    // Reset redirect counter as authentication was successful
    $_SESSION['login_redirect_count'] = 0;
    
    // Get the teacher ID from session
    $teacher_id = $_SESSION['user_id'];
    
    // Development bypass mode (for testing)
    if (isset($_GET['bypass_auth']) && $_GET['bypass_auth'] === 'true') {
        $teacher_id = 999; // Emergency teacher ID
        echo '<!-- ⚠️ DEVELOPMENT MODE: Using emergency teacher account (ID: 999) ⚠️ -->';
    }
    
} catch (Exception $e) {
    // Log any errors that occur during authentication
    error_log('Authentication error in teacher_auth.php: ' . $e->getMessage());
    echo '<!-- Authentication error: ' . $e->getMessage() . ' -->'; 
}
?>
