<?php
// Set current page for active menu highlighting
$current_page = 'confirm-withdrawal.php';

// Include the centralized teacher layout (handles all authentication, session, DB connection)
require_once __DIR__ . '/components/teacher_layout.php';

$message = '';
$success = false;

// Handle withdrawal confirmation
if (isset($_GET['token']) && !empty($_GET['token'])) {
    $token = $_GET['token'];
    
    try {
        // Find the withdrawal request with this token
        $stmt = $conn->prepare("
            SELECT w.*, u.first_name, u.last_name, u.email 
            FROM withdrawal_requests w 
            JOIN users u ON w.teacher_id = u.user_id 
            WHERE w.confirmation_token = ? AND w.status = 'pending_verification'
        ");
        $stmt->bind_param("s", $token);
        $stmt->execute();
        $result = $stmt->get_result();
        $withdrawal = $result->fetch_assoc();
        
        if ($withdrawal) {
            // Check if token is not expired (24 hours)
            $requested_time = strtotime($withdrawal['requested_at']);
            $current_time = time();
            $time_difference = ($current_time - $requested_time) / 3600; // hours
            
            if ($time_difference > 24) {
                $message = "⏰ This confirmation link has expired (24+ hours old). Please submit a new withdrawal request.";
                $success = false;
            } else {
                // Confirm the withdrawal - change status to pending (visible to admin)
                $update_stmt = $conn->prepare("UPDATE withdrawal_requests SET status = 'pending', confirmed_at = NOW() WHERE confirmation_token = ?");
                $update_stmt->bind_param("s", $token);
                
                if ($update_stmt->execute()) {
                    $success = true;
                    $message = "✅ Withdrawal request confirmed successfully! Your request has been forwarded to admin for processing.";
                    
                    // Send admin notification email (optional)
                    $admin_email = "admin@panaditeacademy.com"; // Replace with actual admin email
                    $teacher_name = $withdrawal['first_name'] . ' ' . $withdrawal['last_name'];
                    $amount = number_format($withdrawal['amount'], 2);
                    
                    $admin_subject = "New Withdrawal Request - R{$amount} - {$teacher_name}";
                    $admin_message = "
                    <div style='max-width:600px; margin:0 auto; font-family:Arial,sans-serif;'>
                        <div style='background:#dc3545; color:white; padding:20px; text-align:center;'>
                            <h1>💰 New Withdrawal Request</h1>
                        </div>
                        <div style='padding:20px; background:white;'>
                            <h2>Admin Action Required</h2>
                            <div style='background:#f8f9fa; padding:15px; border-radius:5px; margin:20px 0;'>
                                <h3>Request Details:</h3>
                                <p><strong>Teacher:</strong> {$teacher_name}</p>
                                <p><strong>Amount:</strong> R{$amount}</p>
                                <p><strong>Bank:</strong> {$withdrawal['bank_name']}</p>
                                <p><strong>Account:</strong> {$withdrawal['account_holder']} - ****" . substr($withdrawal['account_number'], -4) . "</p>
                                <p><strong>Branch Code:</strong> {$withdrawal['branch_code']}</p>
                                <p><strong>Type:</strong> {$withdrawal['account_type']}</p>
                            </div>
                            <p style='text-align:center;'>
                                <a href='https://panaditeacademy.com/admin/withdrawals.php' 
                                   style='background:#dc3545; color:white; text-decoration:none; padding:15px 25px; border-radius:8px; font-weight:bold; display:inline-block;'>
                                    Process Withdrawal
                                </a>
                            </p>
                        </div>
                    </div>";
                    
                    // Send admin notification using same method as working emails
                    $headers = "MIME-Version: 1.0" . "\r\n";
                    $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
                    $headers .= "From: noreply@panaditeacademy.com" . "\r\n";
                    mail($admin_email, $admin_subject, $admin_message, $headers);
                    
                } else {
                    $message = "❌ Error confirming withdrawal request. Please try again or contact support.";
                    $success = false;
                }
            }
        } else {
            $message = "❌ Invalid or already processed confirmation link. Please check your email or submit a new request.";
            $success = false;
        }
        
    } catch (Exception $e) {
        error_log("Withdrawal confirmation error: " . $e->getMessage());
        $message = "❌ System error. Please try again or contact support.";
        $success = false;
    }
} else {
    $message = "❌ No confirmation token provided. Please use the link from your email.";
    $success = false;
}
?>

<style>
.confirmation-container {
    max-width: 600px;
    margin: 50px auto;
    padding: 30px;
    background: white;
    border-radius: 12px;
    box-shadow: 0 4px 15px rgba(0,0,0,0.1);
}

.success-message {
    background: #d4edda;
    border: 1px solid #c3e6cb;
    color: #155724;
    padding: 20px;
    border-radius: 8px;
    margin: 20px 0;
    text-align: center;
}

.error-message {
    background: #f8d7da;
    border: 1px solid #f5c6cb;
    color: #721c24;
    padding: 20px;
    border-radius: 8px;
    margin: 20px 0;
    text-align: center;
}

.action-buttons {
    text-align: center;
    margin: 30px 0;
}

.btn {
    display: inline-block;
    padding: 12px 25px;
    text-decoration: none;
    border-radius: 8px;
    font-weight: 600;
    margin: 0 10px;
}

.btn-primary {
    background: #667eea;
    color: white;
}

.btn-success {
    background: #28a745;
    color: white;
}
</style>

<div class="confirmation-container">
    <div style="text-align: center; margin-bottom: 30px;">
        <h1 style="color: #1e40af; margin: 0;">🔐 Withdrawal Confirmation</h1>
        <p style="color: #6b7280; margin: 10px 0 0 0;">Panadite Academy Teacher Portal</p>
    </div>
    
    <?php if ($success): ?>
        <div class="success-message">
            <h2>✅ Confirmation Successful!</h2>
            <p><?php echo $message; ?></p>
        </div>
        
        <div style="background: #e8f5e8; padding: 20px; border-radius: 8px; margin: 20px 0;">
            <h3 style="color: #2d5a2d; margin: 0 0 15px 0;">📋 What Happens Next?</h3>
            <ul style="color: #2d5a2d; margin: 0; padding-left: 20px;">
                <li>✅ Your withdrawal request is now with our admin team</li>
                <li>⏰ Processing time: 1-2 business days</li>
                <li>📧 You'll receive email updates on status changes</li>
                <li>💰 Funds will be transferred to your confirmed bank account</li>
            </ul>
        </div>
    <?php else: ?>
        <div class="error-message">
            <h2>❌ Confirmation Failed</h2>
            <p><?php echo $message; ?></p>
        </div>
    <?php endif; ?>
    
    <div class="action-buttons">
        <?php if ($success): ?>
            <a href="dashboard.php" class="btn btn-success">
                📊 View Dashboard
            </a>
            <a href="request-withdrawal.php" class="btn btn-primary">
                💰 New Withdrawal Request
            </a>
        <?php else: ?>
            <a href="request-withdrawal.php" class="btn btn-primary">
                💰 Submit New Request
            </a>
            <a href="dashboard.php" class="btn btn-success">
                📊 Back to Dashboard
            </a>
        <?php endif; ?>
    </div>
</div>

<?php
// Include the centralized teacher layout end (closes HTML structure)
require_once __DIR__ . '/components/teacher_layout_end.php';
?>
