<?php
/**
 * Content Creation Workflow
 * 
 * Multi-step process for teachers to create video-specific content
 * (quizzes, assignments, materials, notes) after video upload.
 */

// Set current page for active menu highlighting
$current_page = 'upload-video.php';

// Include the centralized teacher layout start
require_once __DIR__ . '/components/teacher_layout.php';
require_once __DIR__ . '/../includes/id_encryption.php';

// Check if workflow session exists
if (!isset($_SESSION['content_workflow'])) {
    $_SESSION['error_message'] = 'No content creation workflow found.';
    header('Location: upload-video.php');
    exit();
}

$workflow = $_SESSION['content_workflow'];
$encrypted_video_id = $_GET['video_id'] ?? '';
$video_id = IdEncryption::decrypt($encrypted_video_id);

// Validate video ID matches workflow
if ($video_id != $workflow['video_id']) {
    $_SESSION['error_message'] = 'Invalid video ID for workflow.';
    header('Location: upload-video.php');
    exit();
}

$content_types = $workflow['content_types'];
$current_step = $workflow['current_step'];
$video_title = $workflow['video_title'];
$course_id = $workflow['course_id'];

// Content type information
$content_info = [
    'quiz' => [
        'title' => 'Interactive Quiz',
        'description' => 'Create a quiz to test student understanding of this video',
        'icon' => 'fas fa-brain',
        'color' => 'warning',
        'url' => 'create_quiz.php'
    ],
    'assignment' => [
        'title' => 'Practice Assignment', 
        'description' => 'Create a hands-on assignment based on this video content',
        'icon' => 'fas fa-edit',
        'color' => 'primary',
        'url' => 'assignments.php'
    ],
    'materials' => [
        'title' => 'Supplementary Resources',
        'description' => 'Upload additional materials related to this video',
        'icon' => 'fas fa-file-alt',
        'color' => 'success',
        'url' => 'materials.php'
    ],
    'notes' => [
        'title' => 'Lecture Notes',
        'description' => 'Add lecture notes or transcript for this video',
        'icon' => 'fas fa-sticky-note',
        'color' => 'info',
        'url' => 'notes.php'
    ]
];
?>

<div class="container-xl py-4">
    <div class="row justify-content-center">
        <div class="col-lg-10">
            <!-- Header -->
            <div class="text-center mb-4">
                <h2 class="fw-bold text-primary">
                    <i class="fas fa-magic me-2"></i>Content Creation Workflow
                </h2>
                <p class="text-muted">Create additional learning materials for your video: <strong><?= htmlspecialchars($video_title) ?></strong></p>
            </div>

            <!-- Progress Bar -->
            <div class="card mb-4">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-center mb-3">
                        <span class="fw-bold">Progress</span>
                        <span class="text-muted"><?= count($workflow['completed_steps'] ?? []) ?> of <?= count($content_types) ?> completed</span>
                    </div>
                    <div class="progress" style="height: 8px;">
                        <div class="progress-bar bg-success" style="width: <?= count($content_types) > 0 ? (count($workflow['completed_steps'] ?? []) / count($content_types)) * 100 : 0 ?>%"></div>
                    </div>
                </div>
            </div>

            <!-- Content Creation Cards -->
            <div class="row">
                <?php foreach ($content_types as $index => $type): ?>
                    <?php 
                    $info = $content_info[$type];
                    $is_completed = in_array($type, $workflow['completed_steps'] ?? []);
                    $is_current = $index == $current_step && !$is_completed;
                    ?>
                    <div class="col-md-6 mb-4">
                        <div class="card h-100 <?= $is_current ? 'border-' . $info['color'] . ' border-2' : '' ?> <?= $is_completed ? 'border-success' : '' ?>">
                            <div class="card-body text-center">
                                <div class="content-workflow-icon mb-3 <?= $is_completed ? 'bg-success' : 'bg-' . $info['color'] ?>">
                                    <?php if ($is_completed): ?>
                                        <i class="fas fa-check text-white"></i>
                                    <?php else: ?>
                                        <i class="<?= $info['icon'] ?> text-white"></i>
                                    <?php endif; ?>
                                </div>
                                
                                <h5 class="card-title fw-bold"><?= $info['title'] ?></h5>
                                <p class="card-text text-muted"><?= $info['description'] ?></p>
                                
                                <?php if ($is_completed): ?>
                                    <div class="alert alert-success py-2">
                                        <i class="fas fa-check-circle me-1"></i>
                                        <small>Completed</small>
                                    </div>
                                    <a href="<?= $info['url'] ?>?video_id=<?= $encrypted_video_id ?>&edit=1" class="btn btn-outline-success btn-sm">
                                        <i class="fas fa-edit me-1"></i>Edit
                                    </a>
                                <?php elseif ($is_current): ?>
                                    <div class="alert alert-primary py-2">
                                        <i class="fas fa-arrow-right me-1"></i>
                                        <small>Ready to create</small>
                                    </div>
                                    <a href="<?= $info['url'] ?>?video_id=<?= $encrypted_video_id ?>" class="btn btn-<?= $info['color'] ?> btn-lg">
                                        <i class="<?= $info['icon'] ?> me-2"></i>Create Now
                                    </a>
                                <?php else: ?>
                                    <div class="alert alert-light py-2">
                                        <i class="fas fa-clock me-1"></i>
                                        <small>Waiting</small>
                                    </div>
                                    <button class="btn btn-outline-secondary btn-sm" disabled>
                                        <i class="fas fa-lock me-1"></i>Pending
                                    </button>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>

            <!-- Action Buttons -->
            <div class="text-center mt-4">
                <div class="btn-group" role="group">
                    <a href="videos.php" class="btn btn-outline-secondary">
                        <i class="fas fa-list me-2"></i>View All Videos
                    </a>
                    
                    <?php if (count($workflow['completed_steps'] ?? []) == count($content_types)): ?>
                        <a href="finish-content-workflow.php?video_id=<?= $encrypted_video_id ?>" class="btn btn-success">
                            <i class="fas fa-check-double me-2"></i>Complete Workflow
                        </a>
                    <?php else: ?>
                        <button class="btn btn-outline-warning" onclick="skipWorkflow()">
                            <i class="fas fa-forward me-2"></i>Skip Remaining
                        </button>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Help Section -->
            <div class="card mt-4 border-info">
                <div class="card-header bg-info bg-opacity-10">
                    <h6 class="card-title mb-0 text-info">
                        <i class="fas fa-question-circle me-2"></i>How it works
                    </h6>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6">
                            <ul class="list-unstyled">
                                <li class="mb-2"><i class="fas fa-check text-success me-2"></i>Click "Create Now" on each content type</li>
                                <li class="mb-2"><i class="fas fa-check text-success me-2"></i>You'll be taken to the creation form</li>
                                <li class="mb-2"><i class="fas fa-check text-success me-2"></i>The form will be pre-filled with video details</li>
                            </ul>
                        </div>
                        <div class="col-md-6">
                            <ul class="list-unstyled">
                                <li class="mb-2"><i class="fas fa-check text-success me-2"></i>Content will be automatically linked to your video</li>
                                <li class="mb-2"><i class="fas fa-check text-success me-2"></i>Students will see badges on the video</li>
                                <li class="mb-2"><i class="fas fa-check text-success me-2"></i>You can edit content anytime later</li>
                            </ul>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
.content-workflow-icon {
    width: 60px;
    height: 60px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.5rem;
    margin: 0 auto;
}

.card.border-2 {
    box-shadow: 0 4px 12px rgba(0,0,0,0.1);
}

.card.border-success {
    background: rgba(25, 135, 84, 0.02);
}
</style>

<script>
function skipWorkflow() {
    if (confirm('Are you sure you want to skip the remaining content creation? You can always add content later.')) {
        window.location.href = 'finish-content-workflow.php?video_id=<?= $encrypted_video_id ?>&skip=1';
    }
}
</script>

<?php
// Include the centralized teacher layout end
require_once __DIR__ . '/components/teacher_layout_end.php';
?>
