<?php
// Set current page for active menu highlighting
$current_page = 'course-materials.php';

// Essential includes without HTML output
require_once '../includes/session_start.php';
require_once '../config/db_connect.php';
require_once '../auth/functions.php';
requireRole('teacher');

// Show detailed errors for debugging
ini_set('display_errors', 1);
ini_set('log_errors', 1);
ini_set('error_log', '../logs/course_materials_errors.log');
error_reporting(E_ALL);

// Create logs directory if it doesn't exist
$log_dir = '../logs';
if (!is_dir($log_dir)) {
    mkdir($log_dir, 0755, true);
}

// Debug function (no HTML output to avoid headers already sent)
function debug_log($message, $data = null) {
    $timestamp = date('Y-m-d H:i:s');
    $log_message = "[$timestamp] DEBUG: $message";
    if ($data !== null) {
        $log_message .= " | Data: " . print_r($data, true);
    }
    error_log($log_message . "\n", 3, '../logs/course_materials_debug.log');
}

debug_log("Page loaded, starting course-materials.php");

// Set up error handler
set_error_handler(function($severity, $message, $file, $line) {
    debug_log("PHP Error: $message in $file on line $line");
    return false;
});

register_shutdown_function(function() {
    $error = error_get_last();
    if ($error && in_array($error['type'], [E_ERROR, E_CORE_ERROR, E_COMPILE_ERROR, E_PARSE])) {
        debug_log("Fatal Error: " . $error['message'] . " in " . $error['file'] . " on line " . $error['line']);
    }
});

// Increase file upload limits
ini_set('upload_max_filesize', '50M');
ini_set('post_max_size', '55M');
ini_set('memory_limit', '256M');
ini_set('max_execution_time', '300');
ini_set('max_input_time', '300');

// Get teacher ID from session
$teacher_id = $_SESSION['user_id'];

// Define course materials upload directory
$upload_dir = '../uploads/materials/';
if (!is_dir($upload_dir)) {
    mkdir($upload_dir, 0755, true);
}

// Initialize variables
$material_id = null;
$title = '';
$description = '';
$category = '';
$course_id = '';
$file_name = '';
$existing_file_path = '';
$error = '';
$success = '';

// Get courses for dropdown
$stmt = $conn->prepare("SELECT course_id, title AS course_name, course_id AS course_code FROM courses WHERE teacher_id = ? ORDER BY title");
$stmt->bind_param("i", $teacher_id);
$stmt->execute();
$courses_result = $stmt->get_result();
$courses = [];
while ($row = $courses_result->fetch_assoc()) {
    $courses[] = $row;
}

// Process form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    debug_log("Form submitted", $_POST);
    
    // Check if it's a delete request
    if (isset($_POST['action']) && $_POST['action'] === 'delete' && isset($_POST['material_id'])) {
        debug_log("Delete request detected");
        $delete_id = intval($_POST['material_id']);
        
        // Get file path before deleting the record
        $file_query = $conn->prepare("SELECT file_path FROM course_materials WHERE material_id = ? AND teacher_id = ?");
        $file_query->bind_param("ii", $delete_id, $teacher_id);
        $file_query->execute();
        $file_result = $file_query->get_result();
        
        if ($file_row = $file_result->fetch_assoc()) {
            $file_to_delete = $file_row['file_path'];
            
            // Delete the record from the database
            $delete_stmt = $conn->prepare("DELETE FROM course_materials WHERE material_id = ? AND teacher_id = ?");
            $delete_stmt->bind_param("ii", $delete_id, $teacher_id);
            
            if ($delete_stmt->execute()) {
                // Attempt to delete the physical file if it exists
                if (file_exists($file_to_delete)) {
                    @unlink($file_to_delete);
                }
                $success = "Material deleted successfully.";
            } else {
                $error = "Failed to delete material. Please try again.";
            }
        } else {
            $error = "Material not found or you don't have permission to delete it.";
        }
    } else {
        debug_log("Processing create/update request");
        
        // This is a create/update request
        $material_id = isset($_POST['material_id']) ? intval($_POST['material_id']) : null;
        $title = trim($_POST['title']);
        $description = trim($_POST['description']);
        $category = trim($_POST['category']);
        $course_id = !empty($_POST['course_id']) ? intval($_POST['course_id']) : null;
        $video_id = !empty($_POST['video_id']) ? intval($_POST['video_id']) : null;
        
        debug_log("Form data extracted", [
            'material_id' => $material_id,
            'title' => $title,
            'category' => $category,
            'course_id' => $course_id,
            'video_id' => $video_id
        ]);
        
        // Validate form data
        if (empty($title)) {
            $error = "Please enter a title for the material.";
        } elseif (empty($category)) {
            $error = "Please select or enter a category.";
        } else {
            // Check if we're updating an existing material
            if ($material_id) {
                // Get current file path
                $file_query = $conn->prepare("SELECT file_path FROM course_materials WHERE material_id = ? AND teacher_id = ?");
                $file_query->bind_param("ii", $material_id, $teacher_id);
                $file_query->execute();
                $file_result = $file_query->get_result();
                
                if ($file_row = $file_result->fetch_assoc()) {
                    $existing_file_path = $file_row['file_path'];
                } else {
                    $error = "Material not found or you don't have permission to edit it.";
                }
            }
            
            // Process file upload if there's no error so far
            if (empty($error)) {
                $file_path = $existing_file_path;
                
                // Check if base64 file data was provided
                if (isset($_POST['base64_file']) && !empty($_POST['base64_file']) && isset($_POST['base64_filename']) && !empty($_POST['base64_filename'])) {
                    // Process base64 file upload
                    $base64_data = $_POST['base64_file'];
                    $original_filename = $_POST['base64_filename'];
                    $file_extension = strtolower(pathinfo($original_filename, PATHINFO_EXTENSION));
                    
                    // Generate a unique filename
                    $new_file_name = uniqid('material_') . '_' . basename($original_filename);
                    $upload_path = $upload_dir . $new_file_name;
                    
                    // Allowed file extensions
                    $allowed_extensions = ['pdf', 'doc', 'docx', 'xls', 'xlsx', 'ppt', 'pptx', 'txt', 'zip', 'rar', 'jpg', 'jpeg', 'png', 'gif', 'mp3', 'mp4'];
                    
                    if (!in_array($file_extension, $allowed_extensions)) {
                        $error = "Invalid file type. Allowed types: " . implode(', ', $allowed_extensions);
                    } else {
                        try {
                            // Ensure upload directory exists and is writable
                            if (!is_dir($upload_dir)) {
                                mkdir($upload_dir, 0755, true);
                            }
                            
                            if (!is_writable($upload_dir)) {
                                $error = "Upload directory is not writable. Please check permissions.";
                            } else {
                                // Properly handle the Base64 data
                                // Check if the string contains a data URL scheme
                                if (strpos($base64_data, 'data:') === 0) {
                                    // Extract the base64 part from data URL
                                    $base64_parts = explode(',', $base64_data, 2);
                                    if (count($base64_parts) == 2) {
                                        $base64_content = $base64_parts[1];
                                    } else {
                                        $error = "Invalid Base64 data URL format.";
                                    }
                                } else {
                                    // Assume the whole string is Base64
                                    $base64_content = $base64_data;
                                }
                                
                                if (empty($error)) {
                                    // Decode and save the file
                                    $file_content = base64_decode($base64_content, true);
                                    
                                    if ($file_content === false) {
                                        $error = "Failed to decode the Base64 data. Invalid encoding.";
                                    } else {
                                        // Save the file
                                        if (file_put_contents($upload_path, $file_content) !== false) {
                                            $file_path = $upload_path;
                                            
                                            // Delete old file if we're updating
                                            if (!empty($existing_file_path) && file_exists($existing_file_path)) {
                                                @unlink($existing_file_path);
                                            }
                                        } else {
                                            $error = "Failed to save the file. Please check directory permissions.";
                                        }
                                    }
                                }
                            }
                        } catch (Exception $e) {
                            $error = "Error processing file: " . $e->getMessage();
                        }
                    }
                }
                // Traditional file upload as fallback
                elseif (isset($_FILES['material_file']) && $_FILES['material_file']['error'] !== UPLOAD_ERR_NO_FILE) {
                    if ($_FILES['material_file']['error'] !== UPLOAD_ERR_OK) {
                        $error = "File upload error. Error code: " . $_FILES['material_file']['error'];
                    } else {
                        $file_tmp = $_FILES['material_file']['tmp_name'];
                        $file_name = basename($_FILES['material_file']['name']);
                        $file_extension = strtolower(pathinfo($file_name, PATHINFO_EXTENSION));
                        
                        // Generate a unique filename
                        $new_file_name = uniqid('material_') . '_' . $file_name;
                        $upload_path = $upload_dir . $new_file_name;
                        
                        // Allowed file extensions
                        $allowed_extensions = ['pdf', 'doc', 'docx', 'xls', 'xlsx', 'ppt', 'pptx', 'txt', 'zip', 'rar', 'jpg', 'jpeg', 'png', 'gif', 'mp3', 'mp4'];
                        
                        if (!in_array($file_extension, $allowed_extensions)) {
                            $error = "Invalid file type. Allowed types: " . implode(', ', $allowed_extensions);
                        } else {
                            // Upload successful, update file path
                            if (move_uploaded_file($file_tmp, $upload_path)) {
                                $file_path = $upload_path;
                                
                                // Delete old file if we're updating
                                if (!empty($existing_file_path) && file_exists($existing_file_path)) {
                                    @unlink($existing_file_path);
                                }
                            } else {
                                $error = "Failed to upload file. Please check directory permissions.";
                            }
                        }
                    }
                } elseif (empty($file_path)) {
                    // No file uploaded and no existing file (for new material)
                    $error = "Please upload a file.";
                }
                
                // Save to database if there's no error
                if (empty($error)) {
                    debug_log("About to execute database operation", [
                        'material_id' => $material_id,
                        'file_path' => $file_path
                    ]);
                    
                    try {
                        if ($material_id) {
                            debug_log("Preparing UPDATE query");
                            // Update existing material
                            $stmt = $conn->prepare("
                                UPDATE course_materials 
                                SET title = ?, description = ?, category = ?, course_id = ?, video_id = ?, file_path = ?, date_modified = NOW() 
                                WHERE material_id = ? AND teacher_id = ?
                            ");
                            $stmt->bind_param("sssiisii", $title, $description, $category, $course_id, $video_id, $file_path, $material_id, $teacher_id);
                        } else {
                            debug_log("Preparing INSERT query");
                            // Insert new material
                            $stmt = $conn->prepare("
                                INSERT INTO course_materials (teacher_id, title, description, category, course_id, video_id, file_path, date_added) 
                                VALUES (?, ?, ?, ?, ?, ?, ?, NOW())
                            ");
                            $stmt->bind_param("isssiss", $teacher_id, $title, $description, $category, $course_id, $video_id, $file_path);
                        }
                        
                        debug_log("Executing database query");
                        
                        if ($stmt->execute()) {
                            debug_log("Database operation successful");
                            // Store success message in session for main page display
                            $_SESSION['success_message'] = $material_id ? "Material updated successfully!" : "Material uploaded successfully!";
                            
                            // Reset form for new submission
                            if (!$material_id) {
                                $title = '';
                                $description = '';
                                $category = '';
                                $course_id = '';
                                $video_id = '';
                            }
                            
                            debug_log("About to redirect to course-materials.php");
                            // Redirect back to this page instead of manage-course-materials.php
                            header("Location: course-materials.php");
                            exit;
                        } else {
                            debug_log("Database execution failed", $stmt->error);
                            $error = "Database error: " . $stmt->error;
                        }
                    } catch (Exception $e) {
                        debug_log("Exception in database operation", $e->getMessage());
                        $error = "Exception: " . $e->getMessage();
                    }
                }
            }
        }
    }
}

// Check if we're editing an existing material
if (isset($_GET['edit']) && is_numeric($_GET['edit'])) {
    $material_id = intval($_GET['edit']);
    $stmt = $conn->prepare("
        SELECT m.*, c.course_name, c.course_code 
        FROM course_materials m
        LEFT JOIN courses c ON m.course_id = c.course_id
        WHERE m.material_id = ? AND m.teacher_id = ?
    ");
    $stmt->bind_param("ii", $material_id, $teacher_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($material = $result->fetch_assoc()) {
        $title = $material['title'];
        $description = $material['description'];
        $category = $material['category'];
        $course_id = $material['course_id'];
        $video_id = $material['video_id'];
        $existing_file_path = $material['file_path'];
        $file_name = basename($existing_file_path);
    } else {
        $error = "Material not found or you don't have permission to edit it.";
        $material_id = null;
    }
}

// Get existing categories for autocomplete
$stmt = $conn->prepare("SELECT DISTINCT category FROM course_materials WHERE teacher_id = ? ORDER BY category");
$stmt->bind_param("i", $teacher_id);
$stmt->execute();
$categories_result = $stmt->get_result();
$existing_categories = [];
while ($row = $categories_result->fetch_assoc()) {
    if (!empty($row['category'])) {
        $existing_categories[] = $row['category'];
    }
}

// Include the teacher layout after form processing is complete
require_once __DIR__ . '/components/teacher_layout.php';

// Include the header content
include_once 'components/header_content.php';
?>

<!-- Custom CSS for Course Materials Upload page -->
<style>
    /* Modern Header Styles with Clean Panadite Design */
    .materials-upload-header {
        background: linear-gradient(135deg, #4361ee 0%, #3b82f6 100%);
        padding: 2rem 0;
        color: white;
        border-radius: 20px;
        margin-bottom: 2.5rem;
        position: relative;
        overflow: hidden;
        box-shadow: 0 15px 30px rgba(106, 17, 203, 0.15);
        transition: all 0.5s ease;
    }
    
    .materials-upload-header:hover {
        box-shadow: 0 20px 40px rgba(106, 17, 203, 0.25);
        transform: translateY(-5px);
    }
    
    .materials-upload-header h1 {
        font-weight: 700;
        margin-bottom: 1rem;
        position: relative;
        z-index: 2;
        font-size: 2.5rem;
        letter-spacing: -0.5px;
        text-shadow: 0 2px 10px rgba(0,0,0,0.1);
    }
    
    .materials-upload-header p {
        opacity: 0.95;
        max-width: 700px;
        position: relative;
        z-index: 2;
        font-size: 1.1rem;
        line-height: 1.6;
        font-weight: 400;
    }
    
    /* Animated Floating Shapes */
    .floating-shape {
        position: absolute;
        background: rgba(255, 255, 255, 0.1);
        border-radius: 50%;
        z-index: 1;
    }
    
    .shape-1 {
        width: 120px;
        height: 120px;
        top: -30px;
        right: 10%;
        animation: float 8s ease-in-out infinite;
    }
    
    .shape-2 {
        width: 80px;
        height: 80px;
        bottom: -20px;
        right: 20%;
        animation: float 6s ease-in-out infinite 1s;
    }
    
    .shape-3 {
        width: 60px;
        height: 60px;
        top: 40%;
        right: 30%;
        animation: float 7s ease-in-out infinite 0.5s;
    }
    
    .shape-4 {
        width: 40px;
        height: 40px;
        bottom: 30%;
        right: 40%;
        animation: float 5s ease-in-out infinite 1.5s;
    }
    
    @keyframes float {
        0% { transform: translateY(0) rotate(0deg); }
        50% { transform: translateY(-20px) rotate(10deg); }
        100% { transform: translateY(0) rotate(0deg); }
    }
    
    /* Document/File Icon Animation */
    .header-icon {
        position: absolute;
        right: 10%;
        top: 50%;
        transform: translateY(-50%);
        width: 200px;
        height: 200px;
        background: rgba(255, 255, 255, 0.1);
        border-radius: 20px;
        display: flex;
        align-items: center;
        justify-content: center;
        z-index: 1;
        animation: pulse 3s ease-in-out infinite;
    }
    
    @keyframes pulse {
        0% { transform: translateY(-50%) scale(1); opacity: 0.7; }
        50% { transform: translateY(-50%) scale(1.05); opacity: 0.9; }
        100% { transform: translateY(-50%) scale(1); opacity: 0.7; }
    }
    
    /* Card Styles */
    .upload-card {
        border: none;
        border-radius: 12px;
        box-shadow: 0 8px 25px rgba(67, 97, 238, 0.1);
        transition: all 0.3s ease;
        background: #fff;
        backdrop-filter: blur(20px);
        overflow: hidden;
    }
    
    .upload-card:hover {
        box-shadow: 0 12px 30px rgba(67, 97, 238, 0.15);
        transform: translateY(-4px);
    }
    
    .upload-card .card-header {
        background: linear-gradient(to right, rgba(67, 97, 238, 0.05), rgba(59, 130, 246, 0.05));
        border-bottom: 1px solid rgba(0,0,0,0.05);
        padding: 1.75rem;
    }
    
    .upload-card .card-header h5 {
        margin-bottom: 0;
        font-weight: 600;
        display: flex;
        align-items: center;
        font-size: 1.25rem;
        color: #333;
    }
    
    .upload-card .card-header svg {
        margin-right: 12px;
        width: 24px;
        height: 24px;
        color: #4361ee;
        stroke-width: 2;
    }
    
    .upload-card .card-body {
        padding: 2.5rem;
        background-color: #fff;
    }
    
    /* Form Element Styles */
    .form-group {
        margin-bottom: 2rem;
    }
    
    .form-label {
        font-weight: 600;
        margin-bottom: 0.75rem;
        color: #333;
        font-size: 0.95rem;
        display: inline-block;
    }
    
    .form-control {
        border-radius: 12px;
        padding: 0.85rem 1.25rem;
        border: 1px solid #e0e0e0;
        transition: all 0.3s;
        font-size: 1rem;
        box-shadow: 0 2px 5px rgba(0,0,0,0.02);
    }
    
    .form-control:focus {
        border-color: #6a11cb;
        box-shadow: 0 0 0 0.25rem rgba(106, 17, 203, 0.15);
    }
    
    .form-text {
        font-size: 0.85rem;
        color: #6c757d;
        margin-top: 0.5rem;
    }
    
    /* Enhanced File Drop Area */
    .file-drop-area {
        border: 2px dashed #d1d5db;
        border-radius: 12px;
        padding: 2rem;
        text-align: center;
        background: linear-gradient(145deg, #f8fafc, #f1f5f9);
        transition: all 0.3s ease;
        cursor: pointer;
        position: relative;
        overflow: hidden;
    }
    .file-drop-area:hover {
        border-color: #4361ee;
        background-color: rgba(67, 97, 238, 0.03);
        transform: translateY(-2px);
    }
    
    .file-drop-area.highlight {
        border-color: #4361ee;
        background: linear-gradient(135deg, rgba(67, 97, 238, 0.1), rgba(59, 130, 246, 0.05));
    }
    
    .file-message {
        margin-bottom: 0;
        font-size: 1.05rem;
        color: #495057;
        font-weight: 500;
    }
    
    .file-message-secondary {
        margin-top: 0.5rem;
        color: #6c757d;
        font-size: 0.9rem;
    }
    
    .file-icon {
        margin-bottom: 1rem;
        color: #6a11cb;
        font-size: 2.5rem;
        transition: transform 0.3s ease;
    }
    
    .file-drop-area:hover .file-icon {
        transform: scale(1.1);
    }
    
    .file-preview {
        display: none;
        margin: 1.25rem auto 0;
        max-width: 100%;
        text-align: left;
        animation: fadeIn 0.5s ease;
    }
    
    @keyframes fadeIn {
        from { opacity: 0; transform: translateY(10px); }
        to { opacity: 1; transform: translateY(0); }
    }
    
    .file-info {
        display: flex;
        align-items: center;
        padding: 12px 16px;
        background: #f8f9fa;
        border-radius: 10px;
        border: 1px solid #e9ecef;
        margin-bottom: 12px;
        transition: all 0.3s ease;
        box-shadow: 0 2px 5px rgba(0,0,0,0.05);
    }
    
    .file-info:hover {
        background: #fff;
        box-shadow: 0 5px 15px rgba(106, 17, 203, 0.1);
        border-color: #6a11cb;
    }
    
    /* Modern Button Styles */
    .btn-primary {
        background: linear-gradient(135deg, #4361ee 0%, #3b82f6 100%);
        border: none;
        border-radius: 8px;
        padding: 12px 24px;
        font-weight: 600;
        transition: all 0.3s ease;
        position: relative;
        overflow: hidden;
    }
    
    .btn-primary:hover {
        background: linear-gradient(135deg, #3451db 0%, #2563eb 100%);
        transform: translateY(-2px);
        box-shadow: 0 8px 20px rgba(67, 97, 238, 0.3);
    }
    
    .btn-primary:active {
        transform: translateY(-1px);
    }
    
    .file-preview .file-name {
        font-weight: 600;
        font-size: 0.95rem;
        margin-bottom: 2px;
        color: #333;
    }
    
    .file-preview .btn-close {
        font-size: 0.75rem;
        transition: all 0.3s ease;
        opacity: 0.6;
    }
    
    .file-preview .btn-close:hover {
        transform: rotate(90deg);
        opacity: 1;
    }
    
    .file-preview .file-remove {
        margin-left: auto;
        color: #6c757d;
        cursor: pointer;
        font-size: 18px;
    }
    
    .btn-panadite {
        background: linear-gradient(135deg, #6a11cb 0%, #8c67f4 100%);
        border: none;
        color: white;
        padding: 0.6rem 1.5rem;
        border-radius: 8px;
        font-weight: 500;
        transition: all 0.3s;
    }
    
    .btn-panadite:hover {
        box-shadow: 0 5px 15px rgba(106, 17, 203, 0.4);
        transform: translateY(-2px);
        color: white;
    }
    
    .btn-outline-panadite {
        border: 1px solid #6a11cb;
        color: #6a11cb;
        background-color: transparent;
        padding: 0.6rem 1.5rem;
        border-radius: 8px;
        font-weight: 500;
        transition: all 0.3s;
    }
    
    .btn-outline-panadite:hover {
        background-color: rgba(106, 17, 203, 0.1);
    }
    
    /* Loading animation */
    .file-loading {
        display: none;
        text-align: center;
    }
    
    .file-loading .spinner {
        width: 40px;
        height: 40px;
        margin: 0 auto 10px;
        border: 4px solid #f3f3f3;
        border-top: 4px solid #6a11cb;
        border-radius: 50%;
        animation: spin 1s linear infinite;
    }
    
    @keyframes spin {
        0% { transform: rotate(0deg); }
        100% { transform: rotate(360deg); }
    }
    
    /* Tips Section Styles */
    .tips-container {
        border-radius: 20px;
        box-shadow: 0 10px 25px rgba(106, 17, 203, 0.08);
        overflow: hidden;
        margin-bottom: 2rem;
        background: linear-gradient(to right, rgba(255, 255, 255, 0.9), rgba(255, 255, 255, 0.95));
        border: 1px solid rgba(140, 103, 244, 0.2);
        transition: all 0.3s ease;
    }
    
    .tips-container:hover {
        box-shadow: 0 15px 35px rgba(106, 17, 203, 0.15);
    }
    
    .tips-header {
        padding: 1.25rem 1.75rem;
        background: linear-gradient(135deg, rgba(106, 17, 203, 0.08) 0%, rgba(140, 103, 244, 0.08) 100%);
        display: flex;
        align-items: center;
        cursor: pointer;
        position: relative;
        transition: all 0.3s ease;
    }
    
    .tips-header:hover {
        background: linear-gradient(135deg, rgba(67, 97, 238, 0.12) 0%, rgba(59, 130, 246, 0.12) 100%);
    }
    
    .tips-icon {
        width: 60px;
        height: 60px;
        background: linear-gradient(135deg, #4361ee 0%, #3b82f6 100%);
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        color: white;
        font-size: 24px;
        margin-right: 1rem;
        box-shadow: 0 8px 25px rgba(67, 97, 238, 0.3);
    }
    
    .tips-icon i {
        font-size: 1.5rem;
    }
    
    .tips-header h5 {
        margin: 0;
        font-weight: 600;
        color: #333;
        flex-grow: 1;
    }
    
    .tips-toggle-icon {
        transition: transform 0.3s ease;
    }
    
    .tips-toggle-icon i {
        font-size: 1.5rem;
        color: #6a11cb;
    }
    
    .tips-content {
        max-height: 0;
        overflow: hidden;
        transition: max-height 0.5s ease, padding 0.5s ease;
    }
    
    .tips-content.active {
        max-height: 1000px;
        padding: 1.75rem;
        border-top: 1px solid rgba(140, 103, 244, 0.1);
    }
    
    .tip-item {
        display: flex;
        padding: 1.25rem;
        background: white;
        border-radius: 16px;
        box-shadow: 0 5px 15px rgba(0, 0, 0, 0.04);
        height: 100%;
        transition: all 0.3s ease;
    }
    
    .tip-item:hover {
        transform: translateY(-5px);
        box-shadow: 0 10px 20px rgba(106, 17, 203, 0.1);
    }
    
    .tip-icon {
        width: 48px;
        height: 48px;
        min-width: 48px;
        display: flex;
        align-items: center;
        justify-content: center;
        background: linear-gradient(135deg, rgba(106, 17, 203, 0.1) 0%, rgba(140, 103, 244, 0.1) 100%);
        border-radius: 12px;
        margin-right: 1rem;
    }
    
    .tip-icon i {
        font-size: 1.75rem;
        color: #6a11cb;
    }
    
    .tip-text h6 {
        font-weight: 600;
        margin-bottom: 0.5rem;
        color: #4361ee;
        font-weight: 600;
        font-size: 0.9rem;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        margin-bottom: 0.75rem;
    }
    
    .tip-text p {
        font-size: 0.9rem;
        color: #6c757d;
        margin-bottom: 0;
        line-height: 1.5;
    }
    
    .tips-footer {
        text-align: center;
        padding-top: 1.5rem;
        margin-top: 1rem;
        border-top: 1px dashed rgba(140, 103, 244, 0.2);
        font-size: 0.9rem;
        color: #6c757d;
    }
    
    .tips-footer a {
        color: #6a11cb;
        text-decoration: none;
        font-weight: 500;
        transition: all 0.2s ease;
    }
    
    .tips-footer a:hover {
        text-decoration: underline;
    }
</style>

<div class="container-fluid">
    <!-- Statistics Cards - Compact Design -->
<div class="row mb-4">
    <div class="col-md-4 mb-3">
        <div class="card border-0 shadow-sm" style="border-radius: 10px;">
            <div class="card-body p-3">
                <div class="d-flex align-items-center">
                    <div class="icon-box me-3" style="width: 40px; height: 40px; background: rgba(67, 97, 238, 0.1); border-radius: 8px; display: flex; align-items: center; justify-content: center;">
                        <i class="bx bx-file" style="font-size: 20px; color: #4361ee;"></i>
                    </div>
                    <div>
                        <div class="small text-muted text-uppercase fw-bold mb-1">Total Materials</div>
                        <div class="fs-4 fw-bold text-dark mb-0">
                            <?php
                            $stmt = $conn->prepare("SELECT COUNT(*) as total FROM course_materials WHERE teacher_id = ?");
                            if ($stmt) {
                                $stmt->bind_param("i", $teacher_id);
                                $stmt->execute();
                                $result = $stmt->get_result();
                                $row = $result->fetch_assoc();
                                echo $row['total'];
                            } else {
                                echo '0';
                            }
                            ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <div class="col-md-4 mb-3">
        <div class="card border-0 shadow-sm" style="border-radius: 10px;">
            <div class="card-body p-3">
                <div class="d-flex align-items-center">
                    <div class="icon-box me-3" style="width: 40px; height: 40px; background: rgba(34, 197, 94, 0.1); border-radius: 8px; display: flex; align-items: center; justify-content: center;">
                        <i class="bx bx-category" style="font-size: 20px; color: #22c55e;"></i>
                    </div>
                    <div>
                        <div class="small text-muted text-uppercase fw-bold mb-1">Categories</div>
                        <div class="fs-4 fw-bold text-dark mb-0">
                            <?php
                            $stmt = $conn->prepare("SELECT COUNT(DISTINCT category) as total FROM course_materials WHERE teacher_id = ? AND category != ''");
                            if ($stmt) {
                                $stmt->bind_param("i", $teacher_id);
                                $stmt->execute();
                                $result = $stmt->get_result();
                                $row = $result->fetch_assoc();
                                echo $row['total'];
                            } else {
                                echo '0';
                            }
                            ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <div class="col-md-4 mb-3">
        <div class="card border-0 shadow-sm" style="border-radius: 10px;">
            <div class="card-body p-3">
                <div class="d-flex align-items-center">
                    <div class="icon-box me-3" style="width: 40px; height: 40px; background: rgba(59, 130, 246, 0.1); border-radius: 8px; display: flex; align-items: center; justify-content: center;">
                        <i class="bx bx-download" style="font-size: 20px; color: #3b82f6;"></i>
                    </div>
                    <div>
                        <div class="small text-muted text-uppercase fw-bold mb-1">Downloads</div>
                        <div class="fs-4 fw-bold text-dark mb-0">
                            <?php
                            $stmt = $conn->prepare("SELECT SUM(COALESCE(download_count, 0)) as total FROM course_materials WHERE teacher_id = ?");
                            if ($stmt) {
                                $stmt->bind_param("i", $teacher_id);
                                $stmt->execute();
                                $result = $stmt->get_result();
                                $row = $result->fetch_assoc();
                                echo $row['total'] ?: 0;
                            } else {
                                echo '0';
                            }
                            ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Success Message on Main Page -->
<?php if (isset($_SESSION['success_message'])): ?>
<div class="alert alert-success alert-dismissible fade show" role="alert">
    <i class="bx bx-check-circle me-2"></i>
    <?= $_SESSION['success_message'] ?>
    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
</div>
<?php 
    unset($_SESSION['success_message']); 
endif; 
?>

<!-- Materials Library -->
<div class="card border-0 shadow-sm" style="border-radius: 12px;">
    <div class="card-header bg-white border-0 py-3">
        <div class="d-flex justify-content-between align-items-center">
            <h5 class="mb-0 fw-bold text-dark d-flex align-items-center">
                <i class="bx bx-library me-2" style="color: #4361ee;"></i>
                Course Material Management
            </h5>
            <div class="d-flex gap-2 align-items-center">
                <select class="form-select form-select-sm" style="width: auto; border-radius: 8px; border: 1px solid #e5e7eb;">
                    <option>All Categories</option>
                    <?php
                    $stmt = $conn->prepare("SELECT DISTINCT category FROM course_materials WHERE teacher_id = ? AND category != '' ORDER BY category");
                    if ($stmt) {
                        $stmt->bind_param("i", $teacher_id);
                        $stmt->execute();
                        $result = $stmt->get_result();
                        while($row = $result->fetch_assoc()) {
                            echo "<option value='{$row['category']}'>{$row['category']}</option>";
                        }
                    }
                    ?>
                </select>
                <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#uploadModal" style="background: #4361ee; border: none; border-radius: 8px; padding: 8px 16px; font-weight: 500;">
                    <i class="bx bx-plus me-1"></i> Add Material
                </button>
            </div>
        </div>
    </div>
    <div class="card-body p-4">
        <?php
        // Fetch materials for display
        $stmt = $conn->prepare("SELECT m.*, c.course_name as course_title FROM course_materials m LEFT JOIN courses c ON m.course_id = c.course_id WHERE m.teacher_id = ? ORDER BY m.date_added DESC");
        if ($stmt) {
            $stmt->bind_param("i", $teacher_id);
            $stmt->execute();
            $materials_result = $stmt->get_result();
        } else {
            // Prepare failed, create empty result
            $materials_result = null;
        }
        
        if ($materials_result && $materials_result->num_rows > 0): ?>
        <div class="table-responsive">
            <table class="table table-hover" style="border: none;">
                <thead style="background-color: #f8fafc; border-radius: 8px;">
                    <tr>
                        <th class="border-0 text-muted fw-bold" style="padding: 16px; font-size: 13px; text-transform: uppercase; letter-spacing: 0.5px;">Title</th>
                        <th class="border-0 text-muted fw-bold" style="padding: 16px; font-size: 13px; text-transform: uppercase; letter-spacing: 0.5px;">Category</th>
                        <th class="border-0 text-muted fw-bold" style="padding: 16px; font-size: 13px; text-transform: uppercase; letter-spacing: 0.5px;">Course</th>
                        <th class="border-0 text-muted fw-bold" style="padding: 16px; font-size: 13px; text-transform: uppercase; letter-spacing: 0.5px;">Upload Date</th>
                        <th class="border-0 text-muted fw-bold" style="padding: 16px; font-size: 13px; text-transform: uppercase; letter-spacing: 0.5px;">Downloads</th>
                        <th class="border-0 text-muted fw-bold" style="padding: 16px; font-size: 13px; text-transform: uppercase; letter-spacing: 0.5px;">Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php while ($material = $materials_result->fetch_assoc()): ?>
                    <tr>
                        <td>
                            <div class="d-flex align-items-center">
                                <i class="bx bx-file me-2 text-primary"></i>
                                <div>
                                    <div class="fw-bold"><?= htmlspecialchars($material['title']) ?></div>
                                    <?php if (!empty($material['description'])): ?>
                                    <small class="text-muted"><?= htmlspecialchars(substr($material['description'], 0, 50)) ?>...</small>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </td>
                        <td>
                            <?php if (!empty($material['category'])): ?>
                            <span class="badge bg-secondary"><?= htmlspecialchars($material['category']) ?></span>
                            <?php else: ?>
                            <span class="text-muted">-</span>
                            <?php endif; ?>
                        </td>
                        <td>
                            <?php if (!empty($material['course_title'])): ?>
                            <?= htmlspecialchars($material['course_title']) ?>
                            <?php else: ?>
                            <span class="text-muted">All Courses</span>
                            <?php endif; ?>
                        </td>
                        <td><?= date('M j, Y', strtotime($material['upload_date'])) ?></td>
                        <td>
                            <span class="badge bg-info"><?= $material['download_count'] ?: 0 ?></span>
                        </td>
                        <td>
                            <div class="btn-group" role="group">
                                <a href="<?= htmlspecialchars($material['file_path']) ?>" target="_blank" class="btn btn-sm btn-outline-primary" title="View">
                                    <i class="bx bx-show"></i>
                                </a>
                                <a href="course-materials.php?edit=<?= $material['material_id'] ?>" class="btn btn-sm btn-outline-secondary" title="Edit">
                                    <i class="bx bx-edit"></i>
                                </a>
                                <form method="POST" style="display: inline;" onsubmit="return confirm('Are you sure you want to delete this material?')">
                                    <input type="hidden" name="action" value="delete">
                                    <input type="hidden" name="material_id" value="<?= $material['material_id'] ?>">
                                    <button type="submit" class="btn btn-sm btn-outline-danger" title="Delete">
                                        <i class="bx bx-trash"></i>
                                    </button>
                                </form>
                            </div>
                        </td>
                    </tr>
                    <?php endwhile; ?>
                </tbody>
            </table>
        </div>
        <?php else: ?>
        <div class="text-center py-5">
            <div class="empty-state">
                <div class="mb-4" style="opacity: 0.6;">
                    <i class="bx bx-file-blank" style="font-size: 4rem; color: #9ca3af;"></i>
                </div>
                <h5 class="text-muted mb-2 fw-bold">No materials uploaded yet</h5>
                <p class="text-muted mb-4">Manage and organize your course materials from the dedicated materials page.</p>
                <a href="manage-course-materials.php" class="btn btn-primary" style="background: #4361ee; border: none; border-radius: 8px; padding: 10px 20px; font-weight: 500; text-decoration: none;">
                    <i class="bx bx-folder-open me-2"></i>View Materials
                </a>
            </div>
        </div>
        <?php endif; ?>
    </div>
</div>
</div>

<!-- Upload Modal -->
<div class="modal fade" id="uploadModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">
                    <i class="bx bx-upload me-2"></i>
                    <?= $material_id ? 'Edit Material' : 'Upload New Material' ?>
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <!-- Alert Messages -->
                <?php if (!empty($error)): ?>
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    <i class="bx bx-error-circle me-2"></i>
                    <?= $error ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php endif; ?>
                
                <?php if (!empty($success)): ?>
                <div class="alert alert-success alert-dismissible fade show" role="alert">
                    <i class="bx bx-check-circle me-2"></i>
                    <?= $success ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php endif; ?>

                <form id="materialForm" action="course-materials.php" method="post" enctype="multipart/form-data">
                    <?php if ($material_id): ?>
                    <input type="hidden" name="material_id" value="<?= $material_id ?>">
                    <?php endif; ?>
                    <div class="row">
                        <div class="col-md-8">
                            <div class="mb-3">
                                <label for="title" class="form-label">Title <span class="text-danger">*</span></label>
                                <input type="text" class="form-control" id="title" name="title" value="<?= htmlspecialchars($title) ?>" required>
                                <div class="form-text">Give your material a clear, descriptive title</div>
                            </div>
                        </div>
                        
                        <div class="col-md-4">
                            <div class="mb-3">
                                <label for="category" class="form-label">Category</label>
                                <input type="text" class="form-control" id="category" name="category" value="<?= htmlspecialchars($category) ?>" placeholder="e.g., Worksheets, Presentations">
                            </div>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="description" class="form-label">Description</label>
                        <textarea class="form-control" id="description" name="description" rows="3"><?= htmlspecialchars($description) ?></textarea>
                        <div class="form-text">Briefly describe what this material contains</div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="course_id" class="form-label">Associated Course</label>
                                <select class="form-select" id="course_id" name="course_id" onchange="loadCourseVideos()">
                                    <option value="">None (Available to all students)</option>
                                    <?php foreach ($courses as $course): ?>
                                    <option value="<?= $course['course_id'] ?>" <?= $course_id == $course['course_id'] ? 'selected' : '' ?>>
                                        <?= htmlspecialchars($course['course_name']) ?>
                                    </option>
                                    <?php endforeach; ?>
                                </select>
                                <div class="form-text">Link this material to a specific course</div>
                            </div>
                        </div>
                        
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="video_id" class="form-label">Associated Video Lesson <span class="badge bg-info">New</span></label>
                                <select class="form-select" id="video_id" name="video_id" disabled>
                                    <option value="">Select course first...</option>
                                </select>
                                <div class="form-text">Associate this material with a specific video lesson (optional)</div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Material File <span class="text-danger">*</span></label>
                        <div class="file-drop-area" id="dropArea">
                            <input class="file-input" type="file" name="material_file" id="fileInput" accept=".pdf,.doc,.docx,.xls,.xlsx,.ppt,.pptx,.txt,.zip,.rar,.jpg,.jpeg,.png,.gif,.mp3,.mp4" style="display: none;">
                            
                            <div class="drop-zone-content">
                                <div class="upload-icon">
                                    <i class="bx bx-cloud-upload"></i>
                                </div>
                                
                                <div class="upload-text">Drop your file here</div>
                                <div class="upload-subtext text-muted">or</div>
                                <button type="button" class="btn btn-outline-primary" onclick="document.getElementById('fileInput').click()">
                                    <i class="bx bx-folder-open me-2"></i>Browse Files
                                </button>
                                
                                <div class="file-types mt-3">
                                    <small class="text-muted">Supported formats: PDF, DOC, DOCX, XLS, XLSX, PPT, PPTX, TXT, ZIP, RAR, JPG, PNG, GIF, MP3, MP4</small>
                                </div>
                            </div>
                            
                            <div class="file-preview" id="filePreview" style="display: none;">
                                <div class="d-flex align-items-center justify-content-between">
                                    <div class="d-flex align-items-center">
                                        <i class="bx bx-file-blank me-2 fs-3 text-primary"></i>
                                        <div>
                                            <div class="fw-bold" id="fileName"></div>
                                            <small class="text-muted" id="fileSize"></small>
                                        </div>
                                    </div>
                                    <button type="button" class="btn btn-sm btn-outline-danger" id="removeFile">
                                        <i class="bx bx-x"></i>
                                    </button>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Hidden inputs for Base64 file data -->
                        <input type="hidden" name="base64_file" id="base64FileInput">
                        <input type="hidden" name="base64_filename" id="fileNameInput">
                        <input type="hidden" name="base64_filetype" id="fileTypeInput">
                        
                        <?php if (!empty($existing_file_path)): ?>
                        <div class="alert alert-info mt-3">
                            <i class="bx bx-info-circle me-2"></i>
                            Current file: <strong><?= basename($existing_file_path) ?></strong>
                            <br><small class="text-muted">Upload a new file to replace the current one, or leave empty to keep the existing file.</small>
                        </div>
                        <?php endif; ?>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="submit" form="materialForm" class="btn btn-primary">
                    <i class="bx bx-save me-1"></i>
                    <?= $material_id ? 'Update Material' : 'Upload Material' ?>
                </button>
            </div>
        </div>
    </div>
</div>

<script>
    document.addEventListener('DOMContentLoaded', function() {
        const dropArea = document.getElementById('dropArea');
        const fileInput = document.getElementById('fileInput');
        const filePreview = document.getElementById('filePreview');
        const fileName = document.getElementById('fileName');
        const fileIcon = document.getElementById('fileIcon');
        const removeFileBtn = document.getElementById('removeFileBtn');
        const fileSize = document.getElementById('fileSize');
        
        // Handle click on drop area
        dropArea.addEventListener('click', function(e) {
            if (e.target.classList.contains('choose-file-button') || e.target === dropArea) {
                fileInput.click();
            }
        });
        
        // Prevent default drag behaviors
        ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
            dropArea.addEventListener(eventName, preventDefaults, false);
        });
        
        // Highlight drop area when item is dragged over it
        ['dragenter', 'dragover'].forEach(eventName => {
            dropArea.addEventListener(eventName, function(e) {
                e.preventDefault();
                e.stopPropagation();
                dropArea.classList.add('highlight');
            }, false);
        });
        
        // Remove highlight when drag leaves
        ['dragleave', 'drop'].forEach(eventName => {
            dropArea.addEventListener(eventName, function(e) {
                e.preventDefault();
                e.stopPropagation();
                dropArea.classList.remove('highlight');
            }, false);
        });
        
        // Handle dropped files
        dropArea.addEventListener('drop', function(e) {
            e.preventDefault();
            e.stopPropagation();
            const dt = e.dataTransfer;
            const files = dt.files;
            
            if (files.length) {
                fileInput.files = files;
                handleFiles();
            }
        }, false);
        
        // Handle file input change
        fileInput.addEventListener('change', handleFiles, false);
        
        // Handle remove file button
        removeFileBtn.addEventListener('click', function() {
            fileInput.value = '';
            filePreview.style.display = 'none';
        });
        
        function preventDefaults(e) {
            e.preventDefault();
            e.stopPropagation();
        }
        
        function handleFiles() {
            if (fileInput.files.length > 0) {
                const file = fileInput.files[0]; // Only handle the first file
                
                // Check file size (50MB limit)
                const maxSize = 50 * 1024 * 1024; // 50MB in bytes
                if (file.size > maxSize) {
                    alert('File size exceeds the limit of 50MB.');
                    return;
                }
        // Animate header elements with slight delay
        setTimeout(() => {
            const headerTitle = document.querySelector('.materials-upload-header h1');
            const headerDesc = document.querySelector('.materials-upload-header p');
            
            if (headerTitle && headerDesc) {
                headerTitle.style.opacity = '0';
                headerTitle.style.transform = 'translateY(20px)';
                headerDesc.style.opacity = '0';
                headerDesc.style.transform = 'translateY(20px)';
                
                // Fade in header elements
                setTimeout(() => {
                    headerTitle.style.transition = 'all 0.6s ease';
                    headerTitle.style.opacity = '1';
                    headerTitle.style.transform = 'translateY(0)';
                    
                    setTimeout(() => {
                        headerDesc.style.transition = 'all 0.6s ease';
                        headerDesc.style.opacity = '1';
                        headerDesc.style.transform = 'translateY(0)';
                    }, 200);
                }, 100);
            }
        }, 300);
    });
</script>

<!-- Footer removed as it was causing errors -->

<!-- Base64 File Upload Script with cache busting timestamp: <?php echo time(); ?> -->
<script>
// Base64 File Upload Functionality - Improved Implementation
document.addEventListener('DOMContentLoaded', function() {
    // Initialize video dropdown functionality first
    initializeVideoDropdown();
    
    // Basic element selection
    var dropArea = document.getElementById('dropArea');
    if (!dropArea) return; // Exit if element doesn't exist
    
    var fileInput = document.getElementById('fileInput');
    var chooseButton = document.querySelector('.choose-file-button');
    var filePreview = document.getElementById('filePreview');
    var fileName = document.getElementById('fileName');
    var fileSize = document.getElementById('fileSize');
    var fileIcon = document.getElementById('fileIcon');
    var removeFileBtn = document.getElementById('removeFileBtn');
    var materialForm = document.getElementById('materialForm');
    
    // Create or get hidden fields for base64 data
    var base64FileInput, fileNameInput, fileTypeInput;
    
    if (materialForm) {
        // Remove any existing base64 inputs to avoid duplicates
        var existingBase64Input = document.getElementById('base64_file');
        var existingNameInput = document.getElementById('base64_filename');
        var existingTypeInput = document.getElementById('base64_filetype');
        
        if (existingBase64Input) materialForm.removeChild(existingBase64Input);
        if (existingNameInput) materialForm.removeChild(existingNameInput);
        if (existingTypeInput) materialForm.removeChild(existingTypeInput);
        
        // Create hidden input for base64 data
        base64FileInput = document.createElement('input');
        base64FileInput.type = 'hidden';
        base64FileInput.name = 'base64_file';
        base64FileInput.id = 'base64_file';
        materialForm.appendChild(base64FileInput);
        
        // Create hidden input for file name
        fileNameInput = document.createElement('input');
        fileNameInput.type = 'hidden';
        fileNameInput.name = 'base64_filename';
        fileNameInput.id = 'base64_filename';
        materialForm.appendChild(fileNameInput);
        
        // Create hidden input for file type
        fileTypeInput = document.createElement('input');
        fileTypeInput.type = 'hidden';
        fileTypeInput.name = 'base64_filetype';
        fileTypeInput.id = 'base64_filetype';
        materialForm.appendChild(fileTypeInput);
    } else {
        console.error('Material form not found!');
    }
    
    // Exit if required elements don't exist
    if (!fileInput || !filePreview) return;

    // Make drop area clickable to select files
    if (dropArea) {
        dropArea.addEventListener('click', function() {
            fileInput.click();
        });
    }

    // Handle "browse files" text click
    if (chooseButton) {
        chooseButton.addEventListener('click', function(e) {
            e.stopPropagation();
            fileInput.click();
        });
    }

    // Handle file selection through input
    if (fileInput) {
        fileInput.addEventListener('change', function() {
            if (this.files && this.files.length > 0) {
                processSelectedFile(this.files[0]);
            }
        });
    }

    // Prevent defaults for drag events
    function preventDefaults(e) {
        e.preventDefault();
        e.stopPropagation();
    }
    
    // Add drag event listeners if dropArea exists
    if (dropArea) {
        ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(function(eventName) {
            dropArea.addEventListener(eventName, preventDefaults, false);
        });
        
        // Highlight functions for drag events
        function highlight() { dropArea.classList.add('highlight'); }
        function unhighlight() { dropArea.classList.remove('highlight'); }
        
        // Add highlight/unhighlight events
        dropArea.addEventListener('dragenter', highlight, false);
        dropArea.addEventListener('dragover', highlight, false);
        dropArea.addEventListener('dragleave', unhighlight, false);
        dropArea.addEventListener('drop', unhighlight, false);
        
        // Handle file drop
        dropArea.addEventListener('drop', function(e) {
            var dt = e.dataTransfer;
            if (dt.files && dt.files.length > 0) {
                processSelectedFile(dt.files[0]);
            }
        }, false);
    }
    
    // Convert file to Base64 (improved)
    function readFileAsBase64(file, callback) {
        console.log('Converting file to Base64:', file.name, file.type, file.size);
        
        var reader = new FileReader();
        reader.onload = function(event) {
            var base64 = event.target.result;
            console.log('Base64 conversion successful. Length:', base64.length);
            callback(base64);
        };
        reader.onerror = function(error) {
            console.error('Error reading file:', error);
            alert('Error reading file: ' + error);
        };
        reader.readAsDataURL(file);
    }

    // Process the selected file with better error handling
    function processSelectedFile(file) {
        if (!file) {
            console.error('No file selected');
            return;
        }
        
        console.log('Processing file:', file.name, file.type, file.size);
        
        // Update file info display
        if (fileName) fileName.textContent = file.name;
        if (fileSize) fileSize.textContent = formatFileSize(file.size);
        updateFileIcon(file.name);
        showFilePreview();
        
        // Convert file to Base64
        readFileAsBase64(file, function(base64) {
            // Store base64 data in hidden inputs
            if (base64FileInput) {
                base64FileInput.value = base64;
                console.log('Base64 data set in hidden input');
            } else {
                console.error('base64FileInput not found');
            }
            
            if (fileNameInput) {
                fileNameInput.value = file.name;
            } else {
                console.error('fileNameInput not found');
            }
            
            if (fileTypeInput) {
                fileTypeInput.value = file.type;
            } else {
                console.error('fileTypeInput not found');
            }
        });
    }

    // Format file size for display
    function formatFileSize(bytes) {
        if (bytes === 0) return '0 Bytes';
        var k = 1024;
        var sizes = ['Bytes', 'KB', 'MB', 'GB'];
        var i = Math.floor(Math.log(bytes) / Math.log(k));
        return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
    }

    // Update file icon based on extension
    function updateFileIcon(filename) {
        if (!fileIcon) return;
        
        var extension = filename.split('.').pop().toLowerCase();
        fileIcon.className = 'bx me-3';

        // Simple icon mapping
        if (extension === 'pdf') {
            fileIcon.classList.add('bxs-file-pdf', 'text-danger');
        } else if (extension === 'doc' || extension === 'docx') {
            fileIcon.classList.add('bxs-file-doc', 'text-primary');
        } else if (extension === 'xls' || extension === 'xlsx') {
            fileIcon.classList.add('bxs-file-excel', 'text-success');
        } else if (extension === 'ppt' || extension === 'pptx') {
            fileIcon.classList.add('bxs-file-ppt', 'text-warning');
        } else if (extension === 'jpg' || extension === 'jpeg' || extension === 'png' || extension === 'gif') {
            fileIcon.classList.add('bxs-file-image', 'text-info');
        } else if (extension === 'mp4' || extension === 'avi' || extension === 'mov') {
            fileIcon.classList.add('bxs-video', 'text-purple');
        } else if (extension === 'mp3' || extension === 'wav') {
            fileIcon.classList.add('bxs-music', 'text-pink');
        } else if (extension === 'zip' || extension === 'rar') {
            fileIcon.classList.add('bxs-file-archive', 'text-secondary');
        } else {
            fileIcon.classList.add('bxs-file', 'text-muted');
        }
    }

    // Show the file preview
    function showFilePreview() {
        if (filePreview) filePreview.style.display = 'block';
        if (dropArea) dropArea.classList.add('has-file');
    }

    // Hide the file preview
    function hideFilePreview() {
        if (filePreview) filePreview.style.display = 'none';
        if (dropArea) dropArea.classList.remove('has-file');
        if (fileInput) fileInput.value = '';
        
        // Clear hidden base64 inputs
        if (base64FileInput) base64FileInput.value = '';
        if (fileNameInput) fileNameInput.value = '';
        if (fileTypeInput) fileTypeInput.value = '';
    }

    // Handle file removal button
    if (removeFileBtn) {
        removeFileBtn.addEventListener('click', function(e) {
            e.stopPropagation();
            hideFilePreview();
        });
    }
    
    // Hook into form submission with better handling
    if (materialForm) {
        materialForm.addEventListener('submit', function(e) {
            // Check if we have a file selected but no Base64 data yet
            if (fileInput.files.length > 0 && (!base64FileInput || !base64FileInput.value)) {
                // Prevent the form from submitting
                e.preventDefault();
                
                console.log('File selected but Base64 not ready, converting now...');
                
                // Convert the file to Base64 and submit the form when done
                readFileAsBase64(fileInput.files[0], function(base64) {
                    if (base64FileInput) base64FileInput.value = base64;
                    if (fileNameInput) fileNameInput.value = fileInput.files[0].name;
                    if (fileTypeInput) fileTypeInput.value = fileInput.files[0].type;
                    
                    console.log('Base64 conversion completed, submitting form');
                    materialForm.submit();
                });
            } else {
                // If we already have the Base64 data or no file is selected,
                // make sure to clear the file input to avoid sending the file twice
                if (fileInput.files.length > 0 && base64FileInput && base64FileInput.value) {
                    console.log('Clearing file input to avoid double upload');
                    // We're submitting Base64 data, so clear the file input
                    fileInput.value = '';
                }
            }
        });
    }
    
    // Initialize video dropdown on page load if editing or course is pre-selected
    const courseSelect = document.getElementById('course_id');
    if (courseSelect && courseSelect.value) {
        <?php if (!empty($course_id)): ?>
        loadCourseVideos(<?= $video_id ?? 'null' ?>);
        <?php else: ?>
        loadCourseVideos();
        <?php endif; ?>
    }
});

// Initialize video dropdown functionality
function initializeVideoDropdown() {
    const courseSelect = document.getElementById('course_id');
    const videoSelect = document.getElementById('video_id');
    
    if (!courseSelect || !videoSelect) {
        console.log('Course or video select elements not found');
        return;
    }
    
    // Check if course is already selected on page load
    if (courseSelect.value) {
        console.log('Course pre-selected:', courseSelect.value);
        loadCourseVideos(<?= !empty($video_id) ? $video_id : 'null' ?>);
    }
    
    // Add change event listener
    courseSelect.addEventListener('change', function() {
        console.log('Course changed to:', this.value);
        loadCourseVideos();
    });
}

// Function to load videos for selected course
function loadCourseVideos(selectedVideoId = null) {
    const courseSelect = document.getElementById('course_id');
    const videoSelect = document.getElementById('video_id');
    
    if (!courseSelect || !videoSelect) return;
    
    const courseId = courseSelect.value;
    
    if (!courseId) {
        videoSelect.innerHTML = '<option value="">Select course first...</option>';
        videoSelect.disabled = true;
        return;
    }

    console.log('Loading videos for course ID:', courseId);

    // Show loading state
    videoSelect.innerHTML = '<option value="">Loading videos...</option>';
    videoSelect.disabled = true;

    // Make AJAX request to get videos for this course
    fetch('ajax/get_course_videos.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: 'course_id=' + courseId
    })
    .then(response => {
        console.log('Response status:', response.status);
        return response.text(); // Get as text first to see what we're receiving
    })
    .then(text => {
        console.log('Raw response:', text);
        try {
            const data = JSON.parse(text);
            console.log('Parsed data:', data);
            
            if (data.success) {
                // Clear and populate video options
                videoSelect.innerHTML = '<option value="">-- Select Video Lesson (Optional) --</option>';
                
                if (data.videos && data.videos.length > 0) {
                    data.videos.forEach(video => {
                        const option = document.createElement('option');
                        option.value = video.video_id;
                        option.textContent = video.title;
                        if (selectedVideoId && video.video_id == selectedVideoId) {
                            option.selected = true;
                        }
                        videoSelect.appendChild(option);
                    });
                    console.log('Loaded', data.videos.length, 'videos');
                } else {
                    console.log('No videos found in response');
                }
                
                videoSelect.disabled = false;
            } else {
                videoSelect.innerHTML = '<option value="">No videos available</option>';
                videoSelect.disabled = true;
                console.error('Error loading videos:', data.message);
            }
        } catch (parseError) {
            console.error('JSON Parse Error:', parseError);
            console.error('Raw response was:', text);
            videoSelect.innerHTML = '<option value="">Error parsing response</option>';
            videoSelect.disabled = true;
        }
    })
    .catch(error => {
        videoSelect.innerHTML = '<option value="">Error loading videos</option>';
        videoSelect.disabled = true;
        console.error('AJAX Error:', error);
    });
}
</script>



<script>


// Load course videos without debugging
function loadCourseVideos(selectedVideoId = null) {
    const courseSelect = document.getElementById('course_id');
    const videoSelect = document.getElementById('video_id');
    
    if (!courseSelect || !videoSelect) {
        return;
    }

    const courseId = courseSelect.value;
    
    if (!courseId) {
        videoSelect.innerHTML = '<option value="">Select course first...</option>';
        videoSelect.disabled = true;
        return;
    }

    videoSelect.innerHTML = '<option value="">Loading videos...</option>';
    videoSelect.disabled = true;

    // Make AJAX request to get videos for this course
    fetch('ajax/get_course_videos.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: 'course_id=' + courseId
    })
    .then(response => response.text())
    .then(text => {
        try {
            const data = JSON.parse(text);
            
            if (data.success) {
                videoSelect.innerHTML = '<option value="">-- Select Video Lesson (Optional) --</option>';
                
                if (data.videos && data.videos.length > 0) {
                    data.videos.forEach(video => {
                        const option = document.createElement('option');
                        option.value = video.video_id;
                        option.textContent = video.title;
                        if (selectedVideoId && video.video_id == selectedVideoId) {
                            option.selected = true;
                        }
                        videoSelect.appendChild(option);
                    });
                }
                
                videoSelect.disabled = false;
            } else {
                videoSelect.innerHTML = '<option value="">No videos available</option>';
                videoSelect.disabled = true;
            }
        } catch (parseError) {
            videoSelect.innerHTML = '<option value="">Error parsing response</option>';
            videoSelect.disabled = true;
        }
    })
    .catch(error => {
        videoSelect.innerHTML = '<option value="">Error loading videos</option>';
        videoSelect.disabled = true;
    });
}

// Initialize video dropdown functionality
document.addEventListener('DOMContentLoaded', function() {
    setTimeout(function() {
        const courseSelect = document.getElementById('course_id');
        const videoSelect = document.getElementById('video_id');
        
        if (!courseSelect || !videoSelect) {
            return;
        }
        
        // Check if course is already selected
        if (courseSelect.value) {
            loadCourseVideos(<?= !empty($video_id) ? $video_id : 'null' ?>);
        }
        
        // Add change event listener
        courseSelect.addEventListener('change', function() {
            loadCourseVideos();
        });
        
        updateDebugStatus('Initialization complete');
    }, 500);
});
</script>

<style>
/* Targeted fixes for course-materials page only */
.page-content {
    padding: 20px 30px;
}

.container-fluid {
    padding-left: 15px;
    padding-right: 15px;
}

.row {
    margin-left: 0;
    margin-right: 0;
}
</style>

<?php
// Include the centralized teacher layout end
require_once __DIR__ . '/components/teacher_layout_end.php';
?>
