<?php
// Professional Assignment Creation System - Panadite Academy
require_once('../includes/session_start.php');
require_once('../config/db_connect.php');
require_once('../auth/functions.php');

// Require teacher role
requireRole('teacher');

$current_page = 'assignments';
$message = '';

// Process form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        // Validate required fields
        $required_fields = ['title', 'course_id', 'due_date', 'points', 'description'];
        $errors = [];
        
        foreach ($required_fields as $field) {
            if (empty($_POST[$field])) {
                $errors[] = ucfirst(str_replace('_', ' ', $field)) . ' is required';
            }
        }
        
        // Validate submission types separately (array field)
        if (empty($_POST['submission_types']) || !is_array($_POST['submission_types'])) {
            $errors[] = 'At least one submission type is required';
        }
        
        if (!empty($errors)) {
            $message = '<div class="alert alert-danger"><strong>Validation Error:</strong> ' . implode(', ', $errors) . '</div>';
        } else {
            // Get teacher ID
            $teacher_id = $_SESSION['user_id'];
            
            // Prepare and validate data for insertion
            $title = trim(mysqli_real_escape_string($conn, $_POST['title']));
            $course_id = (int)$_POST['course_id'];
            $description = trim(mysqli_real_escape_string($conn, $_POST['description']));
            $instructions = trim(mysqli_real_escape_string($conn, $_POST['instructions'] ?? ''));
            
            // Format dates properly for MySQL DATETIME
            $due_date = mysqli_real_escape_string($conn, $_POST['due_date']);
            $start_date = !empty($_POST['start_date']) ? mysqli_real_escape_string($conn, $_POST['start_date']) : NULL;
            
            // Validate and set numeric values
            $points = max(1, (int)$_POST['points']); // Minimum 1 point
            $max_file_size = max(1, min(500, (int)($_POST['max_file_size'] ?? 50))); // 1-500 MB range
            $max_group_size = max(1, min(10, (int)($_POST['max_group_size'] ?? 1))); // 1-10 people range
            $late_penalty = max(0, min(100, (float)($_POST['late_penalty'] ?? 0))); // 0-100% range
            
            // Handle multiple submission types
            $submission_types = [];
            if (isset($_POST['submission_types']) && is_array($_POST['submission_types'])) {
                // Validate each submission type against allowed values
                $valid_types = ['file_upload', 'video', 'document', 'image', 'excel', 'pdf', 'presentation', 'code', 'text_entry', 'multiple_files'];
                foreach ($_POST['submission_types'] as $type) {
                    if (in_array($type, $valid_types)) {
                        $submission_types[] = $type;
                    }
                }
            }
            $submission_type_json = json_encode($submission_types);
            
            // Boolean values
            $allow_late = isset($_POST['allow_late']) ? 1 : 0;
            $group_assignment = isset($_POST['group_assignment']) ? 1 : 0;
            $is_published = isset($_POST['is_published']) ? 1 : 0;
            $contributes_to_final_grade = isset($_POST['contributes_to_final_grade']) ? 1 : 0;
            
            // Clean grading rubric
            $grading_rubric = trim(mysqli_real_escape_string($conn, $_POST['grading_rubric'] ?? ''));
            
            // Handle allowed formats - combine formats from all selected submission types
            $allowed_formats = [];
            if (isset($_POST['allowed_formats']) && is_array($_POST['allowed_formats'])) {
                $allowed_formats = $_POST['allowed_formats'];
            } else {
                // Auto-generate formats based on selected submission types
                foreach ($submission_types as $type) {
                    switch ($type) {
                        case 'document':
                            $allowed_formats = array_merge($allowed_formats, ['pdf', 'doc', 'docx', 'txt']);
                            break;
                        case 'excel':
                            $allowed_formats = array_merge($allowed_formats, ['xlsx', 'xls', 'csv']);
                            break;
                        case 'presentation':
                            $allowed_formats = array_merge($allowed_formats, ['ppt', 'pptx', 'pdf']);
                            break;
                        case 'image':
                            $allowed_formats = array_merge($allowed_formats, ['jpg', 'jpeg', 'png', 'gif', 'bmp']);
                            break;
                        case 'video':
                            $allowed_formats = array_merge($allowed_formats, ['mp4', 'avi', 'mov', 'mkv']);
                            break;
                        case 'code':
                            $allowed_formats = array_merge($allowed_formats, ['zip', 'rar', 'py', 'js', 'html', 'css', 'php']);
                            break;
                    }
                }
                $allowed_formats = array_unique($allowed_formats); // Remove duplicates
            }
            $allowed_formats_json = json_encode($allowed_formats);
            
            // Handle file upload if provided
            $file_path = null;
            if (isset($_FILES['assignment_file']) && $_FILES['assignment_file']['error'] == 0) {
                $upload_dir = '../uploads/assignments/';
                if (!file_exists($upload_dir)) {
                    mkdir($upload_dir, 0755, true);
                }
                
                $file_extension = pathinfo($_FILES['assignment_file']['name'], PATHINFO_EXTENSION);
                $file_name = 'assignment_' . time() . '_' . uniqid() . '.' . $file_extension;
                $full_path = $upload_dir . $file_name;
                
                if (move_uploaded_file($_FILES['assignment_file']['tmp_name'], $full_path)) {
                    $file_path = 'uploads/assignments/' . $file_name;
                }
            }
            

            
            // Insert assignment into database
            $insert_query = "
                INSERT INTO assignments (
                    teacher_id, course_id, title, description, instructions, 
                    due_date, start_date, points, submission_type, allowed_formats, 
                    max_file_size, allow_late, late_penalty, group_assignment, 
                    max_group_size, grading_rubric, is_published, file_path, contributes_to_final_grade
                ) VALUES (
                    ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?
                )
            ";
            
            $stmt = $conn->prepare($insert_query);
            
            // Check if prepare failed
            if (!$stmt) {
                throw new Exception('Prepare failed: ' . $conn->error);
            }
            
            // Parameter types: i=integer, s=string, d=decimal
            // 1:teacher_id(i), 2:course_id(i), 3:title(s), 4:description(s), 5:instructions(s),
            // 6:due_date(s), 7:start_date(s), 8:points(i), 9:submission_type_json(s), 10:allowed_formats_json(s),
            // 11:max_file_size(i), 12:allow_late(i), 13:late_penalty(d), 14:group_assignment(i),
            // 15:max_group_size(i), 16:grading_rubric(s), 17:is_published(i), 18:file_path(s), 19:contributes_to_final_grade(i)
            
            $stmt->bind_param(
                'iisssssissiidiisisi',
                $teacher_id, $course_id, $title, $description, $instructions,
                $due_date, $start_date, $points, $submission_type_json, $allowed_formats_json,
                $max_file_size, $allow_late, $late_penalty, $group_assignment,
                $max_group_size, $grading_rubric, $is_published, $file_path, $contributes_to_final_grade
            );
            
            if ($stmt->execute()) {
                $assignment_id = $conn->insert_id;
                $message = '<div class="alert alert-success"><i class="fas fa-check-circle me-2"></i><strong>Success!</strong> Assignment "' . htmlspecialchars($title) . '" has been created successfully. <a href="assignments.php" class="alert-link">View All Assignments</a></div>';
                
                // Clear form data on success
                $_POST = [];
            } else {
                throw new Exception('Failed to create assignment. SQL Error: ' . $stmt->error . ' | MySQL Error: ' . $conn->error);
            }
        }
    } catch (Exception $e) {
        $message = '<div class="alert alert-danger"><strong>Error:</strong> ' . $e->getMessage() . '</div>';
    }
}

// Get teacher's courses for dropdown
$teacher_id = $_SESSION['user_id'];
$courses_query = "SELECT course_id, title FROM courses WHERE teacher_id = ? ORDER BY title";
$courses_stmt = $conn->prepare($courses_query);
$courses_stmt->bind_param("i", $teacher_id);
$courses_stmt->execute();
$courses_result = $courses_stmt->get_result();

$page_title = "Create Assignment";
include __DIR__ . '/components/header.php';
?>

<link rel="stylesheet" href="assets/css/pages/assignments.css">
<style>
.assignment-form {
    background: #fff;
    border-radius: 12px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    padding: 2rem;
    margin-bottom: 2rem;
}

.form-section {
    margin-bottom: 2rem;
    padding-bottom: 1.5rem;
    border-bottom: 1px solid #e9ecef;
}

.form-section:last-child {
    border-bottom: none;
    margin-bottom: 0;
}

.section-title {
    color: #2c3e50;
    font-size: 1.2rem;
    font-weight: 600;
    margin-bottom: 1rem;
    display: flex;
    align-items: center;
}

.section-title i {
    margin-right: 0.5rem;
    color: #3498db;
}

.submission-type-cards {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1rem;
    margin-top: 1rem;
}

.submission-card {
    border: 2px solid #e9ecef;
    border-radius: 8px;
    padding: 1rem;
    text-align: center;
    cursor: pointer;
    transition: all 0.3s ease;
    background: #fff;
    position: relative;
}

.submission-card:hover {
    border-color: #3498db;
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(52, 152, 219, 0.15);
}

.submission-card.selected {
    border-color: #3498db;
    background: #f8f9fa;
}

.selection-indicator {
    position: absolute;
    top: 8px;
    right: 8px;
    background: #3498db;
    color: white;
    border-radius: 50%;
    width: 24px;
    height: 24px;
    display: none;
    align-items: center;
    justify-content: center;
    font-size: 12px;
}

.submission-card.selected .selection-indicator {
    display: flex;
}

.submission-card i {
    font-size: 2rem;
    color: #3498db;
    margin-bottom: 0.5rem;
}

.format-tags {
    display: flex;
    flex-wrap: wrap;
    gap: 0.5rem;
    margin-top: 1rem;
}

.format-tag {
    background: #e9ecef;
    padding: 0.25rem 0.5rem;
    border-radius: 4px;
    font-size: 0.875rem;
    cursor: pointer;
    transition: background-color 0.2s;
}

.format-tag:hover {
    background: #dee2e6;
}

.format-tag.selected {
    background: #3498db;
    color: white;
}

.advanced-settings {
    background: #f8f9fa;
    border-radius: 8px;
    padding: 1.5rem;
    margin-top: 1rem;
}

.btn-create {
    background: linear-gradient(135deg, #3498db, #2980b9);
    border: none;
    padding: 0.75rem 2rem;
    font-size: 1.1rem;
    font-weight: 600;
    border-radius: 8px;
    transition: all 0.3s ease;
}

.btn-create:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(52, 152, 219, 0.3);
    background: linear-gradient(135deg, #2980b9, #3498db);
}
</style>

<div class="page-content">
    <div class="container-fluid py-4">
        <div class="row">
            <div class="col-12">
                <!-- Page Header -->
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <div>
                        <h1 class="h3 mb-0">Create New Assignment</h1>
                        <p class="text-muted mb-0">Design professional assignments for real-world applications</p>
                    </div>
                    <a href="assignments.php" class="btn btn-outline-secondary">
                        <i class="fas fa-arrow-left me-2"></i>Back to Assignments
                    </a>
                </div>

                <!-- Success/Error Messages -->
                <?php if (!empty($message)): ?>
                    <?= $message ?>
                <?php endif; ?>

                <!-- Assignment Creation Form -->
                <form method="POST" enctype="multipart/form-data" class="assignment-form">
                    
                    <!-- Basic Information Section -->
                    <div class="form-section">
                        <h3 class="section-title">
                            <i class="fas fa-info-circle"></i>
                            Basic Information
                        </h3>
                        
                        <div class="row">
                            <div class="col-md-8">
                                <div class="mb-3">
                                    <label for="title" class="form-label">Assignment Title *</label>
                                    <input type="text" class="form-control" id="title" name="title" 
                                           value="<?= htmlspecialchars($_POST['title'] ?? '') ?>" 
                                           placeholder="Enter a clear, descriptive title for your assignment" required>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="mb-3">
                                    <label for="course_id" class="form-label">Course *</label>
                                    <select class="form-select" id="course_id" name="course_id" required>
                                        <option value="">Select Course</option>
                                        <?php while ($course = $courses_result->fetch_assoc()): ?>
                                            <option value="<?= $course['course_id'] ?>" 
                                                    <?= (isset($_POST['course_id']) && $_POST['course_id'] == $course['course_id']) ? 'selected' : '' ?>>
                                                <?= htmlspecialchars($course['title']) ?>
                                            </option>
                                        <?php endwhile; ?>
                                    </select>
                                </div>
                            </div>
                        </div>

                        <div class="mb-3">
                            <label for="description" class="form-label">Assignment Description *</label>
                            <textarea class="form-control" id="description" name="description" rows="4" 
                                      placeholder="Provide a comprehensive description of what students need to accomplish..." required><?= htmlspecialchars($_POST['description'] ?? '') ?></textarea>
                        </div>

                        <div class="mb-3">
                            <label for="instructions" class="form-label">Detailed Instructions</label>
                            <textarea class="form-control" id="instructions" name="instructions" rows="6" 
                                      placeholder="Provide step-by-step instructions, requirements, and expectations..."><?= htmlspecialchars($_POST['instructions'] ?? '') ?></textarea>
                        </div>
                    </div>

                    <!-- Submission Type Section -->
                    <div class="form-section">
                        <h3 class="section-title">
                            <i class="fas fa-upload"></i>
                            Submission Requirements
                        </h3>
                        
                        <div class="mb-3">
                            <label class="form-label">How should students submit their work? * <small class="text-muted">(Select multiple options if needed)</small></label>
                            <div class="submission-type-cards">
                                <div class="submission-card" data-type="document">
                                    <i class="fas fa-file-word"></i>
                                    <div><strong>Document</strong></div>
                                    <small>Word docs, PDFs, text files</small>
                                    <div class="selection-indicator"><i class="fas fa-check"></i></div>
                                </div>
                                <div class="submission-card" data-type="excel">
                                    <i class="fas fa-file-excel"></i>
                                    <div><strong>Spreadsheet</strong></div>
                                    <small>Excel, CSV files</small>
                                    <div class="selection-indicator"><i class="fas fa-check"></i></div>
                                </div>
                                <div class="submission-card" data-type="presentation">
                                    <i class="fas fa-file-powerpoint"></i>
                                    <div><strong>Presentation</strong></div>
                                    <small>PowerPoint, PDF slides</small>
                                    <div class="selection-indicator"><i class="fas fa-check"></i></div>
                                </div>
                                <div class="submission-card" data-type="video">
                                    <i class="fas fa-video"></i>
                                    <div><strong>Video</strong></div>
                                    <small>MP4, AVI, MOV files</small>
                                    <div class="selection-indicator"><i class="fas fa-check"></i></div>
                                </div>
                                <div class="submission-card" data-type="image">
                                    <i class="fas fa-image"></i>
                                    <div><strong>Images</strong></div>
                                    <small>Photos, diagrams, screenshots</small>
                                    <div class="selection-indicator"><i class="fas fa-check"></i></div>
                                </div>
                                <div class="submission-card" data-type="code">
                                    <i class="fas fa-code"></i>
                                    <div><strong>Code Project</strong></div>
                                    <small>Source code, ZIP archives</small>
                                    <div class="selection-indicator"><i class="fas fa-check"></i></div>
                                </div>
                                <div class="submission-card" data-type="multiple_files">
                                    <i class="fas fa-folder"></i>
                                    <div><strong>Multiple Files</strong></div>
                                    <small>Portfolio, project bundle</small>
                                    <div class="selection-indicator"><i class="fas fa-check"></i></div>
                                </div>
                                <div class="submission-card" data-type="text_entry">
                                    <i class="fas fa-keyboard"></i>
                                    <div><strong>Text Entry</strong></div>
                                    <small>Type directly in browser</small>
                                    <div class="selection-indicator"><i class="fas fa-check"></i></div>
                                </div>
                            </div>
                            <div id="selected-types-display" class="mt-2" style="display: none;">
                                <small class="text-muted">Selected: </small>
                                <div id="selected-types-list" class="d-inline"></div>
                            </div>
                            <div id="submission-types-inputs">
                                <!-- Hidden inputs for submission types will be created here by JavaScript -->
                            </div>
                        </div>

                        <div id="format-selection" class="mb-3" style="display: none;">
                            <label class="form-label">Allowed File Formats</label>
                            <div class="format-tags" id="format-tags-container"></div>
                            <input type="hidden" id="allowed_formats" name="allowed_formats[]">
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="max_file_size" class="form-label">Maximum File Size (MB)</label>
                                    <input type="number" class="form-control" id="max_file_size" name="max_file_size" 
                                           value="<?= htmlspecialchars($_POST['max_file_size'] ?? '50') ?>" min="1" max="500">
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="points" class="form-label">Total Points *</label>
                                    <input type="number" class="form-control" id="points" name="points" 
                                           value="<?= htmlspecialchars($_POST['points'] ?? '100') ?>" min="1" max="1000" required>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Timing Section -->
                    <div class="form-section">
                        <h3 class="section-title">
                            <i class="fas fa-clock"></i>
                            Assignment Timing
                        </h3>
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="start_date" class="form-label">Available From</label>
                                    <input type="datetime-local" class="form-control" id="start_date" name="start_date" 
                                           value="<?= htmlspecialchars($_POST['start_date'] ?? '') ?>">
                                    <small class="form-text text-muted">Leave empty to make available immediately</small>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="due_date" class="form-label">Due Date *</label>
                                    <input type="datetime-local" class="form-control" id="due_date" name="due_date" 
                                           value="<?= htmlspecialchars($_POST['due_date'] ?? '') ?>" required>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-check mb-3">
                                    <input class="form-check-input" type="checkbox" id="allow_late" name="allow_late" 
                                           <?= isset($_POST['allow_late']) ? 'checked' : '' ?>>
                                    <label class="form-check-label" for="allow_late">
                                        Allow Late Submissions
                                    </label>
                                </div>
                            </div>
                            <div class="col-md-6" id="late-penalty-section" style="display: none;">
                                <div class="mb-3">
                                    <label for="late_penalty" class="form-label">Late Penalty (% per day)</label>
                                    <input type="number" class="form-control" id="late_penalty" name="late_penalty" 
                                           value="<?= htmlspecialchars($_POST['late_penalty'] ?? '5') ?>" min="0" max="100" step="0.1">
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Advanced Settings -->
                    <div class="form-section">
                        <h3 class="section-title">
                            <i class="fas fa-cogs"></i>
                            Advanced Settings
                        </h3>
                        
                        <div class="advanced-settings">
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="form-check mb-3">
                                        <input class="form-check-input" type="checkbox" id="group_assignment" name="group_assignment" 
                                               <?= isset($_POST['group_assignment']) ? 'checked' : '' ?>>
                                        <label class="form-check-label" for="group_assignment">
                                            <strong>Group Assignment</strong>
                                        </label>
                                    </div>
                                </div>
                                <div class="col-md-6" id="group-size-section" style="display: none;">
                                    <div class="mb-3">
                                        <label for="max_group_size" class="form-label">Maximum Group Size</label>
                                        <input type="number" class="form-control" id="max_group_size" name="max_group_size" 
                                               value="<?= htmlspecialchars($_POST['max_group_size'] ?? '4') ?>" min="2" max="10">
                                    </div>
                                </div>
                            </div>

                            <div class="mb-3">
                                <label for="grading_rubric" class="form-label">Grading Rubric</label>
                                <textarea class="form-control" id="grading_rubric" name="grading_rubric" rows="5" 
                                          placeholder="Define your grading criteria and expectations..."><?= htmlspecialchars($_POST['grading_rubric'] ?? '') ?></textarea>
                            </div>

                            <div class="mb-3">
                                <label class="form-label">Attach Reference File (Optional)</label>
                                <input type="file" class="form-control" id="assignment_file" name="assignment_file" 
                                       accept=".pdf,.doc,.docx,.ppt,.pptx,.xlsx,.xls">
                                <small class="form-text text-muted">Upload a reference document, template, or sample file</small>
                            </div>

                            <div class="form-check mb-3">
                                <input class="form-check-input" type="checkbox" id="contributes_to_final_grade" name="contributes_to_final_grade" 
                                       <?= isset($_POST['contributes_to_final_grade']) ? 'checked' : 'checked' ?>>
                                <label class="form-check-label" for="contributes_to_final_grade">
                                    <strong>🎓 Contributes to Final Grade/Certificate</strong>
                                    <small class="d-block text-muted">This assignment will count towards the student's final grade and certificate calculation</small>
                                </label>
                            </div>
                            
                            <div class="alert alert-info mb-3">
                                <i class="fas fa-info-circle me-2"></i>
                                <strong>Grade Contribution:</strong>
                                <ul class="mb-0 mt-2">
                                    <li><strong>✅ Checked:</strong> Formal assessment - counts towards final certificate</li>
                                    <li><strong>❌ Unchecked:</strong> Practice assignment - for learning only, won't affect final grade</li>
                                </ul>
                            </div>

                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" id="is_published" name="is_published" 
                                       <?= isset($_POST['is_published']) ? 'checked' : '' ?>>
                                <label class="form-check-label" for="is_published">
                                    <strong>📢 Publish Immediately</strong>
                                    <small class="d-block text-muted">Students will be able to see and submit to this assignment right away</small>
                                </label>
                            </div>
                        </div>
                    </div>

                    <!-- Submit Button -->
                    <div class="text-center">
                        <button type="submit" class="btn btn-primary btn-create">
                            <i class="fas fa-plus me-2"></i>Create Assignment
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Multiple submission type selection
    const submissionCards = document.querySelectorAll('.submission-card');
    const formatSelection = document.getElementById('format-selection');
    const formatContainer = document.getElementById('format-tags-container');
    const selectedTypesDisplay = document.getElementById('selected-types-display');
    const selectedTypesList = document.getElementById('selected-types-list');
    
    let selectedTypes = [];
    
    // File format options for each type
    const formatOptions = {
        document: ['pdf', 'doc', 'docx', 'txt', 'rtf'],
        excel: ['xlsx', 'xls', 'csv', 'ods'],
        presentation: ['ppt', 'pptx', 'pdf', 'odp'],
        image: ['jpg', 'jpeg', 'png', 'gif', 'bmp', 'svg', 'webp'],
        video: ['mp4', 'avi', 'mov', 'mkv', 'wmv', 'flv'],
        code: ['zip', 'rar', '7z', 'py', 'js', 'html', 'css', 'php', 'java', 'cpp'],
        multiple_files: ['zip', 'rar', '7z', 'tar'],
        text_entry: []
    };
    
    submissionCards.forEach(card => {
        card.addEventListener('click', function() {
            const type = this.dataset.type;
            
            // Toggle selection
            if (this.classList.contains('selected')) {
                // Remove from selection
                this.classList.remove('selected');
                selectedTypes = selectedTypes.filter(t => t !== type);
            } else {
                // Add to selection
                this.classList.add('selected');
                selectedTypes.push(type);
            }
            
            // Update hidden input
            updateSubmissionTypesInput();
            
            // Update display
            updateSelectedTypesDisplay();
            
            // Update format options
            if (selectedTypes.length > 0) {
                const hasTextEntry = selectedTypes.includes('text_entry');
                const hasFileTypes = selectedTypes.some(t => t !== 'text_entry');
                
                if (hasFileTypes) {
                    formatSelection.style.display = 'block';
                    displayCombinedFormatOptions();
                } else if (hasTextEntry && selectedTypes.length === 1) {
                    formatSelection.style.display = 'none';
                }
            } else {
                formatSelection.style.display = 'none';
            }
        });
    });
    
    function updateSubmissionTypesInput() {
        // Get the container for submission type inputs
        const container = document.getElementById('submission-types-inputs');
        
        // Clear existing inputs
        container.innerHTML = '';
        
        // Create hidden inputs for each selected type
        selectedTypes.forEach(type => {
            const input = document.createElement('input');
            input.type = 'hidden';
            input.name = 'submission_types[]';
            input.value = type;
            container.appendChild(input);
        });
        
        console.log('Updated submission types:', selectedTypes);
    }
    
    function updateSelectedTypesDisplay() {
        if (selectedTypes.length > 0) {
            selectedTypesDisplay.style.display = 'block';
            const typeNames = selectedTypes.map(type => {
                const card = document.querySelector(`[data-type="${type}"]`);
                return card.querySelector('strong').textContent;
            });
            selectedTypesList.textContent = typeNames.join(', ');
        } else {
            selectedTypesDisplay.style.display = 'none';
        }
    }
    
    function displayCombinedFormatOptions() {
        const allFormats = new Set();
        
        selectedTypes.forEach(type => {
            if (type !== 'text_entry') {
                const formats = formatOptions[type] || [];
                formats.forEach(format => allFormats.add(format));
            }
        });
        
        formatContainer.innerHTML = '';
        
        Array.from(allFormats).sort().forEach(format => {
            const tag = document.createElement('span');
            tag.className = 'format-tag selected';
            tag.textContent = format.toUpperCase();
            tag.dataset.format = format;
            
            tag.addEventListener('click', function() {
                this.classList.toggle('selected');
                updateAllowedFormats();
            });
            
            formatContainer.appendChild(tag);
        });
        
        updateAllowedFormats();
    }
    
    function displayFormatOptions(type) {
        const formats = formatOptions[type] || [];
        formatContainer.innerHTML = '';
        
        formats.forEach(format => {
            const tag = document.createElement('span');
            tag.className = 'format-tag selected';
            tag.textContent = format.toUpperCase();
            tag.dataset.format = format;
            
            tag.addEventListener('click', function() {
                this.classList.toggle('selected');
                updateAllowedFormats();
            });
            
            formatContainer.appendChild(tag);
        });
        
        updateAllowedFormats();
    }
    
    function updateAllowedFormats() {
        const selectedFormats = Array.from(document.querySelectorAll('.format-tag.selected'))
            .map(tag => tag.dataset.format);
        document.getElementById('allowed_formats').value = JSON.stringify(selectedFormats);
    }
    
    // Late submission toggle
    const allowLateCheckbox = document.getElementById('allow_late');
    const latePenaltySection = document.getElementById('late-penalty-section');
    
    allowLateCheckbox.addEventListener('change', function() {
        latePenaltySection.style.display = this.checked ? 'block' : 'none';
    });
    
    // Group assignment toggle
    const groupAssignmentCheckbox = document.getElementById('group_assignment');
    const groupSizeSection = document.getElementById('group-size-section');
    
    groupAssignmentCheckbox.addEventListener('change', function() {
        groupSizeSection.style.display = this.checked ? 'block' : 'none';
    });
    
    // Initialize based on current values
    if (allowLateCheckbox.checked) {
        latePenaltySection.style.display = 'block';
    }
    
    if (groupAssignmentCheckbox.checked) {
        groupSizeSection.style.display = 'block';
    }
    
    // Form submission validation
    const form = document.querySelector('form');
    form.addEventListener('submit', function(e) {
        console.log('Form submission started');
        console.log('Selected types:', selectedTypes);
        
        // Check if at least one submission type is selected
        if (selectedTypes.length === 0) {
            e.preventDefault();
            alert('Please select at least one submission type before submitting.');
            
            // Scroll to submission types section
            document.querySelector('.submission-type-cards').scrollIntoView({
                behavior: 'smooth',
                block: 'center'
            });
            
            return false;
        }
        
        // Ensure hidden inputs are created
        updateSubmissionTypesInput();
        
        // Debug: Show what will be submitted
        const formData = new FormData(form);
        const submissionTypes = formData.getAll('submission_types[]');
        console.log('Submission types being sent:', submissionTypes);
        
        if (submissionTypes.length === 0) {
            e.preventDefault();
            alert('Error: Submission types not properly set. Please try selecting them again.');
            return false;
        }
    });
    
    // Initialize form based on any existing values
    // (This section can be expanded later if needed for edit functionality)
});
</script>

<?php include __DIR__ . '/components/footer.php'; ?>
