<?php
// Define current page for sidebar highlighting
$current_page = 'create-course.php';

// Include the centralized teacher layout
require_once __DIR__ . '/components/teacher_layout.php';

// Include additional required files
require_once __DIR__ . '/../includes/functions.php';

// Error handling and reporting during development
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Fatal error handler to capture and display errors
function create_course_shutdown_handler() {
    $error = error_get_last();
    if ($error !== NULL && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
        echo '<div class="alert alert-danger">';
        echo '<h4>Fatal Error Occurred:</h4>';
        echo '<p>' . $error['message'] . '</p>';
        echo '<p>File: ' . $error['file'] . ' on line ' . $error['line'] . '</p>';
        echo '</div>';
    }
}
register_shutdown_function('create_course_shutdown_handler');

// Get teacher data with robust fallback for emergency/test accounts
function create_course_get_teacher($conn, $user_id) {
    // For emergency login user_id (999)
    if ($user_id == 999) {
        return [
            'user_id' => 999,
            'username' => 'teacher',
            'first_name' => 'Emergency',
            'last_name' => 'Teacher',
            'email' => 'emergency@example.com',
            'bio' => 'Emergency teacher account',
            'qualifications' => 'Emergency credentials',
            'expertise' => 'System testing',
            'status' => 'active'
        ];
    }
    
    // Ensure we have a valid database connection
    if (!$conn || $conn->connect_error) {
        error_log('Database connection not available in create_course_get_teacher');
        return [
            'user_id' => $user_id,
            'username' => 'db_error_user',
            'first_name' => 'Database', 
            'last_name' => 'Error',
            'email' => 'error@example.com',
            'bio' => 'Database connection error',
            'qualifications' => 'N/A',
            'expertise' => 'N/A',
            'status' => 'active'
        ];
    }
    
    try {
        // First check if the user exists
        $user_check = $conn->query("SELECT * FROM users WHERE user_id = {$user_id}");
        if (!$user_check || $user_check->num_rows == 0) {
            error_log("User with ID {$user_id} not found in create_course_get_teacher");
            // Return fallback data
            return [
                'user_id' => $user_id,
                'username' => 'unknown',
                'first_name' => 'Unknown',
                'last_name' => 'Teacher',
                'email' => 'unknown@example.com'
            ];
        }
        
        $stmt = $conn->prepare("SELECT u.*, tp.bio, tp.qualifications, tp.expertise, tp.status 
                               FROM users u 
                               LEFT JOIN teacher_profiles tp ON u.user_id = tp.teacher_id 
                               WHERE u.user_id = ?");
                              
        // Check if prepare was successful
        if ($stmt === false) {
            error_log("Prepare failed: (" . $conn->errno . ") " . $conn->error);
            // Fall back to a simpler query without join
            $stmt = $conn->prepare("SELECT * FROM users WHERE user_id = ?");
            if ($stmt === false) {
                error_log("Even simple prepare failed: (" . $conn->errno . ") " . $conn->error);
                // Return data from the earlier query since prepare failed
                return $user_check->fetch_assoc();
            }
        }
        
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result && $result->num_rows > 0) {
            $teacher = $result->fetch_assoc();
            
            // Add default values for missing fields
            $teacher['bio'] = $teacher['bio'] ?? 'No biography available';
            $teacher['qualifications'] = $teacher['qualifications'] ?? 'Not specified';
            $teacher['expertise'] = $teacher['expertise'] ?? 'Not specified';
            $teacher['status'] = $teacher['status'] ?? 'active';
            
            return $teacher;
        }
    } catch (Exception $e) {
        error_log("Error getting teacher: " . $e->getMessage());
    }
    
    // Fallback data if query fails
    return [
        'user_id' => $user_id,
        'username' => 'unknown',
        'first_name' => 'Unknown',
        'last_name' => 'Teacher',
        'email' => 'unknown@example.com',
        'bio' => 'No data available',
        'qualifications' => 'Not available',
        'expertise' => 'Not available',
        'status' => 'active'
    ];
}

$teacher_id = $_SESSION['user_id'];
$teacher = create_course_get_teacher($conn, $teacher_id);

// Check teacher identity verification status - REQUIRED FOR COURSE CREATION
$verification_query = "SELECT identity_verification_status, identity_document_path, identity_uploaded_at, identity_rejection_reason FROM users WHERE user_id = ?";
$verification_stmt = $conn->prepare($verification_query);
$verification_stmt->bind_param("i", $teacher_id);
$verification_stmt->execute();
$verification_result = $verification_stmt->get_result();
$verification_data = $verification_result->fetch_assoc() ?: [];

$identity_status = $verification_data['identity_verification_status'] ?? 'pending';
$is_verified = ($identity_status === 'verified');

// Get course and video counts with error handling for nav badges
function robust_count_teacher_courses($conn, $teacher_id) {
    if ($teacher_id == 999) {
        return 3; // Demo count for emergency account
    }
    
    if (!$conn || $conn->connect_error) {
        return 0;
    }
    
    try {
        // Check if courses table exists
        $check_table = $conn->query("SHOW TABLES LIKE 'courses'");
        if ($check_table && $check_table->num_rows == 0) {
            error_log("courses table does not exist in count_teacher_courses");
            return 0;
        }
        
        // Prepare statement
        $stmt = $conn->prepare("SELECT COUNT(*) as count FROM courses WHERE teacher_id = ?");
        
        // Check if prepare was successful
        if ($stmt === false) {
            error_log("Course count prepare failed: (" . $conn->errno . ") " . $conn->error);
            return 0;
        }
        
        $stmt->bind_param("i", $teacher_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result && $row = $result->fetch_assoc()) {
            return $row['count'];
        }
    } catch (Exception $e) {
        error_log("Error counting courses: " . $e->getMessage());
    }
    
    return 0;
}

function robust_count_teacher_videos($conn, $teacher_id) {
    if ($teacher_id == 999) {
        return 5; // Demo count for emergency account
    }
    
    if (!$conn || $conn->connect_error) {
        return 0;
    }
    
    try {
        // Check if videos table exists
        $check_table = $conn->query("SHOW TABLES LIKE 'videos'");
        if ($check_table && $check_table->num_rows == 0) {
            error_log("videos table does not exist in count_teacher_videos");
            return 0;
        }
        
        // Prepare statement
        $stmt = $conn->prepare("SELECT COUNT(*) as count FROM videos WHERE teacher_id = ?");
        
        // Check if prepare was successful
        if ($stmt === false) {
            error_log("Video count prepare failed: (" . $conn->errno . ") " . $conn->error);
            return 0;
        }
        
        $stmt->bind_param("i", $teacher_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result && $row = $result->fetch_assoc()) {
            return $row['count'];
        }
    } catch (Exception $e) {
        error_log("Error counting videos: " . $e->getMessage());
    }
    
    return 0;
}

// Get course and video counts
$course_count = robust_count_teacher_courses($conn, $teacher_id);
$video_count = robust_count_teacher_videos($conn, $teacher_id);

// Set current page for sidebar highlighting
$current_page = 'create-course.php';

// Page specific CSS for create course page
?>
<style>
    /* Page-specific styles for create course */
    }

    * {
        font-family: 'Inter', 'Poppins', sans-serif !important;
    }

    .container-fluid {
        padding: 0 30px !important;
    }

    /* Page Header */
    h1.h3 {
        font-size: 28px !important;
        font-weight: 700 !important;
        color: #1f2937 !important;
        margin-bottom: 0 !important;
        letter-spacing: -0.025em !important;
    }

    /* Modern Create Course Page Styling - Panadite Colors */
    .card {
        border: none !important;
        border-radius: 16px !important;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.08) !important;
        background: #ffffff !important;
        transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1) !important;
        overflow: hidden !important;
    }

    .card:hover {
        box-shadow: 0 8px 25px rgba(0, 0, 0, 0.12) !important;
        transform: translateY(-2px) !important;
    }

    .card-header {
        background: linear-gradient(135deg, #f8faff, #e1e8ff) !important;
        border-bottom: 1px solid #e1e8ff !important;
        padding: 20px 24px !important;
        border-radius: 16px 16px 0 0 !important;
    }

    .card-title {
        font-size: 18px !important;
        font-weight: 600 !important;
        color: #1f2937 !important;
        display: flex !important;
        align-items: center !important;
        gap: 8px !important;
    }

    .card-title i {
        font-size: 20px !important;
        color: #4361ee !important;
    }

    .card-body {
        padding: 24px !important;
    }

    /* Form Styling */
    .form-label {
        font-weight: 600 !important;
        color: #374151 !important;
        margin-bottom: 8px !important;
        font-size: 14px !important;
    }

    .form-control, .form-select {
        border: 2px solid #e5e7eb !important;
        border-radius: 12px !important;
        padding: 12px 16px !important;
        font-size: 14px !important;
        transition: all 0.2s ease !important;
        background: #ffffff !important;
    }

    .form-control:focus, .form-select:focus {
        border-color: #4361ee !important;
        box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.1) !important;
        outline: none !important;
    }

    .form-control::placeholder {
        color: #9ca3af !important;
    }

    /* Textarea specific */
    textarea.form-control {
        min-height: 120px !important;
        resize: vertical !important;
    }

    /* File input styling */
    .form-control[type="file"] {
        padding: 10px !important;
    }

    /* Button Styling */
    .btn {
        border-radius: 12px !important;
        padding: 12px 24px !important;
        font-weight: 600 !important;
        font-size: 14px !important;
        transition: all 0.2s ease !important;
        border: none !important;
        box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1) !important;
    }

    .btn-primary {
        background: linear-gradient(135deg, #4361ee, #5a72f0) !important;
        color: white !important;
    }

    .btn-primary:hover {
        background: linear-gradient(135deg, #3b4de8, #4361ee) !important;
        transform: translateY(-1px) !important;
        box-shadow: 0 4px 12px rgba(67, 97, 238, 0.3) !important;
    }

    .btn-secondary {
        background: #f3f4f6 !important;
        color: #374151 !important;
    }

    .btn-secondary:hover {
        background: #e5e7eb !important;
        color: #1f2937 !important;
    }

    /* Tips Card Styling */
    .list-unstyled li {
        padding: 8px 0 !important;
        display: flex !important;
        align-items: center !important;
        font-size: 14px !important;
    }

    .list-unstyled li i {
        margin-right: 12px !important;
        font-size: 16px !important;
    }

    .text-success {
        color: #10b981 !important;
    }

    .text-warning {
        color: #f59e0b !important;
    }

    .text-info {
        color: #3b82f6 !important;
    }

    /* Status Badge */
    .badge {
        padding: 6px 12px !important;
        border-radius: 8px !important;
        font-weight: 600 !important;
        font-size: 12px !important;
    }

    .bg-secondary {
        background: #6b7280 !important;
    }

    /* Loading overlay */
    .loading-overlay {
        background: rgba(255, 255, 255, 0.95) !important;
        backdrop-filter: blur(10px) !important;
        border-radius: 16px !important;
    }

    .spinner-border {
        color: #4361ee !important;
    }

    /* Thumbnail Preview */
    #thumbnailPreview img {
        max-width: 200px !important; 
        max-height: 150px !important;
        margin-top: 16px !important;
        border-radius: 12px !important;
        box-shadow: 0 4px 8px rgba(0, 0, 0, 0.1) !important;
    }

    /* Progress Bar */
    .progress {
        height: 8px !important;
        border-radius: 6px !important;
        background: #f3f4f6 !important;
    }

    .progress-bar {
        background: linear-gradient(90deg, #4361ee, #5a72f0) !important;
        border-radius: 6px !important;
    }

    /* Row spacing */
    .row.mb-3 {
        margin-bottom: 24px !important;
    }

    /* Invalid feedback */
    .invalid-feedback {
        color: #ef4444 !important;
        font-size: 12px !important;
        font-weight: 500 !important;
    }

    .form-control.is-invalid, .form-select.is-invalid {
        border-color: #ef4444 !important;
        box-shadow: 0 0 0 3px rgba(239, 68, 68, 0.1) !important;
    }

    /* Section spacing */
    .mb-4 {
        margin-bottom: 32px !important;
    }

    /* Responsive adjustments */
    @media (max-width: 768px) {
        .container-fluid {
            padding: 0 20px !important;
        }
        
        .card-body {
            padding: 20px !important;
        }
        
        .card-header {
            padding: 16px 20px !important;
        }
        
        h1.h3 {
            font-size: 24px !important;
        }
    }

    /* Custom checkbox/radio styling */
    .form-check-input:checked {
        background-color: #4361ee !important;
        border-color: #4361ee !important;
    }

    /* Success messages */
    .alert {
        border: none !important;
        border-radius: 12px !important;
        padding: 16px 20px !important;
        font-weight: 500 !important;
    }

    .alert-success {
        background: linear-gradient(135deg, #ecfdf5, #d1fae5) !important;
        color: #065f46 !important;
    }

    .alert-danger {
        background: linear-gradient(135deg, #fef2f2, #fecaca) !important;
        color: #991b1b !important;
    }

    /* Timeline Styling */
    .timeline {
        position: relative;
        padding-left: 30px;
    }

    .timeline::before {
        content: '';
        position: absolute;
        left: 15px;
        top: 10px;
        bottom: 10px;
        width: 2px;
        background: #e9ecef;
    }

    .timeline-item {
        position: relative;
        margin-bottom: 25px;
    }

    .timeline-item.active .timeline-marker {
        background: #007bff !important;
        box-shadow: 0 0 0 4px rgba(0, 123, 255, 0.2);
    }

    .timeline-item.completed .timeline-marker {
        background: #28a745 !important;
    }

    .timeline-marker {
        position: absolute;
        left: -22px;
        top: 5px;
        width: 12px;
        height: 12px;
        border-radius: 50%;
        border: 2px solid #fff;
        z-index: 1;
    }

    .timeline-content h6 {
        margin-bottom: 5px;
        font-weight: 600;
    }

    .timeline-content p {
        margin-bottom: 0;
        font-size: 0.875rem;
    }

    /* Sidebar cards styling */
    .col-lg-4 .card {
        border: none;
        box-shadow: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
        transition: box-shadow 0.15s ease-in-out;
    }

    .col-lg-4 .card:hover {
        box-shadow: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    }

    .col-lg-4 .card-header {
        background: linear-gradient(135deg, #f8f9fa, #e9ecef);
        border-bottom: 1px solid #dee2e6;
    }
/*Course Creation Tips  */
    /* Simple Header Styles */
    .simple-header {
        text-align: left;
        margin-bottom: 30px;
    }

    .simple-heading {
        font-size: 2rem;
        font-weight: 700;
        color: #000000;
        margin-bottom: 8px;
        line-height: 1.2;
    }

    .simple-subheading {
        font-size: 1rem;
        font-weight: 400;
        color: #000000;
        margin: 0;
        line-height: 1.5;
        opacity: 0.8;
    }
</style>
<!-- Page title and content -->
<div class="container-fluid">
    <!-- Simple Page Header -->
    <div class="simple-header mb-4">
        <h1 class="simple-heading">Create Course</h1>
        <p class="simple-subheading">Build and share your educational content with students worldwide.</p>
    </div>
    
    <div class="row">
        <!-- Main Content Column -->
        <div class="col-lg-8">
            <div class="card shadow-sm">
                <div class="card-header">
                    <h6 class="card-title mb-0"><i class="fas fa-book-open me-2 text-primary"></i>Create a New Course</h6>
                </div>
                <div class="card-body">
                    <?php if (!$is_verified): ?>
                                <!-- IDENTITY VERIFICATION REQUIRED -->
                                <div class="text-center py-5">
                                    <?php if ($identity_status === 'pending' && !empty($verification_data['identity_document_path'])): ?>
                                        <!-- Document uploaded, waiting for approval -->
                                        <div class="mb-4">
                                            <svg xmlns="http://www.w3.org/2000/svg" width="64" height="64" fill="currentColor" class="bi bi-clock text-warning mb-3" viewBox="0 0 16 16">
                                                <path d="M8 3.5a.5.5 0 0 0-1 0V9a.5.5 0 0 0 .252.434l3.5 2a.5.5 0 0 0 .496-.868L8 8.71V3.5z"/>
                                                <path d="M8 16A8 8 0 1 0 8 0a8 8 0 0 0 0 16zm7-8A7 7 0 1 1 1 8a7 7 0 0 1 14 0z"/>
                                            </svg>
                                        </div>
                                        <h4 class="text-warning mb-3">🕐 Verification In Progress</h4>
                                        <p class="text-muted mb-4">
                                            Your identity document was submitted on <strong><?= date('M j, Y', strtotime($verification_data['identity_uploaded_at'])) ?></strong><br>
                                            Our team is reviewing your documents. This usually takes 1-3 business days.
                                        </p>
                                        <div class="alert alert-info">
                                            <i class="bi bi-info-circle me-2"></i>
                                            <strong>Course creation will be enabled once your identity is verified.</strong><br>
                                            You'll receive an email notification when verification is complete.
                                        </div>
                                    <?php elseif ($identity_status === 'rejected'): ?>
                                        <!-- Document rejected -->
                                        <div class="mb-4">
                                            <svg xmlns="http://www.w3.org/2000/svg" width="64" height="64" fill="currentColor" class="bi bi-x-circle text-danger mb-3" viewBox="0 0 16 16">
                                                <path d="M8 15A7 7 0 1 1 8 1a7 7 0 0 1 0 14zm0 1A8 8 0 1 0 8 0a8 8 0 0 0 0 16z"/>
                                                <path d="M4.646 4.646a.5.5 0 0 1 .708 0L8 7.293l2.646-2.647a.5.5 0 0 1 .708.708L8.707 8l2.647 2.646a.5.5 0 0 1-.708.708L8 8.707l-2.646 2.647a.5.5 0 0 1-.708-.708L7.293 8 4.646 5.354a.5.5 0 0 1 0-.708z"/>
                                            </svg>
                                        </div>
                                        <h4 class="text-danger mb-3">❌ Verification Rejected</h4>
                                        <div class="alert alert-danger mb-4">
                                            <strong>Rejection Reason:</strong><br>
                                            <?= htmlspecialchars($verification_data['identity_rejection_reason'] ?? 'No reason provided') ?>
                                        </div>
                                        <p class="text-muted mb-4">
                                            Please upload a new identity document to proceed with course creation.
                                        </p>
                                        <a href="profile.php#identity-verification" class="btn btn-primary">
                                            <i class="bi bi-upload me-2"></i>Upload New Document
                                        </a>
                                    <?php else: ?>
                                        <!-- No document uploaded yet -->
                                        <div class="mb-4">
                                            <svg xmlns="http://www.w3.org/2000/svg" width="64" height="64" fill="currentColor" class="bi bi-shield-exclamation text-warning mb-3" viewBox="0 0 16 16">
                                                <path d="M5.338 1.59a61.44 61.44 0 0 0-2.837.856.481.481 0 0 0-.328.39c-.554 4.157.726 7.19 2.253 9.188a10.725 10.725 0 0 0 2.287 2.233c.346.244.652.42.893.533.12.057.218.095.293.118a.55.55 0 0 0 .101.025.615.615 0 0 0 .1-.025c.076-.023.174-.061.294-.118.24-.113.547-.29.893-.533a10.726 10.726 0 0 0 2.287-2.233c1.527-1.997 2.807-5.031 2.253-9.188a.48.48 0 0 0-.328-.39c-.651-.213-1.75-.56-2.837-.855C9.552 1.29 8.531 1.067 8 1.067c-.53 0-1.552.223-2.662.524zM5.072.56C6.157.265 7.31 0 8 0s1.843.265 2.928.56c1.11.3 2.229.655 2.887.87a1.54 1.54 0 0 1 1.044 1.262c.596 4.477-.787 7.795-2.465 9.99a11.775 11.775 0 0 1-2.517 2.453 7.159 7.159 0 0 1-1.048.625c-.28.132-.581.24-.829.24s-.548-.108-.829-.24a7.158 7.158 0 0 1-1.048-.625 11.777 11.777 0 0 1-2.517-2.453C1.928 10.487.545 7.169 1.141 2.692A1.54 1.54 0 0 1 2.185 1.43 62.456 62.456 0 0 1 5.072.56z"/>
                                                <path d="M7.001 11a1 1 0 1 1 2 0 1 1 0 0 1-2 0zM7.1 4.995a.905.905 0 1 1 1.8 0l-.35 3.507a.552.552 0 0 1-1.1 0L7.1 4.995z"/>
                                            </svg>
                                        </div>
                                        <h4 class="text-warning mb-3">🆔 Identity Verification Required</h4>
                                        <p class="text-muted mb-4">
                                            To maintain trust and security on our platform, teachers must verify their identity before creating courses.<br>
                                            Please upload a government-issued ID document to get started.
                                        </p>
                                        <div class="alert alert-info mb-4">
                                            <i class="bi bi-shield-check me-2"></i>
                                            <strong>Accepted Documents:</strong> National ID, Passport, Driver's License
                                        </div>
                                        <a href="profile.php#identity-verification" class="btn btn-primary btn-lg">
                                            <i class="bi bi-upload me-2"></i>Upload Identity Document
                                        </a>
                                    <?php endif; ?>
                                </div>
                    </div>
                </div>
            </div>
        </div><!-- Verification Process--->


    </div>
    
</div><!-- End container-fluid -->
        </div><!-- End content-wrapper -->
    </div><!-- End dashboard-container -->
    
    <!-- JavaScript Libraries -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/toastify-js"></script>
    
    <script>
        // Safe element access function for this page
        function safeGetElement(id) {
            return document.getElementById(id) || null;
        }
    </script>
    
<?php require_once 'components/teacher_layout_end.php'; ?>
<?php else: ?>
                <!-- VERIFIED TEACHER CONTENT STARTS HERE -->
                                <!-- VERIFIED TEACHER - SHOW COURSE CREATION FORM -->
                                <div class="alert alert-success mb-4">
                                    <i class="bi bi-shield-check-fill me-2"></i>
                                    <strong>Identity Verified!</strong> You can now create and sell courses on our platform.
                                </div>
                            
                            <form action="process-create-course.php" method="POST" enctype="multipart/form-data" id="createCourseForm" class="needs-validation" novalidate> 
                                <!-- MAX_FILE_SIZE must precede the file input field -->
                                <input type="hidden" name="MAX_FILE_SIZE" value="5242880" /><!-- 5MB -->
                                <div id="formMessages" class="mb-4"></div>
                                <div id="loadingOverlay" class="position-fixed top-0 start-0 w-100 h-100 d-none" style="background: rgba(0,0,0,0.5); z-index: 9999;">
                                    <div class="position-absolute top-50 start-50 translate-middle text-white text-center">
                                        <div class="spinner-border text-primary mb-2" style="width: 3rem; height: 3rem;" role="status">
                                            <span class="visually-hidden">Loading...</span>
                                        </div>
                                        <p>Creating course... Please wait</p>
                                    </div>
                                </div>
                                <div class="row mb-3">
                                    <div class="col-md-6 mb-3 mb-md-0">
                                        <label for="courseTitle" class="form-label">Course Title</label>
                                        <input type="text" class="form-control" id="courseTitle" name="title" required placeholder="Enter course title">
                                    </div>
                                    <div class="col-md-6">
                                        <label for="courseCategory" class="form-label">Category</label>
                                        <select class="form-select" id="courseCategory" name="category">
                                            <option value="">Select a category</option>
                                            <?php
                                            // Fetch categories from database
                                            $categoriesResult = $conn->query("SELECT category_id, category_name FROM categories ORDER BY category_name");
                                            if ($categoriesResult && $categoriesResult->num_rows > 0) {
                                                while ($category = $categoriesResult->fetch_assoc()) {
                                                    echo '<option value="' . $category['category_id'] . '">' . htmlspecialchars($category['category_name']) . '</option>';
                                                }
                                            } else {
                                                // Fallback static options if database query fails
                                                echo '<option value="1">Programming</option>';
                                                echo '<option value="2">Mathematics</option>';
                                                echo '<option value="3">Business</option>';
                                                echo '<option value="4">Science</option>';
                                                echo '<option value="5">Languages</option>';
                                                echo '<option value="6">Technology</option>';
                                                echo '<option value="7">Arts</option>';
                                            }
                                            ?>
                                        </select>
                                        <div class="invalid-feedback">Please select a category</div>
                                    </div>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="courseDescription" class="form-label">Course Description</label>
                                    <textarea class="form-control" id="courseDescription" name="description" rows="4" placeholder="Enter course description"></textarea>
                                </div>
                                
                                <div class="row mb-3">
                                    <div class="col-md-6 mb-3 mb-md-0">
                                        <label for="coursePrice" class="form-label">Price (ZAR)</label>
                                        <div class="input-group">
                                            <span class="input-group-text">R</span>
                                            <input type="number" class="form-control" id="coursePrice" name="price" min="0" step="0.01" placeholder="0.00">
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <label for="courseLevel" class="form-label">Difficulty Level</label>
                                        <select class="form-select" id="courseLevel" name="level">
                                            <option value="beginner">Beginner</option>
                                            <option value="intermediate">Intermediate</option>
                                            <option value="advanced">Advanced</option>
                                            <option value="all">All Levels</option>
                                        </select>
                                    </div>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="courseThumbnail" class="form-label">Course Cover Image</label>
                                    <input type="file" class="form-control" id="courseThumbnail" name="thumbnail" accept="image/jpeg,image/png,image/jpg">
                                    <div id="thumbnailHelp" class="form-text">Recommended size: 1280x720 pixels (16:9 ratio). Accepted formats: JPG, JPEG, PNG</div>
                                    <div id="thumbnailPreview" class="mt-2"></div>
                                    <!-- Hidden inputs for base64 data -->
                                    <input type="hidden" name="thumbnail_base64" id="thumbnail_base64">
                                    <input type="hidden" name="thumbnail_filename" id="thumbnail_filename">
                                </div>
                                
                                <div class="mb-3">
                                    <label for="learningOutcomes" class="form-label">Learning Outcomes</label>
                                    <textarea class="form-control" id="learningOutcomes" name="outcomes" rows="3" placeholder="What students will learn (one point per line)"></textarea>
                                    <div class="form-text">Enter each outcome on a new line</div>
                                </div>
                                
                                <div class="row mb-3">
                                    <div class="col-md-6 mb-3 mb-md-0">
                                        <label for="courseDuration" class="form-label">Estimated Duration (hours)</label>
                                        <input type="number" class="form-control" id="courseDuration" name="duration" min="1" placeholder="E.g. 10">
                                    </div>
                                    <div class="col-md-6">
                                        <label for="courseStatus" class="form-label">Initial Status</label>
                                        <select class="form-select" id="courseStatus" name="status">
                                            <option value="draft">Draft</option>
                                            <option value="published" selected>Published</option>
                                        </select>
                                    </div>
                                </div>

                                <!-- Grade Weighting Section -->
                                <div class="card mb-4" style="border-left: 4px solid #28a745;">
                                    <div class="card-header" style="background: linear-gradient(135deg, #e8f5e8 0%, #c8e6c9 100%); border-bottom: 1px solid #28a745;">
                                        <h5 class="card-title mb-0" style="color: #1b5e20;">
                                            <i class="fas fa-balance-scale" style="margin-right: 0.5rem;"></i>
                                            Grade Weighting Configuration
                                        </h5>
                                        <small class="text-muted">Specify how much each assessment type contributes to the final grade</small>
                                    </div>
                                    <div class="card-body" style="background: #f1f8e9;">
                                        <!-- Hidden assignment weight (always 0 for quiz-only system) -->
                                        <input type="hidden" id="assignmentWeight" name="assignment_weight" value="0">
                                        
                                        <div class="row justify-content-center">
                                            <div class="col-md-8">
                                                <div class="text-center mb-3">
                                                    <div class="badge bg-success fs-6 px-3 py-2">
                                                        <i class="fas fa-star me-2"></i>Quiz-Only Certificate System
                                                    </div>
                                                </div>
                                                <label for="quizWeight" class="form-label text-center d-block">
                                                    <i class="fas fa-question-circle text-success me-2"></i>
                                                    <strong>Quiz Contribution to Final Grade</strong>
                                                </label>
                                                <div class="input-group input-group-lg">
                                                    <input type="number" 
                                                           class="form-control text-center" 
                                                           id="quizWeight" 
                                                           name="quiz_weight" 
                                                           value="100" 
                                                           readonly
                                                           style="font-size: 2rem; font-weight: 700; color: #28a745; background-color: #f8fff9; border-color: #28a745;">
                                                    <span class="input-group-text bg-success text-white" style="font-size: 1.5rem; font-weight: 600;">%</span>
                                                </div>
                                                <div class="form-text text-center mt-2" style="font-size: 1rem;">
                                                    <i class="fas fa-info-circle text-success me-1"></i>
                                                    <strong>Quizzes determine 100% of the final grade for certificate eligibility</strong>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <!-- Course Passing Grade Threshold -->
                                        <div class="row mt-3">
                                            <div class="col-md-6 offset-md-3">
                                                <label for="coursePassingGrade" class="form-label text-center d-block">
                                                    <i class="fas fa-trophy text-warning me-2"></i>
                                                    <strong>Course Passing Grade (%)</strong>
                                                </label>
                                                <div class="input-group input-group-lg">
                                                    <input type="number" 
                                                           class="form-control text-center" 
                                                           id="coursePassingGrade" 
                                                           name="course_passing_grade" 
                                                           value="60" 
                                                           min="1" 
                                                           max="100" 
                                                           step="0.01"
                                                           required
                                                           style="font-size: 1.5rem; font-weight: 600; color: #f6ad55; border-color: #f6ad55;">
                                                    <span class="input-group-text bg-warning text-white" style="font-size: 1.2rem; font-weight: 600;">%</span>
                                                </div>
                                                <div class="form-text text-center mt-2">
                                                    <i class="fas fa-info-circle text-warning me-1"></i>
                                                    <strong>Minimum grade students need to earn their certificate</strong>
                                                </div>
                                            </div>
                                        </div>

                                        <!-- Quiz-Only System Information -->
                                        <div class="row mt-3">
                                            <div class="col-12">
                                                <div class="alert alert-info border-0" style="background-color: #e8f5e8; border-left: 4px solid #28a745 !important;">
                                                    <div class="row align-items-center">
                                                        <div class="col-md-8">
                                                            <h6 class="mb-1" style="color: #155724; font-weight: 600;">
                                                                <i class="fas fa-graduation-cap me-2"></i>
                                                                Certificate-Ready Course
                                                            </h6>
                                                            <p class="mb-0" style="color: #155724; font-size: 0.9rem;">
                                                                Students earn certificates when their quiz average meets the passing grade threshold.
                                                            </p>
                                                        </div>
                                                        <div class="col-md-4 text-end">
                                                            <div class="badge bg-success px-3 py-2" style="font-size: 1rem;">
                                                                <i class="fas fa-check-circle me-1"></i>
                                                                100% Quiz-Based
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="d-flex justify-content-end">
                                    <button type="button" class="btn btn-light me-2" onclick="window.history.back()">Cancel</button>
                                    <button type="submit" class="btn btn-primary" id="createCourseBtn">
                                        <span class="spinner-border spinner-border-sm d-none" id="submitSpinner" role="status" aria-hidden="true"></span>
                                        Create Course
                                    </button>
                                </div>
                            </form>
                            
                            <!-- Form messages and loading overlay -->
                            <div id="formMessages" class="mb-3"></div>
                            <div id="loadingOverlay" class="position-fixed d-none" style="top:0;left:0;right:0;bottom:0;background:rgba(255,255,255,0.7);z-index:1000;display:flex;justify-content:center;align-items:center;">
                                <div class="spinner-border text-primary" style="width: 3rem; height: 3rem;" role="status">
                                    <span class="visually-hidden">Loading...</span>
                                </div>
                            </div>
                            
                            <script>
                            // Grade weighting validation and interaction functions
                            function validateWeights() {
                                const assignmentWeight = parseFloat(document.getElementById('assignmentWeight').value) || 0;
                                const quizWeight = parseFloat(document.getElementById('quizWeight').value) || 0;
                                const total = assignmentWeight + quizWeight;
                                
                                const totalSpan = document.getElementById('totalWeight');
                                const validationDiv = document.getElementById('weightValidation');
                                
                                totalSpan.textContent = total.toFixed(2);
                                
                                if (Math.abs(total - 100) < 0.01) { // Allow for tiny floating point differences
                                    validationDiv.className = 'alert alert-success';
                                    validationDiv.innerHTML = '<i class="fas fa-check-circle me-2"></i><strong>Total: ' + total.toFixed(2) + '%</strong> - Perfect! Weights add up correctly.';
                                    return true;
                                } else if (total > 100) {
                                    validationDiv.className = 'alert alert-danger';
                                    validationDiv.innerHTML = '<i class="fas fa-exclamation-triangle me-2"></i><strong>Total: ' + total.toFixed(2) + '%</strong> - Too high! Total must equal 100%.';
                                    return false;
                                } else {
                                    validationDiv.className = 'alert alert-warning';
                                    validationDiv.innerHTML = '<i class="fas fa-exclamation-triangle me-2"></i><strong>Total: ' + total.toFixed(2) + '%</strong> - Too low! Total must equal 100%.';
                                    return false;
                                }
                            }
                            
                            function setWeights(assignmentPercent, quizPercent) {
                                document.getElementById('assignmentWeight').value = assignmentPercent;
                                document.getElementById('quizWeight').value = quizPercent;
                                validateWeights();
                                
                                // Add visual feedback for the clicked example
                                document.querySelectorAll('.example-card').forEach(card => {
                                    card.style.backgroundColor = '';
                                    card.style.borderColor = '#dee2e6';
                                });
                                
                                event.target.closest('.example-card').style.backgroundColor = '#e8f5e8';
                                event.target.closest('.example-card').style.borderColor = '#28a745';
                                
                                setTimeout(() => {
                                    event.target.closest('.example-card').style.backgroundColor = '';
                                    event.target.closest('.example-card').style.borderColor = '#dee2e6';
                                }, 2000);
                            }
                            
                            $(document).ready(function() {
                                // Grade weighting real-time validation
                                $('#assignmentWeight, #quizWeight').on('input change', function() {
                                    validateWeights();
                                });
                                
                                // Example card hover effects
                                $('.example-card').hover(
                                    function() {
                                        $(this).css({
                                            'backgroundColor': '#f8f9fa',
                                            'borderColor': '#6c757d',
                                            'transform': 'translateY(-2px)',
                                            'boxShadow': '0 4px 8px rgba(0,0,0,0.1)'
                                        });
                                    },
                                    function() {
                                        $(this).css({
                                            'backgroundColor': '',
                                            'borderColor': '#dee2e6',
                                            'transform': '',
                                            'boxShadow': ''
                                        });
                                    }
                                );
                                
                                // File input handling with Base64 conversion
                                $('#courseThumbnail').on('change', function(e) {
                                    const file = e.target.files[0];
                                    if (!file) return;
                                    
                                    // Store original filename
                                    $('#thumbnail_filename').val(file.name);
                                    
                                    // Show preview
                                    const reader = new FileReader();
                                    reader.onload = function(e) {
                                        const base64Data = e.target.result;
                                        $('#thumbnail_base64').val(base64Data);
                                        
                                        // Display preview
                                        $('#thumbnailPreview').html(`<img src="${base64Data}" class="img-thumbnail" alt="Thumbnail Preview">`);
                                    };
                                    reader.readAsDataURL(file);
                                });
                                
                                // Enhanced form validation with better error handling
                                $('#createCourseForm').on('submit', function(event) {
                                    let formValid = true;
                                    
                                    // Reset previous validations
                                    $('.form-control, .form-select').removeClass('is-invalid');
                                    $('.invalid-feedback').remove();
                                    
                                    // Validate grade weighting
                                    const assignmentWeight = parseFloat($('#assignmentWeight').val()) || 0;
                                    const quizWeight = parseFloat($('#quizWeight').val()) || 0;
                                    const total = assignmentWeight + quizWeight;
                                    
                                    if (Math.abs(total - 100) >= 0.01) {
                                        formValid = false;
                                        $('#assignmentWeight, #quizWeight').addClass('is-invalid');
                                        
                                        const errorMsg = total > 100 ? 
                                            `Total weight is ${total.toFixed(2)}% - must equal exactly 100%` :
                                            `Total weight is ${total.toFixed(2)}% - must equal exactly 100%`;
                                            
                                        $('#weightValidation').after(`<div class="invalid-feedback d-block">${errorMsg}</div>`);
                                        
                                        // Scroll to the grade weighting section
                                        $('html, body').animate({
                                            scrollTop: $('#assignmentWeight').offset().top - 100
                                        }, 500);
                                    }
                                    
                                    // Check required fields
                                    $('form .form-control[required], form .form-select[required]').each(function() {
                                        if (!$(this).val() || $(this).val().trim() === '') {
                                            formValid = false;
                                            $(this).addClass('is-invalid');
                                            $('<div class="invalid-feedback">This field is required</div>').insertAfter($(this));
                                        } else {
                                            $(this).removeClass('is-invalid');
                                        }
                                    });
                                    
                                    // Validate price is numeric and positive
                                    const price = $('#coursePrice').val();
                                    if (price && (isNaN(price) || parseFloat(price) < 0)) {
                                        formValid = false;
                                        $('#coursePrice').addClass('is-invalid');
                                        $('<div class="invalid-feedback">Price must be a positive number</div>').insertAfter($('#coursePrice'));
                                    }
                                    
                                    // Validate duration is numeric and positive
                                    const duration = $('#courseDuration').val();
                                    if (duration && (isNaN(duration) || parseInt(duration) < 1)) {
                                        formValid = false;
                                        $('#courseDuration').addClass('is-invalid');
                                        $('<div class="invalid-feedback">Duration must be a positive number</div>').insertAfter($('#courseDuration'));
                                    }
                                    
                                    // Check file is selected
                                    if (!$('#courseThumbnail').val()) {
                                        formValid = false;
                                        $('#courseThumbnail').addClass('is-invalid');
                                        $('<div class="invalid-feedback">Course thumbnail is required</div>').insertAfter($('#courseThumbnail'));
                                    }
                                    
                                    if (!formValid) {
                                        event.preventDefault();
                                        Toastify({
                                            text: "Please correct the errors in the form",
                                            duration: 3000,
                                            gravity: "top",
                                            position: "right",
                                            backgroundColor: "#e74a3b",
                                        }).showToast();
                                        return false;
                                    }
                                    
                                    // Show loading state
                                    $('#createCourseBtn').html('<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span> Creating course...');
                                    $('#createCourseBtn').prop('disabled', true);
                                    
                                    // Let the form submit normally - process-create-course.php will handle it
                                });
                            });
                            </script>
                            <script>
                                $(document).ready(function() {
                                    $('#createCourseBtn').on('click', function(e) {
                                        e.preventDefault();
                                        
                                        // Show loading state
                                        $(this).prop('disabled', true);
                                        $('#submitSpinner').removeClass('d-none');
                                        
                                        // Check if we need to process the file
                                        if ($('#thumbnailPreview img').length === 0 && $('#courseThumbnail')[0].files.length > 0) {
                                            // Process the file to base64 if not already done
                                            const file = $('#courseThumbnail')[0].files[0];
                                            const reader = new FileReader();
                                            reader.onload = function(e) {
                                                $('#thumbnail_base64').val(e.target.result);
                                                $('#thumbnail_filename').val(file.name);
                                                $('#createCourseForm').submit();
                                            };
                                            reader.onerror = function() {
                                                showError('Failed to process image. Please try again with a smaller image.');
                                                $('#createCourseBtn').prop('disabled', false);
                                                $('#submitSpinner').addClass('d-none');
                                            };
                                            reader.readAsDataURL(file);
                                        } else {
                                            // Submit the form if we have base64 data or no file
                                            $('#createCourseForm').submit();
                                        }
                                        return false;
                                    });
                                    // Define error display function
                                    function showError(message) {
                                        $('#formMessages').html(`<div class="alert alert-danger alert-dismissible fade show"><strong>Error:</strong> ${message}<button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button></div>`);
                                    }
                                    
                                    // Ensure thumbnail preview container exists
                                    var $thumbnailInputContainer = $('#courseThumbnail').parent();
                                    if ($thumbnailInputContainer.length > 0 && $('#thumbnailPreview').length === 0) {
                                        $thumbnailInputContainer.append('<div id="thumbnailPreview" class="mt-2"></div>');
                                    }
                                    
                                    // Preview thumbnail with improved handling
                                    $('#courseThumbnail').on('change', function(e) {
                                        if (!e.target || !e.target.files) {
                                            console.warn('File input event has no files property');
                                            return;
                                        }
                                        const file = e.target.files[0];
                                    if (file) {
                                        // Validate file type
                                        const validTypes = ['image/jpeg', 'image/png', 'image/jpg'];
                                        if (!validTypes.includes(file.type)) {
                                            $('#thumbnailPreview').html(`
                                                <div class="mt-2 text-danger">
                                                    <i class="fas fa-exclamation-circle"></i> 
                                                    Invalid file type. Please select a JPG, JPEG, or PNG file.
                                                </div>
                                            `);
                                            return;
                                        }
                                        
                                        // Show loading indicator
                                        $('#thumbnailPreview').html(`
                                            <div class="mt-2">
                                                <div class="spinner-border spinner-border-sm text-primary" role="status">
                                                    <span class="visually-hidden">Loading...</span>
                                                </div>
                                                <span class="ms-2">Loading preview...</span>
                                            </div>
                                        `);
                                        
                                        // Read and display image
                                        const reader = new FileReader();
                                        reader.onload = function(e) {
                                            $('#thumbnailPreview').html(`
                                                <div class="mt-2">
                                                    <img src="${e.target.result}" class="img-thumbnail" style="max-width: 200px; max-height: 150px;">
                                                    <div class="text-muted small mt-1">${file.name} (${(file.size / 1024).toFixed(1)} KB)</div>
                                                </div>
                                            `);
                                        }
                                        reader.onerror = function() {
                                            $('#thumbnailPreview').html(`
                                                <div class="mt-2 text-danger">
                                                    <i class="fas fa-exclamation-circle"></i> 
                                                    Error loading preview.
                                                </div>
                                            `);
                                        }
                                        reader.readAsDataURL(file);
                                    } else {
                                        $('#thumbnailPreview').empty();
                                    }
                                });
                                
                                // Add input validation highlighting
                                $('.form-control, .form-select').on('input change', function() {
                                    if ($(this).val() && $(this).val().trim() !== '') {
                                        $(this).removeClass('is-invalid');
                                    }
                                });
                            });
                            </script>
                        </div>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>

    </div>
    
</div><!-- End container-fluid -->
        </div><!-- End content-wrapper -->
    </div><!-- End dashboard-container -->
    
    <!-- JavaScript Libraries -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/toastify-js"></script>
    
    <!-- Error handling for AJAX responses -->
    <script>
    $(document).ready(function() {
        // Global AJAX error handler
        $(document).ajaxError(function(event, jqxhr, settings, thrownError) {
            console.error('AJAX Error:', thrownError);
            Toastify({
                text: "Server error occurred. Please try again.",
                duration: 5000,
                gravity: "top",
                position: "right",
                backgroundColor: "#e74a3b",
            }).showToast(); 
        });
    });
    </script>
    
    <script>
        // Safe element access function for this page
        function safeGetElement(id) {
            return document.getElementById(id) || null;
        }
    </script>
    
<?php require_once 'components/teacher_layout_end.php'; ?>
