<?php
session_start();
// Use the correct path to database connection file
include_once "../config/db_connect.php";
include_once "../includes/functions.php";

// Check if user is logged in and is a teacher (optional security check)
if (isset($_SESSION['user_id']) && $_SESSION['user_role'] === 'teacher') {
    $is_teacher = true;
} else {
    $is_teacher = false;
}

echo '<!DOCTYPE html>
<html>
<head>
    <title>Teacher Settings Table Setup</title>
    <style>
        body {
            font-family: "Poppins", sans-serif;
            line-height: 1.6;
            max-width: 800px;
            margin: 0 auto;
            padding: 20px;
            background-color: #f9fafb;
        }
        .container {
            background-color: #fff;
            border-radius: 10px;
            padding: 20px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
            margin-top: 30px;
        }
        h1 {
            color: #6366f1;
            text-align: center;
        }
        .output-line {
            margin: 10px 0;
            padding: 8px;
            border-left: 4px solid #6366f1;
            background-color: #f8f9ff;
        }
        .success {
            border-left: 4px solid #10b981;
            background-color: #ecfdf5;
        }
        .error {
            border-left: 4px solid #ef4444;
            background-color: #fef2f2;
        }
        .info {
            border-left: 4px solid #3b82f6;
            background-color: #eff6ff;
        }
        .back-link {
            display: inline-block;
            margin-top: 20px;
            background-color: #6366f1;
            color: white;
            padding: 8px 20px;
            text-decoration: none;
            border-radius: 5px;
            font-weight: 500;
        }
        .back-link:hover {
            background-color: #4f46e5;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>Teacher Settings Table Setup</h1>';

if (!isset($conn)) {
    die('<div class="output-line error">❌ Database connection failed. The $conn variable is not available.</div>');
}

echo '<div class="output-line success">✅ Database connection successful!</div>';

// SQL to create teacher_settings table if it doesn't exist
$sql = "CREATE TABLE IF NOT EXISTS teacher_settings (
  id INT PRIMARY KEY AUTO_INCREMENT,
  user_id INT NOT NULL,
  email_notifications TINYINT(1) DEFAULT 1,
  sms_notifications TINYINT(1) DEFAULT 0,
  dashboard_theme VARCHAR(20) DEFAULT 'light',
  timezone VARCHAR(50) DEFAULT 'UTC',
  language VARCHAR(10) DEFAULT 'en',
  enable_2fa TINYINT(1) DEFAULT 0,
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
)";

// Execute the query
if ($conn->query($sql) === TRUE) {
    echo '<div class="output-line success">✅ Table \'teacher_settings\' created successfully or already exists.</div>';
    
    // Check if there are any teachers without settings and create default settings for them
    $find_teachers = "SELECT u.id, CONCAT(u.first_name, ' ', u.last_name) as name
                     FROM users u 
                     LEFT JOIN teacher_settings ts ON u.id = ts.user_id 
                     WHERE u.role = 'teacher' AND ts.id IS NULL";
    
    $result = $conn->query($find_teachers);
    
    if ($result && $result->num_rows > 0) {
        echo '<div class="output-line info">ℹ️ Found ' . $result->num_rows . ' teachers without settings entries.</div>';
        
        // Prepare the insert statement
        $insert_stmt = $conn->prepare("INSERT INTO teacher_settings 
                                    (user_id, email_notifications, sms_notifications, dashboard_theme, timezone, language, enable_2fa) 
                                    VALUES (?, 1, 0, 'light', 'UTC', 'en', 0)");
        
        $insert_stmt->bind_param("i", $teacher_id);
        $success_count = 0;
        
        while ($row = $result->fetch_assoc()) {
            $teacher_id = $row['id'];
            if($insert_stmt->execute()) {
                echo '<div class="output-line">✓ Created settings for teacher: ' . htmlspecialchars($row['name']) . ' (ID: ' . $teacher_id . ')</div>';
                $success_count++;
            } else {
                echo '<div class="output-line error">✗ Failed to create settings for ' . htmlspecialchars($row['name']) . ' (ID: ' . $teacher_id . ') - Error: ' . $insert_stmt->error . '</div>';
            }
        }
        
        echo '<div class="output-line success">✅ Default settings created for ' . $success_count . ' teacher(s).</div>';
    } else {
        echo '<div class="output-line info">ℹ️ All teachers already have settings entries.</div>';
    }
} else {
    echo '<div class="output-line error">❌ Error creating table: ' . $conn->error . '</div>';
}

echo '<div class="output-line">Database connection closed.</div>';
echo '<a href="settings.php" class="back-link">Go to Settings Page</a>
    </div>
</body>
</html>';
?>
