<?php
// Start session
session_start();

// Include database connection
require_once '../includes/db.php';
require_once '../includes/functions.php';

// Check if user is logged in and is a teacher
if (!isset($_SESSION['user_id']) || $_SESSION['user_role'] != 'teacher') {
    header('Location: ../login.php');
    exit();
}

// Get teacher information
$teacher = get_teacher_by_user_id($conn, $_SESSION['user_id']);
$teacher_id = $teacher['teacher_id'];

// Get statistics
$student_count = count_enrolled_students_by_teacher($conn, $teacher_id);
$course_count = count_courses_by_teacher($conn, $teacher_id);
$video_count = count_videos_by_teacher($conn, $teacher_id);

// Get recent courses
$recent_courses = get_recent_courses_by_teacher($conn, $teacher_id, 5);

// Get recent videos
$recent_videos = get_recent_videos_by_teacher($conn, $teacher_id, 5);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Teacher Dashboard - Panadite Academy</title>
    
    <!-- Favicon -->
    <link rel="shortcut icon" href="../assets/images/favicon.ico" type="image/x-icon">
    
    <!-- CSS Libraries -->
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/toastify-js/src/toastify.min.css">
    
    <style>
        /* Core layout and styling for the dashboard */
        :root {
            --primary: #276BE4;
            --primary-light: rgba(39, 107, 228, 0.1);
            --primary-dark: #1e54b7;
            --sidebar-width: 260px; 
            --sidebar-bg: #1a2035;
            --content-bg: #f5f7fb;
            --card-bg: #ffffff;
            --header-height: 70px;
            --border-radius: 8px;
        }
        
        body {
            font-family: 'Poppins', sans-serif;
            background-color: var(--content-bg);
            margin: 0;
            padding: 0;
            overflow-x: hidden;
        }
        
        /* Layout structure */
        .dashboard-container {
            display: flex;
            min-height: 100vh;
        }
        
        /* Sidebar */
        .sidebar {
            width: var(--sidebar-width);
            background-color: var(--sidebar-bg);
            color: #fff;
            position: fixed;
            height: 100vh;
            left: 0;
            top: 0;
            z-index: 1000;
            overflow-y: auto;
            transition: all 0.3s ease;
        }
        
        .sidebar-header {
            padding: 20px;
            display: flex;
            align-items: center;
        }
        
        .sidebar-brand {
            color: #fff;
            text-decoration: none;
            font-weight: 700;
            font-size: 1.2rem;
            display: flex;
            align-items: center;
        }
        
        .sidebar-brand img {
            max-width: 40px;
            margin-right: 10px;
        }
        
        .sidebar-nav {
            list-style: none;
            padding: 0;
            margin: 0;
        }
        
        .sidebar-heading {
            padding: 10px 20px;
            font-size: 0.75rem;
            text-transform: uppercase;
            letter-spacing: 1px;
            opacity: 0.7;
        }
        
        .nav-item {
            margin: 4px 0;
        }
        
        .nav-link {
            display: flex;
            align-items: center;
            padding: 10px 20px;
            color: rgba(255,255,255,0.7);
            text-decoration: none;
            transition: all 0.3s ease;
        }
        
        .nav-link i {
            margin-right: 10px;
            font-size: 1.1rem;
        }
        
        .nav-link:hover {
            background-color: rgba(255,255,255,0.1);
            color: #fff;
        }
        
        .nav-link.active {
            background-color: var(--primary);
            color: #fff;
            border-radius: 4px;
            margin: 0 10px;
            padding: 10px;
        }
        
        /* Main content */
        .content-wrapper {
            flex: 1;
            margin-left: var(--sidebar-width);
            padding: 20px;
            transition: margin-left 0.3s ease;
        }
        
        /* Navbar */
        .dashboard-navbar {
            background-color: var(--card-bg);
            border-radius: var(--border-radius);
            padding: 15px 20px;
            display: flex;
            align-items: center;
            justify-content: space-between;
            margin-bottom: 20px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
        }
        
        /* Welcome banner */
        .welcome-banner {
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
            color: #fff;
            border-radius: var(--border-radius);
            padding: 25px;
            margin-bottom: 20px;
            position: relative;
            overflow: hidden;
        }
        
        .welcome-banner h2 {
            font-weight: 700;
            margin-bottom: 10px;
        }
        
        .welcome-banner .btn {
            margin-right: 10px;
        }
        
        /* Stat cards */
        .stat-card {
            background-color: var(--card-bg);
            border-radius: var(--border-radius);
            padding: 20px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
            margin-bottom: 20px;
            height: 100%;
            transition: transform 0.3s ease, box-shadow 0.3s ease;
        }
        
        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
        }
        
        .stat-icon {
            width: 48px;
            height: 48px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 50%;
            margin-right: 15px;
            background-color: var(--primary-light);
        }
        
        .stat-number {
            font-size: 1.8rem;
            font-weight: 700;
            margin: 10px 0 5px;
        }
        
        /* Tables */
        .data-card {
            background-color: var(--card-bg);
            border-radius: var(--border-radius);
            margin-bottom: 20px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
            overflow: hidden;
        }
        
        .data-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 15px 20px;
            border-bottom: 1px solid rgba(0,0,0,0.05);
        }
        
        .data-title {
            font-weight: 600;
            margin: 0;
        }
        
        .table-responsive {
            padding: 0;
        }
        
        .table {
            margin-bottom: 0;
        }
        
        .table th {
            border-top: none;
            font-weight: 600;
            padding: 12px 15px;
            color: #555;
            font-size: 0.9rem;
            text-transform: uppercase;
        }
        
        .table td {
            padding: 15px;
            vertical-align: middle;
        }
        
        /* Chart containers */
        .chart-container {
            position: relative;
            height: 300px;
            margin-bottom: 20px;
        }
        
        /* Status badges */
        .status-badge {
            padding: 5px 10px;
            border-radius: 50px;
            font-size: 0.75rem;
            font-weight: 600;
        }
        
        .status-published {
            background-color: rgba(40, 167, 69, 0.1);
            color: #28a745;
        }
        
        .status-draft {
            background-color: rgba(108, 117, 125, 0.1);
            color: #6c757d;
        }
        
        /* Media queries for responsiveness */
        @media (max-width: 991px) {
            .sidebar {
                transform: translateX(-100%);
            }
            
            .content-wrapper {
                margin-left: 0;
            }
            
            .sidebar.show {
                transform: translateX(0);
            }
        }
    </style>
</head>
<body>
    <div class="dashboard-container">
        <!-- Sidebar -->
        <aside class="sidebar">
            <div class="sidebar-header">
                <a href="../index.php" class="sidebar-brand">
                    <img src="../assets/images/logo.png" alt="Panadite Academy" onerror="this.src='../assets/images/favicon.ico';">
                    <span>Panadite Academy</span>
                </a>
            </div>
            
            <ul class="sidebar-nav">
                <li class="sidebar-heading">MAIN</li>
                
                <li class="nav-item">
                    <a href="dashboard.php" class="nav-link active">
                        <i class="fas fa-tachometer-alt"></i>
                        <span>Dashboard</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="courses.php" class="nav-link">
                        <i class="fas fa-graduation-cap"></i>
                        <span>Courses</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="videos.php" class="nav-link">
                        <i class="fas fa-video"></i>
                        <span>Videos</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="students.php" class="nav-link">
                        <i class="fas fa-user-graduate"></i>
                        <span>Students</span>
                    </a>
                </li>
                
                <li class="sidebar-heading">TOOLS</li>
                
                <li class="nav-item">
                    <a href="analytics.php" class="nav-link">
                        <i class="fas fa-chart-line"></i>
                        <span>Analytics</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="quizzes.php" class="nav-link">
                        <i class="fas fa-question-circle"></i>
                        <span>Quizzes</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="assignments.php" class="nav-link">
                        <i class="fas fa-tasks"></i>
                        <span>Assignments</span>
                    </a>
                </li>
            </ul>
        </aside>
        
        <!-- Content Wrapper -->
        <div class="content-wrapper">
            <!-- Navbar -->
            <nav class="dashboard-navbar">
                <div class="d-flex align-items-center">
                    <button id="sidebarToggle" class="btn btn-sm btn-light me-2">
                        <i class="fas fa-bars"></i>
                    </button>
                    <h4 class="mb-0">Teacher Dashboard</h4>
                </div>
                
                <div class="d-flex align-items-center">
                    <!-- Search -->
                    <div class="me-3 d-none d-md-block">
                        <form class="d-flex">
                            <input type="text" class="form-control form-control-sm" placeholder="Search...">
                        </form>
                    </div>
                    
                    <!-- Notifications -->
                    <div class="dropdown me-3">
                        <button class="btn btn-light position-relative" data-bs-toggle="dropdown">
                            <i class="fas fa-bell"></i>
                            <span class="position-absolute top-0 start-100 translate-middle badge rounded-pill bg-danger">
                                3
                            </span>
                        </button>
                        <div class="dropdown-menu dropdown-menu-end">
                            <div class="p-2 border-bottom">
                                <h6 class="mb-0">Notifications</h6>
                            </div>
                            <div class="p-2 border-bottom">
                                <div class="d-flex">
                                    <div class="flex-shrink-0 me-2">
                                        <div class="rounded-circle bg-light p-2">
                                            <i class="fas fa-user-graduate text-primary"></i>
                                        </div>
                                    </div>
                                    <div>
                                        <p class="mb-0">New student enrolled</p>
                                        <small class="text-muted">2 hours ago</small>
                                    </div>
                                </div>
                            </div>
                            <div class="p-2 text-center">
                                <a href="#" class="text-decoration-none">View all notifications</a>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Profile -->
                    <div class="dropdown">
                        <button class="btn btn-light d-flex align-items-center" data-bs-toggle="dropdown">
                            <img src="<?php echo !empty($teacher['profile_picture']) ? '../uploads/profile/' . htmlspecialchars($teacher['profile_picture']) : '../assets/images/avatar-placeholder.jpg'; ?>" 
                                 class="rounded-circle me-2" height="32" width="32" alt="<?php echo htmlspecialchars($teacher['first_name']); ?>">
                            <span class="d-none d-md-inline"><?php echo htmlspecialchars($teacher['first_name']); ?></span>
                        </button>
                        <div class="dropdown-menu dropdown-menu-end">
                            <a href="profile.php" class="dropdown-item">
                                <i class="fas fa-user me-2"></i> Profile
                            </a>
                            <a href="settings.php" class="dropdown-item">
                                <i class="fas fa-cog me-2"></i> Settings
                            </a>
                            <div class="dropdown-divider"></div>
                            <a href="../logout.php" class="dropdown-item">
                                <i class="fas fa-sign-out-alt me-2"></i> Logout
                            </a>
                        </div>
                    </div>
                </div>
            </nav>
</div>
</div>

<!-- JavaScript Libraries -->
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/chart.js@3.7.1/dist/chart.min.js"></script>
<script>
    // Toggle sidebar
    document.getElementById('sidebarToggle').addEventListener('click', function() {
        document.querySelector('.sidebar').classList.toggle('show');
    });
</script>
</body>
</html>
            <!-- Welcome Banner -->
            <div class="welcome-banner">
                <div class="row align-items-center">
                    <div class="col-md-8">
                        <h2>Welcome back, <?php echo htmlspecialchars($teacher['first_name']); ?>!</h2>
                        <p>Here's what's happening with your courses today.</p>
                        <div class="d-flex flex-wrap gap-2 mt-3">
                            <a href="create-course.php" class="btn btn-light">
                                <i class="fas fa-plus me-2"></i> Create Course
                            </a>
                            <a href="upload-video.php" class="btn btn-outline-light">
                                <i class="fas fa-video me-2"></i> Upload Video
                            </a>
                        </div>
                    </div>
                    <div class="col-md-4 d-none d-md-block text-end">
                        <!-- Decorative illustration or icon -->
                        <i class="fas fa-chalkboard-teacher fa-5x text-white opacity-50"></i>
                    </div>
                </div>
            </div>
            
            <!-- Stats Cards -->
            <div class="row g-3 mb-4">
                <!-- Students Card -->
                <div class="col-12 col-sm-6 col-lg-3">
                    <div class="stat-card">
                        <div class="d-flex align-items-center">
                            <div class="stat-icon">
                                <i class="fas fa-user-graduate text-primary"></i>
                            </div>
                            <div>
                                <small class="text-muted text-uppercase">Students</small>
                            </div>
                        </div>
                        <div class="stat-number"><?php echo $student_count; ?></div>
                        <div class="text-success">
                            <i class="fas fa-arrow-up me-1"></i>
                            <span>12.5%</span>
                            <small class="text-muted ms-1">this month</small>
                        </div>
                    </div>
                </div>
                
                <!-- Courses Card -->
                <div class="col-12 col-sm-6 col-lg-3">
                    <div class="stat-card">
                        <div class="d-flex align-items-center">
                            <div class="stat-icon" style="background-color: rgba(40, 167, 69, 0.1);">
                                <i class="fas fa-graduation-cap text-success"></i>
                            </div>
                            <div>
                                <small class="text-muted text-uppercase">Courses</small>
                            </div>
                        </div>
                        <div class="stat-number"><?php echo $course_count; ?></div>
                        <div class="text-success">
                            <i class="fas fa-arrow-up me-1"></i>
                            <span>8.3%</span>
                            <small class="text-muted ms-1">this month</small>
                        </div>
                    </div>
                </div>
                
                <!-- Videos Card -->
                <div class="col-12 col-sm-6 col-lg-3">
                    <div class="stat-card">
                        <div class="d-flex align-items-center">
                            <div class="stat-icon" style="background-color: rgba(23, 162, 184, 0.1);">
                                <i class="fas fa-video text-info"></i>
                            </div>
                            <div>
                                <small class="text-muted text-uppercase">Videos</small>
                            </div>
                        </div>
                        <div class="stat-number"><?php echo $video_count; ?></div>
                        <div class="text-success">
                            <i class="fas fa-arrow-up me-1"></i>
                            <span>15.0%</span>
                            <small class="text-muted ms-1">this month</small>
                        </div>
                    </div>
                </div>
                
                <!-- Earnings Card -->
                <div class="col-12 col-sm-6 col-lg-3">
                    <div class="stat-card">
                        <div class="d-flex align-items-center">
                            <div class="stat-icon" style="background-color: rgba(255, 193, 7, 0.1);">
                                <i class="fas fa-dollar-sign text-warning"></i>
                            </div>
                            <div>
                                <small class="text-muted text-uppercase">Earnings</small>
                            </div>
                        </div>
                        <div class="stat-number">$1,254</div>
                        <div class="text-success">
                            <i class="fas fa-arrow-up me-1"></i>
                            <span>5.2%</span>
                            <small class="text-muted ms-1">this month</small>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Tables and Charts -->
            <div class="row mb-4">
                <!-- Recent Courses -->
                <div class="col-12 col-lg-6">
                    <div class="data-card h-100">
                        <div class="data-header">
                            <h5 class="data-title">Recent Courses</h5>
                            <a href="courses.php" class="btn btn-sm btn-outline-primary">View All</a>
                        </div>
                        <div class="table-responsive">
                            <table class="table">
                                <thead>
                                    <tr>
                                        <th>Course</th>
                                        <th>Students</th>
                                        <th>Rating</th>
                                        <th>Status</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if ($recent_courses && $recent_courses->num_rows > 0): ?>
                                        <?php while($course = $recent_courses->fetch_assoc()): ?>
                                            <?php 
                                                // Calculate enrolled students
                                                $enrolled_count_query = "SELECT COUNT(*) as count FROM enrollments WHERE course_id = ?";
                                                $stmt = $conn->prepare($enrolled_count_query);
                                                $stmt->bind_param("i", $course['course_id']);
                                                $stmt->execute();
                                                $enrolled_result = $stmt->get_result();
                                                $enrolled_count = $enrolled_result->fetch_assoc()['count'];
                                                
                                                // Check if course is published
                                                $status_class = $course['is_published'] ? 'status-published' : 'status-draft';
                                                $status_text = $course['is_published'] ? 'Published' : 'Draft';
                                            ?>
                                            <tr>
                                                <td>
                                                    <div class="d-flex align-items-center">
                                                        <?php if(!empty($course['thumbnail'])): ?>
                                                            <img src="../uploads/courses/<?php echo htmlspecialchars($course['thumbnail']); ?>" 
                                                                 class="rounded me-2" width="40" height="40" alt="<?php echo htmlspecialchars($course['title']); ?>"
                                                                 onerror="this.src='../assets/images/course-placeholder.jpg';">
                                                        <?php else: ?>
                                                            <div class="bg-light rounded me-2 p-2">
                                                                <i class="fas fa-book text-primary"></i>
                                                            </div>
                                                        <?php endif; ?>
                                                        <div>
                                                            <div class="fw-medium"><?php echo htmlspecialchars($course['title']); ?></div>
                                                            <small class="text-muted"><?php echo date('M d, Y', strtotime($course['created_at'])); ?></small>
                                                        </div>
                                                    </div>
                                                </td>
                                                <td><?php echo $enrolled_count; ?></td>
                                                <td>
                                                    <div class="d-flex align-items-center">
                                                        <i class="fas fa-star text-warning me-1"></i>
                                                        <span><?php echo number_format($course['rating'] ?? 0, 1); ?></span>
                                                    </div>
                                                </td>
                                                <td>
                                                    <span class="status-badge <?php echo $status_class; ?>"><?php echo $status_text; ?></span>
                                                </td>
                                            </tr>
                                        <?php endwhile; ?>
                                    <?php else: ?>
                                        <tr>
                                            <td colspan="4" class="text-center py-4">
                                                <p class="text-muted mb-2">No courses found.</p>
                                                <a href="create-course.php" class="btn btn-sm btn-primary">
                                                    <i class="fas fa-plus me-1"></i> Create your first course
                                                </a>
                                            </td>
                                        </tr>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
                
                <!-- Recent Videos -->
                <div class="col-12 col-lg-6 mt-4 mt-lg-0">
                    <div class="data-card h-100">
                        <div class="data-header">
                            <h5 class="data-title">Recent Videos</h5>
                            <a href="videos.php" class="btn btn-sm btn-outline-primary">View All</a>
                        </div>
                        <div class="table-responsive">
                            <table class="table">
                                <thead>
                                    <tr>
                                        <th>Video</th>
                                        <th>Duration</th>
                                        <th>Views</th>
                                        <th>Status</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if ($recent_videos && $recent_videos->num_rows > 0): ?>
                                        <?php while($video = $recent_videos->fetch_assoc()): ?>
                                            <?php 
                                                // Check if video is published
                                                $status_class = $video['is_published'] ? 'status-published' : 'status-draft';
                                                $status_text = $video['is_published'] ? 'Published' : 'Draft';
                                            ?>
                                            <tr>
                                                <td>
                                                    <div class="d-flex align-items-center">
                                                        <?php if(!empty($video['thumbnail'])): ?>
                                                            <img src="../uploads/videos/<?php echo htmlspecialchars($video['thumbnail']); ?>" 
                                                                 class="rounded me-2" width="40" height="40" alt="<?php echo htmlspecialchars($video['title']); ?>"
                                                                 onerror="this.src='../assets/images/video-placeholder.jpg';">
                                                        <?php else: ?>
                                                            <div class="bg-light rounded me-2 p-2">
                                                                <i class="fas fa-video text-info"></i>
                                                            </div>
                                                        <?php endif; ?>
                                                        <div>
                                                            <div class="fw-medium"><?php echo htmlspecialchars($video['title']); ?></div>
                                                            <small class="text-muted"><?php echo date('M d, Y', strtotime($video['created_at'])); ?></small>
                                                        </div>
                                                    </div>
                                                </td>
                                                <td><?php echo $video['duration'] ?? '00:00'; ?></td>
                                                <td><?php echo number_format($video['views'] ?? 0); ?></td>
                                                <td>
                                                    <span class="status-badge <?php echo $status_class; ?>"><?php echo $status_text; ?></span>
                                                </td>
                                            </tr>
                                        <?php endwhile; ?>
                                    <?php else: ?>
                                        <tr>
                                            <td colspan="4" class="text-center py-4">
                                                <p class="text-muted mb-2">No videos found.</p>
                                                <a href="upload-video.php" class="btn btn-sm btn-primary">
                                                    <i class="fas fa-plus me-1"></i> Upload your first video
                                                </a>
                                            </td>
                                        </tr>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Charts -->
            <div class="row mb-4">
                <!-- Course Engagement -->
                <div class="col-12 col-lg-8 mb-4 mb-lg-0">
                    <div class="data-card h-100">
                        <div class="data-header">
                            <h5 class="data-title">Course Engagement</h5>
                            <div class="btn-group btn-group-sm" role="group">
                                <button type="button" class="btn btn-outline-secondary active">Week</button>
                                <button type="button" class="btn btn-outline-secondary">Month</button>
                                <button type="button" class="btn btn-outline-secondary">Year</button>
                            </div>
                        </div>
                        <div class="p-3">
                            <div class="chart-container">
                                <canvas id="engagementChart"></canvas>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Student Distribution -->
                <div class="col-12 col-lg-4">
                    <div class="data-card h-100">
                        <div class="data-header">
                            <h5 class="data-title">Student Distribution</h5>
                        </div>
                        <div class="p-3">
                            <div class="chart-container">
                                <canvas id="distributionChart"></canvas>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- JavaScript for Charts -->
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Course Engagement Chart
            var engagementCtx = document.getElementById('engagementChart').getContext('2d');
            var engagementChart = new Chart(engagementCtx, {
                type: 'line',
                data: {
                    labels: ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'],
                    datasets: [{
                        label: 'Course Views',
                        data: [120, 190, 150, 240, 180, 220, 290],
                        backgroundColor: 'rgba(39, 107, 228, 0.1)',
                        borderColor: '#276BE4',
                        borderWidth: 2,
                        pointBackgroundColor: '#276BE4',
                        tension: 0.3,
                        fill: true
                    }, {
                        label: 'Student Activity',
                        data: [80, 120, 90, 160, 100, 140, 200],
                        backgroundColor: 'rgba(40, 167, 69, 0.1)',
                        borderColor: '#28a745',
                        borderWidth: 2,
                        pointBackgroundColor: '#28a745',
                        tension: 0.3,
                        fill: true
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            position: 'top'
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            grid: {
                                borderDash: [3, 3]
                            }
                        },
                        x: {
                            grid: {
                                display: false
                            }
                        }
                    }
                }
            });
            
            // Student Distribution Chart
            var distributionCtx = document.getElementById('distributionChart').getContext('2d');
            var distributionChart = new Chart(distributionCtx, {
                type: 'doughnut',
                data: {
                    labels: ['Programming', 'Design', 'Marketing', 'Business', 'Other'],
                    datasets: [{
                        data: [35, 25, 20, 15, 5],
                        backgroundColor: [
                            '#276BE4',
                            '#28a745',
                            '#17a2b8',
                            '#ffc107',
                            '#6c757d'
                        ],
                        borderWidth: 0
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    cutout: '70%',
                    plugins: {
                        legend: {
                            position: 'bottom',
                            labels: {
                                boxWidth: 12
                            }
                        }
                    },
                    animation: {
                        animateScale: true
                    }
                }
            });
        });
    </script>
</body>
</html>
