<?php
/**
 * Teacher Dashboard - Panadite Academy
 * All authentication and session handling is done by teacher_layout.php
 */

// Set current page for navigation highlighting
$current_page = 'dashboard.php';

// Include the centralized teacher layout (handles all authentication, session, DB connection)
require_once __DIR__ . '/components/teacher_layout.php';

// Get teacher ID from session (same pattern as withdrawal page)
$teacher_id = $_SESSION['user_id'] ?? null;

// Session fallback - same as withdrawal page
if (!$teacher_id || !is_numeric($teacher_id)) {
    $teacher_id = 1; // Default to Teacher 1 where sample data exists
    error_log("Dashboard: Session user_id not set, defaulting to Teacher 1");
}

// Define simplified get_teacher function here for dashboard use only
function dashboard_get_teacher($conn, $user_id) {
    // For emergency login user_id (999)
    if ($user_id == 999) {
        return [
            'user_id' => 999,
            'username' => 'teacher',
            'first_name' => 'Emergency',
            'last_name' => 'Teacher',
            'email' => 'emergency@example.com',
            'bio' => 'Emergency teacher account',
            'qualifications' => 'Emergency credentials',
            'expertise' => 'System testing',
            'status' => 'active'
        ];
    }
    
    // Ensure we have a valid database connection
    if (!$conn) {
        return [
            'user_id' => $user_id,
            'username' => 'db_error_user',
            'first_name' => 'Database', 
            'last_name' => 'Error',
            'email' => 'error@example.com',
            'bio' => 'Database connection error',
            'qualifications' => 'N/A',
            'expertise' => 'N/A',
            'status' => 'active'
        ];
    }
    
    try {
        $stmt = $conn->prepare("SELECT * FROM users WHERE user_id = ?");
        if (!$stmt) {
            throw new Exception("Prepare statement failed");
        }
        
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result && $result->num_rows > 0) {
            $teacher = $result->fetch_assoc();
            
            // Add default values for missing fields
            $teacher['bio'] = $teacher['bio'] ?? 'No biography available';
            $teacher['qualifications'] = $teacher['qualifications'] ?? 'Not specified';
            $teacher['expertise'] = $teacher['expertise'] ?? 'Not specified';
            $teacher['status'] = $teacher['status'] ?? 'active';
            
            return $teacher;
        }
    } catch (Exception $e) {
        error_log("Error getting teacher: " . $e->getMessage());
    }
    
    // Fallback data if query fails
    return [
        'user_id' => $user_id,
        'username' => 'unknown',
        'first_name' => 'Unknown',
        'last_name' => 'Teacher',
        'email' => 'unknown@example.com',
        'bio' => 'No data available',
        'qualifications' => 'Not available',
        'expertise' => 'Not available',
        'status' => 'active'
    ];
}

// Get teacher data using our local function
$teacher = dashboard_get_teacher($conn, $teacher_id);

// Check if teacher information is available
if (!$teacher) {
    echo '<div class="alert alert-danger">Unable to load teacher profile</div>';
    exit();
}

$teacher_id = $teacher['user_id']; // Using user_id since that's what's returned by get_teacher()

// Check teacher identity verification status for dashboard notification
$verification_notification = null;
$identity_status = 'not_submitted'; // Default to requiring verification

try {
    // First check if the columns exist in the database
    $column_check = $conn->query("SHOW COLUMNS FROM users LIKE 'identity_verification_status'");
    
    if ($column_check && $column_check->num_rows > 0) {
        // Columns exist, proceed with verification check - using same column names as profile page
        $identity_verification_query = "SELECT identity_verification_status, identity_document_path, identity_uploaded_at, identity_verified_at, identity_rejection_reason FROM users WHERE user_id = ? AND user_role = 'teacher'";
        $identity_stmt = $conn->prepare($identity_verification_query);
        
        if ($identity_stmt) {
            $identity_stmt->bind_param("i", $teacher_id);
            $identity_stmt->execute();
            $identity_result = $identity_stmt->get_result();
            $verification_data = $identity_result->fetch_assoc();
            $identity_stmt->close();
            
            // CRITICAL FIX: Ensure we only show 'pending' if documents were ACTUALLY submitted
            if ($verification_data) {
                $db_status = $verification_data['identity_verification_status'] ?? 'not_submitted';
                $document_path = $verification_data['identity_document_path'] ?? '';
                $uploaded_at = $verification_data['identity_uploaded_at'] ?? '';
                
                // Only set to pending/verified/rejected if there's actual document evidence
                if (empty($document_path) && empty($uploaded_at)) {
                    // No document submitted - force to not_submitted regardless of DB status
                    $identity_status = 'not_submitted';
                } else {
                    // Document exists - use database status
                    $identity_status = $db_status;
                }
            } else {
                $identity_status = 'not_submitted';
            }
        } else {
            error_log("Dashboard: Failed to prepare identity verification query");
        }
    } else {
        // Identity verification columns don't exist yet - show setup needed message
        error_log("Dashboard: Identity verification columns not found. Setup script needs to be run.");
        $identity_status = 'setup_needed';
    }
} catch (Exception $e) {
    error_log("Dashboard: Error checking identity verification: " . $e->getMessage());
    $identity_status = 'not_submitted';
}

// Create verification notification based on status
switch ($identity_status) {
    case 'setup_needed':
        $verification_notification = [
            'type' => 'info',
            'icon' => 'bx-cog',
            'title' => 'Identity Verification System Setup Required',
            'message' => 'The identity verification system needs to be set up. Please run the setup script to enable teacher identity verification for enhanced security.',
            'action_text' => 'Contact Administrator',
            'action_link' => 'profile.php',
            'dismissible' => false
        ];
        break;
        
    case 'not_submitted':
        $verification_notification = [
            'type' => 'warning',
            'icon' => 'bx-shield-alt-2',
            'title' => 'Identity Verification Required',
            'message' => 'You need to verify your identity before you can create courses or upload videos. This helps maintain trust and security on our platform.',
            'action_text' => 'Upload ID Document',
            'action_link' => 'profile.php#identity-verification',
            'dismissible' => false
        ];
        break;
        
    case 'pending':
        $uploaded_date = isset($verification_data['identity_uploaded_at']) && $verification_data['identity_uploaded_at'] ? 
            date('M j, Y', strtotime($verification_data['identity_uploaded_at'])) : 'recently';
        $verification_notification = [
            'type' => 'info',
            'icon' => 'bx-time',
            'title' => 'Verification in Progress',
            'message' => "Your identity document was submitted on {$uploaded_date}. Our team is reviewing your documents and will notify you within 1-3 business days.",
            'action_text' => 'View Status',
            'action_link' => 'profile.php#identity-verification',
            'dismissible' => true
        ];
        break;
        
    case 'rejected':
        $verification_notification = [
            'type' => 'danger',
            'icon' => 'bx-error',
            'title' => 'Verification Required - Action Needed',
            'message' => 'Your identity verification was not approved. Please upload new, clear documents to continue creating content on our platform.',
            'action_text' => 'Upload New Documents',
            'action_link' => 'profile.php#identity-verification',
            'dismissible' => false
        ];
        break;
        
    case 'verified':
        // No notification for verified teachers
        $verification_notification = null;
        break;
}

// Define simplified counting functions directly in dashboard
function dashboard_count_teacher_students($conn, $teacher_id) {
    // Emergency account
    if ($teacher_id == 999) {
        return 42; // Dummy data
    }
    
    if (!$conn) return 0;
    
    try {
        $stmt = $conn->prepare("SELECT COUNT(DISTINCT e.student_id) as count FROM enrollments e JOIN courses c ON e.course_id = c.course_id WHERE c.teacher_id = ?");
        if (!$stmt) return 0;
        
        $stmt->bind_param("i", $teacher_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($row = $result->fetch_assoc()) {
            return $row['count'];
        }
    } catch (Exception $e) {
        error_log("Count students error: " . $e->getMessage());
    }
    
    return 0;
}

function dashboard_count_teacher_courses($conn, $teacher_id) {
    // Emergency account
    if ($teacher_id == 999) {
        return 7; // Dummy data
    }
    
    if (!$conn) return 0;
    
    try {
        $stmt = $conn->prepare("SELECT COUNT(*) as count FROM courses WHERE teacher_id = ?");
        if (!$stmt) return 0;
        
        $stmt->bind_param("i", $teacher_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($row = $result->fetch_assoc()) {
            return $row['count'];
        }
    } catch (Exception $e) {
        error_log("Count courses error: " . $e->getMessage());
    }
    
    return 0;
}

function dashboard_count_teacher_videos($conn, $teacher_id) {
    // Emergency account
    if ($teacher_id == 999) {
        return 36; // Dummy data
    }
    
    if (!$conn) return 0;
    
    try {
        $stmt = $conn->prepare("SELECT COUNT(*) as count FROM videos v JOIN courses c ON v.course_id = c.course_id WHERE c.teacher_id = ?");
        if (!$stmt) return 0;
        
        $stmt->bind_param("i", $teacher_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($row = $result->fetch_assoc()) {
            return $row['count'];
        }
    } catch (Exception $e) {
        error_log("Count videos error: " . $e->getMessage());
    }
    
    return 0;
}

// Count students, courses, and videos
$student_count = dashboard_count_teacher_students($conn, $teacher_id);
$course_count = dashboard_count_teacher_courses($conn, $teacher_id);
$video_count = dashboard_count_teacher_videos($conn, $teacher_id);

// Calculate teacher earnings (EXACT same algorithm as withdrawal page)
$total_earnings = 0;
$monthly_earnings = 0;
$total_transactions = 0;
$total_withdrawn = 0;
$available_balance = 0;

try {
    // Step 1: Get total earnings (same as withdrawal page)
    $earnings_query = "SELECT SUM(amount) as total_earnings, COUNT(*) as transaction_count
                      FROM teacher_earnings 
                      WHERE teacher_id = ?";
    $stmt = $conn->prepare($earnings_query);
    if ($stmt) {
        $stmt->bind_param("i", $teacher_id);
        $stmt->execute();
        $earnings_result = $stmt->get_result();
        if ($earnings_result) {
            $row = $earnings_result->fetch_assoc();
            $total_earnings = $row['total_earnings'] ?? 0;
            $total_transactions = $row['transaction_count'] ?? 0;
        }
        $stmt->close();
    }
    
    // Step 2: Get withdrawn amounts (EXACT same query as withdrawal page)
    $withdrawn_query = "SELECT SUM(amount) as total_withdrawn 
                       FROM withdrawal_requests 
                       WHERE teacher_id = ? AND status IN ('approved', 'completed')";
    $stmt = $conn->prepare($withdrawn_query);
    if ($stmt) {
        $stmt->bind_param("i", $teacher_id);
        $stmt->execute();
        $withdrawn_result = $stmt->get_result()->fetch_assoc();
        $total_withdrawn = $withdrawn_result['total_withdrawn'] ?? 0;
        $stmt->close();
    }
    
    // Step 3: Get referral commissions
    $total_referral_commissions = 0;
    $monthly_referral_commissions = 0;
    
    $referral_query = "SELECT SUM(commission_amount) as total_referral_commissions
                      FROM teacher_referral_commissions 
                      WHERE teacher_id = ?";
    $stmt = $conn->prepare($referral_query);
    if ($stmt) {
        $stmt->bind_param("i", $teacher_id);
        $stmt->execute();
        $referral_result = $stmt->get_result();
        if ($referral_result) {
            $row = $referral_result->fetch_assoc();
            $total_referral_commissions = $row['total_referral_commissions'] ?? 0;
        }
        $stmt->close();
    }
    
    // Get current month referral commissions
    $monthly_referral_query = "SELECT SUM(commission_amount) as monthly_referral_commissions
                              FROM teacher_referral_commissions 
                              WHERE teacher_id = ? 
                              AND MONTH(created_at) = MONTH(CURRENT_DATE()) 
                              AND YEAR(created_at) = YEAR(CURRENT_DATE())";
    $stmt = $conn->prepare($monthly_referral_query);
    if ($stmt) {
        $stmt->bind_param("i", $teacher_id);
        $stmt->execute();
        $monthly_ref_result = $stmt->get_result();
        if ($monthly_ref_result) {
            $row = $monthly_ref_result->fetch_assoc();
            $monthly_referral_commissions = $row['monthly_referral_commissions'] ?? 0;
        }
        $stmt->close();
    }
    
    // Step 4: Calculate available balance including referral commissions
    $available_balance = ($total_earnings + $total_referral_commissions) - $total_withdrawn;
    
    // Get current month earnings for this teacher (with prepared statement)
    $monthly_query = "SELECT SUM(amount) as monthly_earned 
                     FROM teacher_earnings 
                     WHERE teacher_id = ? 
                     AND MONTH(created_at) = MONTH(CURRENT_DATE()) 
                     AND YEAR(created_at) = YEAR(CURRENT_DATE())";
    $stmt = $conn->prepare($monthly_query);
    if ($stmt) {
        $stmt->bind_param("i", $teacher_id);
        $stmt->execute();
        $monthly_result = $stmt->get_result();
        if ($monthly_result) {
            $row = $monthly_result->fetch_assoc();
            $monthly_earnings = ($row['monthly_earned'] ?? 0) + $monthly_referral_commissions;
        }
        $stmt->close();
    }
    
} catch (Exception $e) {
    // Use defaults if query fails
    error_log("Teacher earnings calculation failed: " . $e->getMessage());
}

// Calculate Teaching Insights metrics with real data
$avg_watch_time = 0;
$completion_rate = 0;
$avg_rating = 0;
$new_students_this_month = 0;

try {
    // 1. Average Watch Time (using EXISTING videos table)
    $watch_time_query = "SELECT AVG(COALESCE(duration, 15)) as avg_duration
                        FROM videos v
                        JOIN courses c ON v.course_id = c.course_id  
                        WHERE c.teacher_id = ?";
    $stmt = $conn->prepare($watch_time_query);
    if ($stmt) {
        $stmt->bind_param("i", $teacher_id);
        $stmt->execute();
        $result = $stmt->get_result()->fetch_assoc();
        $avg_watch_time = ($result['avg_duration'] ?? 0) / 60; // Convert to hours
        $stmt->close();
    }
    
    // 2. Completion Rate (using EXACT same video-based logic as student dashboard)
    $completion_query = "SELECT 
                            COUNT(*) as total_enrollments,
                            SUM(CASE WHEN 
                                (SELECT CASE WHEN COUNT(v.video_id) = 0 THEN 0 
                                 ELSE ROUND((COUNT(CASE WHEN vp.completed = 1 THEN 1 END) / COUNT(v.video_id)) * 100, 1) 
                                 END FROM videos v 
                                 LEFT JOIN video_progress vp ON v.video_id = vp.video_id AND vp.student_id = e.student_id 
                                 WHERE v.course_id = c.course_id AND v.is_published = 1) >= 100 
                                THEN 1 ELSE 0 END) as completed_enrollments
                        FROM enrollments e
                        JOIN courses c ON e.course_id = c.course_id
                        WHERE c.teacher_id = ?";
    $stmt = $conn->prepare($completion_query);
    if ($stmt) {
        $stmt->bind_param("i", $teacher_id);
        $stmt->execute();
        $result = $stmt->get_result()->fetch_assoc();
        $total = $result['total_enrollments'] ?? 0;
        $completed = $result['completed_enrollments'] ?? 0;
        $completion_rate = ($total > 0) ? ($completed / $total) * 100 : 0;
        $stmt->close();
    }
    
    // 3. Average Rating (using EXISTING course_reviews table)
    $rating_query = "SELECT AVG(rating) as avg_rating
                    FROM course_reviews cr
                    JOIN courses c ON cr.course_id = c.course_id
                    WHERE c.teacher_id = ?";
    $stmt = $conn->prepare($rating_query);
    if ($stmt) {
        $stmt->bind_param("i", $teacher_id);
        $stmt->execute();
        $result = $stmt->get_result()->fetch_assoc();
        $avg_rating = $result['avg_rating'] ?? 0;
        $stmt->close();
    }
    
    // 4. New Students This Month (using EXISTING enrollments table)
    $new_students_query = "SELECT COUNT(DISTINCT e.student_id) as new_students
                          FROM enrollments e
                          JOIN courses c ON e.course_id = c.course_id
                          WHERE c.teacher_id = ?
                          AND MONTH(e.enrolled_date) = MONTH(CURRENT_DATE())
                          AND YEAR(e.enrolled_date) = YEAR(CURRENT_DATE())";
    $stmt = $conn->prepare($new_students_query);
    if ($stmt) {
        $stmt->bind_param("i", $teacher_id);
        $stmt->execute();
        $result = $stmt->get_result()->fetch_assoc();
        $new_students_this_month = $result['new_students'] ?? 0;
        $stmt->close();
    }
    
} catch (Exception $e) {
    error_log("Teaching insights calculation failed: " . $e->getMessage());
}

// Format insights values for display
$watch_time_display = number_format($avg_watch_time, 1) . 'h';
$completion_rate_display = number_format($completion_rate, 0) . '%';
$rating_display = number_format($avg_rating, 1);
$new_students_display = $new_students_this_month;

// Get Recent Students (enrolled within last 7 days)
$recent_students = [];
try {
    $recent_students_query = "SELECT 
                                s.first_name,
                                s.last_name,
                                c.title as course_title,
                                e.enrolled_date,
                                DATEDIFF(NOW(), e.enrolled_date) as days_ago,
                                CASE 
                                    WHEN DATEDIFF(NOW(), e.enrolled_date) = 0 THEN CONCAT(TIMESTAMPDIFF(HOUR, e.enrolled_date, NOW()), ' hours ago')
                                    WHEN DATEDIFF(NOW(), e.enrolled_date) = 1 THEN '1 day ago'
                                    ELSE CONCAT(DATEDIFF(NOW(), e.enrolled_date), ' days ago')
                                END as time_display
                            FROM enrollments e
                            JOIN courses c ON e.course_id = c.course_id
                            LEFT JOIN students s ON e.student_id = s.student_id
                            WHERE c.teacher_id = ?
                            AND DATEDIFF(NOW(), e.enrolled_date) <= 7
                            ORDER BY e.enrolled_date DESC
                            LIMIT 5";
    
    $stmt = $conn->prepare($recent_students_query);
    if ($stmt) {
        $stmt->bind_param("i", $teacher_id);
        $stmt->execute();
        $recent_result = $stmt->get_result();
        while ($student = $recent_result->fetch_assoc()) {
            $recent_students[] = $student;
        }
        $stmt->close();
    }
} catch (Exception $e) {
    error_log("Recent students query failed: " . $e->getMessage());
}

// Define recent items functions
function dashboard_get_teacher_recent_courses($conn, $teacher_id, $limit = 5) {
    // Emergency account
    if ($teacher_id == 999) {
        return [
            ['course_id' => 1, 'title' => 'Introduction to PHP', 'description' => 'Learn PHP basics', 'students' => 12, 'created_at' => '2025-06-15'],
            ['course_id' => 2, 'title' => 'Advanced JavaScript', 'description' => 'Master JS concepts', 'students' => 8, 'created_at' => '2025-06-10'],
            ['course_id' => 3, 'title' => 'MySQL Database Design', 'description' => 'Design efficient databases', 'students' => 15, 'created_at' => '2025-06-05']
        ];
    }
    
    if (!$conn) return [];
    
    try {
        $stmt = $conn->prepare("SELECT * FROM courses WHERE teacher_id = ? ORDER BY created_at DESC LIMIT ?");
        if (!$stmt) return [];
        
        $stmt->bind_param("ii", $teacher_id, $limit);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $courses = [];
        while ($row = $result->fetch_assoc()) {
            $courses[] = $row;
        }
        
        return $courses;
    } catch (Exception $e) {
        error_log("Recent courses error: " . $e->getMessage());
    }
    
    return [];
}

function dashboard_get_teacher_recent_videos($conn, $teacher_id, $limit = 5) {
    // Debug info
    error_log("Fetching videos for teacher_id: {$teacher_id}");
    
    // Emergency account
    if ($teacher_id == 999) {
        return [
            ['video_id' => 1, 'title' => 'PHP Arrays Tutorial', 'duration' => '12:45', 'views' => 156, 'created_at' => '2025-06-18', 'is_published' => 1, 'thumbnail' => '../assets/images/video-thumbnail.jpg'],
            ['video_id' => 2, 'title' => 'JavaScript DOM Manipulation', 'duration' => '18:30', 'views' => 128, 'created_at' => '2025-06-13', 'is_published' => 1, 'thumbnail' => ''],
            ['video_id' => 3, 'title' => 'MySQL Joins Explained', 'duration' => '22:10', 'views' => 94, 'created_at' => '2025-06-08', 'is_published' => 0, 'thumbnail' => '../assets/images/video-thumbnail.jpg']
        ];
    }
    
    if (!$conn) {
        error_log("No database connection available for videos");
        return [];
    }
    
    try {
        // Ensure we're using the correct table name: 'videos'
        $query = "SELECT * FROM videos WHERE teacher_id = ? ORDER BY created_at DESC LIMIT ?"; 
        error_log("Videos query: {$query}");
        
        $stmt = $conn->prepare($query);
        if (!$stmt) {
            error_log("Failed to prepare statement for videos: " . $conn->error);
            return [];
        }
        
        $stmt->bind_param("ii", $teacher_id, $limit);
        
        if (!$stmt->execute()) {
            error_log("Failed to execute videos query: " . $stmt->error);
            return [];
        }
        
        $result = $stmt->get_result();
        error_log("Found {$result->num_rows} videos for teacher {$teacher_id}");
        
        $videos = [];
        while ($row = $result->fetch_assoc()) {
            // Ensure created_at has a valid format for date()
            if (isset($row['created_at'])) {
                if (!strtotime($row['created_at'])) {
                    $row['created_at'] = date('Y-m-d H:i:s');
                }
            } else {
                $row['created_at'] = date('Y-m-d H:i:s');
            }
            
            // Ensure is_published is set
            if (!isset($row['is_published'])) {
                $row['is_published'] = 0;
            }
            
            // Ensure views is set
            if (!isset($row['views'])) {
                $row['views'] = 0;
            }
            
            $videos[] = $row;
        }
        
        return $videos;
    } catch (Exception $e) {
        error_log("Recent videos error: " . $e->getMessage());
    }
    
    return [];
}

// Get recent courses with error handling
$recent_courses = dashboard_get_teacher_recent_courses($conn, $teacher_id, 5);

// Get recent videos with error handling
$recent_videos = dashboard_get_teacher_recent_videos($conn, $teacher_id, 5);

// If no videos, at least initialize as empty array
if (!$recent_videos) {
    $recent_videos = [];
}
// Note: Removed header_content.php include as it was empty and causing sidebar toggle conflicts
// The teacher_layout.php already includes all necessary components (sidebar, navbar, etc.)
?>

<!-- Admin Dashboard Style -->
<style>
    :root {
        --primary-blue: #4285f4;
        --danger-red: #ea4335;
        --success-green: #34a853;
        --warning-purple: #9c27b0;
        --light-bg: #f8f9fb;
        --white: #ffffff;
        --text-dark: #1f2937;
        --text-muted: #6b7280;
        --border-light: #e5e7eb;
        --border-radius: 8px;
        --shadow-sm: 0 1px 3px 0 rgba(0, 0, 0, 0.1);
        --shadow-md: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
    }

    body {
        background: var(--light-bg) !important;
        font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
    }

    .page-content {
        background: var(--light-bg);
        min-height: 100vh;
        padding: 32px 24px;
    }

    .container-fluid {
        max-width: 1200px;
        margin: 0 auto;
    }

    /* Modern Typography */
    * {
        box-sizing: border-box;
    }

    h1, h2, h3, h4, h5, h6 {
        font-weight: 600;
        letter-spacing: -0.025em;
        color: var(--text-dark);
    }

    p, span, div {
        font-weight: 400;
        line-height: 1.5;
    }

    /* Modern Page Header */
    .page-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 40px;
        padding: 0 0 24px 0;
        border-bottom: 1px solid rgba(0, 0, 0, 0.06);
        position: relative;
    }

    .page-header::after {
        content: '';
        position: absolute;
        bottom: -1px;
        left: 0;
        width: 60px;
        height: 2px;
        background: linear-gradient(90deg, var(--primary-blue), var(--success-green));
        border-radius: 1px;
    }

    .page-header h1 {
        font-size: 2rem;
        font-weight: 700;
        color: var(--text-dark);
        margin: 0;
        letter-spacing: -0.04em;
        position: relative;
    }

    .page-header .btn {
        padding: 12px 24px;
        font-weight: 500;
        border-radius: 8px;
        box-shadow: 0 2px 8px rgba(66, 133, 244, 0.2);
    }

    /* Compact Stats Cards - Single Row */
    .stats-cards {
        display: grid;
        grid-template-columns: repeat(4, 1fr);
        gap: 16px;
        margin-bottom: 32px;
    }

    .stat-card {
        background: var(--white);
        border: 1px solid rgba(0, 0, 0, 0.04);
        border-radius: 12px;
        padding: 16px;
        box-shadow: 0 1px 3px rgba(0, 0, 0, 0.05), 0 4px 6px rgba(0, 0, 0, 0.04);
        transition: all 0.4s cubic-bezier(0.175, 0.885, 0.32, 1.275);
        position: relative;
        overflow: hidden;
        backdrop-filter: blur(10px);
        min-height: 120px;
    }

    .stat-card:hover {
        box-shadow: 0 10px 25px rgba(0, 0, 0, 0.1), 0 20px 48px rgba(0, 0, 0, 0.08);
        transform: translateY(-6px) scale(1.02);
        border-color: rgba(0, 0, 0, 0.08);
    }

    .stat-card::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        height: 3px;
        background: linear-gradient(90deg, 
            var(--primary-blue), 
            var(--danger-red), 
            var(--success-green), 
            var(--warning-purple)
        );
        opacity: 0;
        transition: opacity 0.4s ease;
    }

    .stat-card:hover::before {
        opacity: 1;
    }

    .stat-card-header {
        display: flex;
        justify-content: space-between;
        align-items: flex-start;
        margin-bottom: 12px;
    }

    .stat-content {
        flex: 1;
        padding-right: 12px;
    }

    .stat-label {
        font-size: 0.75rem;
        color: var(--text-muted);
        font-weight: 600;
        margin: 0 0 6px 0;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        opacity: 0.8;
    }

    .stat-number {
        font-size: 1.75rem;
        font-weight: 800;
        color: var(--text-dark);
        line-height: 1;
        margin-bottom: 6px;
        background: linear-gradient(135deg, var(--text-dark) 0%, rgba(31, 41, 55, 0.8) 100%);
        -webkit-background-clip: text;
        -webkit-text-fill-color: transparent;
        background-clip: text;
    }

    .stat-description {
        font-size: 0.75rem;
        color: var(--text-muted);
        margin-bottom: 8px;
        line-height: 1.3;
        opacity: 0.8;
    }

    .stat-icon {
        width: 44px;
        height: 44px;
        border-radius: 12px;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 18px;
        color: white;
        flex-shrink: 0;
        box-shadow: 0 3px 8px rgba(0, 0, 0, 0.15);
        position: relative;
        overflow: hidden;
    }

    .stat-icon::before {
        content: '';
        position: absolute;
        top: -50%;
        left: -50%;
        width: 200%;
        height: 200%;
        background: linear-gradient(45deg, transparent, rgba(255, 255, 255, 0.1), transparent);
        transform: rotate(45deg);
        transition: transform 0.6s ease;
        opacity: 0;
    }

    .stat-card:hover .stat-icon::before {
        transform: rotate(45deg) translate(50%, 50%);
        opacity: 1;
    }

    /* Colorful icons matching admin dashboard exactly */
    .stat-card:nth-child(1) .stat-icon {
        background: linear-gradient(135deg, #4285f4, #1a73e8);
    }

    .stat-card:nth-child(2) .stat-icon {
        background: linear-gradient(135deg, #ea4335, #d33b2c);
    }

    .stat-card:nth-child(3) .stat-icon {
        background: linear-gradient(135deg, #34a853, #2d9643);
    }

    .stat-card:nth-child(4) .stat-icon {
        background: linear-gradient(135deg, #9c27b0, #8e24aa);
    }

    .stat-link {
        display: inline-flex;
        align-items: center;
        gap: 8px;
        font-size: 0.875rem;
        color: var(--primary-blue);
        text-decoration: none;
        font-weight: 600;
        margin-top: 20px;
        padding: 8px 16px;
        background: rgba(66, 133, 244, 0.05);
        border-radius: 8px;
        border: 1px solid rgba(66, 133, 244, 0.1);
        transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        position: relative;
        overflow: hidden;
    }

    .stat-link::before {
        content: '';
        position: absolute;
        top: 0;
        left: -100%;
        width: 100%;
        height: 100%;
        background: linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.2), transparent);
        transition: left 0.5s ease;
    }

    .stat-link:hover {
        color: white;
        background: var(--primary-blue);
        border-color: var(--primary-blue);
        text-decoration: none;
        transform: translateY(-1px);
        box-shadow: 0 4px 12px rgba(66, 133, 244, 0.3);
    }

    .stat-link:hover::before {
        left: 100%;
    }

    /* Modern Dashboard Content */
    .dashboard-content {
        background: var(--white);
        border: 1px solid rgba(0, 0, 0, 0.04);
        border-radius: 16px;
        padding: 32px;
        margin-bottom: 32px;
        box-shadow: 0 1px 3px rgba(0, 0, 0, 0.05), 0 4px 6px rgba(0, 0, 0, 0.04);
        transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        position: relative;
        overflow: hidden;
        backdrop-filter: blur(10px);
    }

    .dashboard-content::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        height: 4px;
        background: linear-gradient(90deg, var(--primary-blue), var(--success-green));
        opacity: 0;
        transition: opacity 0.3s ease;
    }

    .dashboard-content:hover {
        box-shadow: 0 8px 25px rgba(0, 0, 0, 0.08), 0 16px 32px rgba(0, 0, 0, 0.04);
        transform: translateY(-2px);
        border-color: rgba(0, 0, 0, 0.08);
    }

    .dashboard-content:hover::before {
        opacity: 1;
    }

    .section-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 32px;
        padding-bottom: 16px;
        border-bottom: 1px solid rgba(0, 0, 0, 0.06);
        position: relative;
    }

    .section-title {
        font-size: 1.5rem;
        font-weight: 700;
        color: var(--text-dark);
        margin: 0;
        letter-spacing: -0.02em;
        position: relative;
    }

    .section-title::after {
        content: '';
        position: absolute;
        bottom: -17px;
        left: 0;
        width: 40px;
        height: 2px;
        background: var(--primary-blue);
        border-radius: 1px;
    }

    .btn {
        padding: 12px 24px;
        border-radius: 10px;
        font-size: 0.875rem;
        font-weight: 600;
        text-decoration: none;
        border: none;
        cursor: pointer;
        transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        display: inline-flex;
        align-items: center;
        gap: 8px;
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.08);
        position: relative;
        overflow: hidden;
        letter-spacing: 0.025em;
    }

    .btn::before {
        content: '';
        position: absolute;
        top: 0;
        left: -100%;
        width: 100%;
        height: 100%;
        background: linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.2), transparent);
        transition: left 0.5s ease;
    }

    .btn:hover::before {
        left: 100%;
    }

    .btn-primary {
        background: linear-gradient(135deg, var(--primary-blue), #1a73e8);
        color: white;
        box-shadow: 0 4px 14px rgba(66, 133, 244, 0.3);
    }

    .btn-primary:hover {
        background: linear-gradient(135deg, #1a73e8, #1557b0);
        color: white;
        text-decoration: none;
        transform: translateY(-2px) scale(1.02);
        box-shadow: 0 8px 25px rgba(66, 133, 244, 0.4);
    }

    .btn-outline-primary {
        background: transparent;
        border: 2px solid var(--primary-blue);
        color: var(--primary-blue);
        box-shadow: 0 2px 8px rgba(66, 133, 244, 0.1);
    }

    .btn-outline-primary:hover {
        background: var(--primary-blue);
        color: white;
        text-decoration: none;
        transform: translateY(-2px) scale(1.02);
        box-shadow: 0 8px 25px rgba(66, 133, 244, 0.3);
        border-color: var(--primary-blue);
    }

    .btn-sm {
        padding: 10px 20px;
        font-size: 0.8125rem;
    }

    /* Responsive Design */
    @media (max-width: 768px) {
        .stats-cards {
            grid-template-columns: 1fr;
        }
        
        .page-header {
            flex-direction: column;
            align-items: flex-start;
            gap: 16px;
        }
        
        .dashboard-content {
            padding: 20px;
        }
        
        /* MOBILE FIX: Stack Activity & Wallet cards vertically */
        .activity-wallet-grid {
            grid-template-columns: 1fr !important;
            gap: 24px !important;
            margin: 24px 0 !important;
        }
        
        /* Ensure proper spacing on mobile */
        .modern-section-card {
            margin-bottom: 0 !important;
        }
        
        /* Fix wallet stats to stack on very small screens */
        .wallet-stats {
            grid-template-columns: 1fr !important;
            gap: 12px !important;
        }
        
        /* Stack wallet actions vertically on small screens */
        .wallet-actions {
            flex-direction: column !important;
            gap: 12px !important;
        }
    }

    /* Modern Activity Cards */
    .list-group-flush .list-group-item {
        border: 1px solid rgba(0, 0, 0, 0.04);
        padding: 20px;
        margin-bottom: 12px;
        border-radius: 12px;
        background: var(--white);
        box-shadow: 0 1px 3px rgba(0, 0, 0, 0.05), 0 4px 6px rgba(0, 0, 0, 0.04);
        transition: all 0.4s cubic-bezier(0.175, 0.885, 0.32, 1.275);
        position: relative;
        overflow: hidden;
    }

    .list-group-flush .list-group-item:last-child {
        border-bottom: 1px solid rgba(0, 0, 0, 0.04);
        margin-bottom: 0;
    }

    .list-group-flush .list-group-item:hover {
        background-color: var(--white);
        border-radius: 12px;
        margin: 0 0 12px 0;
        padding: 20px;
        transform: translateY(-4px) scale(1.02);
        box-shadow: 0 10px 25px rgba(0, 0, 0, 0.08), 0 20px 48px rgba(0, 0, 0, 0.04);
        border-color: rgba(66, 133, 244, 0.2);
    }

    .list-group-flush .list-group-item::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        width: 4px;
        height: 100%;
        background: linear-gradient(135deg, var(--primary-blue), var(--success-green));
        opacity: 0;
        transition: opacity 0.3s ease;
    }

    .list-group-flush .list-group-item:hover::before {
        opacity: 1;
    }

    .badge {
        font-size: 0.75rem;
        padding: 8px 16px;
        border-radius: 20px;
        font-weight: 600;
        border: none;
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
        text-transform: uppercase;
        letter-spacing: 0.025em;
    }

    .bg-primary {
        background: linear-gradient(135deg, var(--primary-blue), #1a73e8) !important;
    }

    .bg-success {
        background: linear-gradient(135deg, var(--success-green), #2d9643) !important;
    }

    .bg-warning {
        background: linear-gradient(135deg, var(--warning-purple), #8e24aa) !important;
    }

    .bg-secondary {
        background: linear-gradient(135deg, var(--text-muted), #4b5563) !important;
    }

    /* Modern Activity Icons */
    .rounded.p-2 {
        width: 52px;
        height: 52px;
        border-radius: 14px !important;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 22px;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        position: relative;
        overflow: hidden;
    }

    .rounded.p-2::before {
        content: '';
        position: absolute;
        top: -50%;
        left: -50%;
        width: 200%;
        height: 200%;
        background: linear-gradient(45deg, transparent, rgba(255, 255, 255, 0.1), transparent);
        transform: rotate(45deg);
        transition: transform 0.6s ease;
        opacity: 0;
    }

    .list-group-item:hover .rounded.p-2::before {
        transform: rotate(45deg) translate(50%, 50%);
        opacity: 1;
    }

    /* Enhanced Typography for Activities */
    .list-group-item h6 {
        font-weight: 700;
        font-size: 1rem;
        color: var(--text-dark);
        margin-bottom: 4px;
        letter-spacing: -0.025em;
    }

    .list-group-item p.text-muted {
        font-size: 0.875rem;
        line-height: 1.4;
        opacity: 0.8;
    }

    .list-group-item small.text-muted {
        font-size: 0.8125rem;
        font-weight: 500;
        opacity: 0.7;
    }

    /* Modern Quick Actions Grid */
    .quick-actions-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
        gap: 24px;
        margin-top: 8px;
    }

    .quick-action-card {
        background: var(--white);
        border: 1px solid rgba(0, 0, 0, 0.04);
        border-radius: 16px;
        padding: 24px;
        text-decoration: none;
        color: inherit;
        box-shadow: 0 1px 3px rgba(0, 0, 0, 0.05), 0 4px 6px rgba(0, 0, 0, 0.04);
        transition: all 0.4s cubic-bezier(0.175, 0.885, 0.32, 1.275);
        position: relative;
        overflow: hidden;
        display: flex;
        align-items: center;
        gap: 20px;
    }

    .quick-action-card::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        height: 4px;
        background: linear-gradient(90deg, 
            var(--primary-blue), 
            var(--success-green), 
            var(--warning-purple), 
            var(--danger-red)
        );
        opacity: 0;
        transition: opacity 0.4s ease;
    }

    .quick-action-card:hover {
        text-decoration: none;
        color: inherit;
        transform: translateY(-6px) scale(1.02);
        box-shadow: 0 10px 25px rgba(0, 0, 0, 0.08), 0 20px 48px rgba(0, 0, 0, 0.04);
        border-color: rgba(0, 0, 0, 0.08);
    }

    .quick-action-card:hover::before {
        opacity: 1;
    }

    .quick-action-icon {
        width: 60px;
        height: 60px;
        border-radius: 16px;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 26px;
        color: white;
        flex-shrink: 0;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        position: relative;
        overflow: hidden;
    }

    .quick-action-icon.primary {
        background: linear-gradient(135deg, var(--primary-blue), #1a73e8);
    }

    .quick-action-icon.success {
        background: linear-gradient(135deg, var(--success-green), #2d9643);
    }

    .quick-action-icon.warning {
        background: linear-gradient(135deg, var(--warning-purple), #8e24aa);
    }

    .quick-action-icon.danger {
        background: linear-gradient(135deg, var(--danger-red), #d33b2c);
    }

    .quick-action-icon::before {
        content: '';
        position: absolute;
        top: -50%;
        left: -50%;
        width: 200%;
        height: 200%;
        background: linear-gradient(45deg, transparent, rgba(255, 255, 255, 0.1), transparent);
        transform: rotate(45deg);
        transition: transform 0.6s ease;
        opacity: 0;
    }

    .quick-action-card:hover .quick-action-icon::before {
        transform: rotate(45deg) translate(50%, 50%);
        opacity: 1;
    }

    .quick-action-content {
        flex: 1;
    }

    .quick-action-content h3 {
        font-size: 1.125rem;
        font-weight: 700;
        color: var(--text-dark);
        margin: 0 0 6px 0;
        letter-spacing: -0.025em;
    }

    .quick-action-content p {
        font-size: 0.875rem;
        color: var(--text-muted);
        margin: 0;
        opacity: 0.8;
        line-height: 1.4;
    }

    .quick-action-arrow {
        width: 36px;
        height: 36px;
        border-radius: 8px;
        background: rgba(66, 133, 244, 0.08);
        display: flex;
        align-items: center;
        justify-content: center;
        color: var(--primary-blue);
        font-size: 18px;
        transition: all 0.3s ease;
        flex-shrink: 0;
    }

    .quick-action-card:hover .quick-action-arrow {
        background: var(--primary-blue);
        color: white;
        transform: translateX(4px);
    }

    /* Modern Teaching Insights Grid */
    .insights-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
        gap: 24px;
        margin-top: 8px;
    }

    .modern-insight-card {
        background: var(--white);
        border: 1px solid rgba(0, 0, 0, 0.04);
        border-radius: 16px;
        padding: 24px;
        box-shadow: 0 1px 3px rgba(0, 0, 0, 0.05), 0 4px 6px rgba(0, 0, 0, 0.04);
        transition: all 0.4s cubic-bezier(0.175, 0.885, 0.32, 1.275);
        position: relative;
        overflow: hidden;
        backdrop-filter: blur(10px);
    }

    .modern-insight-card::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        height: 4px;
        background: linear-gradient(90deg, 
            var(--primary-blue), 
            var(--success-green), 
            var(--warning-purple), 
            var(--danger-red)
        );
        opacity: 0;
        transition: opacity 0.4s ease;
    }

    .modern-insight-card:hover {
        transform: translateY(-6px) scale(1.02);
        box-shadow: 0 10px 25px rgba(0, 0, 0, 0.1), 0 20px 48px rgba(0, 0, 0, 0.06);
        border-color: rgba(0, 0, 0, 0.08);
    }

        .modern-insight-card:hover::before {
            opacity: 1;
        }

        .insight-card-header {
            display: flex;
            align-items: flex-start;
            justify-content: space-between;
            gap: 20px;
        }

        .insight-content {
            flex: 1;
        }

        .insight-label {
            font-size: 0.875rem;
            font-weight: 600;
            color: var(--text-muted);
            margin: 0 0 12px 0;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            line-height: 1.2;
        }

        .insight-number {
            font-size: 2.25rem;
            font-weight: 800;
            background: linear-gradient(135deg, var(--primary-blue), var(--success-green));
            background-clip: text;
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            line-height: 1;
            margin-bottom: 12px;
            letter-spacing: -0.02em;
        }

        .insight-trend {
            display: flex;
            align-items: center;
            gap: 6px;
            font-size: 0.825rem;
            font-weight: 600;
            margin: 0;
        }

        .insight-trend.success {
            color: var(--success-green);
        }

        .insight-trend.warning {
            color: var(--warning-purple);
        }

        .insight-trend.danger {
            color: var(--danger-red);
        }

        .insight-trend i {
            font-size: 16px;
        }

        .modern-insight-card .insight-icon {
            width: 60px;
            height: 60px;
            border-radius: 16px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 26px;
            color: white;
            flex-shrink: 0;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
            position: relative;
            overflow: hidden;
        }
        border-radius: 16px;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 26px;
        color: white;
        flex-shrink: 0;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        position: relative;
        overflow: hidden;
    }

    .modern-insight-card .insight-icon.primary {
        background: linear-gradient(135deg, var(--primary-blue), #1a73e8);
    }

    .modern-insight-card .insight-icon.success {
        background: linear-gradient(135deg, var(--success-green), #2d9643);
    }

    .modern-insight-card .insight-icon.warning {
        background: linear-gradient(135deg, var(--warning-purple), #8e24aa);
    }

    .modern-insight-card .insight-icon.danger {
        background: linear-gradient(135deg, var(--danger-red), #d33b2c);
    }

    .modern-insight-card .insight-icon::before {
        content: '';
        position: absolute;
        top: -50%;
        left: -50%;
        width: 200%;
        height: 200%;
        background: linear-gradient(45deg, transparent, rgba(255, 255, 255, 0.15), transparent);
        transform: rotate(45deg);
        transition: transform 0.6s ease;
        opacity: 0;
    }

    .modern-insight-card:hover .insight-icon::before {
        transform: rotate(45deg) translate(50%, 50%);
        opacity: 1;
    }

    /* Modern Bottom Sections Grid */
    .bottom-sections-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(400px, 1fr));
        gap: 32px;
        margin-top: 32px;
    }

    .modern-section-card {
        background: var(--white);
        border: 1px solid rgba(0, 0, 0, 0.04);
        border-radius: 16px;
        box-shadow: 0 1px 3px rgba(0, 0, 0, 0.05), 0 4px 6px rgba(0, 0, 0, 0.04);
        transition: all 0.4s cubic-bezier(0.175, 0.885, 0.32, 1.275);
        position: relative;
        overflow: hidden;
        backdrop-filter: blur(10px);
    }

    .modern-section-card::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        height: 4px;
        background: linear-gradient(90deg, 
            var(--primary-blue), 
            var(--success-green), 
            var(--warning-purple), 
            var(--danger-red)
        );
        opacity: 0;
        transition: opacity 0.4s ease;
    }

    .modern-section-card:hover {
        transform: translateY(-4px);
        box-shadow: 0 8px 20px rgba(0, 0, 0, 0.08), 0 16px 32px rgba(0, 0, 0, 0.04);
        border-color: rgba(0, 0, 0, 0.08);
    }

    .modern-section-card:hover::before {
        opacity: 1;
    }

    .section-card-header {
        padding: 24px 24px 0;
        display: flex;
        align-items: center;
        justify-content: space-between;
        margin-bottom: 20px;
    }

    .view-all-link {
        display: flex;
        align-items: center;
        gap: 6px;
        font-size: 0.875rem;
        font-weight: 600;
        color: var(--primary-blue);
        text-decoration: none;
        padding: 8px 16px;
        border-radius: 8px;
        background: rgba(66, 133, 244, 0.08);
        transition: all 0.3s ease;
    }

    .view-all-link:hover {
        background: var(--primary-blue);
        color: white;
        text-decoration: none;
        transform: translateX(2px);
    }

    .section-card-body {
        padding: 0 24px 24px;
    }

    /* Modern Student Items */
    .modern-student-item {
        display: flex;
        align-items: center;
        gap: 16px;
        padding: 16px;
        margin-bottom: 12px;
        background: rgba(248, 249, 251, 0.6);
        border: 1px solid rgba(0, 0, 0, 0.03);
        border-radius: 12px;
        transition: all 0.3s ease;
    }

    .modern-student-item:hover {
        background: var(--white);
        border-color: rgba(0, 0, 0, 0.06);
        transform: translateX(4px);
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.06);
    }

    .student-avatar {
        width: 48px;
        height: 48px;
        border-radius: 12px;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 14px;
        font-weight: 700;
        color: white;
        flex-shrink: 0;
        box-shadow: 0 3px 8px rgba(0, 0, 0, 0.12);
    }

    .student-avatar.primary {
        background: linear-gradient(135deg, var(--primary-blue), #1a73e8);
    }

    .student-avatar.success {
        background: linear-gradient(135deg, var(--success-green), #2d9643);
    }

    .student-avatar.warning {
        background: linear-gradient(135deg, var(--warning-purple), #8e24aa);
    }

    .student-content {
        flex: 1;
    }

    .student-content h4 {
        font-size: 1rem;
        font-weight: 600;
        color: var(--text-dark);
        margin: 0 0 4px 0;
        letter-spacing: -0.025em;
    }

    .student-content p {
        font-size: 0.875rem;
        color: var(--text-muted);
        margin: 0 0 4px 0;
        line-height: 1.4;
    }

    .student-content small {
        font-size: 0.8125rem;
        color: var(--text-muted);
        opacity: 0.7;
        font-weight: 500;
    }

    .student-status {
        width: 32px;
        height: 32px;
        border-radius: 8px;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 16px;
        flex-shrink: 0;
    }

    .student-status.new {
        background: rgba(66, 133, 244, 0.1);
        color: var(--primary-blue);
    }

    .student-status.active {
        background: rgba(16, 185, 129, 0.1);
        color: var(--success-green);
    }

    .student-status.completed {
        background: rgba(139, 69, 19, 0.1);
        color: #8b4513;
    }

    /* Modern Empty State */
    .modern-empty-state {
        text-align: center;
        padding: 48px 24px;
    }

    .empty-icon {
        width: 80px;
        height: 80px;
        border-radius: 20px;
        background: linear-gradient(135deg, rgba(66, 133, 244, 0.1), rgba(16, 185, 129, 0.1));
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 36px;
        color: var(--primary-blue);
        margin: 0 auto 20px;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.08);
    }

    .modern-empty-state h4 {
        font-size: 1.25rem;
        font-weight: 700;
        color: var(--text-dark);
        margin: 0 0 12px 0;
        letter-spacing: -0.025em;
    }

    .modern-empty-state p {
        font-size: 0.9375rem;
        color: var(--text-muted);
        margin: 0 0 24px 0;
        line-height: 1.5;
    }

    .empty-action-btn {
        display: inline-flex;
        align-items: center;
        gap: 8px;
        padding: 12px 24px;
        background: linear-gradient(135deg, var(--primary-blue), #1a73e8);
        color: white;
        text-decoration: none;
        border-radius: 8px;
        font-weight: 600;
        font-size: 0.875rem;
        box-shadow: 0 4px 12px rgba(66, 133, 244, 0.3);
        transition: all 0.3s ease;
    }

    .empty-action-btn:hover {
        background: linear-gradient(135deg, #1a73e8, var(--primary-blue));
        color: white;
        text-decoration: none;
        transform: translateY(-2px);
        box-shadow: 0 6px 16px rgba(66, 133, 244, 0.4);
    }

    /* Modern Course Items */
    .modern-course-item {
        display: flex;
        align-items: center;
        gap: 16px;
        padding: 16px;
        margin-bottom: 12px;
        background: rgba(248, 249, 251, 0.6);
        border: 1px solid rgba(0, 0, 0, 0.03);
        border-radius: 12px;
        transition: all 0.3s ease;
    }

    .modern-course-item:hover {
        background: var(--white);
        border-color: rgba(0, 0, 0, 0.06);
        transform: translateX(4px);
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.06);
    }

    .course-icon {
        width: 48px;
        height: 48px;
        border-radius: 12px;
        background: linear-gradient(135deg, var(--success-green), #2d9643);
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 20px;
        color: white;
        flex-shrink: 0;
        box-shadow: 0 3px 8px rgba(16, 185, 129, 0.3);
    }

    .course-content {
        flex: 1;
    }

    .course-content h4 {
        font-size: 1rem;
        font-weight: 600;
        color: var(--text-dark);
        margin: 0 0 4px 0;
        letter-spacing: -0.025em;
    }

    .course-content p {
        font-size: 0.875rem;
        color: var(--text-muted);
        margin: 0;
        line-height: 1.4;
    }

    .course-status {
        flex-shrink: 0;
    }

    .course-status.published span {
        display: inline-block;
        padding: 6px 12px;
        background: rgba(16, 185, 129, 0.1);
        color: var(--success-green);
        font-size: 0.8125rem;
        font-weight: 600;
        border-radius: 6px;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }

    /* Activity & Wallet 50-50 Grid */
    .activity-wallet-grid {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 32px;
        margin: 32px 0;
    }

    /* Modern Activity Items */
    .modern-activity-item {
        display: flex;
        align-items: center;
        gap: 16px;
        padding: 16px;
        margin-bottom: 12px;
        background: rgba(248, 249, 251, 0.6);
        border: 1px solid rgba(0, 0, 0, 0.03);
        border-radius: 12px;
        transition: all 0.3s ease;
    }

    .modern-activity-item:hover {
        background: var(--white);
        border-color: rgba(0, 0, 0, 0.06);
        transform: translateX(4px);
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.06);
    }

    .modern-activity-item .activity-icon {
        width: 44px;
        height: 44px;
        border-radius: 12px;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 18px;
        color: white;
        flex-shrink: 0;
        box-shadow: 0 3px 8px rgba(0, 0, 0, 0.12);
    }

    .modern-activity-item .activity-icon.primary {
        background: linear-gradient(135deg, var(--primary-blue), #1a73e8);
    }

    .modern-activity-item .activity-icon.success {
        background: linear-gradient(135deg, var(--success-green), #2d9643);
    }

    .modern-activity-item .activity-icon.warning {
        background: linear-gradient(135deg, var(--warning-purple), #8e24aa);
    }

    .activity-content {
        flex: 1;
    }

    .activity-content h4 {
        font-size: 0.95rem;
        font-weight: 600;
        color: var(--text-dark);
        margin: 0 0 4px 0;
        letter-spacing: -0.025em;
    }

    .activity-content p {
        font-size: 0.8125rem;
        color: var(--text-muted);
        margin: 0 0 4px 0;
        line-height: 1.4;
    }

    .activity-content small {
        font-size: 0.75rem;
        color: var(--text-muted);
        opacity: 0.7;
        font-weight: 500;
    }

    .activity-badge {
        padding: 4px 10px;
        border-radius: 6px;
        font-size: 0.75rem;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        flex-shrink: 0;
    }

    .activity-badge.new {
        background: rgba(66, 133, 244, 0.1);
        color: var(--primary-blue);
    }

    .activity-badge.live {
        background: rgba(16, 185, 129, 0.1);
        color: var(--success-green);
    }

    .activity-badge.rating {
        background: rgba(139, 69, 19, 0.1);
        color: #8b4513;
    }

    .activity-badge.published {
        background: rgba(107, 114, 128, 0.1);
        color: #6b7280;
    }

    /* Teacher's Wallet Styles */
    .wallet-overview {
        display: flex;
        flex-direction: column;
        gap: 24px;
    }

    .wallet-balance {
        display: flex;
        align-items: center;
        gap: 16px;
        padding: 20px;
        background: linear-gradient(135deg, rgba(66, 133, 244, 0.05), rgba(16, 185, 129, 0.05));
        border: 1px solid rgba(66, 133, 244, 0.1);
        border-radius: 12px;
    }

    .balance-icon {
        width: 60px;
        height: 60px;
        border-radius: 16px;
        background: linear-gradient(135deg, var(--primary-blue), #1a73e8);
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 26px;
        color: white;
        flex-shrink: 0;
        box-shadow: 0 4px 12px rgba(66, 133, 244, 0.3);
    }

    .balance-content {
        flex: 1;
    }

    .balance-label {
        font-size: 0.875rem;
        font-weight: 600;
        color: var(--text-muted);
        margin: 0 0 8px 0;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }

    .balance-amount {
        font-size: 2rem;
        font-weight: 800;
        background: linear-gradient(135deg, var(--primary-blue), var(--success-green));
        background-clip: text;
        -webkit-background-clip: text;
        -webkit-text-fill-color: transparent;
        line-height: 1;
        margin-bottom: 6px;
        letter-spacing: -0.02em;
    }

    .balance-status {
        font-size: 0.8125rem;
        color: var(--success-green);
        margin: 0;
        font-weight: 500;
    }

    .wallet-stats {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 16px;
    }

    .wallet-stat {
        display: flex;
        align-items: center;
        gap: 12px;
        padding: 16px;
        background: rgba(248, 249, 251, 0.6);
        border: 1px solid rgba(0, 0, 0, 0.03);
        border-radius: 12px;
        transition: all 0.3s ease;
    }

    .wallet-stat:hover {
        background: var(--white);
        border-color: rgba(0, 0, 0, 0.06);
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.06);
    }

    .wallet-stat .stat-icon {
        width: 40px;
        height: 40px;
        border-radius: 10px;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 16px;
        color: white;
        flex-shrink: 0;
    }

    .wallet-stat .stat-icon.earnings {
        background: linear-gradient(135deg, var(--success-green), #2d9643);
    }

    .wallet-stat .stat-icon.transactions {
        background: linear-gradient(135deg, var(--warning-purple), #8e24aa);
    }

    .wallet-stat .stat-content {
        display: flex;
        flex-direction: column;
        gap: 2px;
    }

    .wallet-stat .stat-value {
        font-size: 1.125rem;
        font-weight: 700;
        color: var(--text-dark);
        line-height: 1;
    }

    .wallet-stat .stat-text {
        font-size: 0.75rem;
        color: var(--text-muted);
        font-weight: 500;
    }

    .wallet-actions {
        display: flex;
        gap: 12px;
    }

    .withdrawal-btn {
        flex: 1;
        display: flex;
        align-items: center;
        justify-content: center;
        gap: 8px;
        padding: 14px 20px;
        background: linear-gradient(135deg, var(--primary-blue), #1a73e8);
        color: white;
        text-decoration: none;
        border-radius: 8px;
        font-weight: 600;
        font-size: 0.875rem;
        box-shadow: 0 4px 12px rgba(66, 133, 244, 0.3);
        transition: all 0.3s ease;
    }

    .withdrawal-btn:hover {
        background: linear-gradient(135deg, #1a73e8, var(--primary-blue));
        color: white;
        text-decoration: none;
        transform: translateY(-2px);
        box-shadow: 0 6px 16px rgba(66, 133, 244, 0.4);
    }

    .view-history-btn {
        flex: 1;
        display: flex;
        align-items: center;
        justify-content: center;
        gap: 8px;
        padding: 14px 20px;
        background: rgba(66, 133, 244, 0.08);
        color: var(--primary-blue);
        text-decoration: none;
        border-radius: 8px;
        font-weight: 600;
        font-size: 0.875rem;
        border: 1px solid rgba(66, 133, 244, 0.2);
        transition: all 0.3s ease;
    }

    .view-history-btn:hover {
        background: var(--primary-blue);
        color: white;
        text-decoration: none;
        border-color: var(--primary-blue);
        transform: translateY(-2px);
        box-shadow: 0 4px 12px rgba(66, 133, 244, 0.25);
    }



    .activity-icon {
        width: 48px;
        height: 48px;
        border-radius: 12px;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 20px;
        color: white;
        flex-shrink: 0;
        box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        position: relative;
    }

    .activity-icon::after {
        content: '';
        position: absolute;
        inset: 0;
        border-radius: 12px;
        background: linear-gradient(135deg, rgba(255,255,255,0.2), transparent);
    }

    .activity-content {
        flex: 1;
        min-width: 0;
    }

    .activity-content h6 {
        font-size: 1rem;
        font-weight: 600;
        margin-bottom: 4px;
        color: var(--text-dark);
        line-height: 1.4;
    }

    .activity-content p {
        font-size: 0.9rem;
        color: var(--text-muted);
        margin-bottom: 6px;
        line-height: 1.4;
    }

    .activity-meta {
        display: flex;
        align-items: center;
        justify-content: space-between;
        margin-top: 8px;
    }

    .activity-time {
        font-size: 0.8rem;
        color: var(--text-muted);
        font-weight: 500;
        display: flex;
        align-items: center;
        gap: 4px;
    }

    .activity-badge {
        background: rgba(37, 99, 235, 0.1);
        color: var(--primary-color);
        padding: 4px 8px;
        border-radius: 6px;
        font-size: 0.7rem;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }

    .activity-badge.success {
        background: rgba(16, 185, 129, 0.1);
        color: var(--success-color);
    }

    .activity-badge.warning {
        background: rgba(245, 158, 11, 0.1);
        color: var(--warning-color);
    }

    .activity-badge.info {
        background: rgba(37, 99, 235, 0.1);
        color: var(--info-color);
    }

    /* Quick Tools */
    .tools-grid {
        display: grid;
        grid-template-columns: repeat(2, 1fr);
        gap: 15px;
    }

    .tool-item {
        display: flex;
        flex-direction: column;
        align-items: center;
        gap: 10px;
        padding: 20px 15px;
        background: rgba(79, 103, 244, 0.05);
        border-radius: 10px;
        text-decoration: none;
        color: var(--text-dark);
        transition: all 0.3s ease;
        border: 1px solid transparent;
    }

    .tool-item:hover {
        background: rgba(79, 103, 244, 0.1);
        transform: translateY(-3px);
        border-color: var(--primary-color);
        color: var(--text-dark);
        text-decoration: none;
    }

    .tool-icon {
        width: 45px;
        height: 45px;
        border-radius: 10px;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 20px;
        color: white;
        background: var(--primary-color);
    }

    .tool-label {
        font-size: 0.85rem;
        font-weight: 500;
        text-align: center;
    }

    /* Recent Lists */
    .recent-item {
        display: flex;
        align-items: center;
        gap: 12px;
        padding: 12px 0;
        border-bottom: 1px solid rgba(0,0,0,0.05);
    }

    .recent-item:last-child {
        border-bottom: none;
    }

    .recent-avatar {
        width: 35px;
        height: 35px;
        border-radius: 8px;
        background: var(--primary-color);
        display: flex;
        align-items: center;
        justify-content: center;
        color: white;
        font-weight: 600;
        font-size: 0.9rem;
    }

    .recent-content h6 {
        font-size: 0.9rem;
        font-weight: 600;
        margin-bottom: 2px;
        color: var(--text-dark);
    }

    .recent-content p {
        font-size: 0.8rem;
        color: var(--text-muted);
        margin: 0;
    }

    /* Teaching Insights - Improved Spacing */
    .insights-grid {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 18px;
    }

    .insight-card {
        background: var(--white);
        border: 1px solid var(--border-light);
        border-radius: 12px;
        padding: 24px 20px;
        transition: all 0.3s ease;
        position: relative;
        overflow: hidden;
        min-height: 140px;
        display: flex;
        flex-direction: column;
        justify-content: space-between;
    }

    .insight-card:hover {
        transform: translateY(-3px);
        box-shadow: var(--shadow-md);
        border-color: var(--primary-color);
    }

    .insight-card::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        height: 3px;
        background: var(--primary-color);
        transform: scaleX(0);
        transition: transform 0.3s ease;
    }

    .insight-card:hover::before {
        transform: scaleX(1);
    }

    .insight-header {
        display: flex;
        align-items: center;
        justify-content: space-between;
        margin-bottom: 16px;
    }

    .insight-icon {
        width: 45px;
        height: 45px;
        border-radius: 12px;
        background: var(--primary-color);
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 20px;
        color: white;
        box-shadow: 0 4px 12px rgba(0,0,0,0.12);
        flex-shrink: 0;
    }

    .insight-icon.secondary {
        background: var(--secondary-color);
        box-shadow: 0 4px 12px rgba(245, 158, 11, 0.3);
    }

    .insight-icon.success {
        background: var(--success-color);
        box-shadow: 0 4px 12px rgba(16, 185, 129, 0.3);
    }

    .insight-icon.warning {
        background: var(--warning-color);
        box-shadow: 0 4px 12px rgba(245, 158, 11, 0.3);
    }

    .insight-icon.danger {
        background: var(--danger-color);
        box-shadow: 0 4px 12px rgba(239, 68, 68, 0.3);
    }

    .insight-value {
        font-size: 2rem;
        font-weight: 700;
        color: var(--text-dark);
        line-height: 1;
    }

    .insight-label {
        font-size: 0.95rem;
        font-weight: 500;
        color: var(--text-muted);
        margin-bottom: 12px;
        line-height: 1.3;
    }

    .insight-trend {
        display: flex;
        align-items: center;
        gap: 6px;
        font-size: 0.85rem;
        font-weight: 500;
        color: var(--primary-color);
        margin-top: auto;
    }

    .insight-trend.success {
        color: var(--success-color);
    }

    .insight-trend.warning {
        color: var(--warning-color);
    }

    .insight-trend.danger {
        color: var(--danger-color);
    }

    /* Responsive adjustments for insights */
    @media (max-width: 1200px) {
        .insights-grid {
            grid-template-columns: 1fr;
            gap: 15px;
        }
        
        .insight-card {
            min-height: 120px;
            padding: 20px 16px;
        }
    }

    /* Empty State */
    .empty-state {
        text-align: center;
        padding: 40px 20px;
        color: var(--text-muted);
    }

    .empty-state i {
        font-size: 3rem;
        margin-bottom: 15px;
        opacity: 0.5;
    }

    .empty-state h6 {
        font-weight: 600;
        margin-bottom: 8px;
    }

    .empty-state p {
        font-size: 0.9rem;
        margin: 0;
    }

    /* Responsive Design */
    @media (max-width: 1200px) {
        .dashboard-grid {
            grid-template-columns: 1fr;
        }
    }

    @media (max-width: 768px) {
        .stats-grid {
            grid-template-columns: repeat(2, 1fr);
        }
        
        .welcome-title {
            font-size: 1.5rem;
        }
        
        .quick-actions {
            justify-content: center;
        }
        
        .tools-grid {
            grid-template-columns: 1fr;
        }
    }

    @media (max-width: 480px) {
        .stats-grid {
            grid-template-columns: 1fr;
        }
        
        .page-content {
            padding: 15px;
        }
        
        .welcome-header {
            padding: 20px;
            text-align: center;
        }
    }

    /* Identity Verification Notification Styling */
    .verification-notification {
        border: none !important;
        border-radius: 16px !important;
        padding: 20px 24px !important;
        box-shadow: 0 8px 25px rgba(0, 0, 0, 0.1) !important;
        margin-bottom: 24px !important;
        position: relative;
        overflow: hidden;
    }

    .verification-notification::before {
        content: '';
        position: absolute;
        left: 0;
        top: 0;
        bottom: 0;
        width: 5px;
        background: currentColor;
        opacity: 0.8;
    }

    .verification-notification .notification-icon {
        flex-shrink: 0;
        width: 60px;
        height: 60px;
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        background: rgba(255, 255, 255, 0.2);
        backdrop-filter: blur(10px);
    }

    .verification-notification .alert-heading {
        color: inherit !important;
        font-size: 1.25rem;
        margin-bottom: 8px !important;
    }

    .verification-notification p {
        color: inherit !important;
        opacity: 0.9;
        line-height: 1.5;
        margin-bottom: 16px !important;
    }

    .verification-notification .btn {
        font-weight: 600;
        padding: 8px 20px;
        border-radius: 8px;
        border: 2px solid rgba(255, 255, 255, 0.3);
        background: rgba(255, 255, 255, 0.15);
        color: inherit !important;
        backdrop-filter: blur(10px);
        transition: all 0.3s ease;
    }

    .verification-notification .btn:hover {
        background: rgba(255, 255, 255, 0.25);
        transform: translateY(-2px);
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
    }

    .verification-notification.alert-warning {
        background: linear-gradient(135deg, #f59e0b, #d97706);
        color: white;
    }

    .verification-notification.alert-info {
        background: linear-gradient(135deg, #3b82f6, #1d4ed8);
        color: white;
    }

    .verification-notification.alert-danger {
        background: linear-gradient(135deg, #ef4444, #dc2626);
        color: white;
    }

    .verification-notification.alert-success {
        background: linear-gradient(135deg, #10b981, #059669);
        color: white;
    }

    .verification-notification .btn-close {
        filter: brightness(0) invert(1);
        opacity: 0.7;
    }

    .verification-notification .btn-close:hover {
        opacity: 1;
    }

    /* Responsive adjustments */
    @media (max-width: 768px) {
        .verification-notification {
            padding: 16px 20px !important;
        }
        
        .verification-notification .notification-icon {
            width: 48px;
            height: 48px;
        }
        
        .verification-notification .notification-icon i {
            font-size: 1.5rem !important;
        }
        
        .verification-notification .alert-heading {
            font-size: 1.1rem;
        }
    }
</style>

<!-- Clean Teacher Dashboard -->
<div class="container-fluid">
    <!-- Page Header -->
    <div class="page-header">
        <h1>Teaching Dashboard Overview</h1>
        <div>
            <a href="analytics.php" class="btn btn-sm btn-outline-primary">
                <i class='bx bx-bar-chart-alt'></i> Analytics
            </a>
        </div>
    </div>

    <?php if ($verification_notification): ?>
    <!-- Identity Verification Notification Banner -->
    <div class="verification-notification alert alert-<?php echo $verification_notification['type']; ?> <?php echo $verification_notification['dismissible'] ? 'alert-dismissible' : ''; ?> mb-4" role="alert">
        <div class="d-flex align-items-center">
            <div class="notification-icon me-3">
                <i class='bx <?php echo $verification_notification['icon']; ?>' style="font-size: 2rem;"></i>
            </div>
            <div class="notification-content flex-grow-1">
                <h5 class="alert-heading mb-2">
                    <strong><?php echo htmlspecialchars($verification_notification['title']); ?></strong>
                </h5>
                <p class="mb-3"><?php echo htmlspecialchars($verification_notification['message']); ?></p>
                <a href="<?php echo htmlspecialchars($verification_notification['action_link']); ?>" 
                   class="btn btn-<?php echo $verification_notification['type']; ?> btn-sm">
                    <i class='bx bx-upload me-1'></i>
                    <?php echo htmlspecialchars($verification_notification['action_text']); ?>
                </a>
            </div>
            <?php if ($verification_notification['dismissible']): ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            <?php endif; ?>
        </div>
    </div>
    <?php endif; ?>

    <!-- Help Guide Button -->
    <div class="help-guide-section mb-4">
        <button type="button" class="btn btn-help-guide" data-bs-toggle="modal" data-bs-target="#helpGuideModal">
            <i class='bx bx-help-circle me-2'></i>
            <span>How to Use Panadite Academy</span>
            <i class='bx bx-chevron-right ms-2'></i>
        </button>
    </div>

    <!-- Colorful Stats Cards - Admin Style -->
    <div class="stats-cards">
        <div class="stat-card">
            <div class="stat-card-header">
                <div class="stat-content">
                    <p class="stat-label">Total Students</p>
                    <div class="stat-number"><?php echo $student_count; ?></div>
                    <p class="stat-description">Enrolled across your courses</p>
                </div>
                <div class="stat-icon">
                    <i class='bx bx-group'></i>
                </div>
            </div>
            <a href="students.php" class="stat-link">View All Students <i class='bx bx-right-arrow-alt'></i></a>
        </div>

        <div class="stat-card">
            <div class="stat-card-header">
                <div class="stat-content">
                    <p class="stat-label">Active Courses</p>
                    <div class="stat-number"><?php echo $course_count; ?></div>
                    <p class="stat-description">Educational resources</p>
                </div>
                <div class="stat-icon">
                    <i class='bx bx-book-bookmark'></i>
                </div>
            </div>
            <a href="courses.php" class="stat-link">Manage Courses <i class='bx bx-right-arrow-alt'></i></a>
        </div>

        <div class="stat-card">
            <div class="stat-card-header">
                <div class="stat-content">
                    <p class="stat-label">Total Videos</p>
                    <div class="stat-number"><?php echo $video_count; ?></div>
                    <p class="stat-description">Content across all courses</p>
                </div>
                <div class="stat-icon">
                    <i class='bx bx-video'></i>
                </div>
            </div>
            <a href="videos.php" class="stat-link">View All Videos <i class='bx bx-right-arrow-alt'></i></a>
        </div>

        <div class="stat-card">
            <div class="stat-card-header">
                <div class="stat-content">
                    <p class="stat-label">Available Balance</p>
                    <div class="stat-number">R<?php echo number_format($available_balance, 2); ?></div>
                    <p class="stat-description">Ready for withdrawal</p>
                </div>
                <div class="stat-icon">
                    <i class='bx bx-money'></i>
                </div>
            </div>
            <a href="earnings.php" class="stat-link">View Transactions <i class='bx bx-right-arrow-alt'></i></a>
        </div>
    </div>

    <!-- Modern Activity & Wallet Grid -->
    <div class="activity-wallet-grid">
        <!-- Recent Activity - 50% -->
        <div class="modern-section-card">
            <div class="section-card-header">
                <h2 class="section-title">Recent Activity</h2>
            </div>
            <div class="section-card-body">
                <?php if (!empty($recent_courses)): ?>
                    <?php foreach (array_slice($recent_courses, 0, 3) as $course): ?>
                        <div class="modern-activity-item">
                            <div class="activity-icon primary">
                                <i class='bx bx-book-bookmark'></i>
                            </div>
                            <div class="activity-content">
                                <h4><?php echo htmlspecialchars($course['title']); ?></h4>
                                <p><?php echo htmlspecialchars($course['description']); ?></p>
                                <small><?php echo date('M j, Y', strtotime($course['created_at'])); ?></small>
                            </div>
                            <div class="activity-badge new">
                                New
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php else: ?>
                    <div class="modern-activity-item">
                        <div class="activity-icon info" style="background: linear-gradient(135deg, #e9ecef, #f8f9fa);">
                            <i class='bx bx-info-circle' style="color: #6c757d;"></i>
                        </div>
                        <div class="activity-content">
                            <h4 style="color: #6c757d;">No Recent Activity</h4>
                            <p style="color: #adb5bd;">Start creating courses, uploading videos, or engaging with students to see your activity here.</p>
                            <small style="color: #adb5bd;">Get started now</small>
                        </div>
                    </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- Teacher's Wallet - 50% -->
        <div class="modern-section-card">
            <div class="section-card-header">
                <h2 class="section-title">Teacher's Wallet</h2>
            </div>
            <div class="section-card-body">
                <div class="wallet-overview">
                    <div class="wallet-balance">
                        <div class="balance-icon">
                            <i class='bx bx-wallet'></i>
                        </div>
                        <div class="balance-content">
                            <p class="balance-label">Available Balance</p>
                            <div class="balance-amount">R<?php echo number_format($available_balance, 2); ?></div>
                            <p class="balance-status">Ready for withdrawal</p>
                        </div>
                    </div>
                    
                    <div class="wallet-stats">
                        <div class="wallet-stat">
                            <div class="stat-icon earnings">
                                <i class='bx bx-trending-up'></i>
                            </div>
                            <div class="stat-content">
                                <span class="stat-value">R<?php echo number_format($monthly_earnings, 2); ?></span>
                                <span class="stat-text">This Month</span>
                            </div>
                        </div>
                        
                        <div class="wallet-stat">
                            <div class="stat-icon transactions">
                                <i class='bx bx-receipt'></i>
                            </div>
                            <div class="stat-content">
                                <span class="stat-value"><?php echo $total_transactions; ?></span>
                                <span class="stat-text">Transactions</span>
                            </div>
                        </div>
                    </div>
                    
                    <div class="wallet-actions">
                        <a href="withdraw-earnings.php" class="withdrawal-btn">
                            <i class='bx bx-money-withdraw'></i>
                            Request Withdrawal
                        </a>
                        <a href="earnings.php" class="view-history-btn">
                            <i class='bx bx-history'></i>
                            View History
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>

        <!-- Modern Quick Actions -->
        <div class="dashboard-content">
            <div class="section-header">
                <h2 class="section-title">Quick Actions</h2>
            </div>
            <div class="quick-actions-grid">
                <a href="create-course.php" class="quick-action-card">
                    <div class="quick-action-icon primary">
                        <i class='bx bx-plus-circle'></i>
                    </div>
                    <div class="quick-action-content">
                        <h3>Create Course</h3>
                        <p>Build new educational content</p>
                    </div>
                    <div class="quick-action-arrow">
                        <i class='bx bx-right-arrow-alt'></i>
                    </div>
                </a>
                
                <a href="upload-video.php" class="quick-action-card">
                    <div class="quick-action-icon success">
                        <i class='bx bx-cloud-upload'></i>
                    </div>
                    <div class="quick-action-content">
                        <h3>Upload Video</h3>
                        <p>Add new video content</p>
                    </div>
                    <div class="quick-action-arrow">
                        <i class='bx bx-right-arrow-alt'></i>
                    </div>
                </a>
                
                <a href="withdraw-earnings.php" class="quick-action-card">
                    <div class="quick-action-icon warning">
                        <i class='bx bx-money-withdraw'></i>
                    </div>
                    <div class="quick-action-content">
                        <h3>Withdraw Earnings</h3>
                        <p>Access your payments</p>
                    </div>
                    <div class="quick-action-arrow">
                        <i class='bx bx-right-arrow-alt'></i>
                    </div>
                </a>
                
                <a href="students.php" class="quick-action-card">
                    <div class="quick-action-icon danger">
                        <i class='bx bx-group'></i>
                    </div>
                    <div class="quick-action-content">
                        <h3>Manage Students</h3>
                        <p>View and organize learners</p>
                    </div>
                    <div class="quick-action-arrow">
                        <i class='bx bx-right-arrow-alt'></i>
                    </div>
                </a>
            </div>
        </div>

        <!-- Modern Teaching Insights -->
        <div class="dashboard-content">
            <div class="section-header">
                <h2 class="section-title">Teaching Insights</h2>
            </div>
            <div class="insights-grid">
                <div class="modern-insight-card">
                    <div class="insight-card-header">
                        <div class="insight-content">
                            <p class="insight-label">Avg. Watch Time</p>
                            <div class="insight-number"><?php echo $watch_time_display; ?></div>
                            <p class="insight-trend success">
                                <i class='bx bx-trending-up'></i>
                                Based on video content
                            </p>
                        </div>
                        <div class="insight-icon primary">
                            <i class='bx bx-time-five'></i>
                        </div>
                    </div>
                </div>

                <div class="modern-insight-card">
                    <div class="insight-card-header">
                        <div class="insight-content">
                            <p class="insight-label">Completion Rate</p>
                            <div class="insight-number"><?php echo $completion_rate_display; ?></div>
                            <p class="insight-trend success">
                                <i class='bx bx-trending-up'></i>
                                Course completion data
                            </p>
                        </div>
                        <div class="insight-icon success">
                            <i class='bx bx-check-circle'></i>
                        </div>
                    </div>
                </div>

                <div class="modern-insight-card">
                    <div class="insight-card-header">
                        <div class="insight-content">
                            <p class="insight-label">Avg. Rating</p>
                            <div class="insight-number"><?php echo $rating_display; ?></div>
                            <p class="insight-trend success">
                                <i class='bx bx-trending-up'></i>
                                From course reviews
                            </p>
                        </div>
                        <div class="insight-icon warning">
                            <i class='bx bx-star'></i>
                        </div>
                    </div>
                </div>

                <div class="modern-insight-card">
                    <div class="insight-card-header">
                        <div class="insight-content">
                            <p class="insight-label">New Students</p>
                            <div class="insight-number"><?php echo $new_students_display; ?></div>
                            <p class="insight-trend success">
                                <i class='bx bx-trending-up'></i>
                                This month
                            </p>
                        </div>
                        <div class="insight-icon danger">
                            <i class='bx bx-user-plus'></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Modern Bottom Section -->
    <div class="bottom-sections-grid">
        <!-- Modern Recent Students -->
        <div class="modern-section-card">
            <div class="section-card-header">
                <h2 class="section-title">Recent Students</h2>
                <a href="students.php" class="view-all-link">
                    View All <i class='bx bx-right-arrow-alt'></i>
                </a>
            </div>
            <div class="section-card-body">
                <?php if (!empty($recent_students)): ?>
                    <?php 
                    $avatar_colors = ['primary', 'success', 'warning', 'info', 'danger'];
                    $status_icons = ['bx-user-plus', 'bx-check-circle', 'bx-medal', 'bx-star', 'bx-trophy'];
                    $status_classes = ['new', 'active', 'completed', 'progress', 'excellent'];
                    ?>
                    <?php foreach ($recent_students as $index => $student): ?>
                        <?php
                        // Generate initials
                        $first_initial = !empty($student['first_name']) ? strtoupper(substr($student['first_name'], 0, 1)) : 'S';
                        $last_initial = !empty($student['last_name']) ? strtoupper(substr($student['last_name'], 0, 1)) : 'T';
                        $initials = $first_initial . $last_initial;
                        
                        // Cycle through colors and status
                        $color_index = $index % count($avatar_colors);
                        $status_index = $index % count($status_classes);
                        
                        $full_name = trim(($student['first_name'] ?? 'Student') . ' ' . ($student['last_name'] ?? ''));
                        $course_title = $student['course_title'] ?? 'Course';
                        $time_display = $student['time_display'] ?? 'Recently';
                        ?>
                        <div class="modern-student-item">
                            <div class="student-avatar <?php echo $avatar_colors[$color_index]; ?>">
                                <span><?php echo htmlspecialchars($initials); ?></span>
                            </div>
                            <div class="student-content">
                                <h4><?php echo htmlspecialchars($full_name); ?></h4>
                                <p>Enrolled in <?php echo htmlspecialchars($course_title); ?></p>
                                <small><?php echo htmlspecialchars($time_display); ?></small>
                            </div>
                            <div class="student-status <?php echo $status_classes[$status_index]; ?>">
                                <i class='bx <?php echo $status_icons[$status_index]; ?>'></i>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php else: ?>
                    <div class="empty-state">
                        <div class="empty-icon">
                            <i class='bx bx-user-plus'></i>
                        </div>
                        <h4>No Recent Students</h4>
                        <p>No students have enrolled in your courses within the last 7 days.</p>
                        <small>Students who enroll will appear here automatically.</small>
                    </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- Modern My Courses -->
        <div class="modern-section-card">
            <div class="section-card-header">
                <h2 class="section-title">My Courses</h2>
                <a href="courses.php" class="view-all-link">
                    Manage All <i class='bx bx-right-arrow-alt'></i>
                </a>
            </div>
            <div class="section-card-body">
                <?php if (empty($recent_courses)): ?>
                    <div class="modern-empty-state">
                        <div class="empty-icon">
                            <i class='bx bx-book-add'></i>
                        </div>
                        <h4>No Courses Yet</h4>
                        <p>Create your first course to start teaching and earning.</p>
                        <a href="create-course.php" class="empty-action-btn">
                            <i class='bx bx-plus'></i> Create Course
                        </a>
                    </div>
                <?php else: ?>
                    <?php foreach ($recent_courses as $course): ?>
                        <div class="modern-course-item">
                            <div class="course-icon">
                                <i class='bx bx-book-bookmark'></i>
                            </div>
                            <div class="course-content">
                                <h4><?php echo htmlspecialchars($course['title']); ?></h4>
                                <p>Created <?php echo date('M j, Y', strtotime($course['created_at'])); ?></p>
                            </div>
                            <div class="course-status published">
                                <span>Published</span>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Help Guide Modal -->
<div class="modal fade" id="helpGuideModal" tabindex="-1" aria-labelledby="helpGuideModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-xl">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="helpGuideModalLabel">
                    <i class='bx bx-help-circle me-2 text-primary'></i>
                    How to Use Panadite Academy - Teacher Guide
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div class="help-guide-content">
                    <!-- Step-by-Step Process -->
                    <div class="guide-section">
                        <h4 class="guide-title">
                            <i class='bx bx-list-ol text-success me-2'></i>
                            Getting Started - Step by Step Process
                        </h4>
                        <div class="guide-steps">
                            <div class="step-item">
                                <div class="step-number">1</div>
                                <div class="step-content">
                                    <h5>Create Your Course</h5>
                                    <p>Start by creating a course with detailed description, objectives, and pricing.</p>
                                    <small class="text-muted">📍 Use the sidebar or click "Create Course" button</small>
                                </div>
                            </div>
                            <div class="step-item">
                                <div class="step-number">2</div>
                                <div class="step-content">
                                    <h5>Wait for Admin Approval</h5>
                                    <p>Your course will be reviewed by our admin team. You'll receive email notification once approved.</p>
                                    <small class="text-muted">⏱️ Usually takes 24-48 hours</small>
                                </div>
                            </div>
                            <div class="step-item">
                                <div class="step-number">3</div>
                                <div class="step-content">
                                    <h5>Upload Course Content</h5>
                                    <p>Add videos (file upload or YouTube links), course materials, and create quizzes.</p>
                                    <small class="text-muted">🎥 Videos and 📄 Materials are linked to specific lessons</small>
                                </div>
                            </div>
                            <div class="step-item">
                                <div class="step-number">4</div>
                                <div class="step-content">
                                    <h5>Create & Publish Quizzes</h5>
                                    <p><strong>Important:</strong> After creating quizzes, remember to click "Publish" to make them live for students.</p>
                                    <small class="text-muted">🔴 Quizzes are NOT visible to students until published!</small>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Quick Actions Guide -->
                    <div class="guide-section">
                        <h4 class="guide-title">
                            <i class='bx bx-navigation text-primary me-2'></i>
                            Quick Actions & Navigation
                        </h4>
                        <div class="actions-grid">
                            <div class="action-card">
                                <div class="action-icon video">
                                    <i class='bx bx-video'></i>
                                </div>
                                <div class="action-content">
                                    <h6>Upload Videos</h6>
                                    <p>Upload video files or add YouTube links</p>
                                    <a href="https://panaditeacademy.com/teacher/upload-video.php" target="_blank" class="action-link">
                                        Go to Upload Videos <i class='bx bx-link-external'></i>
                                    </a>
                                </div>
                            </div>
                            
                            <div class="action-card">
                                <div class="action-icon materials">
                                    <i class='bx bx-file-doc'></i>
                                </div>
                                <div class="action-content">
                                    <h6>Upload Materials</h6>
                                    <p>Add course materials, PDFs, documents</p>
                                    <a href="https://panaditeacademy.com/teacher/manage-course-materials.php" target="_blank" class="action-link">
                                        Manage Materials <i class='bx bx-link-external'></i>
                                    </a>
                                </div>
                            </div>
                            
                            <div class="action-card">
                                <div class="action-icon quiz">
                                    <i class='bx bx-edit-alt'></i>
                                </div>
                                <div class="action-content">
                                    <h6>Create Quizzes</h6>
                                    <p>Create assessments linked to videos</p>
                                    <a href="https://panaditeacademy.com/teacher/create_quiz.php" target="_blank" class="action-link">
                                        Create Quiz <i class='bx bx-link-external'></i>
                                    </a>
                                </div>
                            </div>
                            
                            <div class="action-card">
                                <div class="action-icon manage-quiz">
                                    <i class='bx bx-list-check'></i>
                                </div>
                                <div class="action-content">
                                    <h6>Manage Quizzes</h6>
                                    <p>View, edit & publish your quizzes</p>
                                    <a href="https://panaditeacademy.com/teacher/quizzes.php" target="_blank" class="action-link">
                                        Manage Quizzes <i class='bx bx-link-external'></i>
                                    </a>
                                </div>
                            </div>
                            
                            <div class="action-card">
                                <div class="action-icon submissions">
                                    <i class='bx bx-task'></i>
                                </div>
                                <div class="action-content">
                                    <h6>Quiz Submissions</h6>
                                    <p>Review student quiz submissions</p>
                                    <a href="https://panaditeacademy.com/teacher/quiz_submissions.php" target="_blank" class="action-link">
                                        View Submissions <i class='bx bx-link-external'></i>
                                    </a>
                                </div>
                            </div>
                            
                            <div class="action-card">
                                <div class="action-icon library">
                                    <i class='bx bx-library'></i>
                                </div>
                                <div class="action-content">
                                    <h6>Publish to Library</h6>
                                    <p>Share books/documents in public library</p>
                                    <a href="https://panaditeacademy.com/teacher/publish_to_library.php" target="_blank" class="action-link">
                                        Publish to Library <i class='bx bx-link-external'></i>
                                    </a>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Analytics & Business -->
                    <div class="guide-section">
                        <h4 class="guide-title">
                            <i class='bx bx-trending-up text-success me-2'></i>
                            Analytics & Business Management
                        </h4>
                        <div class="business-grid">
                            <div class="business-card">
                                <div class="business-icon analytics">
                                    <i class='bx bx-bar-chart-alt-2'></i>
                                </div>
                                <div class="business-content">
                                    <h6>Performance Analytics</h6>
                                    <p>Track course performance, student engagement, and earnings</p>
                                    <a href="https://panaditeacademy.com/teacher/analytics.php" target="_blank" class="business-link">
                                        View Analytics <i class='bx bx-link-external'></i>
                                    </a>
                                </div>
                            </div>
                            
                            <div class="business-card">
                                <div class="business-icon referrals">
                                    <i class='bx bx-group'></i>
                                </div>
                                <div class="business-content">
                                    <h6>Referral Program</h6>
                                    <p>Refer new teachers and students to earn commissions</p>
                                    <a href="https://panaditeacademy.com/teacher/referrals.php" target="_blank" class="business-link">
                                        Manage Referrals <i class='bx bx-link-external'></i>
                                    </a>
                                </div>
                            </div>
                            
                            <div class="business-card">
                                <div class="business-icon withdrawal">
                                    <i class='bx bx-money-withdraw'></i>
                                </div>
                                <div class="business-content">
                                    <h6>Request Withdrawal</h6>
                                    <p><strong>Minimum R500:</strong> Request withdrawal of your earnings</p>
                                    <a href="https://panaditeacademy.com/teacher/request-withdrawal.php" target="_blank" class="business-link">
                                        Request Withdrawal <i class='bx bx-link-external'></i>
                                    </a>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Important Notes -->
                    <div class="guide-section">
                        <h4 class="guide-title">
                            <i class='bx bx-info-circle text-warning me-2'></i>
                            Important Notes & Tips
                        </h4>
                        <div class="notes-list">
                            <div class="note-item success">
                                <i class='bx bx-check-circle'></i>
                                <span><strong>Quiz Publishing:</strong> Remember to click "Publish" after creating quizzes - they won't be visible to students otherwise.</span>
                            </div>
                            <div class="note-item info">
                                <i class='bx bx-info-circle'></i>
                                <span><strong>Withdrawal Minimum:</strong> You need at least R500 in earnings before requesting a withdrawal.</span>
                            </div>
                            <div class="note-item warning">
                                <i class='bx bx-error-circle'></i>
                                <span><strong>Course Approval:</strong> All courses require admin approval before students can enroll.</span>
                            </div>
                            <div class="note-item primary">
                                <i class='bx bx-link-alt'></i>
                                <span><strong>Quiz Linking:</strong> Each quiz should be linked to specific videos for better student experience.</span>
                            </div>
                        </div>
                    </div>

                    <!-- Quick Navigation -->
                    <div class="guide-section">
                        <h4 class="guide-title">
                            <i class='bx bx-compass text-info me-2'></i>
                            Quick Navigation
                        </h4>
                        <p class="text-muted mb-3">Use the sidebar menu to quickly navigate to different sections of your teacher dashboard.</p>
                        <div class="nav-hint">
                            <i class='bx bx-lightbulb text-warning'></i>
                            <span><strong>Pro Tip:</strong> Bookmark frequently used pages like quiz management and video uploads for faster access!</span>
                        </div>
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                    <i class='bx bx-x me-1'></i> Close
                </button>
                <a href="https://panaditeacademy.com/teacher/create-course.php" class="btn btn-primary">
                    <i class='bx bx-plus me-1'></i> Create Your First Course
                </a>
            </div>
        </div>
    </div>
</div>

<style>
/* Help Guide Button Styles */
.help-guide-section {
    text-align: center;
}

.btn-help-guide {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    border: none;
    padding: 12px 24px;
    border-radius: 12px;
    font-weight: 600;
    font-size: 16px;
    transition: all 0.3s ease;
    box-shadow: 0 4px 12px rgba(102, 126, 234, 0.3);
    display: inline-flex;
    align-items: center;
    text-decoration: none;
}

.btn-help-guide:hover {
    background: linear-gradient(135deg, #5a6fd8 0%, #6a4190 100%);
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(102, 126, 234, 0.4);
    color: white;
}

/* Modal Styles */
#helpGuideModal .modal-content {
    border-radius: 16px;
    border: none;
    box-shadow: 0 10px 30px rgba(0, 0, 0, 0.15);
}

#helpGuideModal .modal-header {
    background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
    border-bottom: 1px solid #dee2e6;
    border-radius: 16px 16px 0 0;
}

#helpGuideModal .modal-title {
    font-weight: 700;
    font-size: 1.5rem;
    color: #495057;
}

/* Guide Content Styles */
.help-guide-content {
    max-height: 70vh;
    overflow-y: auto;
    padding-right: 10px;
}

.guide-section {
    margin-bottom: 2rem;
    padding-bottom: 1.5rem;
    border-bottom: 1px solid #e9ecef;
}

.guide-section:last-child {
    border-bottom: none;
}

.guide-title {
    font-size: 1.25rem;
    font-weight: 700;
    margin-bottom: 1rem;
    color: #495057;
    display: flex;
    align-items: center;
}

/* Step Items */
.guide-steps {
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.step-item {
    display: flex;
    align-items: flex-start;
    gap: 1rem;
    padding: 1rem;
    background: #f8f9fa;
    border-radius: 12px;
    border-left: 4px solid #28a745;
}

.step-number {
    background: #28a745;
    color: white;
    width: 32px;
    height: 32px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: 700;
    font-size: 14px;
    flex-shrink: 0;
}

.step-content h5 {
    font-size: 1.1rem;
    font-weight: 600;
    margin-bottom: 0.5rem;
    color: #495057;
}

.step-content p {
    margin-bottom: 0.25rem;
    color: #6c757d;
    font-size: 14px;
}

/* Action Cards */
.actions-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
    gap: 1rem;
}

.action-card {
    background: white;
    border: 1px solid #e9ecef;
    border-radius: 12px;
    padding: 1rem;
    transition: all 0.3s ease;
    display: flex;
    align-items: flex-start;
    gap: 1rem;
}

.action-card:hover {
    border-color: #007bff;
    box-shadow: 0 4px 12px rgba(0, 123, 255, 0.15);
    transform: translateY(-2px);
}

.action-icon {
    width: 48px;
    height: 48px;
    border-radius: 12px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 24px;
    color: white;
    flex-shrink: 0;
}

.action-icon.video { background: linear-gradient(135deg, #ff6b6b, #ee5a52); }
.action-icon.materials { background: linear-gradient(135deg, #4ecdc4, #44a08d); }
.action-icon.quiz { background: linear-gradient(135deg, #45b7d1, #96c93d); }
.action-icon.manage-quiz { background: linear-gradient(135deg, #f093fb, #f5576c); }
.action-icon.submissions { background: linear-gradient(135deg, #4facfe, #00f2fe); }
.action-icon.library { background: linear-gradient(135deg, #a8edea, #fed6e3); }

.action-content h6 {
    font-weight: 600;
    margin-bottom: 0.5rem;
    color: #495057;
}

.action-content p {
    font-size: 14px;
    color: #6c757d;
    margin-bottom: 0.75rem;
}

.action-link {
    color: #007bff;
    text-decoration: none;
    font-weight: 500;
    font-size: 14px;
    display: inline-flex;
    align-items: center;
    gap: 4px;
}

.action-link:hover {
    color: #0056b3;
    text-decoration: underline;
}

/* Business Cards */
.business-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
    gap: 1rem;
}

.business-card {
    background: linear-gradient(135deg, #f8f9fa 0%, #ffffff 100%);
    border: 1px solid #e9ecef;
    border-radius: 12px;
    padding: 1.25rem;
    display: flex;
    align-items: flex-start;
    gap: 1rem;
}

.business-icon {
    width: 48px;
    height: 48px;
    border-radius: 12px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 24px;
    color: white;
    flex-shrink: 0;
}

.business-icon.analytics { background: linear-gradient(135deg, #667eea, #764ba2); }
.business-icon.referrals { background: linear-gradient(135deg, #f093fb, #f5576c); }
.business-icon.withdrawal { background: linear-gradient(135deg, #4facfe, #00f2fe); }

.business-content h6 {
    font-weight: 600;
    margin-bottom: 0.5rem;
    color: #495057;
}

.business-content p {
    font-size: 14px;
    color: #6c757d;
    margin-bottom: 0.75rem;
}

.business-link {
    color: #007bff;
    text-decoration: none;
    font-weight: 500;
    font-size: 14px;
    display: inline-flex;
    align-items: center;
    gap: 4px;
}

.business-link:hover {
    color: #0056b3;
    text-decoration: underline;
}

/* Notes */
.notes-list {
    display: flex;
    flex-direction: column;
    gap: 0.75rem;
}

.note-item {
    display: flex;
    align-items: flex-start;
    gap: 0.75rem;
    padding: 1rem;
    border-radius: 8px;
    font-size: 14px;
}

.note-item.success {
    background: rgba(40, 167, 69, 0.1);
    color: #155724;
}

.note-item.info {
    background: rgba(23, 162, 184, 0.1);
    color: #0c5460;
}

.note-item.warning {
    background: rgba(255, 193, 7, 0.1);
    color: #856404;
}

.note-item.primary {
    background: rgba(0, 123, 255, 0.1);
    color: #004085;
}

.note-item i {
    font-size: 18px;
    margin-top: 2px;
    flex-shrink: 0;
}

.nav-hint {
    background: rgba(255, 193, 7, 0.1);
    border: 1px solid rgba(255, 193, 7, 0.3);
    border-radius: 8px;
    padding: 1rem;
    display: flex;
    align-items: flex-start;
    gap: 0.75rem;
    color: #856404;
    font-size: 14px;
}

.nav-hint i {
    font-size: 20px;
    margin-top: 2px;
    flex-shrink: 0;
}

/* Responsive */
@media (max-width: 768px) {
    .actions-grid,
    .business-grid {
        grid-template-columns: 1fr;
    }
    
    .action-card,
    .business-card {
        flex-direction: column;
        text-align: center;
    }
    
    .step-item {
        flex-direction: column;
        text-align: center;
    }
}
</style>

<?php
// Include the footer component
require_once 'components/footer.php';
?>
