<?php
session_start();
require_once('../includes/db.php');
require_once('../includes/functions.php');

// Check if user is logged in and is a teacher
if (!isset($_SESSION['user_id']) || $_SESSION['user_role'] !== 'teacher') {
    header("Location: ../login.php");
    exit();
}

// Get teacher information
$teacher_id = $_SESSION['user_id'];
$teacher = getUserById($conn, $teacher_id);

// Get count of courses
$stmt = $conn->prepare("SELECT COUNT(*) AS course_count FROM courses WHERE teacher_id = ?");
$stmt->bind_param("i", $teacher_id);
$stmt->execute();
$course_result = $stmt->get_result();
$course_data = $course_result->fetch_assoc();
$course_count = $course_data['course_count'];

// Get count of videos
$stmt = $conn->prepare("SELECT COUNT(*) AS video_count FROM videos WHERE teacher_id = ?");
$stmt->bind_param("i", $teacher_id);
$stmt->execute();
$video_result = $stmt->get_result();
$video_data = $video_result->fetch_assoc();
$video_count = $video_data['video_count'];

// Get count of students enrolled in teacher's courses
$stmt = $conn->prepare("
    SELECT COUNT(DISTINCT e.student_id) AS student_count 
    FROM enrollments e 
    JOIN courses c ON e.course_id = c.course_id 
    WHERE c.teacher_id = ?
");
$stmt->bind_param("i", $teacher_id);
$stmt->execute();
$student_result = $stmt->get_result();
$student_data = $student_result->fetch_assoc();
$student_count = $student_data['student_count'];

// Get recent courses
$stmt = $conn->prepare("
    SELECT * FROM courses 
    WHERE teacher_id = ? 
    ORDER BY created_at DESC 
    LIMIT 5
");
$stmt->bind_param("i", $teacher_id);
$stmt->execute();
$recent_courses = $stmt->get_result();

// Get recent videos
$stmt = $conn->prepare("
    SELECT * FROM videos 
    WHERE teacher_id = ? 
    ORDER BY created_at DESC 
    LIMIT 5
");
$stmt->bind_param("i", $teacher_id);
$stmt->execute();
$recent_videos = $stmt->get_result();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Teacher Dashboard - Panadite Academy</title>
    
    <!-- Favicon -->
    <link rel="shortcut icon" href="../assets/images/favicon.ico" type="image/x-icon">
    
    <!-- CSS Libraries -->
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/toastify-js/src/toastify.min.css">
    
    <!-- Custom CSS -->
    <link rel="stylesheet" href="../assets/css/modern-dashboard-new.css">
    <link rel="stylesheet" href="../assets/css/teacher-dashboard-updates.css">
    <link rel="stylesheet" href="../assets/css/dashboard-fix.css">
    
    <style>
        /* Additional inline styles for dashboard components */
        .welcome-text {
            color: white;
            position: relative;
            z-index: 10;
            padding: 2rem;
        }
        
        .welcome-heading {
            font-size: 2rem;
            font-weight: 700;
            margin-bottom: 0.5rem;
        }
        
        .welcome-subtext {
            font-size: 1rem;
            opacity: 0.8;
            max-width: 600px;
        }
        
        .action-btn {
            padding: 0.5rem 1.25rem;
            border-radius: 50px;
            font-weight: 500;
            transition: all 0.3s ease;
            border: none;
        }
        
        .btn-primary {
            background-color: var(--primary);
            color: white;
        }
        
        .btn-primary:hover {
            background-color: var(--primary-dark);
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(39, 107, 228, 0.3);
        }
        
        .btn-outline-light {
            background-color: rgba(255, 255, 255, 0.1);
            color: white;
            border: 1px solid rgba(255, 255, 255, 0.3);
        }
        
        .btn-outline-light:hover {
            background-color: rgba(255, 255, 255, 0.2);
            transform: translateY(-2px);
        }
        
        .course-card, .video-card {
            border-radius: var(--radius-md);
            overflow: hidden;
            transition: all 0.3s ease;
            height: 100%;
        }
        
        .course-card:hover, .video-card:hover {
            transform: translateY(-5px);
            box-shadow: var(--shadow-md);
        }
        
        .course-image, .video-thumbnail {
            height: 160px;
            object-fit: cover;
        }
        
        .recent-chart {
            height: 300px;
        }
        
        @media (max-width: 768px) {
            .welcome-heading {
                font-size: 1.5rem;
            }
            
            .welcome-text {
                padding: 1.5rem;
            }
        }
    </style>
</head>
<body>
    <div class="modern-dashboard">
        <!-- Sidebar -->
        <aside class="sidebar">
            <div class="sidebar-header">
                <a href="../index.php" class="sidebar-brand">
                    <img src="../assets/images/logo.png" alt="Panadite Academy">
                    <span>Panadite Academy</span>
                </a>
            </div>
            
            <ul class="sidebar-nav">
                <li class="sidebar-heading">MAIN</li>
                
                <li class="nav-item">
                    <a href="dashboard.php" class="nav-link active">
                        <i class="fas fa-tachometer-alt"></i>
                        <span class="nav-text">Dashboard</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="courses.php" class="nav-link">
                        <i class="fas fa-graduation-cap"></i>
                        <span class="nav-text">Courses</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="videos.php" class="nav-link">
                        <i class="fas fa-video"></i>
                        <span class="nav-text">Videos</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="students.php" class="nav-link">
                        <i class="fas fa-users"></i>
                        <span class="nav-text">Students</span>
                    </a>
                </li>
                
                <li class="sidebar-divider-wrapper"></li>
                
                <li class="sidebar-heading">TOOLS</li>
                
                <li class="nav-item">
                    <a href="analytics.php" class="nav-link">
                        <i class="fas fa-chart-bar"></i>
                        <span class="nav-text">Analytics</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="quizzes.php" class="nav-link">
                        <i class="fas fa-question-circle"></i>
                        <span class="nav-text">Quizzes</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="assignments.php" class="nav-link">
                        <i class="fas fa-tasks"></i>
                        <span class="nav-text">Assignments</span>
                    </a>
                </li>
                
                <li class="sidebar-divider-wrapper"></li>
                
                <li class="sidebar-heading">ACCOUNT</li>
                
                <li class="nav-item">
                    <a href="profile.php" class="nav-link">
                        <i class="fas fa-user"></i>
                        <span class="nav-text">Profile</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="settings.php" class="nav-link">
                        <i class="fas fa-cog"></i>
                        <span class="nav-text">Settings</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="../logout.php" class="nav-link">
                        <i class="fas fa-sign-out-alt"></i>
                        <span class="nav-text">Logout</span>
                    </a>
                </li>
            </ul>
        </aside>
        
        <!-- Content Wrapper -->
        <div class="content-wrapper">
            <!-- Navbar -->
            <nav class="navbar">
                <div class="d-flex align-items-center justify-content-between w-100">
                    <div class="d-flex align-items-center">
                        <button id="sidebarToggle" class="sidebar-toggler me-2" aria-label="Toggle Sidebar">
                            <i class="fas fa-bars"></i>
                        </button>
                        
                        <div>
                            <!-- Desktop Title -->
                            <h1 class="navbar-brand mb-0 d-none d-md-block">Teacher Dashboard</h1>
                            <!-- Mobile Title -->
                            <h1 class="navbar-brand mb-0 d-block d-md-none">Dashboard</h1>
                        </div>
                    </div>
                
                <div class="d-flex align-items-center">
                    <!-- Search Bar -->
                    <div class="search-bar me-3 d-none d-md-block">
                        <form action="search.php" method="GET">
                            <div class="input-group">
                                <input type="text" class="form-control" placeholder="Search..." name="q">
                                <button class="btn btn-outline-secondary" type="submit">
                                    <i class="fas fa-search"></i>
                                </button>
                            </div>
                        </form>
                    </div>

                    <!-- Notifications -->
                    <div class="dropdown notification-dropdown me-3">
                        <button class="btn notification-btn position-relative" type="button" data-bs-toggle="dropdown" aria-expanded="false">
                            <i class="fas fa-bell"></i>
                            <span class="position-absolute top-0 start-100 translate-middle badge rounded-pill bg-danger">
                                3
                            </span>
                        </button>
                        <ul class="dropdown-menu notification-dropdown-menu dropdown-menu-end p-0">
                            <li class="notification-header d-flex justify-content-between align-items-center p-3">
                                <span class="fw-bold">Notifications</span>
                                <a href="notifications.php" class="text-muted small">Mark all as read</a>
                            </li>
                            <li><hr class="dropdown-divider m-0"></li>
                            <li class="notification-item unread">
                                <a href="#" class="dropdown-item d-flex align-items-center p-3">
                                    <div class="notification-icon bg-primary-soft rounded-circle me-3">
                                        <i class="fas fa-user-graduate text-primary"></i>
                                    </div>
                                    <div>
                                        <p class="mb-1 fw-medium">New student enrolled</p>
                                        <span class="text-muted small">5 minutes ago</span>
                                    </div>
                                </a>
                            </li>
                            <li><hr class="dropdown-divider m-0"></li>
                            <li class="notification-item unread">
                                <a href="#" class="dropdown-item d-flex align-items-center p-3">
                                    <div class="notification-icon bg-success-soft rounded-circle me-3">
                                        <i class="fas fa-comment text-success"></i>
                                    </div>
                                    <div>
                                        <p class="mb-1 fw-medium">New review on your course</p>
                                        <span class="text-muted small">1 hour ago</span>
                                    </div>
                                </a>
                            </li>
                            <li><hr class="dropdown-divider m-0"></li>
                            <li class="notification-item">
                                <a href="#" class="dropdown-item d-flex align-items-center p-3">
                                    <div class="notification-icon bg-info-soft rounded-circle me-3">
                                        <i class="fas fa-video text-info"></i>
                                    </div>
                                    <div>
                                        <p class="mb-1 fw-medium">Your video has been processed</p>
                                        <span class="text-muted small">3 hours ago</span>
                                    </div>
                                </a>
                            </li>
                            <li><hr class="dropdown-divider m-0"></li>
                            <li class="notification-footer">
                                <a href="notifications.php" class="dropdown-item text-center p-2">
                                    View all notifications
                                </a>
                            </li>
                        </ul>
                    </div>

                    <!-- User Profile -->
                    <div class="dropdown profile-dropdown">
                        <button class="btn p-0 dropdown-toggle" type="button" data-bs-toggle="dropdown" aria-expanded="false">
                            <div class="d-flex align-items-center">
                                <img src="<?php echo !empty($teacher['profile_picture']) ? '../uploads/profile/' . htmlspecialchars($teacher['profile_picture']) : '../assets/images/avatar-placeholder.jpg'; ?>" 
                                     class="avatar-sm rounded-circle me-2" alt="<?php echo htmlspecialchars($teacher['first_name']); ?>">
                                <div class="d-none d-sm-block">
                                    <span class="fw-medium"><?php echo htmlspecialchars($teacher['first_name'] . ' ' . $teacher['last_name']); ?></span>
                                </div>
                            </div>
                        </button>
                        <ul class="dropdown-menu dropdown-menu-end">
                            <li>
                                <a class="dropdown-item" href="profile.php">
                                    <i class="fas fa-user me-2 text-muted"></i> My Profile
                                </a>
                            </li>
                            <li>
                                <a class="dropdown-item" href="settings.php">
                                    <i class="fas fa-cog me-2 text-muted"></i> Settings
                                </a>
                            </li>
                            <li><hr class="dropdown-divider"></li>
                            <li>
                                <a class="dropdown-item" href="../logout.php">
                                    <i class="fas fa-sign-out-alt me-2 text-muted"></i> Logout
                                </a>
                            </li>
                        </ul>
                    </div>
                </div>
                
                <!-- Recent Data Section -->
                <div class="row g-4 mb-4">
                    <!-- Recent Courses -->
                    <div class="col-12 col-lg-6">
                        <div class="data-table-card">
                            <div class="data-table-header">
                                <h5 class="data-table-title">Recent Courses</h5>
                                <a href="courses.php" class="btn btn-sm btn-outline-primary">View All</a>
                            </div>
                            
                            <div class="table-responsive">
                                <table class="table data-table align-middle">
                                    <thead>
                                        <tr>
                                            <th>Course</th>
                                            <th>Students</th>
                                            <th>Rating</th>
                                            <th>Status</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                    <?php if ($recent_courses->num_rows > 0): ?>
                                        <?php while($course = $recent_courses->fetch_assoc()): ?>
                                        <tr>
                                            <td>
                                                <div class="d-flex align-items-center">
                                                    <img src="<?php echo !empty($course['image']) ? '../uploads/courses/' . htmlspecialchars($course['image']) : '../assets/images/course-placeholder.jpg'; ?>" 
                                                         class="rounded me-2" width="40" height="40" alt="Course Image">
                                                    <div>
                                                        <div class="fw-medium"><?php echo htmlspecialchars($course['title']); ?></div>
                                                        <div class="small text-muted"><?php echo date('M d, Y', strtotime($course['created_at'])); ?></div>
                                                    </div>
                                                </div>
                                            </td>
                                            <td>
                                                <?php 
                                                // Get enrollment count
                                                $stmt = $conn->prepare("SELECT COUNT(*) as enrollment_count FROM enrollments WHERE course_id = ?");
                                                $stmt->bind_param("i", $course['course_id']);
                                                $stmt->execute();
                                                $enrollment_result = $stmt->get_result();
                                                $enrollment_data = $enrollment_result->fetch_assoc();
                                                echo $enrollment_data['enrollment_count'];
                                                ?>
                                            </td>
                                            <td>
                                                <div class="d-flex align-items-center">
                                                    <span class="me-1"><?php echo number_format($course['rating'] ?? 0, 1); ?></span>
                                                    <i class="fas fa-star text-warning"></i>
                                                </div>
                                            </td>
                                            <td>
                                                <?php if($course['is_published'] == 1): ?>
                                                    <span class="status-badge status-active">Active</span>
                                                <?php else: ?>
                                                    <span class="status-badge status-pending">Draft</span>
                                                <?php endif; ?>
                                            </td>
                                        </tr>
                                        <?php endwhile; ?>
                                    <?php else: ?>
                                        <tr>
                                            <td colspan="4" class="text-center py-4">No courses found. <a href="create-course.php">Create your first course</a>.</td>
                                        </tr>
                                    <?php endif; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Recent Videos -->
                    <div class="col-12 col-lg-6">
                        <div class="data-table-card">
                            <div class="data-table-header">
                                <h5 class="data-table-title">Recent Videos</h5>
                                <a href="videos.php" class="btn btn-sm btn-outline-primary">View All</a>
                            </div>
                            
                            <div class="table-responsive">
                                <table class="table data-table align-middle">
                                    <thead>
                                        <tr>
                                            <th>Video</th>
                                            <th>Duration</th>
                                            <th>Views</th>
                                            <th>Status</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                    <?php if ($recent_videos->num_rows > 0): ?>
                                        <?php while($video = $recent_videos->fetch_assoc()): ?>
                                        <tr>
                                            <td>
                                                <div class="d-flex align-items-center">
                                                    <div class="rounded me-2 bg-dark position-relative" style="width: 60px; height: 34px;">
                                                        <img src="<?php echo !empty($video['thumbnail']) ? '../uploads/videos/thumbnails/' . htmlspecialchars($video['thumbnail']) : '../assets/images/video-placeholder.jpg'; ?>" 
                                                            class="rounded w-100 h-100 object-fit-cover" alt="Video Thumbnail">
                                                        <div class="position-absolute top-50 start-50 translate-middle text-white">
                                                            <i class="fas fa-play fa-xs"></i>
                                                        </div>
                                                    </div>
                                                    <div>
                                                        <div class="fw-medium"><?php echo htmlspecialchars($video['title']); ?></div>
                                                        <div class="small text-muted"><?php echo date('M d, Y', strtotime($video['created_at'])); ?></div>
                                                    </div>
                                                </div>
                                            </td>
                                            <td>
                                                <?php 
                                                // Format duration (assuming duration is stored in seconds)
                                                $duration = $video['duration'] ?? 0;
                                                $minutes = floor($duration / 60);
                                                $seconds = $duration % 60;
                                                echo sprintf('%d:%02d', $minutes, $seconds);
                                                ?>
                                            </td>
                                            <td>
                                                <?php echo number_format($video['views'] ?? 0); ?>
                                            </td>
                                            <td>
                                                <?php if($video['is_published'] == 1): ?>
                                                    <span class="status-badge status-active">Published</span>
                                                <?php else: ?>
                                                    <span class="status-badge status-pending">Private</span>
                                                <?php endif; ?>
                                            </td>
                                        </tr>
                                        <?php endwhile; ?>
                                    <?php else: ?>
                                        <tr>
                                            <td colspan="4" class="text-center py-4">No videos found. <a href="upload-video.php">Upload your first video</a>.</td>
                                        </tr>
                                    <?php endif; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Engagement & Activity Charts -->
                <div class="row g-4">
                    <div class="col-12 col-lg-8">
                        <div class="data-table-card">
                            <div class="data-table-header">
                                <h5 class="data-table-title">Course Engagement</h5>
                            </div>
                            <canvas id="engagementChart" class="recent-chart"></canvas>
                        </div>
                    </div>
                    <div class="col-12 col-lg-4">
                        <div class="data-table-card">
                            <div class="data-table-header">
                                <h5 class="data-table-title">Student Distribution</h5>
                            </div>
                            <canvas id="distributionChart" class="recent-chart"></canvas>
                        </div>
                    </div>
                </div>
            </div><!-- /.main-content -->
        </div><!-- /.content-wrapper -->
    </div><!-- /.modern-dashboard -->
    
    <!-- JavaScript Libraries -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/toastify-js"></script>
    
    <!-- Custom JavaScript -->
    <script src="../assets/js/night-sky-new.js"></script>
    
    <script>
        // Initialize Chart.js
        document.addEventListener('DOMContentLoaded', function() {
            // Course Engagement Chart
            var engagementCtx = document.getElementById('engagementChart').getContext('2d');
            var engagementChart = new Chart(engagementCtx, {
                type: 'line',
                data: {
                    labels: ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul'],
                    datasets: [{
                        label: 'Course Views',
                        data: [650, 590, 800, 810, 960, 1020, 1180],
                        borderColor: '#276BE4',
                        backgroundColor: 'rgba(39, 107, 228, 0.1)',
                        tension: 0.3,
                        fill: true
                    },
                    {
                        label: 'Student Activity',
                        data: [350, 420, 550, 530, 620, 680, 720],
                        borderColor: '#28a745',
                        backgroundColor: 'rgba(40, 167, 69, 0.1)',
                        tension: 0.3,
                        fill: true
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            position: 'top',
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            grid: {
                                display: true,
                                drawBorder: false
                            }
                        },
                        x: {
                            grid: {
                                display: false
                            }
                        }
                    }
                }
            });
            
            // Student Distribution Chart
            var distributionCtx = document.getElementById('distributionChart').getContext('2d');
            var distributionChart = new Chart(distributionCtx, {
                type: 'doughnut',
                data: {
                    labels: ['Programming', 'Design', 'Marketing', 'Business', 'Other'],
                    datasets: [{
                        data: [35, 25, 20, 15, 5],
                        backgroundColor: [
                            '#276BE4',
                            '#28a745',
                            '#17a2b8',
                            '#ffc107',
                            '#6c757d'
                        ],
                        borderWidth: 0
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    cutout: '70%',
                    plugins: {
                        legend: {
                            position: 'bottom'
                        }
                    }
                }
            });
        });
    </script>
</body>
</html>
                
                <div class="navbar-actions">
                    <!-- Desktop Action Buttons -->
                    <div class="d-none d-md-flex me-2">
                        <a href="create-course.php" class="btn btn-primary action-btn me-2">
                            <i class="fas fa-plus me-1"></i> New Course
                        </a>
                        <a href="upload-video.php" class="btn btn-outline-primary action-btn">
                            <i class="fas fa-upload me-1"></i> Upload Video
                        </a>
                    </div>
                    
                    <!-- Mobile Action Button -->
                    <div class="dropdown d-md-none me-2">
                        <button class="btn btn-primary action-btn" type="button" id="mobileActionButton" data-bs-toggle="dropdown" aria-expanded="false">
                            <i class="fas fa-plus"></i>
                        </button>
                        <ul class="dropdown-menu dropdown-menu-end" aria-labelledby="mobileActionButton">
                            <li><a class="dropdown-item" href="create-course.php">
                                <i class="fas fa-graduation-cap me-1"></i> New Course
                            </a></li>
                            <li><a class="dropdown-item" href="upload-video.php">
                                <i class="fas fa-video me-1"></i> Upload Video
                            </a></li>
                        </ul>
                    </div>
                    
                    <!-- Notification Bell -->
                    <div class="dropdown">
                        <button class="notification-bell" type="button" id="notificationDropdown" data-bs-toggle="dropdown" aria-expanded="false">
                            <i class="fas fa-bell"></i>
                            <span class="notification-badge">3</span>
                        </button>
                        <div class="dropdown-menu dropdown-menu-end notification-dropdown" aria-labelledby="notificationDropdown">
                            <div class="dropdown-header d-flex justify-content-between align-items-center p-3">
                                <span>Notifications</span>
                                <a href="#" class="text-decoration-none small">Mark all as read</a>
                            </div>
                            <div class="notification-item unread">
                                <div class="notification-icon bg-light-primary">
                                    <i class="fas fa-user"></i>
                                </div>
                                <div>
                                    <div class="notification-content">New student enrolled in <strong>Advanced JavaScript</strong></div>
                                    <span class="notification-time">2 hours ago</span>
                                </div>
                            </div>
                            <div class="notification-item unread">
                                <div class="notification-icon bg-light-success">
                                    <i class="fas fa-comment"></i>
                                </div>
                                <div>
                                    <div class="notification-content">New comment on your <strong>React Basics</strong> course</div>
                                    <span class="notification-time">5 hours ago</span>
                                </div>
                            </div>
                            <div class="notification-item">
                                <div class="notification-icon bg-light-info">
                                    <i class="fas fa-star"></i>
                                </div>
                                <div>
                                    <div class="notification-content">New 5-star review on <strong>Python for Beginners</strong></div>
                                    <span class="notification-time">Yesterday</span>
                                </div>
                            </div>
                            <a href="notifications.php" class="dropdown-item text-center small text-gray-500 py-2">Show all notifications</a>
                        </div>
                    </div>
                    
                    <!-- User Profile Dropdown -->
                    <div class="dropdown ms-2">
                        <div class="profile-menu" id="userDropdown" role="button" data-bs-toggle="dropdown" aria-expanded="false">
                            <div class="user-name d-none d-sm-block">
                                <span><?php echo htmlspecialchars($teacher['name']); ?></span>
                                <small class="user-role">Teacher</small>
                            </div>
                            <img src="<?php echo !empty($teacher['profile_image']) ? '../uploads/profile/' . htmlspecialchars($teacher['profile_image']) : '../assets/images/default-avatar.png'; ?>" alt="Profile" class="profile-image">
                        </div>
                        <ul class="dropdown-menu dropdown-menu-end" aria-labelledby="userDropdown">
                            <li><a class="dropdown-item" href="profile.php">
                                <i class="fas fa-user me-2 text-gray-400"></i> Profile
                            </a></li>
                            <li><a class="dropdown-item" href="settings.php">
                                <i class="fas fa-cog me-2 text-gray-400"></i> Settings
                            </a></li>
                            <li><hr class="dropdown-divider"></li>
                            <li><a class="dropdown-item" href="../logout.php">
                                <i class="fas fa-sign-out-alt me-2 text-gray-400"></i> Logout
                            </a></li>
                        </ul>
                    </div>
                </div>
                
                <!-- Recent Data Section -->
                <div class="row g-4 mb-4">
                    <!-- Recent Courses -->
                    <div class="col-12 col-lg-6">
                        <div class="data-table-card">
                            <div class="data-table-header">
                                <h5 class="data-table-title">Recent Courses</h5>
                                <a href="courses.php" class="btn btn-sm btn-outline-primary">View All</a>
                            </div>
                            
                            <div class="table-responsive">
                                <table class="table data-table align-middle">
                                    <thead>
                                        <tr>
                                            <th>Course</th>
                                            <th>Students</th>
                                            <th>Rating</th>
                                            <th>Status</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                    <?php if ($recent_courses->num_rows > 0): ?>
                                        <?php while($course = $recent_courses->fetch_assoc()): ?>
                                        <tr>
                                            <td>
                                                <div class="d-flex align-items-center">
                                                    <img src="<?php echo !empty($course['image']) ? '../uploads/courses/' . htmlspecialchars($course['image']) : '../assets/images/course-placeholder.jpg'; ?>" 
                                                         class="rounded me-2" width="40" height="40" alt="Course Image">
                                                    <div>
                                                        <div class="fw-medium"><?php echo htmlspecialchars($course['title']); ?></div>
                                                        <div class="small text-muted"><?php echo date('M d, Y', strtotime($course['created_at'])); ?></div>
                                                    </div>
                                                </div>
                                            </td>
                                            <td>
                                                <?php 
                                                // Get enrollment count
                                                $stmt = $conn->prepare("SELECT COUNT(*) as enrollment_count FROM enrollments WHERE course_id = ?");
                                                $stmt->bind_param("i", $course['course_id']);
                                                $stmt->execute();
                                                $enrollment_result = $stmt->get_result();
                                                $enrollment_data = $enrollment_result->fetch_assoc();
                                                echo $enrollment_data['enrollment_count'];
                                                ?>
                                            </td>
                                            <td>
                                                <div class="d-flex align-items-center">
                                                    <span class="me-1"><?php echo number_format($course['rating'] ?? 0, 1); ?></span>
                                                    <i class="fas fa-star text-warning"></i>
                                                </div>
                                            </td>
                                            <td>
                                                <?php if($course['is_published'] == 1): ?>
                                                    <span class="status-badge status-active">Active</span>
                                                <?php else: ?>
                                                    <span class="status-badge status-pending">Draft</span>
                                                <?php endif; ?>
                                            </td>
                                        </tr>
                                        <?php endwhile; ?>
                                    <?php else: ?>
                                        <tr>
                                            <td colspan="4" class="text-center py-4">No courses found. <a href="create-course.php">Create your first course</a>.</td>
                                        </tr>
                                    <?php endif; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Recent Videos -->
                    <div class="col-12 col-lg-6">
                        <div class="data-table-card">
                            <div class="data-table-header">
                                <h5 class="data-table-title">Recent Videos</h5>
                                <a href="videos.php" class="btn btn-sm btn-outline-primary">View All</a>
                            </div>
                            
                            <div class="table-responsive">
                                <table class="table data-table align-middle">
                                    <thead>
                                        <tr>
                                            <th>Video</th>
                                            <th>Duration</th>
                                            <th>Views</th>
                                            <th>Status</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                    <?php if ($recent_videos->num_rows > 0): ?>
                                        <?php while($video = $recent_videos->fetch_assoc()): ?>
                                        <tr>
                                            <td>
                                                <div class="d-flex align-items-center">
                                                    <div class="rounded me-2 bg-dark position-relative" style="width: 60px; height: 34px;">
                                                        <img src="<?php echo !empty($video['thumbnail']) ? '../uploads/videos/thumbnails/' . htmlspecialchars($video['thumbnail']) : '../assets/images/video-placeholder.jpg'; ?>" 
                                                            class="rounded w-100 h-100 object-fit-cover" alt="Video Thumbnail">
                                                        <div class="position-absolute top-50 start-50 translate-middle text-white">
                                                            <i class="fas fa-play fa-xs"></i>
                                                        </div>
                                                    </div>
                                                    <div>
                                                        <div class="fw-medium"><?php echo htmlspecialchars($video['title']); ?></div>
                                                        <div class="small text-muted"><?php echo date('M d, Y', strtotime($video['created_at'])); ?></div>
                                                    </div>
                                                </div>
                                            </td>
                                            <td>
                                                <?php 
                                                // Format duration (assuming duration is stored in seconds)
                                                $duration = $video['duration'] ?? 0;
                                                $minutes = floor($duration / 60);
                                                $seconds = $duration % 60;
                                                echo sprintf('%d:%02d', $minutes, $seconds);
                                                ?>
                                            </td>
                                            <td>
                                                <?php echo number_format($video['views'] ?? 0); ?>
                                            </td>
                                            <td>
                                                <?php if($video['is_published'] == 1): ?>
                                                    <span class="status-badge status-active">Published</span>
                                                <?php else: ?>
                                                    <span class="status-badge status-pending">Private</span>
                                                <?php endif; ?>
                                            </td>
                                        </tr>
                                        <?php endwhile; ?>
                                    <?php else: ?>
                                        <tr>
                                            <td colspan="4" class="text-center py-4">No videos found. <a href="upload-video.php">Upload your first video</a>.</td>
                                        </tr>
                                    <?php endif; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Engagement & Activity Charts -->
                <div class="row g-4">
                    <div class="col-12 col-lg-8">
                        <div class="data-table-card">
                            <div class="data-table-header">
                                <h5 class="data-table-title">Course Engagement</h5>
                            </div>
                            <canvas id="engagementChart" class="recent-chart"></canvas>
                        </div>
                    </div>
                    <div class="col-12 col-lg-4">
                        <div class="data-table-card">
                            <div class="data-table-header">
                                <h5 class="data-table-title">Student Distribution</h5>
                            </div>
                            <canvas id="distributionChart" class="recent-chart"></canvas>
                        </div>
                    </div>
                </div>
            </div><!-- /.main-content -->
        </div><!-- /.content-wrapper -->
    </div><!-- /.modern-dashboard -->
    
    <!-- JavaScript Libraries -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/toastify-js"></script>
    
    <!-- Custom JavaScript -->
    <script src="../assets/js/night-sky-new.js"></script>
    
    <script>
        // Initialize Chart.js
        document.addEventListener('DOMContentLoaded', function() {
            // Course Engagement Chart
            var engagementCtx = document.getElementById('engagementChart').getContext('2d');
            var engagementChart = new Chart(engagementCtx, {
                type: 'line',
                data: {
                    labels: ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul'],
                    datasets: [{
                        label: 'Course Views',
                        data: [650, 590, 800, 810, 960, 1020, 1180],
                        borderColor: '#276BE4',
                        backgroundColor: 'rgba(39, 107, 228, 0.1)',
                        tension: 0.3,
                        fill: true
                    },
                    {
                        label: 'Student Activity',
                        data: [350, 420, 550, 530, 620, 680, 720],
                        borderColor: '#28a745',
                        backgroundColor: 'rgba(40, 167, 69, 0.1)',
                        tension: 0.3,
                        fill: true
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            position: 'top',
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            grid: {
                                display: true,
                                drawBorder: false
                            }
                        },
                        x: {
                            grid: {
                                display: false
                            }
                        }
                    }
                }
            });
            
            // Student Distribution Chart
            var distributionCtx = document.getElementById('distributionChart').getContext('2d');
            var distributionChart = new Chart(distributionCtx, {
                type: 'doughnut',
                data: {
                    labels: ['Programming', 'Design', 'Marketing', 'Business', 'Other'],
                    datasets: [{
                        data: [35, 25, 20, 15, 5],
                        backgroundColor: [
                            '#276BE4',
                            '#28a745',
                            '#17a2b8',
                            '#ffc107',
                            '#6c757d'
                        ],
                        borderWidth: 0
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    cutout: '70%',
                    plugins: {
                        legend: {
                            position: 'bottom'
                        }
                    }
                }
            });
        });
    </script>
</body>
</html>




















