<?php
/**
 * 🔍 AJAX HANDLER DEBUG SCRIPT
 */

require_once __DIR__ . '/../includes/config.php';

echo "<html><head><title>🔍 AJAX Handler Debug</title></head><body style='font-family: monospace; padding: 20px;'>";
echo "<h1>🔍 AJAX HANDLER DEBUG INVESTIGATION</h1>";
echo "<hr>";

echo "<h2>🎯 TESTING JAVASCRIPT AJAX HANDLER BINDING</h2>";

echo "<div id='testResults'></div>";

echo "<script src='https://code.jquery.com/jquery-3.6.0.min.js'></script>";
echo "<script>";
echo "
console.log('🔍 DEBUG: Starting AJAX handler investigation...');

$(document).ready(function() {
    console.log('✅ DEBUG: jQuery loaded, version:', $.fn.jquery);
    
    // 1. TEST: Check if form exists
    const form = $('#videoUploadForm');
    console.log('🔍 DEBUG: Form found:', form.length > 0);
    if (form.length === 0) {
        console.error('❌ CRITICAL: Form #videoUploadForm NOT FOUND!');
        $('#testResults').append('<p style=\"color: red;\">❌ FORM NOT FOUND: #videoUploadForm missing</p>');
        return;
    }
    
    // 2. TEST: Check form action
    const action = form.attr('action');
    console.log('🔍 DEBUG: Form action:', action);
    $('#testResults').append('<p>Form action: ' + action + '</p>');
    
    // 3. TEST: Check if AJAX handler binds
    let handlerBound = false;
    
    // Try to bind our AJAX handler
    try {
        form.off('submit.ajax-test'); // Remove any existing test handlers
        
        form.on('submit.ajax-test', function(e) {
            console.log('✅ DEBUG: AJAX handler triggered successfully!');
            e.preventDefault();
            handlerBound = true;
            $('#testResults').append('<p style=\"color: green;\">✅ AJAX handler binding works</p>');
            return false;
        });
        
        console.log('✅ DEBUG: AJAX handler bound successfully');
        $('#testResults').append('<p style=\"color: green;\">✅ AJAX handler binding succeeded</p>');
        
    } catch(error) {
        console.error('❌ ERROR binding AJAX handler:', error);
        $('#testResults').append('<p style=\"color: red;\">❌ AJAX binding failed: ' + error.message + '</p>');
    }
    
    // 4. TEST: Simulate form submission
    setTimeout(function() {
        console.log('🧪 DEBUG: Testing form submission...');
        form.trigger('submit');
        
        if (!handlerBound) {
            console.error('❌ CRITICAL: AJAX handler did not trigger on submit!');
            $('#testResults').append('<p style=\"color: red;\">❌ AJAX handler did not trigger</p>');
        }
    }, 1000);
    
    // 5. TEST: Check for conflicting handlers
    const events = $._data(form[0], 'events');
    console.log('🔍 DEBUG: Current form events:', events);
    
    if (events && events.submit) {
        console.log('🔍 DEBUG: Submit handlers found:', events.submit.length);
        $('#testResults').append('<p>Submit handlers: ' + events.submit.length + '</p>');
        
        events.submit.forEach(function(handler, index) {
            console.log('Handler ' + index + ':', handler);
            $('#testResults').append('<p>Handler ' + index + ': ' + (handler.namespace || 'no-namespace') + '</p>');
        });
    } else {
        console.log('⚠️ DEBUG: No submit handlers found');
        $('#testResults').append('<p style=\"color: orange;\">⚠️ No submit handlers found</p>');
    }
    
    // 6. TEST: Check jQuery conflicts
    if (typeof $ !== 'function') {
        console.error('❌ CRITICAL: jQuery $ not available!');
        $('#testResults').append('<p style=\"color: red;\">❌ jQuery $ conflict detected</p>');
    }
    
    // 7. TEST: Check for JavaScript errors
    window.addEventListener('error', function(e) {
        console.error('❌ JavaScript Error:', e.error);
        $('#testResults').append('<p style=\"color: red;\">❌ JS Error: ' + e.message + '</p>');
    });
});

// 8. TEST: Direct form submission override
function testDirectFormSubmission() {
    console.log('🧪 DEBUG: Testing direct form submission override...');
    
    // Get the actual form from upload page
    const uploadPageUrl = 'upload-video.php';
    
    fetch(uploadPageUrl)
        .then(response => response.text())
        .then(html => {
            // Check if form has the right ID and structure
            const hasFormId = html.includes('id=\"videoUploadForm\"');
            const hasAction = html.includes('action=\"process-video-with-content.php\"');
            const hasJQuery = html.includes('jquery');
            
            console.log('🔍 Form ID exists:', hasFormId);
            console.log('🔍 Form action correct:', hasAction);
            console.log('🔍 jQuery included:', hasJQuery);
            
            $('#testResults').append('<p>Form ID in upload page: ' + (hasFormId ? '✅' : '❌') + '</p>');
            $('#testResults').append('<p>Form action in upload page: ' + (hasAction ? '✅' : '❌') + '</p>');
            $('#testResults').append('<p>jQuery in upload page: ' + (hasJQuery ? '✅' : '❌') + '</p>');
        })
        .catch(error => {
            console.error('❌ Error fetching upload page:', error);
            $('#testResults').append('<p style=\"color: red;\">❌ Could not fetch upload page</p>');
        });
}

setTimeout(testDirectFormSubmission, 2000);
";
echo "</script>";

echo "<h2>🎯 MANUAL TESTS</h2>";
echo "<ol>";
echo "<li><strong>Go to upload page:</strong> <a href='upload-video.php' target='_blank'>upload-video.php</a></li>";
echo "<li><strong>Open browser console (F12)</strong></li>";
echo "<li><strong>Look for JavaScript errors</strong></li>";
echo "<li><strong>Check if AJAX handler logs appear</strong></li>";
echo "<li><strong>Try submitting form and see console output</strong></li>";
echo "</ol>";

echo "<h2>🔍 NEXT INVESTIGATION STEPS</h2>";
echo "<ol>";
echo "<li><strong>Check timing issues:</strong> DOM ready vs script execution</li>";
echo "<li><strong>Check jQuery conflicts:</strong> Multiple jQuery versions</li>";
echo "<li><strong>Check form validation:</strong> Bootstrap validation preventing AJAX</li>";
echo "<li><strong>Check backend:</strong> JSON response format issues</li>";
echo "<li><strong>Check event propagation:</strong> Other handlers stopping AJAX</li>";
echo "</ol>";

echo "</body></html>";
?>
