<?php
// Comprehensive Analytics Debug Script
// This script will investigate all data sources for the teacher analytics page

// Error reporting for debugging
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Include database configuration
require_once 'includes/config.php';

// Establish database connection
$conn = new mysqli(DB_SERVER, DB_USERNAME, DB_PASSWORD, DB_NAME);

// Check connection
if ($conn->connect_error) {
    die("Connection failed: " . $conn->connect_error);
}

// Set charset
$conn->set_charset("utf8");

echo "<h1>🔍 Panadite Academy Analytics Debug Report</h1>";
echo "<style>
body { font-family: Arial, sans-serif; margin: 20px; }
.section { margin: 20px 0; padding: 15px; border: 1px solid #ddd; border-radius: 8px; }
.success { background-color: #d4edda; border-color: #c3e6cb; }
.warning { background-color: #fff3cd; border-color: #ffeaa7; }
.error { background-color: #f8d7da; border-color: #f5c6cb; }
.info { background-color: #d1ecf1; border-color: #bee5eb; }
table { width: 100%; border-collapse: collapse; margin: 10px 0; }
th, td { padding: 8px; text-align: left; border: 1px solid #ddd; }
th { background-color: #f2f2f2; }
.query-box { background: #f8f9fa; padding: 10px; border-radius: 4px; font-family: monospace; margin: 10px 0; }
</style>";

// For debugging purposes, let's find and use a teacher ID from the database
$teacher_id = null;

// Try to get a teacher ID from the users table or directly from courses table
$teacher_query = "SELECT DISTINCT teacher_id FROM courses WHERE teacher_id IS NOT NULL LIMIT 1";
$teacher_result = $conn->query($teacher_query);

if ($teacher_result && $teacher_result->num_rows > 0) {
    $teacher_id = $teacher_result->fetch_assoc()['teacher_id'];
    echo "<div class='section info'><h2>👤 Teacher Information (Debug Mode)</h2>";
    echo "Using Teacher ID for analysis: <strong>$teacher_id</strong><br>";
    echo "<em>Note: Running in debug mode from root directory</em></div>";
} else {
    // Fallback: check if there are any users with teacher role
    $user_check = $conn->query("SELECT user_id FROM users WHERE role = 'teacher' LIMIT 1");
    if ($user_check && $user_check->num_rows > 0) {
        $teacher_id = $user_check->fetch_assoc()['user_id'];
        echo "<div class='section info'><h2>👤 Teacher Information (Debug Mode)</h2>";
        echo "Using Teacher ID from users table: <strong>$teacher_id</strong><br>";
        echo "<em>Note: Running in debug mode from root directory</em></div>";
    } else {
        echo "<div class='section error'><h2>❌ No Teacher Found</h2>";
        echo "No teachers found in database. Please ensure teachers exist in the system.</div>";
        // Continue anyway to check database structure
        $teacher_id = 1; // Use ID 1 as fallback for structure analysis
        echo "<div class='section warning'><h2>⚠️ Using Fallback</h2>";
        echo "Using teacher ID <strong>1</strong> for database structure analysis.</div>";
    }
}

// Database Connection Check
echo "<div class='section info'><h2>🗄️ Database Connection</h2>";
if ($conn) {
    echo "✅ Database connection successful<br>";
    echo "Connected to: " . $conn->get_server_info();
} else {
    echo "❌ Database connection failed";
    exit();
}
echo "</div>";

// 1. Check Tables Existence
echo "<div class='section info'><h2>📋 Database Tables Check</h2>";
$tables_to_check = [
    'courses', 'enrollments', 'course_enrollments', 'student_enrollments',
    'videos', 'video_progress', 'student_progress', 
    'quizzes', 'quiz_attempts', 'quiz_results',
    'payments', 'course_views', 'course_analytics',
    'users', 'categories'
];

foreach ($tables_to_check as $table) {
    $result = $conn->query("SHOW TABLES LIKE '$table'");
    if ($result && $result->num_rows > 0) {
        echo "✅ Table '$table' exists<br>";
    } else {
        echo "❌ Table '$table' does not exist<br>";
    }
}
echo "</div>";

// 2. ALL COURSES Analysis (Find ALL courses, not just for one teacher)
echo "<div class='section info'><h2>📚 ALL Courses Analysis (Finding Real Data)</h2>";

// First, let's see ALL courses in the system
$all_courses_query = "SELECT * FROM courses ORDER BY created_at DESC";
echo "<div class='query-box'>$all_courses_query</div>";
$all_courses_result = $conn->query($all_courses_query);

$course_ids = []; // We'll collect ALL course IDs to find data

if ($all_courses_result && $all_courses_result->num_rows > 0) {
    echo "✅ Found " . $all_courses_result->num_rows . " total courses in system<br>";
    echo "<table><tr><th>Course ID</th><th>Title</th><th>Teacher ID</th><th>Category</th><th>Price</th><th>Published</th><th>Created</th></tr>";
    
    while ($course = $all_courses_result->fetch_assoc()) {
        $course_ids[] = $course['course_id']; // Collect ALL course IDs
        echo "<tr>";
        echo "<td>" . $course['course_id'] . "</td>";
        echo "<td>" . htmlspecialchars($course['title']) . "</td>";
        echo "<td>" . $course['teacher_id'] . "</td>";
        echo "<td>" . $course['category_id'] . "</td>";
        echo "<td>$" . ($course['price'] ?? '0.00') . "</td>";
        echo "<td>" . ($course['is_published'] ? 'Yes' : 'No') . "</td>";
        echo "<td>" . $course['created_at'] . "</td>";
        echo "</tr>";
    }
    echo "</table>";
} else {
    echo "❌ No courses found in system";
    $course_ids = [];
}

// Now let's see the specific teacher's courses
echo "<h3>Teacher ID $teacher_id Specific Courses:</h3>";
$teacher_courses_query = "SELECT * FROM courses WHERE teacher_id = $teacher_id";
echo "<div class='query-box'>$teacher_courses_query</div>";
$teacher_courses_result = $conn->query($teacher_courses_query);

if ($teacher_courses_result && $teacher_courses_result->num_rows > 0) {
    echo "✅ Teacher has " . $teacher_courses_result->num_rows . " courses<br>";
} else {
    echo "⚠️ No courses found for teacher ID $teacher_id - checking all teachers...<br>";
    
    // Let's see what teachers exist and their courses
    $all_teachers_query = "SELECT DISTINCT teacher_id, COUNT(*) as course_count FROM courses GROUP BY teacher_id";
    echo "<div class='query-box'>$all_teachers_query</div>";
    $all_teachers_result = $conn->query($all_teachers_query);
    
    if ($all_teachers_result) {
        echo "<table><tr><th>Teacher ID</th><th>Course Count</th></tr>";
        while ($teacher_row = $all_teachers_result->fetch_assoc()) {
            echo "<tr><td>" . $teacher_row['teacher_id'] . "</td><td>" . $teacher_row['course_count'] . "</td></tr>";
        }
        echo "</table>";
    }
}
echo "</div>";

// 3. ALL Student Enrollments Analysis (Find REAL enrollment data)
echo "<div class='section info'><h2>👥 ALL Student Enrollments Analysis (Real Data Hunt)</h2>";

// Try different enrollment table variations - look at ALL data, not limited to specific courses
$enrollment_tables = ['enrollments', 'course_enrollments', 'student_enrollments'];
$total_enrollments = 0;

foreach ($enrollment_tables as $table) {
    $check_table = $conn->query("SHOW TABLES LIKE '$table'");
    if ($check_table && $check_table->num_rows > 0) {
        echo "<h3>🔍 Analyzing table: $table</h3>";
        
        // Check table structure
        $structure = $conn->query("DESCRIBE $table");
        echo "Table structure:<br>";
        while ($col = $structure->fetch_assoc()) {
            echo "- " . $col['Field'] . " (" . $col['Type'] . ")<br>";
        }
        
        // Count ALL enrollments in the system
        $all_enrollment_query = "SELECT COUNT(*) as count FROM $table";
        echo "<div class='query-box'>$all_enrollment_query</div>";
        $all_enrollment_result = $conn->query($all_enrollment_query);
        
        if ($all_enrollment_result) {
            $count = $all_enrollment_result->fetch_assoc()['count'];
            echo "✅ Found $count TOTAL enrollments in $table<br>";
            $total_enrollments += $count;
            
            // Show ALL enrollment data to find the real information
            $all_enrollments_query = "SELECT * FROM $table ORDER BY enrolled_date DESC LIMIT 20";
            echo "<div class='query-box'>$all_enrollments_query</div>";
            $all_enrollments_result = $conn->query($all_enrollments_query);
            
            if ($all_enrollments_result && $all_enrollments_result->num_rows > 0) {
                echo "<h4>📊 Recent Enrollment Records (Last 20):</h4>";
                echo "<table><tr>";
                $first_row = true;
                while ($row = $all_enrollments_result->fetch_assoc()) {
                    if ($first_row) {
                        foreach (array_keys($row) as $header) {
                            echo "<th>$header</th>";
                        }
                        echo "</tr>";
                        $first_row = false;
                    }
                    echo "<tr>";
                    foreach ($row as $value) {
                        echo "<td>$value</td>";
                    }
                    echo "</tr>";
                }
                echo "</table>";
                
                // Group enrollments by course to see distribution
                $course_enrollment_query = "SELECT course_id, COUNT(*) as enrollment_count FROM $table GROUP BY course_id ORDER BY enrollment_count DESC";
                echo "<div class='query-box'>$course_enrollment_query</div>";
                $course_enrollment_result = $conn->query($course_enrollment_query);
                
                if ($course_enrollment_result && $course_enrollment_result->num_rows > 0) {
                    echo "<h4>📈 Enrollments by Course:</h4>";
                    echo "<table><tr><th>Course ID</th><th>Total Enrollments</th></tr>";
                    while ($course_data = $course_enrollment_result->fetch_assoc()) {
                        echo "<tr><td>" . $course_data['course_id'] . "</td><td>" . $course_data['enrollment_count'] . "</td></tr>";
                    }
                    echo "</table>";
                }
            }
        }
        echo "<hr>";
    }
}

echo "<strong>🎯 TOTAL ENROLLMENTS FOUND ACROSS ALL TABLES: $total_enrollments</strong>";
echo "</div>";

// 4. ALL Quiz Performance Analysis (Hunt for REAL quiz data)
echo "<div class='section info'><h2>🧠 ALL Quiz Performance Analysis (Real Data Hunt)</h2>";

// First, find ALL quizzes in the entire system
$all_quiz_query = "SELECT * FROM quizzes ORDER BY created_at DESC";
echo "<div class='query-box'>$all_quiz_query</div>";
$all_quiz_result = $conn->query($all_quiz_query);

if ($all_quiz_result && $all_quiz_result->num_rows > 0) {
    echo "✅ Found " . $all_quiz_result->num_rows . " TOTAL quizzes in system<br>";
    
    echo "<h4>📋 All Quizzes in System:</h4>";
    echo "<table><tr><th>Quiz ID</th><th>Title</th><th>Course ID</th><th>Questions</th><th>Created</th><th>Status</th></tr>";
    
    $quiz_ids = [];
    while ($quiz = $quiz_result->fetch_assoc()) {
        $quiz_ids[] = $quiz['quiz_id'];
        echo "<tr>";
        echo "<td>" . $quiz['quiz_id'] . "</td>";
        echo "<td>" . htmlspecialchars($quiz['title'] ?? 'Untitled') . "</td>";
        echo "<td>" . ($quiz['course_id'] ?? 'N/A') . "</td>";
        echo "<td>" . ($quiz['total_questions'] ?? $quiz['questions'] ?? 'N/A') . "</td>";
        echo "<td>" . ($quiz['created_at'] ?? 'N/A') . "</td>";
        echo "<td>" . ($quiz['status'] ?? 'active') . "</td>";
        echo "</tr>";
    }
    echo "</table>";
    
    // Check quiz attempts/results for ALL quizzes
    $attempt_tables = ['quiz_attempts', 'quiz_results', 'quiz_submissions'];
    foreach ($attempt_tables as $table) {
        $check_table = $conn->query("SHOW TABLES LIKE '$table'");
        if ($check_table && $check_table->num_rows > 0) {
            echo "<h3>🎯 Analyzing $table:</h3>";
            
            // Show table structure
            $structure = $conn->query("DESCRIBE $table");
            echo "Table structure:<br>";
            while ($col = $structure->fetch_assoc()) {
                echo "- " . $col['Field'] . " (" . $col['Type'] . ")<br>";
            }
            
            // Count ALL attempts in this table
            $all_attempts_query = "SELECT COUNT(*) as count FROM $table";
            echo "<div class='query-box'>$all_attempts_query</div>";
            $all_attempts_result = $conn->query($all_attempts_query);
            
            if ($all_attempts_result) {
                $count = $all_attempts_result->fetch_assoc()['count'];
                echo "✅ Found $count TOTAL attempts in $table<br>";
                
                if ($count > 0) {
                    // Show recent attempts with details
                    $recent_attempts_query = "SELECT * FROM $table ORDER BY attempt_date DESC, created_at DESC LIMIT 20";
                    echo "<div class='query-box'>$recent_attempts_query</div>";
                    $recent_attempts_result = $conn->query($recent_attempts_query);
                    
                    if ($recent_attempts_result && $recent_attempts_result->num_rows > 0) {
                        echo "<h4>📊 Recent Quiz Attempts (Last 20):</h4>";
                        echo "<table><tr>";
                        $first_row = true;
                        while ($attempt = $recent_attempts_result->fetch_assoc()) {
                            if ($first_row) {
                                foreach (array_keys($attempt) as $header) {
                                    echo "<th>$header</th>";
                                }
                                echo "</tr>";
                                $first_row = false;
                            }
                            echo "<tr>";
                            foreach ($attempt as $value) {
                                echo "<td>$value</td>";
                            }
                            echo "</tr>";
                        }
                        echo "</table>";
                    }
                    
                    // Show quiz performance statistics
                    $performance_query = "SELECT 
                        quiz_id, 
                        COUNT(*) as total_attempts,
                        AVG(score) as avg_score,
                        MAX(score) as max_score,
                        MIN(score) as min_score,
                        COUNT(CASE WHEN score >= 70 THEN 1 END) as passed_attempts
                        FROM $table 
                        WHERE score IS NOT NULL 
                        GROUP BY quiz_id 
                        ORDER BY total_attempts DESC";
                    echo "<div class='query-box'>$performance_query</div>";
                    $performance_result = $conn->query($performance_query);
                    
                    if ($performance_result && $performance_result->num_rows > 0) {
                        echo "<h4>📈 Quiz Performance Statistics:</h4>";
                        echo "<table><tr><th>Quiz ID</th><th>Total Attempts</th><th>Avg Score</th><th>Max Score</th><th>Min Score</th><th>Passed (≥70)</th></tr>";
                        while ($perf = $performance_result->fetch_assoc()) {
                            echo "<tr>";
                            echo "<td>" . $perf['quiz_id'] . "</td>";
                            echo "<td>" . $perf['total_attempts'] . "</td>";
                            echo "<td>" . round($perf['avg_score'], 1) . "%</td>";
                            echo "<td>" . $perf['max_score'] . "%</td>";
                            echo "<td>" . $perf['min_score'] . "%</td>";
                            echo "<td>" . $perf['passed_attempts'] . " (" . round(($perf['passed_attempts']/$perf['total_attempts'])*100, 1) . "%)</td>";
                            echo "</tr>";
                        }
                        echo "</table>";
                    }
                }
            }
            echo "<hr>";
        }
    }
} else {
    echo "❌ No quizzes found in entire system";
}
echo "</div>";

// 5. Revenue Analysis
echo "<div class='section info'><h2>💰 Revenue Analysis</h2>";
$payment_tables = ['payments', 'course_payments', 'transactions'];
$total_revenue = 0;

foreach ($payment_tables as $table) {
    $check_table = $conn->query("SHOW TABLES LIKE '$table'");
    if ($check_table && $check_table->num_rows > 0) {
        echo "<h3>Checking table: $table</h3>";
        
        if (!empty($course_ids)) {
            $course_list = implode(',', $course_ids);
            $revenue_query = "SELECT SUM(amount) as total FROM $table WHERE course_id IN ($course_list)";
            echo "<div class='query-box'>$revenue_query</div>";
            $revenue_result = $conn->query($revenue_query);
            
            if ($revenue_result) {
                $revenue = $revenue_result->fetch_assoc()['total'] ?? 0;
                echo "Revenue in $table: $" . number_format($revenue, 2) . "<br>";
                $total_revenue += $revenue;
            }
        }
    }
}

echo "<strong>Total Revenue: $" . number_format($total_revenue, 2) . "</strong>";
echo "</div>";

// 6. Course Views Analysis
echo "<div class='section info'><h2>👀 Course Views Analysis</h2>";
$view_tables = ['course_views', 'page_views', 'analytics'];

foreach ($view_tables as $table) {
    $check_table = $conn->query("SHOW TABLES LIKE '$table'");
    if ($check_table && $check_table->num_rows > 0) {
        echo "<h3>Checking table: $table</h3>";
        
        if (!empty($course_ids)) {
            $course_list = implode(',', $course_ids);
            $views_query = "SELECT SUM(views) as total FROM $table WHERE course_id IN ($course_list)";
            echo "<div class='query-box'>$views_query</div>";
            $views_result = $conn->query($views_query);
            
            if ($views_result) {
                $views = $views_result->fetch_assoc()['total'] ?? 0;
                echo "Views in $table: " . number_format($views) . "<br>";
            }
        }
    }
}
echo "</div>";

// 7. Watch Time Analysis
echo "<div class='section info'><h2>⏱️ Watch Time Analysis</h2>";
if (!empty($course_ids)) {
    $course_list = implode(',', $course_ids);
    
    // Check video progress tables
    $progress_tables = ['video_progress', 'student_progress', 'watch_history'];
    
    foreach ($progress_tables as $table) {
        $check_table = $conn->query("SHOW TABLES LIKE '$table'");
        if ($check_table && $check_table->num_rows > 0) {
            echo "<h3>Checking table: $table</h3>";
            
            // Check table structure first
            $structure = $conn->query("DESCRIBE $table");
            echo "Table structure:<br>";
            while ($col = $structure->fetch_assoc()) {
                echo "- " . $col['Field'] . " (" . $col['Type'] . ")<br>";
            }
            
            // Try to calculate watch time
            $watch_query = "SELECT SUM(watch_time) as total FROM $table 
                           WHERE course_id IN ($course_list) OR 
                           video_id IN (SELECT video_id FROM videos WHERE course_id IN ($course_list))";
            echo "<div class='query-box'>$watch_query</div>";
            $watch_result = $conn->query($watch_query);
            
            if ($watch_result) {
                $watch_time = $watch_result->fetch_assoc()['total'] ?? 0;
                $hours = floor($watch_time / 3600);
                $minutes = floor(($watch_time % 3600) / 60);
                echo "Watch time in $table: {$hours}h {$minutes}m<br>";
            }
        }
    }
}
echo "</div>";

// 8. DEEP DIVE: Student Completion Rates Analysis
echo "<div class='section info'><h2>🎯 DEEP DIVE: Student Completion Analysis</h2>";
if (!empty($course_ids)) {
    $course_list = implode(',', $course_ids);
    
    // Get detailed enrollment data with user info
    echo "<h3>📊 Detailed Enrollment Analysis:</h3>";
    $detailed_enrollment_query = "SELECT e.*, u.username, u.email, u.role, u.created_at as user_created 
                                  FROM enrollments e 
                                  LEFT JOIN users u ON e.student_id = u.user_id 
                                  WHERE e.course_id IN ($course_list)";
    echo "<div class='query-box'>$detailed_enrollment_query</div>";
    $detailed_result = $conn->query($detailed_enrollment_query);
    
    if ($detailed_result && $detailed_result->num_rows > 0) {
        echo "<table><tr><th>Enrollment ID</th><th>Student ID</th><th>Username</th><th>Email</th><th>Progress %</th><th>Enrolled Date</th><th>User Role</th></tr>";
        while ($enrollment = $detailed_result->fetch_assoc()) {
            echo "<tr>";
            echo "<td>" . $enrollment['enrollment_id'] . "</td>";
            echo "<td>" . $enrollment['student_id'] . "</td>";
            echo "<td>" . ($enrollment['username'] ?? 'N/A') . "</td>";
            echo "<td>" . ($enrollment['email'] ?? 'N/A') . "</td>";
            echo "<td>" . $enrollment['progress'] . "%</td>";
            echo "<td>" . $enrollment['enrolled_date'] . "</td>";
            echo "<td>" . ($enrollment['role'] ?? 'N/A') . "</td>";
            echo "</tr>";
        }
        echo "</table>";
    }
}
echo "</div>";

// 9. DEEP DIVE: Video Content & Progress Analysis
echo "<div class='section info'><h2>🎥 DEEP DIVE: Video Content & Progress Analysis</h2>";
if (!empty($course_ids)) {
    $course_list = implode(',', $course_ids);
    
    // Get all videos for the course
    echo "<h3>📹 Course Videos:</h3>";
    $videos_query = "SELECT * FROM videos WHERE course_id IN ($course_list)";
    echo "<div class='query-box'>$videos_query</div>";
    $videos_result = $conn->query($videos_query);
    
    if ($videos_result && $videos_result->num_rows > 0) {
        echo "<table><tr><th>Video ID</th><th>Title</th><th>Duration</th><th>Views</th><th>Upload Date</th><th>Status</th></tr>";
        $video_ids = [];
        while ($video = $videos_result->fetch_assoc()) {
            $video_ids[] = $video['video_id'];
            echo "<tr>";
            echo "<td>" . $video['video_id'] . "</td>";
            echo "<td>" . htmlspecialchars($video['title'] ?? 'Untitled') . "</td>";
            echo "<td>" . ($video['duration'] ?? 'N/A') . "</td>";
            echo "<td>" . ($video['views'] ?? '0') . "</td>";
            echo "<td>" . ($video['upload_date'] ?? $video['created_at'] ?? 'N/A') . "</td>";
            echo "<td>" . ($video['status'] ?? 'active') . "</td>";
            echo "</tr>";
        }
        echo "</table>";
        
        // Detailed video progress analysis
        if (!empty($video_ids)) {
            $video_list = implode(',', $video_ids);
            echo "<h3>📊 Video Progress Details:</h3>";
            
            // Video progress breakdown
            $progress_query = "SELECT vp.*, v.title as video_title, u.username 
                              FROM video_progress vp 
                              LEFT JOIN videos v ON vp.video_id = v.video_id 
                              LEFT JOIN users u ON vp.student_id = u.user_id 
                              WHERE vp.video_id IN ($video_list)";
            echo "<div class='query-box'>$progress_query</div>";
            $progress_result = $conn->query($progress_query);
            
            if ($progress_result && $progress_result->num_rows > 0) {
                echo "<table><tr><th>Student</th><th>Video</th><th>Completion %</th><th>Watch Time</th><th>Completed</th><th>Last Watched</th></tr>";
                while ($progress = $progress_result->fetch_assoc()) {
                    echo "<tr>";
                    echo "<td>" . ($progress['username'] ?? 'User ' . $progress['student_id']) . "</td>";
                    echo "<td>" . htmlspecialchars($progress['video_title'] ?? 'Video ' . $progress['video_id']) . "</td>";
                    echo "<td>" . ($progress['completion_percentage'] ?? '0') . "%</td>";
                    echo "<td>" . gmdate("H:i:s", $progress['watch_time'] ?? 0) . "</td>";
                    echo "<td>" . ($progress['completed'] ? '✅' : '❌') . "</td>";
                    echo "<td>" . ($progress['last_watched'] ?? 'Never') . "</td>";
                    echo "</tr>";
                }
                echo "</table>";
            } else {
                echo "❌ No video progress data found";
            }
        }
    } else {
        echo "❌ No videos found for course";
    }
}
echo "</div>";

// 10. DEEP DIVE: Payment & Revenue Analysis
echo "<div class='section info'><h2>💳 DEEP DIVE: Payment & Revenue Analysis</h2>";
if (!empty($course_ids)) {
    $course_list = implode(',', $course_ids);
    
    // Get all payment records
    echo "<h3>💰 Payment Records:</h3>";
    $payments_query = "SELECT p.*, u.username, u.email, c.title as course_title 
                      FROM payments p 
                      LEFT JOIN users u ON p.student_id = u.user_id 
                      LEFT JOIN courses c ON p.course_id = c.course_id 
                      WHERE p.course_id IN ($course_list)";
    echo "<div class='query-box'>$payments_query</div>";
    $payments_result = $conn->query($payments_query);
    
    if ($payments_result && $payments_result->num_rows > 0) {
        echo "<table><tr><th>Payment ID</th><th>Student</th><th>Course</th><th>Amount</th><th>Status</th><th>Date</th><th>Method</th></tr>";
        while ($payment = $payments_result->fetch_assoc()) {
            echo "<tr>";
            echo "<td>" . $payment['payment_id'] . "</td>";
            echo "<td>" . ($payment['username'] ?? 'User ' . $payment['student_id']) . "</td>";
            echo "<td>" . htmlspecialchars($payment['course_title'] ?? 'Course ' . $payment['course_id']) . "</td>";
            echo "<td>$" . number_format($payment['amount'], 2) . "</td>";
            echo "<td>" . ($payment['status'] ?? 'unknown') . "</td>";
            echo "<td>" . ($payment['payment_date'] ?? $payment['created_at'] ?? 'N/A') . "</td>";
            echo "<td>" . ($payment['payment_method'] ?? 'N/A') . "</td>";
            echo "</tr>";
        }
        echo "</table>";
    } else {
        echo "❌ No payment records found - this explains $0.00 revenue";
    }
}
echo "</div>";

// 11. DEEP DIVE: User Activity & Engagement
echo "<div class='section info'><h2>👤 DEEP DIVE: User Activity & Engagement</h2>";
$users_query = "SELECT u.*, COUNT(e.enrollment_id) as enrolled_courses 
                FROM users u 
                LEFT JOIN enrollments e ON u.user_id = e.student_id 
                GROUP BY u.user_id 
                ORDER BY enrolled_courses DESC 
                LIMIT 10";
echo "<div class='query-box'>$users_query</div>";
$users_result = $conn->query($users_query);

if ($users_result && $users_result->num_rows > 0) {
    echo "<table><tr><th>User ID</th><th>Username</th><th>Email</th><th>Role</th><th>Enrolled Courses</th><th>Joined Date</th><th>Last Login</th></tr>";
    while ($user = $users_result->fetch_assoc()) {
        echo "<tr>";
        echo "<td>" . $user['user_id'] . "</td>";
        echo "<td>" . htmlspecialchars($user['username']) . "</td>";
        echo "<td>" . htmlspecialchars($user['email']) . "</td>";
        echo "<td>" . ($user['role'] ?? 'user') . "</td>";
        echo "<td>" . $user['enrolled_courses'] . "</td>";
        echo "<td>" . ($user['created_at'] ?? 'N/A') . "</td>";
        echo "<td>" . ($user['last_login'] ?? 'Never') . "</td>";
        echo "</tr>";
    }
    echo "</table>";
}
echo "</div>";

// 12. DEEP DIVE: Course Completion Rates
echo "<div class='section info'><h2>📈 DEEP DIVE: Course Completion Statistics</h2>";
if (!empty($course_ids)) {
    foreach ($course_ids as $course_id) {
        echo "<h3>Course ID $course_id Completion Analysis:</h3>";
        
        // Get course completion statistics
        $completion_query = "SELECT 
            COUNT(DISTINCT e.student_id) as total_students,
            COUNT(DISTINCT CASE WHEN e.progress >= 100 THEN e.student_id END) as completed_students,
            AVG(e.progress) as avg_progress,
            COUNT(DISTINCT v.video_id) as total_videos,
            COUNT(DISTINCT vp.video_id) as videos_with_progress,
            SUM(vp.watch_time) as total_watch_time,
            COUNT(CASE WHEN vp.completed = 1 THEN 1 END) as completed_video_views
            FROM enrollments e
            LEFT JOIN videos v ON v.course_id = e.course_id
            LEFT JOIN video_progress vp ON vp.video_id = v.video_id AND vp.student_id = e.student_id
            WHERE e.course_id = $course_id";
        
        echo "<div class='query-box'>$completion_query</div>";
        $completion_result = $conn->query($completion_query);
        
        if ($completion_result) {
            $stats = $completion_result->fetch_assoc();
            echo "<table>";
            echo "<tr><th>Metric</th><th>Value</th></tr>";
            echo "<tr><td>Total Students Enrolled</td><td>" . ($stats['total_students'] ?? 0) . "</td></tr>";
            echo "<tr><td>Students Completed Course</td><td>" . ($stats['completed_students'] ?? 0) . "</td></tr>";
            echo "<tr><td>Average Progress</td><td>" . round($stats['avg_progress'] ?? 0, 2) . "%</td></tr>";
            echo "<tr><td>Total Videos in Course</td><td>" . ($stats['total_videos'] ?? 0) . "</td></tr>";
            echo "<tr><td>Videos with Student Progress</td><td>" . ($stats['videos_with_progress'] ?? 0) . "</td></tr>";
            echo "<tr><td>Total Watch Time (seconds)</td><td>" . ($stats['total_watch_time'] ?? 0) . "</td></tr>";
            echo "<tr><td>Completed Video Views</td><td>" . ($stats['completed_video_views'] ?? 0) . "</td></tr>";
            
            $completion_rate = ($stats['total_students'] > 0) ? 
                round(($stats['completed_students'] / $stats['total_students']) * 100, 2) : 0;
            echo "<tr><td><strong>Course Completion Rate</strong></td><td><strong>" . $completion_rate . "%</strong></td></tr>";
            echo "</table>";
        }
    }
}
echo "</div>";

// 13. Sample Data Creation Recommendations
echo "<div class='section warning'><h2>💡 ENHANCED Recommendations to Fix Analytics</h2>";
echo "<h3>🔧 Priority Fixes Based on Deep Analysis:</h3>";
echo "<ol>";
echo "<li><strong>Add More Student Enrollments:</strong> Only 1 student enrolled - need more for meaningful analytics</li>";
echo "<li><strong>Create Payment Records:</strong> No payments found - explains $0.00 revenue</li>";
echo "<li><strong>Add Video Progress Tracking:</strong> Ensure video watching generates progress data</li>";
echo "<li><strong>Create Course Views Table:</strong> Missing course_views table for view tracking</li>";
echo "<li><strong>Add Quizzes:</strong> No quizzes found - create sample quizzes for engagement metrics</li>";
echo "<li><strong>Populate Missing User Data:</strong> Enhance user profiles and activity tracking</li>";
echo "</ol>";

if (!empty($course_ids)) {
    echo "<h3>🚀 Complete Sample Data Creation Scripts:</h3>";
    $sample_course = $course_ids[0];
    echo "<div class='query-box'>";
    echo "-- Add more student enrollments<br>";
    echo "INSERT INTO enrollments (student_id, course_id, enrollment_date, progress) VALUES<br>";
    echo "(2, $sample_course, NOW(), 75),<br>";
    echo "(5, $sample_course, NOW(), 50),<br>";
    echo "(6, $sample_course, NOW(), 100),<br>";
    echo "(7, $sample_course, NOW(), 25);<br><br>";
    
    echo "-- Add payment records<br>";
    echo "INSERT INTO payments (course_id, student_id, amount, status, payment_date) VALUES<br>";
    echo "($sample_course, 2, 5000.00, 'completed', NOW()),<br>";
    echo "($sample_course, 5, 5000.00, 'completed', NOW()),<br>";
    echo "($sample_course, 6, 5000.00, 'completed', NOW());<br><br>";
    
    echo "-- Create course views table<br>";
    echo "CREATE TABLE IF NOT EXISTS course_views (<br>";
    echo "&nbsp;&nbsp;id INT AUTO_INCREMENT PRIMARY KEY,<br>";
    echo "&nbsp;&nbsp;course_id INT,<br>";
    echo "&nbsp;&nbsp;views INT DEFAULT 0,<br>";
    echo "&nbsp;&nbsp;view_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP<br>";
    echo ");<br><br>";
    
    echo "-- Add course views data<br>";
    echo "INSERT INTO course_views (course_id, views) VALUES ($sample_course, 120);<br><br>";
    
    echo "-- Add video progress for existing students<br>";
    echo "INSERT INTO video_progress (student_id, video_id, watch_time, completed, completion_percentage) <br>";
    echo "SELECT e.student_id, v.video_id, 1800, 1, 100.0 <br>";
    echo "FROM enrollments e, videos v <br>";
    echo "WHERE e.course_id = v.course_id AND v.course_id = $sample_course;<br>";
    echo "</div>";
}
echo "</div>";

echo "<div class='section success'><h2>✅ Debug Complete</h2>";
echo "Review the findings above to understand what data is missing for the analytics page.<br>";
echo "Use the recommendations to populate sample data and fix the analytics display.";
echo "</div>";

$conn->close();
?>
