<?php
// Debug script to analyze database structure and relationships
// This will help us understand the foreign key constraints and fix course creation issues

// Start output buffering to capture all output
ob_start();

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Database connection using the same approach as other files
session_start();
require_once "../config/db_connect.php";

// File for saving debug info
$logFile = __DIR__ . '/database_debug.log';

// Function for pretty printing arrays and objects
function prettyPrint($data) {
    echo "<pre>";
    print_r($data);
    echo "</pre>";
}

// Set headers for readability in browser
header("Content-Type: text/html");
echo "<!DOCTYPE html>
<html>
<head>
    <title>Database Debug</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; }
        h1, h2, h3 { margin-top: 30px; }
        table { border-collapse: collapse; width: 100%; margin-bottom: 20px; }
        th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
        th { background-color: #f2f2f2; }
        tr:nth-child(even) { background-color: #f9f9f9; }
        .error { color: red; font-weight: bold; }
        .success { color: green; }
        pre { background-color: #f5f5f5; padding: 10px; border-radius: 5px; overflow: auto; }
    </style>
</head>
<body>
    <h1>Database Debug Information</h1>";

// Connection status
echo "<h2>Database Connection</h2>";
if ($conn->connect_error) {
    echo "<p class='error'>Connection failed: " . $conn->connect_error . "</p>";
    exit;
} else {
    echo "<p class='success'>Database connected successfully!</p>";
    echo "<p>Server info: " . $conn->server_info . "</p>";
    echo "<p>Host info: " . $conn->host_info . "</p>";
}

// ========== COURSES TABLE ==========
echo "<h2>COURSES Table Structure</h2>";

// Get courses table structure
$coursesColumns = [];
$result = $conn->query("DESCRIBE courses");
if ($result) {
    echo "<table>
        <tr>
            <th>Field</th>
            <th>Type</th>
            <th>Null</th>
            <th>Key</th>
            <th>Default</th>
            <th>Extra</th>
        </tr>";
    
    while ($row = $result->fetch_assoc()) {
        echo "<tr>
            <td>{$row['Field']}</td>
            <td>{$row['Type']}</td>
            <td>{$row['Null']}</td>
            <td>{$row['Key']}</td>
            <td>{$row['Default']}</td>
            <td>{$row['Extra']}</td>
        </tr>";
        $coursesColumns[] = $row['Field'];
    }
    echo "</table>";
} else {
    echo "<p class='error'>Error getting courses table structure: " . $conn->error . "</p>";
}

// ========== CATEGORIES TABLE ==========
echo "<h2>CATEGORIES Table Structure</h2>";

// Get categories table structure
$categoriesColumns = [];
$result = $conn->query("DESCRIBE categories");
if ($result) {
    echo "<table>
        <tr>
            <th>Field</th>
            <th>Type</th>
            <th>Null</th>
            <th>Key</th>
            <th>Default</th>
            <th>Extra</th>
        </tr>";
    
    while ($row = $result->fetch_assoc()) {
        echo "<tr>
            <td>{$row['Field']}</td>
            <td>{$row['Type']}</td>
            <td>{$row['Null']}</td>
            <td>{$row['Key']}</td>
            <td>{$row['Default']}</td>
            <td>{$row['Extra']}</td>
        </tr>";
        $categoriesColumns[] = $row['Field'];
    }
    echo "</table>";
} else {
    echo "<p class='error'>Error getting categories table structure: " . $conn->error . "</p>";
}

// ========== FOREIGN KEY CONSTRAINTS ==========
echo "<h2>Foreign Key Constraints</h2>";

// Get foreign key constraints for courses table
$result = $conn->query("
    SELECT 
        CONSTRAINT_NAME,
        TABLE_NAME,
        COLUMN_NAME,
        REFERENCED_TABLE_NAME,
        REFERENCED_COLUMN_NAME
    FROM
        information_schema.KEY_COLUMN_USAGE
    WHERE
        TABLE_SCHEMA = DATABASE()
        AND TABLE_NAME = 'courses'
        AND REFERENCED_TABLE_NAME IS NOT NULL
");

if ($result) {
    if ($result->num_rows > 0) {
        echo "<table>
            <tr>
                <th>Constraint Name</th>
                <th>Table</th>
                <th>Column</th>
                <th>Referenced Table</th>
                <th>Referenced Column</th>
            </tr>";
        
        while ($row = $result->fetch_assoc()) {
            echo "<tr>
                <td>{$row['CONSTRAINT_NAME']}</td>
                <td>{$row['TABLE_NAME']}</td>
                <td>{$row['COLUMN_NAME']}</td>
                <td>{$row['REFERENCED_TABLE_NAME']}</td>
                <td>{$row['REFERENCED_COLUMN_NAME']}</td>
            </tr>";
        }
        echo "</table>";
    } else {
        echo "<p>No foreign key constraints found for courses table.</p>";
    }
} else {
    echo "<p class='error'>Error getting foreign key constraints: " . $conn->error . "</p>";
}

// ========== CATEGORIES DATA ==========
echo "<h2>CATEGORIES Table Data</h2>";

// Get categories data
$result = $conn->query("SELECT * FROM categories");
if ($result) {
    if ($result->num_rows > 0) {
        echo "<table><tr>";
        foreach ($categoriesColumns as $column) {
            echo "<th>{$column}</th>";
        }
        echo "</tr>";
        
        while ($row = $result->fetch_assoc()) {
            echo "<tr>";
            foreach ($categoriesColumns as $column) {
                echo "<td>" . htmlspecialchars($row[$column] ?? 'NULL') . "</td>";
            }
            echo "</tr>";
        }
        echo "</table>";
    } else {
        echo "<p class='error'>No categories found in the database!</p>";
        echo "<p>This is likely why course creation is failing - there are no valid categories to reference.</p>";
    }
} else {
    echo "<p class='error'>Error getting categories data: " . $conn->error . "</p>";
}

// ========== SAMPLE COURSES DATA ==========
echo "<h2>Sample COURSES Data</h2>";

// Get a few sample courses
$result = $conn->query("SELECT * FROM courses LIMIT 5");
if ($result) {
    if ($result->num_rows > 0) {
        echo "<table><tr>";
        foreach ($coursesColumns as $column) {
            echo "<th>{$column}</th>";
        }
        echo "</tr>";
        
        while ($row = $result->fetch_assoc()) {
            echo "<tr>";
            foreach ($coursesColumns as $column) {
                echo "<td>" . htmlspecialchars($row[$column] ?? 'NULL') . "</td>";
            }
            echo "</tr>";
        }
        echo "</table>";
    } else {
        echo "<p>No courses found in the database.</p>";
    }
} else {
    echo "<p class='error'>Error getting courses data: " . $conn->error . "</p>";
}

// ========== TEST QUERY SIMULATION ==========
echo "<h2>Test Query Simulation</h2>";

// Simulate a course insert with a fixed category_id
echo "<h3>Simulating course insert with category_id = 1</h3>";

// First check if category_id 1 exists
$result = $conn->query("SELECT * FROM categories WHERE category_id = 1");
if ($result) {
    if ($result->num_rows > 0) {
        $categoryData = $result->fetch_assoc();
        echo "<p class='success'>Category with ID 1 exists: ";
        echo json_encode($categoryData);
        echo "</p>";
    } else {
        echo "<p class='error'>Category with ID 1 does NOT exist! This would cause a foreign key constraint failure.</p>";
    }
} else {
    echo "<p class='error'>Error checking category existence: " . $conn->error . "</p>";
}

// Show the query that would be executed
$sampleQuery = "INSERT INTO courses (teacher_id, title, description, category_id, thumbnail, price, is_published) 
                VALUES (6, 'Sample Course', 'This is a test description', 1, '../uploads/thumbnails/sample.jpg', 29.99, 1)";

echo "<h3>Sample Query That Would Run:</h3>";
echo "<pre>" . htmlspecialchars($sampleQuery) . "</pre>";

// Try to execute the query without actually inserting (TEST MODE)
echo "<h3>Testing Query Execution (Dry Run):</h3>";
$conn->begin_transaction();
try {
    // Execute without actually committing
    $testResult = $conn->query($sampleQuery);
    if ($testResult) {
        echo "<p class='success'>Query would execute successfully!</p>";
    } else {
        echo "<p class='error'>Query would fail: " . $conn->error . "</p>";
    }
    // Roll back to prevent actual insertion
    $conn->rollback();
    echo "<p>Transaction rolled back, no data was inserted.</p>";
} catch (Exception $e) {
    $conn->rollback();
    echo "<p class='error'>Exception during query execution: " . $e->getMessage() . "</p>";
}

// ========== DROPDOWN CATEGORY VALUES ==========
echo "<h2>HTML Form Category Values</h2>";

// Extract category options from create-course.php
echo "<p>The category dropdown in the form has these values:</p>";
echo "<table>
    <tr>
        <th>Value (sent to server)</th>
        <th>Display Text</th>
        <th>Corresponding Category ID (if found)</th>
    </tr>
    <tr>
        <td>programming</td>
        <td>Programming</td>
        <td id='cat_programming'>Checking...</td>
    </tr>
    <tr>
        <td>design</td>
        <td>Design</td>
        <td id='cat_design'>Checking...</td>
    </tr>
    <tr>
        <td>business</td>
        <td>Business</td>
        <td id='cat_business'>Checking...</td>
    </tr>
    <tr>
        <td>marketing</td>
        <td>Marketing</td>
        <td id='cat_marketing'>Checking...</td>
    </tr>
    <tr>
        <td>language</td>
        <td>Language</td>
        <td id='cat_language'>Checking...</td>
    </tr>
    <tr>
        <td>other</td>
        <td>Other</td>
        <td id='cat_other'>Checking...</td>
    </tr>
</table>";

// Try to match form values with actual category IDs
$categoryNames = ['programming', 'design', 'business', 'marketing', 'language', 'other'];

// Check what columns we can match against (title, name, slug, etc.)
$potentialCategoryColumns = [];
foreach ($categoriesColumns as $column) {
    // Most likely column names that might contain category names
    if (in_array($column, ['name', 'title', 'category_name', 'slug', 'category'])) {
        $potentialCategoryColumns[] = $column;
    }
}

echo "<script>";
if (!empty($potentialCategoryColumns)) {
    foreach ($categoryNames as $catName) {
        $found = false;
        foreach ($potentialCategoryColumns as $colName) {
            // Try to find categories with matching names
            $stmt = $conn->prepare("SELECT category_id FROM categories WHERE LOWER({$colName}) = LOWER(?)");
            if ($stmt) {
                $stmt->bind_param('s', $catName);
                $stmt->execute();
                $result = $stmt->get_result();
                if ($result && $result->num_rows > 0) {
                    $catRow = $result->fetch_assoc();
                    echo "document.getElementById('cat_{$catName}').innerHTML = '{$catRow['category_id']} (matched on {$colName})';";
                    $found = true;
                    break;
                }
                $stmt->close();
            }
        }
        if (!$found) {
            echo "document.getElementById('cat_{$catName}').innerHTML = 'Not found in database!';";
        }
    }
} else {
    echo "document.querySelectorAll('[id^=\"cat_\"]').forEach(el => el.innerHTML = 'Could not determine category columns to check');";
}
echo "</script>";

// ========== RECOMMENDATIONS ==========
echo "<h2>Recommendations</h2>";
echo "<ul>";
echo "<li>Make sure categories table has at least one record with category_id = 1</li>";
echo "<li>Check that the categories table structure matches what's expected by the foreign key constraint</li>";
echo "<li>Consider updating the form dropdown to use actual category IDs as values instead of strings</li>";
echo "<li>Verify all required fields for the courses table are being provided</li>";
echo "</ul>";

echo "<h2>Debug Log</h2>";
echo "<p>Check course_creation_debug.log for detailed operation logs.</p>";
$latestLogEntries = shell_exec('tail -n 50 course_creation_debug.log');
if ($latestLogEntries) {
    echo "<h3>Latest log entries:</h3>";
    echo "<pre>" . htmlspecialchars($latestLogEntries) . "</pre>";
}

echo "</body>
</html>";

// End output buffering and display the page
ob_end_flush();

// Close the database connection
$conn->close();
?>
