<?php
// Debug Script: Student Enrollment Graph Analysis
// This script explains how the enrollment graph works and shows actual data

require_once __DIR__ . '/../includes/config.php';

// Get teacher ID (you can change this or get from session)
$teacher_id = 4; // Teacher 4 from your analytics

// Get date range (same as analytics page)
$date_range = 'last-30-days';
$end_date = date('Y-m-d');
$start_date = date('Y-m-d', strtotime('-30 days'));

echo "<h1>🔍 Student Enrollment Graph Debug Analysis</h1>";
echo "<p><strong>Teacher ID:</strong> $teacher_id</p>";
echo "<p><strong>Date Range:</strong> $start_date to $end_date ($date_range)</p>";
echo "<hr>";

// 1. First, let's see ALL enrollments for this teacher
echo "<h2>📊 Step 1: ALL Student Enrollments for This Teacher</h2>";

$all_enrollments_query = "SELECT 
    e.enrollment_id,
    e.student_id, 
    u.first_name,
    u.last_name,
    e.course_id, 
    c.title as course_title,
    e.enrolled_date,
    DATE(e.enrolled_date) as enrollment_date_only
FROM enrollments e 
JOIN courses c ON e.course_id = c.course_id 
JOIN users u ON e.student_id = u.id
WHERE c.teacher_id = ? 
ORDER BY e.enrolled_date DESC";

$stmt = $conn->prepare($all_enrollments_query);
$stmt->bind_param("i", $teacher_id);
$stmt->execute();
$result = $stmt->get_result();

echo "<table border='1' cellpadding='10' style='border-collapse: collapse; width: 100%;'>";
echo "<tr style='background-color: #f0f0f0;'>";
echo "<th>Enrollment ID</th><th>Student</th><th>Course</th><th>Full DateTime</th><th>Date Only</th><th>Within Range?</th>";
echo "</tr>";

$total_enrollments = 0;
$enrollments_in_range = 0;

while ($row = $result->fetch_assoc()) {
    $total_enrollments++;
    $within_range = ($row['enrollment_date_only'] >= $start_date && $row['enrollment_date_only'] <= $end_date);
    if ($within_range) $enrollments_in_range++;
    
    $range_color = $within_range ? '#e8f5e8' : '#ffe8e8';
    $range_text = $within_range ? '✅ YES' : '❌ NO';
    
    echo "<tr style='background-color: $range_color;'>";
    echo "<td>{$row['enrollment_id']}</td>";
    echo "<td>{$row['first_name']} {$row['last_name']} (ID: {$row['student_id']})</td>";
    echo "<td>{$row['course_title']} (ID: {$row['course_id']})</td>";
    echo "<td>{$row['enrolled_date']}</td>";
    echo "<td>{$row['enrollment_date_only']}</td>";
    echo "<td><strong>$range_text</strong></td>";
    echo "</tr>";
}
echo "</table>";

echo "<p><strong>📈 Summary:</strong></p>";
echo "<ul>";
echo "<li><strong>Total Enrollments:</strong> $total_enrollments</li>";
echo "<li><strong>Enrollments in Date Range ({$start_date} to {$end_date}):</strong> $enrollments_in_range</li>";
echo "</ul>";

// 2. Now let's see exactly what the graph query returns
echo "<hr>";
echo "<h2>📊 Step 2: Enrollment Graph Query (Same as Analytics Page)</h2>";
echo "<p>This is the EXACT query used in your analytics graph:</p>";

$enrollment_chart_query = "SELECT 
    DATE(e.enrolled_date) as enrollment_date, 
    COUNT(*) as daily_enrollments 
FROM enrollments e 
JOIN courses c ON e.course_id = c.course_id 
WHERE c.teacher_id = ? 
AND DATE(e.enrolled_date) BETWEEN ? AND ?
GROUP BY DATE(e.enrolled_date) 
ORDER BY enrollment_date ASC";

echo "<pre style='background-color: #f5f5f5; padding: 10px; border-radius: 5px;'>";
echo htmlspecialchars($enrollment_chart_query);
echo "</pre>";

$stmt = $conn->prepare($enrollment_chart_query);
$stmt->bind_param("iss", $teacher_id, $start_date, $end_date);
$stmt->execute();
$result = $stmt->get_result();

echo "<table border='1' cellpadding='10' style='border-collapse: collapse; width: 100%;'>";
echo "<tr style='background-color: #f0f0f0;'>";
echo "<th>Date</th><th>Number of Enrollments on That Date</th><th>Chart Display</th>";
echo "</tr>";

$chart_data = [];
$total_chart_enrollments = 0;

while ($row = $result->fetch_assoc()) {
    $chart_data[] = $row;
    $total_chart_enrollments += $row['daily_enrollments'];
    
    $formatted_date = date('M d', strtotime($row['enrollment_date']));
    echo "<tr>";
    echo "<td>{$row['enrollment_date']} ($formatted_date)</td>";
    echo "<td><strong>{$row['daily_enrollments']}</strong></td>";
    echo "<td>📊 Bar/point with height {$row['daily_enrollments']}</td>";
    echo "</tr>";
}

if (empty($chart_data)) {
    echo "<tr style='background-color: #fff3cd;'>";
    echo "<td colspan='3'><strong>⚠️ NO DATA FOUND</strong> - This is why your graph shows empty or default data</td>";
    echo "</tr>";
}

echo "</table>";

echo "<p><strong>📊 Chart Summary:</strong></p>";
echo "<ul>";
echo "<li><strong>Number of data points in graph:</strong> " . count($chart_data) . "</li>";
echo "<li><strong>Total enrollments shown in graph:</strong> $total_chart_enrollments</li>";
echo "</ul>";

// 3. Explain the discrepancy
echo "<hr>";
echo "<h2>🤔 Step 3: Why Your Graph Might Show Unexpected Results</h2>";

if ($total_enrollments > $enrollments_in_range) {
    echo "<div style='background-color: #fff3cd; padding: 15px; border-radius: 5px; margin: 10px 0;'>";
    echo "<h3>⚠️ DATE RANGE FILTER ISSUE DETECTED</h3>";
    echo "<p><strong>Problem:</strong> You have $total_enrollments total enrollments, but only $enrollments_in_range fall within your selected date range ($date_range).</p>";
    echo "<p><strong>Solution:</strong> Change your date range to 'All Time' to see all enrollments, or adjust the date range to include older enrollments.</p>";
    echo "</div>";
}

if (count($chart_data) === 1) {
    echo "<div style='background-color: #d4edda; padding: 15px; border-radius: 5px; margin: 10px 0;'>";
    echo "<h3>✅ SINGLE DATE ENROLLMENT DETECTED</h3>";
    echo "<p><strong>Explanation:</strong> All your students enrolled on the same date ({$chart_data[0]['enrollment_date']}), so the graph shows 1 data point with value {$chart_data[0]['daily_enrollments']}.</p>";
    echo "<p><strong>This is normal:</strong> The graph shows daily enrollment counts, not total students.</p>";
    echo "</div>";
}

if (empty($chart_data)) {
    echo "<div style='background-color: #f8d7da; padding: 15px; border-radius: 5px; margin: 10px 0;'>";
    echo "<h3>❌ NO DATA IN SELECTED RANGE</h3>";
    echo "<p><strong>Problem:</strong> No enrollments found within the selected date range ($start_date to $end_date).</p>";
    echo "<p><strong>Solutions:</strong></p>";
    echo "<ul>";
    echo "<li>Change date range to 'All Time' in analytics</li>";
    echo "<li>Check if enrollments exist in the database</li>";
    echo "<li>Verify teacher ID is correct</li>";
    echo "</ul>";
    echo "</div>";
}

// 4. Show how to interpret the graph correctly
echo "<hr>";
echo "<h2>📚 Step 4: How to Read the Enrollment Graph Correctly</h2>";

echo "<div style='background-color: #e7f3ff; padding: 15px; border-radius: 5px; margin: 10px 0;'>";
echo "<h3>📊 Graph Reading Guide:</h3>";
echo "<ul>";
echo "<li><strong>X-axis (horizontal):</strong> Dates when enrollments happened</li>";
echo "<li><strong>Y-axis (vertical):</strong> Number of students who enrolled on each date</li>";
echo "<li><strong>Each bar/point:</strong> Represents enrollments for ONE specific date</li>";
echo "<li><strong>Graph does NOT show:</strong> Cumulative total enrollments</li>";
echo "<li><strong>Graph DOES show:</strong> Daily enrollment activity over time</li>";
echo "</ul>";

echo "<h3>🎯 Examples:</h3>";
echo "<ul>";
echo "<li><strong>Scenario A:</strong> 5 students all enrolled on Nov 1st → Graph shows 1 bar on Nov 1st with height 5</li>";
echo "<li><strong>Scenario B:</strong> 1 student enrolled each day for 5 days → Graph shows 5 bars, each with height 1</li>";
echo "<li><strong>Your case:</strong> ";

if (count($chart_data) === 1 && !empty($chart_data)) {
    echo "All {$chart_data[0]['daily_enrollments']} students enrolled on {$chart_data[0]['enrollment_date']} → Graph shows 1 data point with value {$chart_data[0]['daily_enrollments']}";
} elseif (count($chart_data) > 1) {
    echo "Students enrolled across " . count($chart_data) . " different dates → Graph shows " . count($chart_data) . " data points";
} else {
    echo "No enrollments in selected date range → Graph shows no data or default sample";
}

echo "</li>";
echo "</ul>";
echo "</div>";

// 5. Quick fixes
echo "<hr>";
echo "<h2>🛠️ Step 5: Quick Fixes to Try</h2>";

echo "<div style='background-color: #f8f9fa; padding: 15px; border-radius: 5px; margin: 10px 0;'>";
echo "<ol>";
echo "<li><strong>Change Date Range:</strong> In your analytics page, try selecting 'All Time' instead of 'Last 30 Days'</li>";
echo "<li><strong>Check Multiple Courses:</strong> If you have multiple courses, try 'All Courses' filter</li>";
echo "<li><strong>Verify Data:</strong> The table above shows your actual enrollment data</li>";
echo "</ol>";
echo "</div>";

// 6. Current analytics URL suggestion
echo "<hr>";
echo "<h2>🔗 Step 6: Try These Analytics URLs</h2>";

$base_url = "analytics.php";
echo "<ul>";
echo "<li><a href='{$base_url}?date_range=all-time' target='_blank'>📊 View All Time Enrollments</a></li>";
echo "<li><a href='{$base_url}?date_range=last-90-days' target='_blank'>📊 View Last 90 Days</a></li>";
echo "<li><a href='{$base_url}?date_range=last-year' target='_blank'>📊 View Last Year</a></li>";
echo "</ul>";

echo "<hr>";
echo "<p><em>🎯 This debug script shows you exactly why your enrollment graph displays the way it does. The graph is working correctly - it shows daily enrollment patterns, not cumulative totals!</em></p>";

?>
