<?php
require_once '../config/db_connect.php';

echo "<h1>🔍 File Upload Debug</h1>";

// Check if this is a file upload request
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['videoFile'])) {
    echo "<h2>📂 Processing File Upload</h2>";
    
    // Debug $_FILES data
    echo "<h3>📋 \$_FILES Data:</h3>";
    echo "<pre>";
    print_r($_FILES);
    echo "</pre>";
    
    // Debug $_POST data  
    echo "<h3>📋 \$_POST Data:</h3>";
    echo "<pre>";
    print_r($_POST);
    echo "</pre>";
    
    $uploadedFile = $_FILES['videoFile'];
    
    echo "<h3>🔍 File Validation:</h3>";
    
    // Check file error
    echo "<p><strong>Upload Error Code:</strong> " . $uploadedFile['error'] . "</p>";
    
    switch ($uploadedFile['error']) {
        case UPLOAD_ERR_OK:
            echo "<p style='color: green;'>✅ No upload errors</p>";
            break;
        case UPLOAD_ERR_INI_SIZE:
            echo "<p style='color: red;'>❌ File exceeds upload_max_filesize</p>";
            break;
        case UPLOAD_ERR_FORM_SIZE:
            echo "<p style='color: red;'>❌ File exceeds MAX_FILE_SIZE</p>";
            break;
        case UPLOAD_ERR_PARTIAL:
            echo "<p style='color: red;'>❌ File was only partially uploaded</p>";
            break;
        case UPLOAD_ERR_NO_FILE:
            echo "<p style='color: red;'>❌ No file was uploaded</p>";
            break;
        case UPLOAD_ERR_NO_TMP_DIR:
            echo "<p style='color: red;'>❌ Missing temporary folder</p>";
            break;
        case UPLOAD_ERR_CANT_WRITE:
            echo "<p style='color: red;'>❌ Failed to write file to disk</p>";
            break;
        case UPLOAD_ERR_EXTENSION:
            echo "<p style='color: red;'>❌ File upload stopped by extension</p>";
            break;
        default:
            echo "<p style='color: red;'>❌ Unknown upload error</p>";
    }
    
    if ($uploadedFile['error'] === UPLOAD_ERR_OK) {
        echo "<p><strong>File Name:</strong> " . $uploadedFile['name'] . "</p>";
        echo "<p><strong>File Type:</strong> " . $uploadedFile['type'] . "</p>";
        echo "<p><strong>File Size:</strong> " . number_format($uploadedFile['size'] / 1024 / 1024, 2) . " MB</p>";
        echo "<p><strong>Temporary Path:</strong> " . $uploadedFile['tmp_name'] . "</p>";
        
        // Check if temp file exists
        if (file_exists($uploadedFile['tmp_name'])) {
            echo "<p style='color: green;'>✅ Temporary file exists</p>";
        } else {
            echo "<p style='color: red;'>❌ Temporary file does not exist</p>";
        }
        
        // Validate file type
        $allowedTypes = ['video/mp4', 'video/webm', 'video/quicktime', 'video/x-msvideo'];
        echo "<p><strong>Allowed Types:</strong> " . implode(', ', $allowedTypes) . "</p>";
        
        if (in_array($uploadedFile['type'], $allowedTypes)) {
            echo "<p style='color: green;'>✅ File type is valid</p>";
        } else {
            echo "<p style='color: red;'>❌ File type '{$uploadedFile['type']}' is not allowed</p>";
        }
        
        // Check upload directories
        echo "<h3>📁 Directory Check:</h3>";
        $uploadDir = __DIR__ . '/../uploads/videos/';
        $thumbnailDir = __DIR__ . '/../uploads/thumbnails/';
        
        echo "<p><strong>Video Upload Dir:</strong> " . $uploadDir . "</p>";
        if (is_dir($uploadDir)) {
            echo "<p style='color: green;'>✅ Video directory exists</p>";
            if (is_writable($uploadDir)) {
                echo "<p style='color: green;'>✅ Video directory is writable</p>";
            } else {
                echo "<p style='color: red;'>❌ Video directory is not writable</p>";
            }
        } else {
            echo "<p style='color: orange;'>⚠️ Video directory does not exist (will be created)</p>";
        }
        
        echo "<p><strong>Thumbnail Upload Dir:</strong> " . $thumbnailDir . "</p>";
        if (is_dir($thumbnailDir)) {
            echo "<p style='color: green;'>✅ Thumbnail directory exists</p>";
            if (is_writable($thumbnailDir)) {
                echo "<p style='color: green;'>✅ Thumbnail directory is writable</p>";
            } else {
                echo "<p style='color: red;'>❌ Thumbnail directory is not writable</p>";
            }
        } else {
            echo "<p style='color: orange;'>⚠️ Thumbnail directory does not exist (will be created)</p>";
        }
        
        // Test file move
        echo "<h3>🧪 Test File Move:</h3>";
        $testDir = __DIR__ . '/../uploads/test/';
        if (!is_dir($testDir)) {
            mkdir($testDir, 0755, true);
        }
        
        $videoExtension = pathinfo($uploadedFile['name'], PATHINFO_EXTENSION);
        $testFilename = 'test_' . time() . '.' . $videoExtension;
        $testPath = $testDir . $testFilename;
        
        if (move_uploaded_file($uploadedFile['tmp_name'], $testPath)) {
            echo "<p style='color: green;'>✅ File move successful - Test file created at: " . $testPath . "</p>";
            echo "<p><strong>Test file size:</strong> " . number_format(filesize($testPath) / 1024 / 1024, 2) . " MB</p>";
            
            // Clean up test file
            if (unlink($testPath)) {
                echo "<p>🗑️ Test file cleaned up</p>";
            }
        } else {
            echo "<p style='color: red;'>❌ File move failed</p>";
        }
    }
    
} else {
    // Show upload form for testing
    echo "<h2>📤 File Upload Test Form</h2>";
    echo "<form method='POST' enctype='multipart/form-data'>";
    echo "<p><strong>Select a video file to test upload:</strong></p>";
    echo "<input type='file' name='videoFile' accept='video/*' required>";
    echo "<br><br>";
    echo "<input type='hidden' name='video_type' value='file'>";
    echo "<input type='text' name='title' placeholder='Test Video Title' required>";
    echo "<br><br>";
    echo "<input type='submit' value='🧪 Test Upload' style='background: #007bff; color: white; padding: 10px 20px; border: none; border-radius: 5px;'>";
    echo "</form>";
}

// PHP configuration check
echo "<h2>⚙️ PHP Configuration:</h2>";
echo "<ul>";
echo "<li><strong>upload_max_filesize:</strong> " . ini_get('upload_max_filesize') . "</li>";
echo "<li><strong>post_max_size:</strong> " . ini_get('post_max_size') . "</li>";
echo "<li><strong>max_execution_time:</strong> " . ini_get('max_execution_time') . "s</li>";
echo "<li><strong>memory_limit:</strong> " . ini_get('memory_limit') . "</li>";
echo "<li><strong>file_uploads:</strong> " . (ini_get('file_uploads') ? 'Enabled' : 'Disabled') . "</li>";
echo "<li><strong>upload_tmp_dir:</strong> " . (ini_get('upload_tmp_dir') ?: 'Default') . "</li>";
echo "</ul>";

echo "<hr>";
echo "<p><strong>Debug completed:</strong> " . date('Y-m-d H:i:s') . "</p>";
?>
