<?php
// Set current page for active menu highlighting
$current_page = 'debug_teacher_data.php';

// Include the centralized teacher layout - this handles session properly
require_once __DIR__ . '/components/teacher_layout.php';

// Enable error reporting during development
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);
?>

<!-- Main Content -->
<div class="main-content">
    <div class="container-fluid">
        <div class="row">
            <div class="col-12">
                <div class="card">
                    <div class="card-header bg-primary text-white">
                        <h2 class="mb-0">🧑‍🏫 Teacher Data Analysis & Debug</h2>
                    </div>
                    <div class="card-body">

<?php
// Now we can access session data properly
if (!isset($_SESSION['user_id'])) {
    echo "<div class='alert alert-danger'>";
    echo "❌ No session found. This should not happen if you're viewing this page!";
    echo "</div>";
} else {
    $user_id = $_SESSION['user_id'];
    
    echo "<div class='alert alert-success'>";
    echo "✅ Session active! User ID: <strong>$user_id</strong>";
    echo "</div>";

    // Get current session data
    echo "<h3>📋 Current Session Data:</h3>";
    echo "<div class='table-responsive'>";
    echo "<table class='table table-bordered mb-4'>";
    echo "<tr class='table-info'><th>Session Key</th><th>Value</th></tr>";
    
    foreach ($_SESSION as $key => $value) {
        if ($key !== 'password') {
            $displayValue = is_array($value) ? json_encode($value) : (string)$value;
            echo "<tr><td><strong>$key</strong></td><td>" . htmlspecialchars($displayValue) . "</td></tr>";
        }
    }
    echo "</table>";
    echo "</div>";

    // Check user in users table
    echo "<h3>👤 User Data in 'users' Table:</h3>";
    $userQuery = "SELECT * FROM users WHERE user_id = ?";
    $stmt = $conn->prepare($userQuery);
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($user = $result->fetch_assoc()) {
        echo "<div class='alert alert-success'>✅ User found in users table</div>";
        echo "<div class='table-responsive'>";
        echo "<table class='table table-bordered mb-4'>";
        echo "<tr class='table-primary'><th>Field</th><th>Value</th></tr>";
        
        foreach ($user as $field => $value) {
            if ($field !== 'password') {
                $displayValue = $value ?: '<em class="text-muted">NULL</em>';
                if ($field === 'user_role') {
                    $roleColor = $value === 'teacher' ? 'success' : 'danger';
                    $displayValue = "<span class='badge bg-$roleColor'>$value</span>";
                }
                echo "<tr><td><strong>$field</strong></td><td>$displayValue</td></tr>";
            }
        }
        echo "</table>";
        echo "</div>";
    } else {
        echo "<div class='alert alert-danger'>❌ User not found in users table!</div>";
    }

    // Check if teacher exists in teachers table
    echo "<h3>🎓 Teacher Data in 'teachers' Table:</h3>";
    $teacherQuery = "SELECT * FROM teachers WHERE user_id = ?";
    $stmt2 = $conn->prepare($teacherQuery);
    $stmt2->bind_param("i", $user_id);
    $stmt2->execute();
    $result2 = $stmt2->get_result();

    if ($teacher = $result2->fetch_assoc()) {
        echo "<div class='alert alert-success'>✅ Teacher record EXISTS in teachers table</div>";
        echo "<div class='table-responsive'>";
        echo "<table class='table table-bordered mb-4'>";
        echo "<tr class='table-success'><th>Field</th><th>Value</th></tr>";
        
        foreach ($teacher as $field => $value) {
            $displayValue = $value ?: '<em class="text-muted">NULL</em>';
            echo "<tr><td><strong>$field</strong></td><td>$displayValue</td></tr>";
        }
        echo "</table>";
        echo "</div>";
    } else {
        echo "<div class='alert alert-warning'>";
        echo "⚠️ Teacher record does NOT exist in teachers table!<br>";
        echo "This explains why your profile shows placeholder data.";
        echo "</div>";
        
        // Show what we need to create
        echo "<h4>🔧 What needs to be created:</h4>";
        echo "<div class='alert alert-info'>";
        echo "We need to insert a record in the 'teachers' table with your user_id ($user_id) and real information.";
        echo "</div>";
    }

    // Check teacher_profiles table
    echo "<h3>📝 Teacher Profile Data in 'teacher_profiles' Table:</h3>";
    $profileQuery = "SELECT * FROM teacher_profiles WHERE teacher_id = ?";
    $stmt3 = $conn->prepare($profileQuery);
    $stmt3->bind_param("i", $user_id);
    $stmt3->execute();
    $result3 = $stmt3->get_result();

    if ($profile = $result3->fetch_assoc()) {
        echo "<div class='alert alert-success'>✅ Teacher profile EXISTS</div>";
        echo "<div class='table-responsive'>";
        echo "<table class='table table-bordered mb-4'>";
        echo "<tr class='table-warning'><th>Field</th><th>Value</th></tr>";
        
        foreach ($profile as $field => $value) {
            $displayValue = $value ?: '<em class="text-muted">NULL</em>';
            echo "<tr><td><strong>$field</strong></td><td>$displayValue</td></tr>";
        }
        echo "</table>";
        echo "</div>";
    } else {
        echo "<div class='alert alert-warning'>⚠️ Teacher profile does NOT exist in teacher_profiles table!</div>";
    }

    // Test the actual query used by profile.php
    echo "<h3>🔍 Profile Query Test (Current Implementation):</h3>";
    
    $currentQuery = "SELECT u.user_id, u.first_name, u.last_name, u.email, u.bio, u.profile_picture,
                           t.website, t.social_linkedin, t.social_twitter, t.social_facebook, t.social_youtube,
                           tp.qualifications, tp.expertise
                    FROM users u 
                    LEFT JOIN teachers t ON u.user_id = t.user_id 
                    LEFT JOIN teacher_profiles tp ON u.user_id = tp.teacher_id 
                    WHERE u.user_id = ? AND u.user_role = 'teacher'";

    $stmt4 = $conn->prepare($currentQuery);
    $stmt4->bind_param("i", $user_id);
    $stmt4->execute();
    $result4 = $stmt4->get_result();

    if ($queryResult = $result4->fetch_assoc()) {
        echo "<div class='alert alert-success'>✅ JOIN Query works - returning data</div>";
        echo "<div class='table-responsive'>";
        echo "<table class='table table-bordered mb-4'>";
        echo "<tr class='table-info'><th>Field</th><th>Value</th><th>Source Table</th></tr>";
        
        $fieldSources = [
            'user_id' => 'users', 'first_name' => 'users', 'last_name' => 'users', 
            'email' => 'users', 'bio' => 'users', 'profile_picture' => 'users',
            'website' => 'teachers', 'social_linkedin' => 'teachers', 'social_twitter' => 'teachers', 
            'social_facebook' => 'teachers', 'social_youtube' => 'teachers',
            'qualifications' => 'teacher_profiles', 'expertise' => 'teacher_profiles'
        ];
        
        foreach ($queryResult as $field => $value) {
            $displayValue = $value ?: '<em class="text-muted">NULL</em>';
            $source = $fieldSources[$field] ?? 'unknown';
            
            $rowClass = '';
            if ($source === 'users') $rowClass = 'table-light';
            elseif ($source === 'teachers') $rowClass = 'table-success';
            elseif ($source === 'teacher_profiles') $rowClass = 'table-warning';
            
            echo "<tr class='$rowClass'><td><strong>$field</strong></td><td>$displayValue</td><td>$source</td></tr>";
        }
        echo "</table>";
        echo "</div>";
    } else {
        echo "<div class='alert alert-danger'>";
        echo "❌ JOIN Query returned NO results!<br>";
        echo "This confirms why your profile shows placeholder data.";
        echo "</div>";
    }

    // Action Plan
    echo "<h3>💡 Action Plan:</h3>";
    echo "<div class='alert alert-primary'>";
    echo "<h5>Based on the analysis above, here's what we need to do:</h5>";
    echo "<ol>";
    
    if (!isset($teacher)) {
        echo "<li><strong>🚨 CRITICAL:</strong> Create a teacher record in the 'teachers' table with your real information</li>";
    }
    
    if (!isset($profile)) {
        echo "<li><strong>⚠️ RECOMMENDED:</strong> Create a teacher profile in 'teacher_profiles' table for additional data</li>";
    }
    
    if (isset($user) && (empty($user['first_name']) || $user['first_name'] === 'Academy')) {
        echo "<li><strong>📝 UPDATE:</strong> Update your real name in the 'users' table</li>";
    }
    
    echo "<li><strong>🔧 FIX:</strong> Update the profile.php page to handle missing teacher data properly</li>";
    echo "<li><strong>✅ TEST:</strong> Verify profile display and update functionality works</li>";
    echo "</ol>";
    
    echo "<div class='mt-3'>";
    echo "<button class='btn btn-success' onclick='createTeacherRecord()'>🛠️ Auto-Fix Teacher Record</button> ";
    echo "<button class='btn btn-primary' onclick='location.reload()'>🔄 Refresh Analysis</button>";
    echo "</div>";
    echo "</div>";
}
?>

<script>
function createTeacherRecord() {
    if (confirm('This will create your teacher record with your current user data. Continue?')) {
        fetch('create_teacher_record.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert('✅ Teacher record created successfully!');
                location.reload();
            } else {
                alert('❌ Error: ' + data.message);
            }
        })
        .catch(error => {
            alert('❌ Network error: ' + error);
        });
    }
}
</script>

                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<?php
// Include the centralized teacher layout end (footer, scripts, etc)
require_once __DIR__ . '/components/teacher_layout_end.php';
?>
