<?php
require_once '../config/db_connect.php';

echo "<h1>🔍 Video Table Debug Report</h1>";

// 1. Check table structure
echo "<h2>📋 Table Structure</h2>";
$result = $conn->query("DESCRIBE videos");
if ($result) {
    echo "<table border='1' cellpadding='5' style='border-collapse: collapse;'>";
    echo "<tr style='background: #f0f0f0;'><th>Field</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th><th>Extra</th></tr>";
    while ($row = $result->fetch_assoc()) {
        echo "<tr>";
        echo "<td><strong>" . $row['Field'] . "</strong></td>";
        echo "<td>" . $row['Type'] . "</td>";
        echo "<td>" . $row['Null'] . "</td>";
        echo "<td>" . $row['Key'] . "</td>";
        echo "<td>" . ($row['Default'] ?? 'NULL') . "</td>";
        echo "<td>" . $row['Extra'] . "</td>";
        echo "</tr>";
    }
    echo "</table>";
} else {
    echo "<p style='color: red;'>Error: " . $conn->error . "</p>";
}

// 2. Check recent video data
echo "<h2>📊 Recent Video Data (Last 10 videos)</h2>";
$query = "SELECT video_id, title, video_type, thumbnail, external_url, platform, video_id_external, created_at 
          FROM videos 
          ORDER BY created_at DESC 
          LIMIT 10";

$result = $conn->query($query);
if ($result && $result->num_rows > 0) {
    echo "<table border='1' cellpadding='5' style='border-collapse: collapse; width: 100%;'>";
    echo "<tr style='background: #f0f0f0;'>
            <th>ID</th>
            <th>Title</th>
            <th>Type</th>
            <th>Thumbnail</th>
            <th>External URL</th>
            <th>Platform</th>
            <th>Video ID</th>
            <th>Created</th>
          </tr>";
    
    while ($video = $result->fetch_assoc()) {
        echo "<tr>";
        echo "<td>" . $video['video_id'] . "</td>";
        echo "<td>" . htmlspecialchars(substr($video['title'], 0, 30)) . "...</td>";
        echo "<td><span style='color: " . ($video['video_type'] == 'url' ? 'blue' : 'green') . ";'>" . $video['video_type'] . "</span></td>";
        
        // Thumbnail analysis
        if ($video['thumbnail']) {
            $isExternal = (strpos($video['thumbnail'], 'http') === 0);
            $color = $isExternal ? 'blue' : 'orange';
            $type = $isExternal ? 'External URL' : 'Local Path';
            echo "<td style='color: $color;' title='$type'>" . htmlspecialchars(substr($video['thumbnail'], 0, 40)) . "...</td>";
        } else {
            echo "<td style='color: red;'>NULL</td>";
        }
        
        echo "<td>" . ($video['external_url'] ? htmlspecialchars(substr($video['external_url'], 0, 30)) . "..." : 'NULL') . "</td>";
        echo "<td>" . ($video['platform'] ?: 'NULL') . "</td>";
        echo "<td>" . ($video['video_id_external'] ?: 'NULL') . "</td>";
        echo "<td>" . $video['created_at'] . "</td>";
        echo "</tr>";
    }
    echo "</table>";
} else {
    echo "<p>No videos found or error: " . $conn->error . "</p>";
}

// 3. Analyze thumbnail status
echo "<h2>🎭 Thumbnail Analysis</h2>";

$stats = [
    'total' => 0,
    'with_thumbnail' => 0,
    'null_thumbnail' => 0,
    'external_thumbnail' => 0,
    'local_thumbnail' => 0,
    'youtube_videos' => 0,
    'youtube_with_thumbnail' => 0
];

$result = $conn->query("SELECT thumbnail, video_type, platform FROM videos");
if ($result) {
    while ($row = $result->fetch_assoc()) {
        $stats['total']++;
        
        if ($row['thumbnail']) {
            $stats['with_thumbnail']++;
            if (strpos($row['thumbnail'], 'http') === 0) {
                $stats['external_thumbnail']++;
            } else {
                $stats['local_thumbnail']++;
            }
        } else {
            $stats['null_thumbnail']++;
        }
        
        if ($row['platform'] == 'youtube') {
            $stats['youtube_videos']++;
            if ($row['thumbnail']) {
                $stats['youtube_with_thumbnail']++;
            }
        }
    }
}

echo "<div style='background: #f9f9f9; padding: 15px; border-radius: 5px;'>";
echo "<h3>Summary:</h3>";
echo "<ul>";
echo "<li><strong>Total videos:</strong> " . $stats['total'] . "</li>";
echo "<li><strong>Videos with thumbnails:</strong> " . $stats['with_thumbnail'] . " (" . round(($stats['with_thumbnail']/$stats['total'])*100, 1) . "%)</li>";
echo "<li><strong>Videos without thumbnails:</strong> <span style='color:red;'>" . $stats['null_thumbnail'] . "</span></li>";
echo "<li><strong>External thumbnails (URLs):</strong> <span style='color:blue;'>" . $stats['external_thumbnail'] . "</span></li>";
echo "<li><strong>Local thumbnails (files):</strong> <span style='color:orange;'>" . $stats['local_thumbnail'] . "</span></li>";
echo "<li><strong>YouTube videos:</strong> " . $stats['youtube_videos'] . "</li>";
echo "<li><strong>YouTube videos with thumbnails:</strong> " . $stats['youtube_with_thumbnail'] . " / " . $stats['youtube_videos'] . "</li>";
echo "</ul>";
echo "</div>";

// 4. Test YouTube thumbnail generation
if ($stats['youtube_videos'] > 0) {
    echo "<h2>🧪 YouTube Thumbnail Test</h2>";
    $result = $conn->query("SELECT video_id, title, video_id_external FROM videos WHERE platform = 'youtube' AND video_id_external IS NOT NULL LIMIT 3");
    
    if ($result && $result->num_rows > 0) {
        echo "<div style='display: flex; flex-wrap: wrap; gap: 20px;'>";
        while ($video = $result->fetch_assoc()) {
            $youtube_id = $video['video_id_external'];
            $thumbnail_url = "https://img.youtube.com/vi/{$youtube_id}/maxresdefault.jpg";
            
            echo "<div style='border: 1px solid #ddd; padding: 10px; border-radius: 5px; max-width: 300px;'>";
            echo "<h4>" . htmlspecialchars($video['title']) . "</h4>";
            echo "<p><strong>Video ID:</strong> " . $youtube_id . "</p>";
            echo "<p><strong>Generated Thumbnail URL:</strong><br><small>" . $thumbnail_url . "</small></p>";
            echo "<img src='" . $thumbnail_url . "' alt='YouTube Thumbnail' style='max-width: 200px; height: auto; border: 1px solid #ccc;' onload=\"this.style.border='2px solid green'\" onerror=\"this.style.border='2px solid red'; this.alt='❌ Failed to load'\">";
            echo "</div>";
        }
        echo "</div>";
    }
}

// 5. Check for missing columns that might be needed
echo "<h2>🔧 Column Check</h2>";
$required_columns = ['video_type', 'external_url', 'platform', 'video_id_external', 'embed_code'];
$existing_columns = [];

$result = $conn->query("DESCRIBE videos");
if ($result) {
    while ($row = $result->fetch_assoc()) {
        $existing_columns[] = $row['Field'];
    }
}

echo "<ul>";
foreach ($required_columns as $col) {
    $exists = in_array($col, $existing_columns);
    $color = $exists ? 'green' : 'red';
    $status = $exists ? '✅ EXISTS' : '❌ MISSING';
    echo "<li style='color: $color;'><strong>$col:</strong> $status</li>";
}
echo "</ul>";

echo "<hr>";
echo "<p><strong>Debug completed:</strong> " . date('Y-m-d H:i:s') . "</p>";
?>
