<?php
/**
 * Delete Teacher Feedback File Endpoint
 * Allows teachers to delete their uploaded feedback files
 */

header('Content-Type: application/json');

// Include authentication and database
require_once __DIR__ . '/components/teacher_layout.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

if (!$input || !isset($input['submission_id']) || !isset($input['file_index'])) {
    echo json_encode(['success' => false, 'message' => 'Invalid request data']);
    exit;
}

$submission_id = intval($input['submission_id']);
$file_index = intval($input['file_index']);

try {
    // Verify teacher has permission to delete this file
    $permission_query = "
        SELECT s.teacher_feedback_files, a.assignment_id, c.teacher_id
        FROM assignment_submissions s
        JOIN assignments a ON s.assignment_id = a.assignment_id
        JOIN courses c ON a.course_id = c.course_id
        WHERE s.submission_id = ? AND c.teacher_id = ?
    ";
    
    $stmt = $conn->prepare($permission_query);
    if (!$stmt) {
        throw new Exception("Database prepare failed: " . $conn->error);
    }
    
    $stmt->bind_param("ii", $submission_id, $_SESSION['user_id']);
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    
    if (!$result) {
        echo json_encode(['success' => false, 'message' => 'Access denied or submission not found']);
        exit;
    }
    
    // Parse existing teacher files
    $teacher_files = [];
    if (!empty($result['teacher_feedback_files'])) {
        $teacher_files = json_decode($result['teacher_feedback_files'], true) ?: [];
    }
    
    if (!isset($teacher_files[$file_index])) {
        echo json_encode(['success' => false, 'message' => 'File not found']);
        exit;
    }
    
    // Get file info for deletion
    $file_to_delete = $teacher_files[$file_index];
    $file_path = __DIR__ . '/../uploads/teacher_feedback/' . $file_to_delete['file_path'];
    
    // Remove file from array
    array_splice($teacher_files, $file_index, 1);
    
    // Update database
    $update_query = "UPDATE assignment_submissions SET teacher_feedback_files = ? WHERE submission_id = ?";
    $stmt = $conn->prepare($update_query);
    if (!$stmt) {
        throw new Exception("Update prepare failed: " . $conn->error);
    }
    
    $new_files_json = empty($teacher_files) ? null : json_encode($teacher_files);
    $stmt->bind_param("si", $new_files_json, $submission_id);
    
    if (!$stmt->execute()) {
        throw new Exception("Database update failed: " . $stmt->error);
    }
    
    // Delete physical file
    if (file_exists($file_path)) {
        if (!unlink($file_path)) {
            // File deletion failed, but database was updated
            error_log("Failed to delete physical file: " . $file_path);
        }
    }
    
    echo json_encode([
        'success' => true, 
        'message' => 'File deleted successfully',
        'remaining_files' => count($teacher_files)
    ]);
    
} catch (Exception $e) {
    error_log("Delete teacher feedback file error: " . $e->getMessage());
    echo json_encode(['success' => false, 'message' => 'Server error: ' . $e->getMessage()]);
}

$conn->close();
?>
