<?php
// Include the centralized teacher layout
require_once __DIR__ . '/components/teacher_layout.php';
require_once '../auth/functions.php';
requireRole('teacher');
require_once '../config/db_connect.php';

$teacher_id = $_SESSION['user_id'];
$quiz_id = intval($_GET['id'] ?? 0);

// Verify quiz ownership
$quiz_stmt = $pdo->prepare("SELECT * FROM quizzes WHERE quiz_id = ? AND teacher_id = ?");
$quiz_stmt->execute([$quiz_id, $teacher_id]);
$quiz = $quiz_stmt->fetch(PDO::FETCH_ASSOC);

if (!$quiz) {
    header('Location: quizzes.php');
    exit;
}

// Get quiz questions
$questions_stmt = $pdo->prepare("
    SELECT q.*, GROUP_CONCAT(qo.option_id, ':', qo.option_text, ':', qo.is_correct ORDER BY qo.option_id SEPARATOR '|') as options
    FROM quiz_questions q
    LEFT JOIN quiz_question_options qo ON q.question_id = qo.question_id
    WHERE q.quiz_id = ?
    GROUP BY q.question_id
    ORDER BY q.question_id
");
$questions_stmt->execute([$quiz_id]);
$questions = $questions_stmt->fetchAll(PDO::FETCH_ASSOC);

// Get teacher courses
$courses_stmt = $pdo->prepare("SELECT course_id, title FROM courses WHERE teacher_id = ?");
$courses_stmt->execute([$teacher_id]);
$courses = $courses_stmt->fetchAll(PDO::FETCH_ASSOC);

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $pdo->beginTransaction();
        
        // Update quiz
        $update_stmt = $pdo->prepare("
            UPDATE quizzes SET 
                course_id = ?, title = ?, description = ?, time_limit = ?, 
                passing_score = ?, ai_grading_enabled = ?, auto_grade_mcq = ?, 
                feedback_enabled = ?, updated_at = NOW()
            WHERE quiz_id = ? AND teacher_id = ?
        ");
        
        $update_stmt->execute([
            $_POST['course_id'], $_POST['quiz_title'], $_POST['quiz_description'],
            $_POST['time_limit'], $_POST['passing_score'], 
            isset($_POST['ai_grading_enabled']) ? 1 : 0,
            isset($_POST['auto_grade_mcq']) ? 1 : 0,
            isset($_POST['feedback_enabled']) ? 1 : 0,
            $quiz_id, $teacher_id
        ]);
        
        // Delete existing questions and options
        $pdo->prepare("DELETE FROM quiz_question_options WHERE question_id IN (SELECT question_id FROM quiz_questions WHERE quiz_id = ?)")->execute([$quiz_id]);
        $pdo->prepare("DELETE FROM quiz_questions WHERE quiz_id = ?")->execute([$quiz_id]);
        
        // Insert new questions (reuse logic from create_quiz.php)
        if (isset($_POST['questions'])) {
            foreach ($_POST['questions'] as $question_data) {
                $question_text = trim($question_data['text']);
                if (empty($question_text)) continue;
                
                $question_stmt = $pdo->prepare("INSERT INTO quiz_questions (quiz_id, question_text, question_type, points, created_at) VALUES (?, ?, ?, ?, NOW())");
                $question_stmt->execute([$quiz_id, $question_text, $question_data['type'], $question_data['points']]);
                $question_id = $pdo->lastInsertId();
                
                if ($question_data['type'] === 'multiple_choice' && isset($question_data['options'])) {
                    foreach ($question_data['options'] as $index => $option_text) {
                        if (empty(trim($option_text))) continue;
                        $is_correct = isset($question_data['correct_option']) && $question_data['correct_option'] == $index ? 1 : 0;
                        $pdo->prepare("INSERT INTO quiz_question_options (question_id, option_text, is_correct, created_at) VALUES (?, ?, ?, NOW())")
                            ->execute([$question_id, trim($option_text), $is_correct]);
                    }
                } elseif ($question_data['type'] === 'true_false') {
                    $correct = $question_data['correct_answer'] ?? 'true';
                    $pdo->prepare("INSERT INTO quiz_question_options (question_id, option_text, is_correct, created_at) VALUES (?, 'True', ?, NOW())")
                        ->execute([$question_id, $correct === 'true' ? 1 : 0]);
                    $pdo->prepare("INSERT INTO quiz_question_options (question_id, option_text, is_correct, created_at) VALUES (?, 'False', ?, NOW())")
                        ->execute([$question_id, $correct === 'false' ? 1 : 0]);
                }
            }
        }
        
        $pdo->commit();
        $success = "Quiz updated successfully!";
        
        // Refresh quiz data
        $quiz_stmt->execute([$quiz_id, $teacher_id]);
        $quiz = $quiz_stmt->fetch(PDO::FETCH_ASSOC);
        $questions_stmt->execute([$quiz_id]);
        $questions = $questions_stmt->fetchAll(PDO::FETCH_ASSOC);
        
    } catch (Exception $e) {
        $pdo->rollBack();
        $error = "Error updating quiz: " . $e->getMessage();
    }
}
?>

<!-- Additional CSS for quiz editing -->
    
    <style>
        /* Use same styles as create_quiz.php */
        :root {
            --panadite-primary: #2563eb;
            --panadite-secondary: #1d4ed8;
            --panadite-light: #dbeafe;
            --panadite-dark: #1e40af;
            --panadite-accent: #f59e0b;
            --text-primary: #1f2937;
            --text-secondary: #6b7280;
            --border-color: #e5e7eb;
            --success-color: #10b981;
            --error-color: #ef4444;
            --bg-light: #f8fafc;
        }

        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(135deg, var(--bg-light) 0%, #e2e8f0 100%);
            color: var(--text-primary);
            min-height: 100vh;
        }

        .page-header {
            background: linear-gradient(135deg, var(--panadite-primary) 0%, var(--panadite-secondary) 100%);
            color: white;
            padding: 2rem 0;
            margin-bottom: 2rem;
            box-shadow: 0 4px 20px rgba(37, 99, 235, 0.15);
        }

        .header-content {
            max-width: 1200px;
            margin: 0 auto;
            padding: 0 1rem;
        }

        .page-title {
            font-size: 2.5rem;
            font-weight: 700;
            margin-bottom: 0.5rem;
        }

        .container {
            max-width: 1000px;
            margin: 0 auto;
            padding: 0 1rem 2rem;
        }

        .form-card {
            background: white;
            border-radius: 20px;
            overflow: hidden;
            box-shadow: 0 6px 25px rgba(0,0,0,0.08);
            margin-bottom: 2rem;
        }

        .form-header {
            padding: 2rem;
            background: linear-gradient(135deg, var(--bg-light) 0%, #e2e8f0 50%);
            border-bottom: 1px solid var(--border-color);
        }

        .form-body {
            padding: 2rem;
        }

        .form-control, .form-select {
            padding: 0.75rem 1rem;
            border: 2px solid var(--border-color);
            border-radius: 8px;
            transition: all 0.3s ease;
        }

        .form-control:focus, .form-select:focus {
            border-color: var(--panadite-primary);
            box-shadow: 0 0 0 3px rgba(37, 99, 235, 0.1);
        }

        .btn {
            padding: 0.75rem 1.5rem;
            border: none;
            border-radius: 8px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
        }

        .btn-success {
            background: var(--success-color);
            color: white;
        }

        .btn-outline {
            background: transparent;
            color: var(--panadite-primary);
            border: 2px solid var(--panadite-primary);
        }

        .question-card {
            background: white;
            border: 2px solid var(--border-color);
            border-radius: 12px;
            padding: 1.5rem;
            margin-bottom: 1rem;
        }

        .alert {
            padding: 1rem 1.5rem;
            border-radius: 8px;
            margin-bottom: 1.5rem;
        }

        .alert-success {
            background: #d1fae5;
            color: #059669;
            border: 1px solid #a7f3d0;
        }

        .alert-danger {
            background: #fee2e2;
            color: #dc2626;
            border: 1px solid #fca5a5;
        }
    </style>

    <div class="page-header">
        <div class="header-content">
            <h1 class="page-title">
                <i class="bx bx-edit"></i>
                Edit Quiz: <?= htmlspecialchars($quiz['title']) ?>
            </h1>
        </div>
    </div>

    <div class="container">
        <?php if (isset($success)): ?>
            <div class="alert alert-success">
                <i class="bx bx-check-circle"></i>
                <?= htmlspecialchars($success) ?>
            </div>
        <?php endif; ?>

        <?php if (isset($error)): ?>
            <div class="alert alert-danger">
                <i class="bx bx-error-circle"></i>
                <?= htmlspecialchars($error) ?>
            </div>
        <?php endif; ?>

        <form method="POST" id="quizForm">
            <div class="form-card">
                <div class="form-header">
                    <h3>Quiz Information</h3>
                </div>
                <div class="form-body">
                    <div class="row">
                        <div class="col-md-8">
                            <div class="mb-3">
                                <label class="form-label">Quiz Title *</label>
                                <input type="text" name="quiz_title" class="form-control" required 
                                       value="<?= htmlspecialchars($quiz['title']) ?>">
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="mb-3">
                                <label class="form-label">Course *</label>
                                <select name="course_id" class="form-select" required>
                                    <?php foreach ($courses as $course): ?>
                                        <option value="<?= $course['course_id'] ?>" 
                                                <?= $course['course_id'] == $quiz['course_id'] ? 'selected' : '' ?>>
                                            <?= htmlspecialchars($course['title']) ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Description</label>
                        <textarea name="quiz_description" class="form-control" rows="3"><?= htmlspecialchars($quiz['description']) ?></textarea>
                    </div>

                    <div class="row">
                        <div class="col-md-4">
                            <label class="form-label">Time Limit (minutes)</label>
                            <input type="number" name="time_limit" class="form-control" 
                                   value="<?= $quiz['time_limit'] ?>" min="1" max="300">
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Passing Score (%)</label>
                            <input type="number" name="passing_score" class="form-control" 
                                   value="<?= $quiz['passing_score'] ?>" min="1" max="100">
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Settings</label>
                            <div class="form-check">
                                <input type="checkbox" name="ai_grading_enabled" 
                                       <?= $quiz['ai_grading_enabled'] ? 'checked' : '' ?>>
                                <label>AI Grading</label>
                            </div>
                            <div class="form-check">
                                <input type="checkbox" name="auto_grade_mcq" 
                                       <?= $quiz['auto_grade_mcq'] ? 'checked' : '' ?>>
                                <label>Auto-grade MCQ</label>
                            </div>
                            <div class="form-check">
                                <input type="checkbox" name="feedback_enabled" 
                                       <?= $quiz['feedback_enabled'] ? 'checked' : '' ?>>
                                <label>Enable Feedback</label>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <div class="form-card">
                <div class="form-header">
                    <h3>Questions</h3>
                    <button type="button" class="btn btn-primary" onclick="addQuestion()">
                        <i class="bx bx-plus"></i> Add Question
                    </button>
                </div>
                <div class="form-body">
                    <div id="questionsContainer">
                        <!-- Existing questions will be loaded here -->
                    </div>
                </div>
            </div>

            <div class="d-flex gap-3">
                <button type="submit" class="btn btn-success">
                    <i class="bx bx-save"></i> Update Quiz
                </button>
                <a href="quizzes.php" class="btn btn-outline">
                    <i class="bx bx-arrow-back"></i> Back to Quizzes
                </a>
            </div>
        </form>
    </div>

    <script>
        let questionCount = 0;
        const existingQuestions = <?= json_encode($questions) ?>;

        // Load existing questions
        document.addEventListener('DOMContentLoaded', function() {
            existingQuestions.forEach(question => {
                addQuestion(question);
            });
            
            if (existingQuestions.length === 0) {
                addQuestion();
            }
        });

        function addQuestion(existing = null) {
            questionCount++;
            let questionHtml = `
                <div class="question-card" id="question-${questionCount}">
                    <div class="d-flex justify-content-between align-items-center mb-3">
                        <h5>Question ${questionCount}</h5>
                        <button type="button" class="btn btn-sm btn-danger" onclick="removeQuestion(${questionCount})">
                            <i class="bx bx-x"></i>
                        </button>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Question Text *</label>
                        <textarea name="questions[${questionCount}][text]" class="form-control" rows="2" required>${existing ? existing.question_text : ''}</textarea>
                    </div>

                    <div class="row">
                        <div class="col-md-8">
                            <label class="form-label">Question Type *</label>
                            <select name="questions[${questionCount}][type]" class="form-select" onchange="updateQuestionType(${questionCount}, this.value)" required>
                                <option value="multiple_choice" ${existing && existing.question_type === 'multiple_choice' ? 'selected' : ''}>Multiple Choice</option>
                                <option value="true_false" ${existing && existing.question_type === 'true_false' ? 'selected' : ''}>True/False</option>
                                <option value="short_answer" ${existing && existing.question_type === 'short_answer' ? 'selected' : ''}>Short Answer</option>
                            </select>
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Points</label>
                            <input type="number" name="questions[${questionCount}][points]" class="form-control" 
                                   value="${existing ? existing.points : 1}" min="1" max="100">
                        </div>
                    </div>

                    <div id="options-${questionCount}" class="mt-3">
                    </div>
                </div>
            `;

            document.getElementById('questionsContainer').insertAdjacentHTML('beforeend', questionHtml);
            
            const questionType = existing ? existing.question_type : 'multiple_choice';
            updateQuestionType(questionCount, questionType, existing);
        }

        function updateQuestionType(questionId, type, existing = null) {
            const optionsContainer = document.getElementById(`options-${questionId}`);
            
            if (type === 'multiple_choice') {
                let options = [];
                let correctOption = 0;
                
                if (existing && existing.options) {
                    const optionsList = existing.options.split('|');
                    options = optionsList.map((opt, index) => {
                        const [id, text, isCorrect] = opt.split(':');
                        if (isCorrect === '1') correctOption = index;
                        return text;
                    });
                }
                
                if (options.length === 0) {
                    options = ['', '', '', ''];
                }
                
                let optionsHtml = '<label class="form-label">Answer Options:</label>';
                options.forEach((option, index) => {
                    optionsHtml += `
                        <div class="d-flex align-items-center gap-2 mb-2">
                            <input type="radio" name="questions[${questionId}][correct_option]" value="${index}" ${index === correctOption ? 'checked' : ''}>
                            <input type="text" name="questions[${questionId}][options][]" class="form-control" 
                                   placeholder="Option ${index + 1}" value="${option}" ${index < 2 ? 'required' : ''}>
                        </div>
                    `;
                });
                
                optionsContainer.innerHTML = optionsHtml;
                
            } else if (type === 'true_false') {
                let correctAnswer = 'true';
                if (existing && existing.options) {
                    const optionsList = existing.options.split('|');
                    const trueOption = optionsList.find(opt => opt.includes('True'));
                    if (trueOption && trueOption.split(':')[2] === '0') {
                        correctAnswer = 'false';
                    }
                }
                
                optionsContainer.innerHTML = `
                    <label class="form-label">Correct Answer:</label>
                    <div class="form-check">
                        <input type="radio" name="questions[${questionId}][correct_answer]" value="true" ${correctAnswer === 'true' ? 'checked' : ''}>
                        <label>True</label>
                    </div>
                    <div class="form-check">
                        <input type="radio" name="questions[${questionId}][correct_answer]" value="false" ${correctAnswer === 'false' ? 'checked' : ''}>
                        <label>False</label>
                    </div>
                `;
            } else {
                optionsContainer.innerHTML = '<p class="text-muted">Students will type their answer in a text field.</p>';
            }
        }

        function removeQuestion(questionId) {
            document.getElementById(`question-${questionId}`).remove();
        }
    </script>

<?php include_once __DIR__ . '/components/teacher_layout_end.php'; ?>
