<?php
// Set current page for active menu highlighting
$current_page = 'enrolled_students.php';

// Include the centralized teacher layout start
require_once __DIR__ . '/components/teacher_layout.php';

// Connect to database is already done in teacher_layout.php

$teacher_id = $_SESSION['user_id'];
$courses = [];
$selected_course_id = isset($_GET['course_id']) ? (int)$_GET['course_id'] : 0;
$enrolled_students = [];

// Get teacher's courses
try {
    $stmt = $conn->prepare("SELECT * FROM courses WHERE teacher_id = ? ORDER BY title ASC");
    $stmt->bind_param("i", $teacher_id);
    $stmt->execute();
    $result = $stmt->get_result();
    while ($row = $result->fetch_assoc()) {
        $courses[] = $row;
    }
} catch (Exception $e) {
    $error = "Database error: " . $e->getMessage();
}

// Get students enrolled in the selected course
if ($selected_course_id > 0) {
    try {
        // First check if enrollments table exists
        $tableCheckQuery = "SHOW TABLES LIKE 'enrollments'";
        $tableCheckResult = $conn->query($tableCheckQuery);
        
        if ($tableCheckResult->num_rows > 0) {
            // Get enrolled students with VIDEO-BASED progress calculation (matching student dashboard logic)
            $stmt = $conn->prepare("
                SELECT 
                    e.enrollment_id,
                    e.student_id,
                    e.user_id,
                    COALESCE(s.first_name, u.first_name) as first_name,
                    COALESCE(s.last_name, u.last_name) as last_name,
                    u.email,
                    u.username,
                    e.enrolled_date,
                    COALESCE(
                        (SELECT CASE 
                            WHEN COUNT(v.video_id) = 0 THEN 0 
                            ELSE ROUND((COUNT(CASE WHEN vp.completed = 1 THEN 1 END) / COUNT(v.video_id)) * 100, 1) 
                        END 
                        FROM videos v 
                        LEFT JOIN video_progress vp ON v.video_id = vp.video_id AND vp.student_id = e.student_id 
                        WHERE v.course_id = e.course_id AND v.is_published = 1), 
                        0
                    ) as video_progress,
                    e.progress as basic_progress,
                    DATEDIFF(NOW(), e.enrolled_date) as days_enrolled
                FROM enrollments e
                LEFT JOIN students s ON e.student_id = s.student_id
                LEFT JOIN users u ON COALESCE(s.user_id, e.user_id) = u.user_id
                WHERE e.course_id = ?
                ORDER BY e.enrolled_date DESC
            ");
            $stmt->bind_param("i", $selected_course_id);
            $stmt->execute();
            $result = $stmt->get_result();
            while ($row = $result->fetch_assoc()) {
                $enrolled_students[] = $row;
            }
        }
    } catch (Exception $e) {
        $error = "Database error: " . $e->getMessage();
    }
}

// Get course details if a course is selected
$course_details = null;
if ($selected_course_id > 0) {
    try {
        $stmt = $conn->prepare("SELECT * FROM courses WHERE course_id = ?");
        $stmt->bind_param("i", $selected_course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($row = $result->fetch_assoc()) {
            $course_details = $row;
        }
    } catch (Exception $e) {
        $error = "Database error: " . $e->getMessage();
    }
}
?>

<!-- Custom styles for this page -->
<style>
        /* Enrolled Students Page Custom Styles */
        body {
            background-color: #f8f9fc;
            font-family: 'Nunito', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif;
        }
        
        /* Main content styling */
        .container-fluid {
            padding: 2rem 1.5rem;
        }
        
        /* Page header */
        .page-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 2rem;
        }
        
        .page-header h1 {
            color: #1f2937;
            font-weight: 700;
            font-size: 2rem;
            margin: 0;
        }
        
        /* Table styling */
        .table-container {
            background: white;
            border-radius: 12px;
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
            overflow: hidden;
            border: 1px solid #f3f4f6;
        }
        
        .table-header {
            background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
            padding: 2rem;
            border-bottom: 1px solid #e2e8f0;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .table-header h5 {
            margin: 0;
            color: #1e293b;
            font-weight: 700;
            font-size: 1.25rem;
        }
        
        .export-btn {
            background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%);
            color: white;
            border: none;
            padding: 0.75rem 1.5rem;
            border-radius: 8px;
            font-size: 0.875rem;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.2s ease;
            box-shadow: 0 2px 4px rgba(59, 130, 246, 0.3);
        }
        
        .export-btn:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 8px rgba(59, 130, 246, 0.4);
        }
        
        .export-btn i {
            margin-right: 0.5rem;
        }
        
        .table {
            margin-bottom: 0;
            font-size: 0.9rem;
        }
        
        .table th {
            background: #f8fafc;
            color: #475569;
            font-weight: 700;
            font-size: 0.8rem;
            text-transform: uppercase;
            letter-spacing: 0.05rem;
            border-top: none;
            padding: 1.5rem 1rem;
            border-bottom: 2px solid #e2e8f0;
        }
        
        .table td {
            padding: 1.5rem 1rem;
            vertical-align: middle;
            border-top: 1px solid #f1f5f9;
            color: #334155;
        }
        
        .table tbody tr {
            transition: all 0.2s ease;
        }
        
        .table tbody tr:hover {
            background-color: #f8fafc;
            transform: translateY(-1px);
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.05);
        }
        
        /* Search input styling */
        .search-input {
            border: 2px solid #e5e7eb;
            border-radius: 8px;
            padding: 0.75rem 1rem;
            font-size: 0.875rem;
            transition: all 0.2s ease;
            width: 100%;
        }
        
        .search-input:focus {
            border-color: #6366f1;
            box-shadow: 0 0 0 3px rgba(99, 102, 241, 0.1);
            outline: none;
        }
        
        /* Empty State Styling */
        .empty-state-card {
            background: white;
            border-radius: 12px;
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
            border: 1px solid #f3f4f6;
            padding: 4rem 2rem;
            text-align: center;
            margin: 2rem 0;
        }
        
        .empty-state-icon {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            background: linear-gradient(135deg, #f3f4f6 0%, #e5e7eb 100%);
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 2rem;
            font-size: 2rem;
            color: #9ca3af;
        }
        
        .empty-state-content h5 {
            color: #374151;
            font-weight: 700;
            font-size: 1.25rem;
            margin-bottom: 1rem;
        }
        
        .empty-state-content p {
            color: #6b7280;
            font-size: 1rem;
            margin: 0;
            max-width: 400px;
            margin-left: auto;
            margin-right: auto;
        }
        
        .dashboard-container {
            padding-top: 1.5rem;
        }
        
        .card {
            box-shadow: 0 0.15rem 1.75rem 0 rgba(58, 59, 69, 0.15);
            border: none;
        }
        
        .card-header {
            background-color: #f8f9fc;
            border-bottom: 1px solid #e3e6f0;
        }
        
        .student-card {
            transition: all 0.3s ease;
        }
        
        .student-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 0.5rem 2rem 0 rgba(58, 59, 69, 0.25);
        }
        
        .student-info {
            display: flex;
            align-items: center;
        }
        
        .student-avatar {
            width: 50px;
            height: 50px;
            border-radius: 50%;
            margin-right: 15px;
            background-color: var(--primary);
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.25rem;
            font-weight: bold;
        }
        
        .progress-bar {
            background-color: var(--primary);
            border-radius: 10px;
        }
        
        /* Course selector and details */
        .course-selector {
            background: white;
            padding: 2rem;
            border-radius: 12px;
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
            margin-bottom: 2rem;
            border: 1px solid #e5e7eb;
        }
        
        .course-selector h5 {
            color: #374151;
            font-weight: 600;
            margin-bottom: 0;
        }
        
        .course-selector .form-control {
            border: 2px solid #e5e7eb;
            border-radius: 8px;
            padding: 0.75rem 1rem;
            font-size: 0.875rem;
            transition: all 0.2s ease;
        }
        
        .course-selector .form-control:focus {
            border-color: #6366f1;
            box-shadow: 0 0 0 3px rgba(99, 102, 241, 0.1);
        }
        

        
        /* Stats */
        .stats-container {
            display: flex;
            gap: 2rem;
            margin-bottom: 2.5rem;
            width: 50%;
        }
        
        .stat-card {
            background: white;
            padding: 2.5rem 2rem;
            border-radius: 12px;
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
            text-align: center;
            flex: 1;
            border: 1px solid #f3f4f6;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }
        
        .stat-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 15px -3px rgba(0, 0, 0, 0.1), 0 4px 6px -2px rgba(0, 0, 0, 0.05);
        }
        
        .stat-card {
            background: white;
            padding: 2rem;
            border-radius: 12px;
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
            text-align: left;
            flex: 1;
            border: 1px solid #f3f4f6;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
            display: flex;
            align-items: center;
            gap: 1rem;
        }
        
        .stat-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 15px -3px rgba(0, 0, 0, 0.1), 0 4px 6px -2px rgba(0, 0, 0, 0.05);
        }
        
        .stat-icon {
            width: 48px;
            height: 48px;
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 20px;
            color: white;
            flex-shrink: 0;
        }
        
        .stat-card:first-child .stat-icon {
            background: #3b82f6;
        }
        
        .stat-card:last-child .stat-icon {
            background: #10b981;
        }
        
        .stat-content {
            display: flex;
            flex-direction: column;
        }
        
        .stat-value {
            font-size: 2rem;
            font-weight: 800;
            color: #1f2937;
            margin-bottom: 0.25rem;
            line-height: 1;
        }
        
        .stat-label {
            color: #6b7280;
            font-size: 0.75rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.05rem;
        }
        
        /* Student Avatar Styling */
        .student-info {
            display: flex;
            align-items: center;
            gap: 12px;
        }
        
        .student-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: bold;
            font-size: 14px;
            color: white;
            background: linear-gradient(135deg, #FF6B6B 0%, #FF8E53 100%);
            flex-shrink: 0;
        }
        
        .student-avatar:nth-child(2n) {
            background: linear-gradient(135deg, #4ECDC4 0%, #44A08D 100%);
        }
        
        .student-avatar:nth-child(3n) {
            background: linear-gradient(135deg, #45B7D1 0%, #96C93D 100%);
        }
        
        .student-avatar:nth-child(4n) {
            background: linear-gradient(135deg, #F093FB 0%, #F5576C 100%);
        }
        
        .student-avatar:nth-child(5n) {
            background: linear-gradient(135deg, #FFC371 0%, #FF5F6D 100%);
        }
        
        .student-avatar:nth-child(6n) {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        }
    </style>

<!-- Begin Page Content -->
<div class="container-xl py-4">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h1 class="h3 mb-0 text-gray-800">Enrolled Students</h1>
            <a href="dashboard.php" class="btn btn-sm btn-primary shadow-sm">
                <i class="fas fa-arrow-left fa-sm text-white-50 me-1"></i> Back to Dashboard
            </a>
        </div>
        
        <!-- Course Selector -->
        <div class="course-selector">
            <div class="row align-items-center">
                <div class="col-md-6">
                    <h5 class="mb-0">Select a Course</h5>
                </div>
                <div class="col-md-6">
                    <form action="enrolled_students.php" method="get">
                        <div class="input-group">
                            <select name="course_id" class="form-control" onchange="this.form.submit()">
                                <option value="0">-- Select a course --</option>
                                <?php foreach ($courses as $course): ?>
                                <option value="<?php echo $course['course_id']; ?>" <?php echo $selected_course_id == $course['course_id'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($course['title']); ?>
                                </option>
                                <?php endforeach; ?>
                            </select>
                            <button class="btn btn-primary" type="submit">
                                <i class="fas fa-filter"></i>
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
        
        <?php if ($selected_course_id > 0 && $course_details): ?>

        
        <!-- Stats -->
        <div class="stats-container">
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="fas fa-users"></i>
                </div>
                <div class="stat-content">
                    <div class="stat-value"><?php echo count($enrolled_students); ?></div>
                    <div class="stat-label">Total Enrollments</div>
                </div>
            </div>
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="fas fa-chart-line"></i>
                </div>
                <div class="stat-content">
                    <?php 
                    $total_completion = 0;
                    foreach ($enrolled_students as $student) {
                        $total_completion += $student['video_progress'] ?? 0;
                    }
                    $avg_completion = count($enrolled_students) > 0 ? round($total_completion / count($enrolled_students)) : 0;
                    ?>
                    <div class="stat-value"><?php echo $avg_completion; ?>%</div>
                    <div class="stat-label">Avg. Completion</div>
                </div>
            </div>
        </div>
        
        <!-- Enrolled Students List -->
        <div class="card mb-4">
            <div class="card-header py-3 d-flex flex-row align-items-center justify-content-between">
                <h6 class="m-0 font-weight-bold text-primary">Enrolled Students</h6>
                <div class="dropdown no-arrow">
                    <button class="btn btn-sm btn-outline-primary" id="export-btn">
                        <i class="fas fa-download fa-sm"></i> Export
                    </button>
                </div>
            </div>
            <div class="card-body">
                <?php if (empty($enrolled_students)): ?>
                <div class="empty-state">
                    <i class="fas fa-user-graduate"></i>
                    <h5>No students enrolled yet</h5>
                    <p class="text-muted">Students who enroll in this course will appear here.</p>
                    <a href="../course.php?id=<?php echo $selected_course_id; ?>" class="btn btn-primary" target="_blank">
                        <i class="fas fa-eye me-1"></i> View Course Page
                    </a>
                </div>
                <?php else: ?>
                <div class="table-responsive">
                    <table class="table table-hover" id="students-table">
                        <thead>
                            <tr>
                                <th>Student</th>
                                <th>Email</th>
                                <th>Enrolled Date</th>
                                <th>Progress</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($enrolled_students as $student): ?>
                            <tr>
                                <td>
                                    <div class="student-info">
                                        <div class="student-avatar">
                                            <?php echo strtoupper(substr($student['first_name'], 0, 1) . substr($student['last_name'], 0, 1)); ?>
                                        </div>
                                        <div>
                                            <div class="fw-bold"><?php echo htmlspecialchars($student['first_name'] . ' ' . $student['last_name']); ?></div>
                                            <small class="text-muted"><?php echo htmlspecialchars($student['username']); ?></small>
                                        </div>
                                    </div>
                                </td>
                                <td><?php echo htmlspecialchars($student['email']); ?></td>
                                <td><?php echo date('M d, Y', strtotime($student['enrolled_date'])); ?></td>
                                <td>
                                    <div class="d-flex align-items-center">
                                        <?php 
                                        $progress = $student['video_progress'] ?? 0;
                                        $progress_color = $progress >= 80 ? 'bg-success' : ($progress >= 50 ? 'bg-warning' : 'bg-primary');
                                        ?>
                                        <div class="progress flex-grow-1" style="height: 8px;">
                                            <div class="progress-bar <?php echo $progress_color; ?>" 
                                                 role="progressbar" 
                                                 style="width: <?php echo $progress; ?>%" 
                                                 aria-valuenow="<?php echo $progress; ?>" 
                                                 aria-valuemin="0" 
                                                 aria-valuemax="100">
                                            </div>
                                        </div>
                                        <span class="ms-2 fw-medium"><?php echo $progress; ?>%</span>
                                    </div>
                                </td>
                                <td>
                                    <button class="btn btn-sm btn-outline-primary" onclick="sendMessage(<?php echo $student['user_id']; ?>)">
                                        <i class="fas fa-envelope"></i>
                                    </button>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <?php endif; ?>
            </div>
        </div>
        <?php elseif ($selected_course_id > 0): ?>
        <div class="alert alert-danger">Course not found or you don't have permission to view this course.</div>
        <?php else: ?>
        <!-- Empty state -->
        <div class="empty-state-card">
            <div class="empty-state-icon">
                <i class="fas fa-graduation-cap"></i>
            </div>
            <div class="empty-state-content">
                <h5>Select a course to view enrolled students</h5>
                <p>Choose a course from the dropdown above to see students enrolled in that course.</p>
            </div>
        </div>
        <?php endif; ?>
     
    </div>
    
<?php
// Include additional scripts needed for this page
?>
<script src="https://cdn.datatables.net/1.13.4/js/jquery.dataTables.min.js"></script>
<script src="https://cdn.datatables.net/1.13.4/js/dataTables.bootstrap5.min.js"></script>
<!-- SheetJS (Excel export) -->
<script src="https://cdnjs.cloudflare.com/ajax/libs/xlsx/0.17.4/xlsx.full.min.js"></script>

<script>
    $(document).ready(function() {
        // Initialize DataTable
        var table = $('#students-table').DataTable({
            responsive: true,
            dom: 'Bfrtip',
            buttons: ['copy', 'csv', 'excel'],
            language: {
                search: "_INPUT_",
                searchPlaceholder: "Search students..."
            }
        });
        
        // Export to Excel functionality
        $('#export-btn').on('click', function() {
            // Create a new workbook
            var wb = XLSX.utils.book_new();
            
            // Get the current data from DataTable
            var data = [];
            
            // Add header row
            var headers = [];
            $('#students-table thead th').each(function() {
                headers.push($(this).text());
            });
            data.push(headers);
            
            // Add data rows
            $('#students-table tbody tr').each(function() {
                var row = [];
                $(this).find('td').each(function() {
                    // Get text content without HTML tags
                    row.push($(this).text().trim());
                });
                data.push(row);
            });
            
            // Create worksheet and add to workbook
            var ws = XLSX.utils.aoa_to_sheet(data);
            XLSX.utils.book_append_sheet(wb, ws, "Enrolled Students");
            
            // Generate Excel file and trigger download
            var today = new Date();
            var filename = "enrolled_students_" + 
                today.getFullYear() + "_" + 
                (today.getMonth()+1) + "_" + 
                today.getDate() + ".xlsx";
            
            XLSX.writeFile(wb, filename);
        });
    });
</script>

<?php
// Include the centralized teacher layout end
require_once __DIR__ . '/components/teacher_layout_end.php';
?>
