<?php
// Enhanced debugging and fixing tool for video upload issues
// This will help identify and resolve issues with videos not appearing for the correct teacher

session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Security check - only teachers can access this
if (!isset($_SESSION['user_id']) || $_SESSION['user_role'] !== 'teacher') {
    header("Location: ../login.php");
    exit();
}

// Database connection
$conn = new mysqli("localhost", "root", "", "panadite_academy");
if ($conn->connect_error) {
    die("Connection failed: " . $conn->connect_error);
}

// Get current user ID from session
$current_user_id = $_SESSION['user_id'];

// Handle fixing actions
$message = '';
if (isset($_GET['action'])) {
    if ($_GET['action'] === 'fix_video' && isset($_GET['video_id'])) {
        $video_id = (int)$_GET['video_id'];
        $stmt = $conn->prepare("UPDATE videos SET teacher_id = ? WHERE video_id = ?");
        $stmt->bind_param("ii", $current_user_id, $video_id);
        
        if ($stmt->execute()) {
            $message = "<div class='alert alert-success'>Video #$video_id has been reassigned to you (Teacher ID: $current_user_id)</div>";
        } else {
            $message = "<div class='alert alert-danger'>Failed to update video: " . $conn->error . "</div>";
        }
    }
    
    // Test insertion action
    if ($_GET['action'] === 'test_insert') {
        $test_title = "Test Video " . date("Y-m-d H:i:s");
        $test_url = "test_url_" . time() . ".mp4";
        
        // Get a valid course ID for this teacher
        $course_result = $conn->query("SELECT course_id FROM courses WHERE teacher_id = $current_user_id LIMIT 1");
        $course_id = 0;
        if ($course_result->num_rows > 0) {
            $course_data = $course_result->fetch_assoc();
            $course_id = $course_data['course_id'];
        }
        
        // Insert test video
        $stmt = $conn->prepare("INSERT INTO videos (title, video_url, course_id, teacher_id, description, duration) VALUES (?, ?, ?, ?, ?, ?)");
        $description = "Test video description";
        $duration = "0:30";
        
        $stmt->bind_param("ssiiss", $test_title, $test_url, $course_id, $current_user_id, $description, $duration);
        
        if ($stmt->execute()) {
            $new_id = $conn->insert_id;
            $message = "<div class='alert alert-success'>Test video inserted successfully with ID: $new_id</div>";
        } else {
            $message = "<div class='alert alert-danger'>Failed to insert test video: " . $conn->error . "</div>";
        }
    }
    
    // Fix all videos
    if ($_GET['action'] === 'fix_all' && isset($_GET['target_id'])) {
        $source_teacher_id = (int)$_GET['target_id'];
        $stmt = $conn->prepare("UPDATE videos SET teacher_id = ? WHERE teacher_id = ?");
        $stmt->bind_param("ii", $current_user_id, $source_teacher_id);
        
        if ($stmt->execute()) {
            $affected = $stmt->affected_rows;
            $message = "<div class='alert alert-success'>$affected videos have been reassigned from Teacher ID $source_teacher_id to you (Teacher ID: $current_user_id)</div>";
        } else {
            $message = "<div class='alert alert-danger'>Failed to update videos: " . $conn->error . "</div>";
        }
    }
    
    // Check database connection
    if ($_GET['action'] === 'test_db') {
        try {
            $test_result = $conn->query("SELECT 1");
            if ($test_result) {
                $message = "<div class='alert alert-success'>Database connection is working correctly</div>";
            } else {
                $message = "<div class='alert alert-danger'>Database query failed: " . $conn->error . "</div>";
            }
        } catch (Exception $e) {
            $message = "<div class='alert alert-danger'>Database exception: " . $e->getMessage() . "</div>";
        }
    }
}

// Check for orphaned videos (videos with no valid teacher)
$orphaned_result = $conn->query("SELECT v.* FROM videos v LEFT JOIN users u ON v.teacher_id = u.user_id WHERE u.user_id IS NULL");
$orphaned_count = $orphaned_result->num_rows;

// Get teacher's own videos
$own_videos_result = $conn->query("SELECT * FROM videos WHERE teacher_id = $current_user_id");
$own_videos_count = $own_videos_result->num_rows;

// Get all videos count
$all_videos_result = $conn->query("SELECT COUNT(*) as total FROM videos");
$all_videos_row = $all_videos_result->fetch_assoc();
$all_videos_count = $all_videos_row['total'];

// Get teachers list for reassignment
$teachers_result = $conn->query("SELECT user_id, first_name, last_name FROM users WHERE user_role = 'teacher'");
$teachers = [];
while ($row = $teachers_result->fetch_assoc()) {
    $teachers[] = $row;
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Fix Video Issues - Panadite Academy</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body class="bg-light">
    <div class="container py-5">
        <div class="card shadow-sm mb-4">
            <div class="card-header bg-primary text-white">
                <h3>Video Database Troubleshooter</h3>
            </div>
            <div class="card-body">
                <?php echo $message; ?>
                
                <h4>Current Status</h4>
                <ul class="list-group mb-4">
                    <li class="list-group-item d-flex justify-content-between align-items-center">
                        Your Teacher ID
                        <span class="badge bg-primary rounded-pill"><?php echo $current_user_id; ?></span>
                    </li>
                    <li class="list-group-item d-flex justify-content-between align-items-center">
                        Your Videos
                        <span class="badge bg-primary rounded-pill"><?php echo $own_videos_count; ?></span>
                    </li>
                    <li class="list-group-item d-flex justify-content-between align-items-center">
                        Total Videos in Database
                        <span class="badge bg-primary rounded-pill"><?php echo $all_videos_count; ?></span>
                    </li>
                    <li class="list-group-item d-flex justify-content-between align-items-center">
                        Orphaned Videos
                        <span class="badge bg-danger rounded-pill"><?php echo $orphaned_count; ?></span>
                    </li>
                </ul>
                
                <h4>Diagnostic Tools</h4>
                <div class="btn-group mb-4">
                    <a href="?action=test_db" class="btn btn-outline-primary">Test Database Connection</a>
                    <a href="?action=test_insert" class="btn btn-outline-success">Test Insert Video</a>
                    <a href="videos.php" class="btn btn-outline-secondary">Go to Video Page</a>
                </div>
                
                <?php if ($all_videos_count > 0): ?>
                    <h4>All Videos in Database</h4>
                    <div class="table-responsive">
                        <table class="table table-striped table-hover">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Title</th>
                                    <th>Course ID</th>
                                    <th>Teacher ID</th>
                                    <th>Created</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php
                                // Get all videos with course info
                                $result = $conn->query("SELECT v.*, c.title as course_title 
                                                        FROM videos v 
                                                        LEFT JOIN courses c ON v.course_id = c.course_id 
                                                        ORDER BY v.created_at DESC");
                                
                                if ($result->num_rows > 0) {
                                    while ($row = $result->fetch_assoc()) {
                                        $is_your_video = ($row['teacher_id'] == $current_user_id);
                                        $row_class = $is_your_video ? 'table-success' : '';
                                        
                                        echo "<tr class='$row_class'>";
                                        echo "<td>{$row['video_id']}</td>";
                                        echo "<td>{$row['title']}</td>";
                                        echo "<td>{$row['course_id']} " . (isset($row['course_title']) ? "({$row['course_title']})" : "No course") . "</td>";
                                        echo "<td><strong>{$row['teacher_id']}</strong></td>";
                                        echo "<td>{$row['created_at']}</td>";
                                        echo "<td>";
                                        if (!$is_your_video) {
                                            echo "<a href='?action=fix_video&video_id={$row['video_id']}' class='btn btn-sm btn-warning'>Assign to Me</a>";
                                        } else {
                                            echo "<span class='badge bg-success'>Your Video</span>";
                                        }
                                        echo "</td>";
                                        echo "</tr>";
                                    }
                                } else {
                                    echo "<tr><td colspan='6'>No videos found in database</td></tr>";
                                }
                                ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
                
                <?php if (count($teachers) > 0): ?>
                    <h4 class="mt-4">Fix Videos by Teacher</h4>
                    <div class="row">
                        <?php foreach ($teachers as $teacher): ?>
                            <div class="col-md-4 mb-3">
                                <div class="card">
                                    <div class="card-body">
                                        <h5 class="card-title"><?php echo $teacher['first_name'] . ' ' . $teacher['last_name']; ?></h5>
                                        <p class="card-text">Teacher ID: <?php echo $teacher['user_id']; ?></p>
                                        <?php if ($teacher['user_id'] != $current_user_id): ?>
                                            <a href="?action=fix_all&target_id=<?php echo $teacher['user_id']; ?>" class="btn btn-sm btn-warning">
                                                Transfer All Videos to Me
                                            </a>
                                        <?php else: ?>
                                            <span class="badge bg-info">This is you</span>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
                
                <h4 class="mt-4">Database Structure Analysis</h4>
                <?php
                // Check videos table structure
                $table_result = $conn->query("DESCRIBE videos");
                if ($table_result) {
                    echo "<div class='table-responsive'>";
                    echo "<table class='table table-sm'>";
                    echo "<thead><tr><th>Field</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th><th>Extra</th></tr></thead>";
                    echo "<tbody>";
                    while ($field = $table_result->fetch_assoc()) {
                        echo "<tr>";
                        echo "<td>{$field['Field']}</td>";
                        echo "<td>{$field['Type']}</td>";
                        echo "<td>{$field['Null']}</td>";
                        echo "<td>{$field['Key']}</td>";
                        echo "<td>{$field['Default']}</td>";
                        echo "<td>{$field['Extra']}</td>";
                        echo "</tr>";
                    }
                    echo "</tbody></table></div>";
                } else {
                    echo "<div class='alert alert-danger'>Could not retrieve table structure: " . $conn->error . "</div>";
                }
                ?>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
