<?php
require_once '../config/db_connect.php';

// Turn on all error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

echo "<h1>Fixing Video Thumbnails</h1>";

// 1. Get all available thumbnail files
$thumbnail_dir = __DIR__ . '/../uploads/thumbnails/';
$available_thumbnails = [];

if (is_dir($thumbnail_dir)) {
    $files = scandir($thumbnail_dir);
    foreach ($files as $file) {
        if ($file != '.' && $file != '..' && is_file($thumbnail_dir . '/' . $file)) {
            $available_thumbnails[] = $file;
        }
    }
    
    echo "<p>Found " . count($available_thumbnails) . " thumbnail files in uploads/thumbnails/ directory</p>";
    echo "<ul>";
    foreach ($available_thumbnails as $thumb) {
        echo "<li>" . htmlspecialchars($thumb) . "</li>";
    }
    echo "</ul>";
} else {
    echo "<p>Error: thumbnails directory not found!</p>";
    exit;
}

// 2. Update all videos to use these thumbnails
$videos_query = "SELECT video_id, title, thumbnail FROM videos";
$videos_result = $conn->query($videos_query);

$updated_count = 0;
$errors = [];

if ($videos_result && $videos_result->num_rows > 0) {
    echo "<h2>Updating Videos</h2>";
    echo "<table border='1' cellpadding='5'>";
    echo "<tr><th>ID</th><th>Title</th><th>Old Thumbnail</th><th>New Thumbnail</th></tr>";
    
    while ($video = $videos_result->fetch_assoc()) {
        $video_id = $video['video_id'];
        $old_thumbnail = $video['thumbnail'];
        
        // Choose thumbnail based on video ID to ensure consistent assignment
        $thumbnail_index = ($video_id - 1) % count($available_thumbnails);
        $new_thumbnail_file = $available_thumbnails[$thumbnail_index];
        $new_thumbnail_path = 'uploads/thumbnails/' . $new_thumbnail_file;
        
        // Update database
        $update_query = "UPDATE videos SET thumbnail = ? WHERE video_id = ?";
        $stmt = $conn->prepare($update_query);
        $stmt->bind_param("si", $new_thumbnail_path, $video_id);
        
        $success = $stmt->execute();
        
        echo "<tr>";
        echo "<td>" . $video_id . "</td>";
        echo "<td>" . htmlspecialchars($video['title']) . "</td>";
        echo "<td>" . htmlspecialchars($old_thumbnail ?: 'NULL') . "</td>";
        echo "<td>";
        if ($success) {
            echo "<span style='color:green'>" . htmlspecialchars($new_thumbnail_path) . "</span>";
            $updated_count++;
        } else {
            echo "<span style='color:red'>ERROR: " . $stmt->error . "</span>";
            $errors[] = "Failed to update video ID {$video_id}: " . $stmt->error;
        }
        echo "</td>";
        echo "</tr>";
    }
    
    echo "</table>";
    
    if ($updated_count > 0) {
        echo "<p style='color:green; font-weight:bold;'>Successfully updated {$updated_count} videos with correct thumbnail paths!</p>";
    }
    
    if (!empty($errors)) {
        echo "<h3>Errors</h3>";
        echo "<ul>";
        foreach ($errors as $error) {
            echo "<li style='color:red'>" . htmlspecialchars($error) . "</li>";
        }
        echo "</ul>";
    }
    
    echo "<p><a href='videos.php' style='display:inline-block; background-color:#4CAF50; color:white; padding:10px 20px; text-decoration:none; border-radius:4px; margin-top:20px;'>Go to Videos Page</a></p>";
} else {
    echo "<p>No videos found in the database.</p>";
}
?>
