// Form validation function
function validateForm() {
    let isValid = true;
    const errors = [];
    
    // Required fields validation
    const requiredFields = [
        { id: 'courseTitle', name: 'Course title' },
        { id: 'courseCategory', name: 'Category' },
        { id: 'courseDescription', name: 'Description' }
    ];
    
    requiredFields.forEach(field => {
        const $field = $('#' + field.id);
        if ($field.val() === '') {
            isValid = false;
            errors.push(`Please enter ${field.name}`);
            $field.addClass('is-invalid');
        } else {
            $field.removeClass('is-invalid');
        }
    });
    
    // Thumbnail validation - skip if base64 data exists
    const hasBase64 = $('#thumbnail_base64').length > 0 && $('#thumbnail_base64').val() !== '';
    const thumbnailFile = $('#courseThumbnail')[0].files[0];
    
    if (!thumbnailFile && !hasBase64) {
        isValid = false;
        errors.push('Course thumbnail is required');
        $('#courseThumbnail').addClass('is-invalid');
    } else if (thumbnailFile) {
        // Check file type
        if (!['image/jpeg', 'image/png', 'image/jpg'].includes(thumbnailFile.type)) {
            isValid = false;
            errors.push('Invalid thumbnail format. Please use JPG, JPEG or PNG.');
            $('#courseThumbnail').addClass('is-invalid');
        } else {
            // Also check file size - limit to 1MB
            const maxSizeBytes = 1 * 1024 * 1024; // 1MB
            if (thumbnailFile.size > maxSizeBytes) {
                isValid = false;
                errors.push('Thumbnail must be smaller than 1MB. Please resize your image.');
                $('#courseThumbnail').addClass('is-invalid');
            } else {
                $('#courseThumbnail').removeClass('is-invalid');
            }
        }
    }
    
    // Price validation
    const price = $('#coursePrice').val();
    if (price && (isNaN(price) || parseFloat(price) < 0)) {
        isValid = false;
        errors.push('Price must be a positive number');
        $('#coursePrice').addClass('is-invalid');
    }
    
    // Duration validation
    const duration = $('#courseDuration').val();
    if (duration && (isNaN(duration) || parseInt(duration) < 1)) {
        isValid = false;
        errors.push('Duration must be a positive number');
        $('#courseDuration').addClass('is-invalid');
    }
    
    return { isValid, errors };
}

// Submit form via AJAX
function submitFormViaAjax(formData) {
    const $form = $('#createCourseForm');
    const $submitBtn = $('#createCourseBtn');
    const $spinner = $('#submitSpinner');
    const $messages = $('#formMessages');
    const $loadingOverlay = $('#loadingOverlay');
    
    $.ajax({
        url: 'process-create-course.php',
        type: 'POST',
        data: formData,
        processData: false,
        contentType: false,
        dataType: 'json',
        success: function(response) {
            if (response.success) {
                // Show success message
                Toastify({
                    text: "Course created successfully!",
                    duration: 4000,
                    close: true,
                    gravity: "top",
                    position: "center",
                    style: {
                        background: "#1cc88a"
                    },
                    stopOnFocus: true
                }).showToast();
                
                // Redirect to courses page after delay
                setTimeout(function() {
                    window.location.href = 'courses.php';
                }, 2000);
            } else {
                // Handle errors
                let errorMessage = "Failed to create course. Please try again.";
                let detailedErrors = [];
                
                if (response.errors && response.errors.length) {
                    detailedErrors = response.errors;
                    errorMessage = response.errors[0];
                }
                
                if (response.debug_info) {
                    console.log('Debug info:', response.debug_info);
                }
                
                // Show error notification
                Toastify({
                    text: errorMessage,
                    duration: 4000,
                    close: true,
                    gravity: "top",
                    position: "center",
                    style: {
                        background: "#dc3545"
                    },
                    stopOnFocus: true
                }).showToast();
                
                // Display errors in form
                let errorHtml = `
                    <div class="alert alert-danger">
                        <i class="fas fa-exclamation-triangle me-2"></i>`;
                
                if (detailedErrors.length > 0) {
                    errorHtml += `<ul class="mb-0">`;
                    detailedErrors.forEach(err => {
                        errorHtml += `<li>${err}</li>`;
                    });
                    errorHtml += `</ul>`;
                } else {
                    errorHtml += errorMessage;
                }
                
                errorHtml += '</div>';
                $messages.html(errorHtml).show();
                
                // Scroll to messages
                $('html, body').animate({
                    scrollTop: $messages.offset().top - 100
                }, 300);
            }
        },
        error: function(xhr, status, error) {
            console.error('AJAX error:', status, error);
            
            let errorMessage = "Server error. Please try again later.";
            if (xhr.status === 400) {
                errorMessage = "Invalid request. Please check form data.";
            }
            
            // Show error notification
            Toastify({
                text: errorMessage,
                duration: 4000,
                close: true,
                gravity: "top",
                position: "center",
                style: {
                    background: "#dc3545"
                },
                stopOnFocus: true
            }).showToast();
            
            // Show error message
            $messages.html(`
                <div class="alert alert-danger">
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    <strong>Error!</strong> ${errorMessage}
                </div>
            `);
        },
        complete: function() {
            // Reset form state
            $submitBtn.prop('disabled', false);
            $spinner.hide();
            $loadingOverlay.addClass('d-none');
            $form.data('submitting', false);
        }
    });
}

// Process thumbnail for form submission
function processThumbnail(callback) {
    const thumbnailFile = $('#courseThumbnail')[0].files[0];
    const $form = $('#createCourseForm');
    const $messages = $('#formMessages');
    
    if (thumbnailFile) {
        console.log('Processing thumbnail: ' + thumbnailFile.name + ' (' + thumbnailFile.size + ' bytes)');
        
        // Show processing message
        $messages.html(`
            <div class="alert alert-info">
                <i class="fas fa-spinner fa-spin"></i> Processing image, please wait...
            </div>
        `);
        
        // Convert to base64
        const reader = new FileReader();
        reader.readAsDataURL(thumbnailFile);
        
        reader.onload = function() {
            // Create hidden input for base64 data
            if ($('#thumbnail_base64').length === 0) {
                $('<input>').attr({
                    type: 'hidden',
                    id: 'thumbnail_base64',
                    name: 'thumbnail_base64',
                    value: reader.result
                }).appendTo($form);
            } else {
                $('#thumbnail_base64').val(reader.result);
            }
            
            // Save filename too
            if ($('#thumbnail_filename').length === 0) {
                $('<input>').attr({
                    type: 'hidden',
                    id: 'thumbnail_filename',
                    name: 'thumbnail_filename',
                    value: thumbnailFile.name
                }).appendTo($form);
            } else {
                $('#thumbnail_filename').val(thumbnailFile.name);
            }
            
            // Continue with form submission
            callback(new FormData($form[0]));
        };
        
        reader.onerror = function(error) {
            console.error('Error reading file:', error);
            $messages.html(`
                <div class="alert alert-danger">
                    <strong>Error!</strong> Could not process the image. Please try another image.
                </div>
            `);
            resetFormSubmission();
        };
    } else {
        // No new file but we might have base64 already
        callback(new FormData($form[0]));
    }
}

// Reset form submission state
function resetFormSubmission() {
    const $form = $('#createCourseForm');
    const $submitBtn = $('#createCourseBtn');
    const $spinner = $('#submitSpinner');
    const $loadingOverlay = $('#loadingOverlay');
    
    $submitBtn.prop('disabled', false);
    $spinner.hide();
    $loadingOverlay.addClass('d-none');
    $form.data('submitting', false);
}

// Document ready function
$(document).ready(function() {
    // Form submission handler
    $('#createCourseForm').on('submit', function(e) {
        e.preventDefault();
        
        // Prevent double submission
        if ($(this).data('submitting')) {
            return false;
        }
        $(this).data('submitting', true);
        
        const $form = $(this);
        const $submitBtn = $('#createCourseBtn');
        const $spinner = $('#submitSpinner');
        const $messages = $('#formMessages');
        const $loadingOverlay = $('#loadingOverlay');
        
        // Show loading state
        $loadingOverlay.removeClass('d-none');
        $submitBtn.prop('disabled', true);
        $spinner.show();
        
        // Clear previous messages
        $messages.empty();
        
        // Validate form
        const validation = validateForm();
        if (!validation.isValid) {
            // Show validation errors
            let errorHtml = `
                <div class="alert alert-danger">
                    <ul class="mb-0">
            `;
            
            validation.errors.forEach(err => {
                errorHtml += `<li>${err}</li>`;
            });
            
            errorHtml += `
                    </ul>
                </div>
            `;
            
            $messages.html(errorHtml);
            
            // Reset form submission state
            resetFormSubmission();
            
            // Scroll to messages
            $('html, body').animate({
                scrollTop: $messages.offset().top - 100
            }, 300);
            
            return false;
        }
        
        // Process thumbnail then submit form
        processThumbnail(function(formData) {
            submitFormViaAjax(formData);
        });
    });
    
    // Create Course button click handler
    $('#createCourseBtn').on('click', function(e) {
        e.preventDefault();
        $('#createCourseForm').submit();
    });
    
    // Thumbnail preview functionality
    $('#courseThumbnail').on('change', function() {
        const file = this.files[0];
        const $thumbnailPreview = $('#thumbnailPreview');
        
        if (file) {
            // Check file type
            if (!['image/jpeg', 'image/png', 'image/jpg'].includes(file.type)) {
                $thumbnailPreview.html(`
                    <div class="mt-2 text-danger">
                        <i class="fas fa-exclamation-circle"></i> 
                        Invalid file type. Please select a JPG, JPEG or PNG file.
                    </div>
                `);
                return;
            }
            
            // Show loading indicator
            $thumbnailPreview.html(`
                <div class="mt-2">
                    <div class="spinner-border spinner-border-sm text-primary" role="status">
                        <span class="visually-hidden">Loading...</span>
                    </div>
                    <span class="ms-2">Loading preview...</span>
                </div>
            `);
            
            // Read and display image
            const reader = new FileReader();
            reader.onload = function(e) {
                $thumbnailPreview.html(`
                    <div class="mt-2">
                        <img src="${e.target.result}" class="img-thumbnail" style="max-width: 200px; max-height: 150px;">
                        <div class="text-muted small mt-1">${file.name} (${(file.size / 1024).toFixed(1)} KB)</div>
                    </div>
                `);
            }
            reader.onerror = function() {
                $thumbnailPreview.html(`
                    <div class="mt-2 text-danger">
                        <i class="fas fa-exclamation-circle"></i> 
                        Error loading preview.
                    </div>
                `);
            }
            reader.readAsDataURL(file);
        } else {
            $thumbnailPreview.empty();
        }
    });
    
    // Form field validation highlighting
    $('.form-control, .form-select').on('input change', function() {
        if ($(this).val() && $(this).val().trim() !== '') {
            $(this).removeClass('is-invalid');
        }
    });
});
