<?php
// Set current page for active menu highlighting
$current_page = 'grades.php';

// Include the centralized teacher layout
require_once __DIR__ . '/components/teacher_layout.php';

// Get assignment filter if provided
$assignment_filter = intval($_GET['assignment_id'] ?? 0);

// Get assignment list for filter dropdown
$assignments_query = "SELECT assignment_id, title FROM assignments WHERE teacher_id = ? ORDER BY created_at DESC";
$stmt = $conn->prepare($assignments_query);
$stmt->bind_param("i", $userId);
$stmt->execute();
$assignments = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

// Build the main query based on filter
if ($assignment_filter > 0) {
    // Show grades for specific assignment
    $grades_query = "SELECT 
        a.assignment_id,
        a.title as assignment_title,
        a.due_date,
        u.user_id,
        u.first_name,
        u.last_name,
        u.email,
        s.grade,
        s.feedback,
        s.submission_date,
        CASE 
            WHEN s.grade IS NOT NULL THEN 'Graded'
            WHEN s.submission_id IS NOT NULL THEN 'Submitted'
            ELSE 'Not Submitted'
        END as status
        FROM assignments a
        JOIN courses c ON a.course_id = c.course_id
        LEFT JOIN assignment_submissions s ON a.assignment_id = s.assignment_id
        LEFT JOIN users u ON s.student_id = u.user_id
        WHERE a.assignment_id = ? AND c.teacher_id = ?
        ORDER BY u.last_name, u.first_name";
    
    $stmt = $conn->prepare($grades_query);
    if (!$stmt) {
        die("SQL Error in specific assignment query: " . $conn->error);
    }
    $stmt->bind_param("ii", $assignment_filter, $userId);
} else {
    // Show overview of all assignments and grades
    $grades_query = "SELECT 
        a.assignment_id,
        a.title as assignment_title,
        a.due_date,
        c.title as course_title,
        COUNT(s.submission_id) as total_submissions,
        COUNT(CASE WHEN s.grade IS NOT NULL THEN 1 END) as graded_submissions,
        AVG(s.grade) as average_grade,
        MIN(s.grade) as min_grade,
        MAX(s.grade) as max_grade
        FROM assignments a
        JOIN courses c ON a.course_id = c.course_id
        LEFT JOIN assignment_submissions s ON a.assignment_id = s.assignment_id
        WHERE c.teacher_id = ?
        GROUP BY a.assignment_id, a.title, a.due_date, c.title
        ORDER BY a.created_at DESC";
    
    $stmt = $conn->prepare($grades_query);
    if (!$stmt) {
        die("SQL Error in overview query: " . $conn->error);
    }
    $stmt->bind_param("i", $userId);
}

$stmt->execute();
$grades_data = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

$page_title = $assignment_filter > 0 ? "Assignment Grades" : "Grades Overview";
?>

<style>
.grades-container {
    background: white;
    border-radius: 12px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    padding: 2rem;
    margin-bottom: 2rem;
}

.filter-section {
    background: #f8f9fa;
    padding: 1.5rem;
    border-radius: 8px;
    margin-bottom: 2rem;
}

.grades-table {
    width: 100%;
    border-collapse: collapse;
    margin-top: 1rem;
}

.grades-table th,
.grades-table td {
    padding: 12px;
    text-align: left;
    border-bottom: 1px solid #dee2e6;
}

.grades-table th {
    background: #f8f9fa;
    font-weight: 600;
    color: #495057;
}

.grades-table tbody tr:hover {
    background: #f8f9fa;
}

.status-badge {
    padding: 4px 12px;
    border-radius: 20px;
    font-size: 0.85rem;
    font-weight: 500;
}

.status-graded { background: #d4edda; color: #155724; }
.status-submitted { background: #fff3cd; color: #856404; }
.status-not-submitted { background: #f8d7da; color: #721c24; }

.grade-display {
    font-weight: 600;
    color: #2563eb;
}

.stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1rem;
    margin-bottom: 2rem;
}

.stat-card {
    background: white;
    padding: 1.5rem;
    border-radius: 8px;
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
    text-align: center;
}

.stat-value {
    font-size: 2rem;
    font-weight: 700;
    color: #2563eb;
    margin-bottom: 0.5rem;
}

.stat-label {
    color: #6b7280;
    font-size: 0.9rem;
}

/* Student Details Styling */
.student-details-row {
    background-color: #f8f9fa;
}

.student-details-cell {
    padding: 1rem !important;
}

.student-details-container h6 {
    margin-bottom: 1rem;
    color: #495057;
    font-weight: 600;
}

.students-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
    gap: 1rem;
}

.student-card {
    background: white;
    border: 1px solid #e9ecef;
    border-radius: 8px;
    padding: 1rem;
    display: flex;
    justify-content: space-between;
    align-items: center;
    transition: box-shadow 0.2s;
}

.student-card:hover {
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
}

.student-info {
    flex: 1;
}

.student-info strong {
    color: #212529;
    font-size: 0.95rem;
}

.student-info small {
    font-size: 0.8rem;
    margin-top: 0.25rem;
}

.student-grade {
    margin: 0 1rem;
}

.student-date {
    text-align: right;
    min-width: 100px;
}
</style>

<!-- Main Content Container -->
<div class="container-fluid mt-4">
    <div class="row">
        <div class="col-12">
            <!-- Page Header -->
            <div class="d-flex justify-content-between align-items-center mb-4">
                <div>
                    <h2 class="mb-1"><?= $page_title ?></h2>
                    <nav aria-label="breadcrumb">
                        <ol class="breadcrumb">
                            <li class="breadcrumb-item"><a href="dashboard.php">Dashboard</a></li>
                            <li class="breadcrumb-item active" aria-current="page">Grades</li>
                        </ol>
                    </nav>
                </div>
                <div>
                    <a href="assignments.php" class="btn btn-outline-primary me-2">
                        <i class="bx bx-task"></i> Assignments
                    </a>
                    <?php if ($assignment_filter > 0): ?>
                        <a href="grades.php" class="btn btn-secondary">
                            <i class="bx bx-arrow-back"></i> All Grades
                        </a>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Filter Section -->
            <div class="filter-section">
                <form method="GET" class="row align-items-end">
                    <div class="col-md-6">
                        <label for="assignment_filter" class="form-label">Filter by Assignment:</label>
                        <select name="assignment_id" id="assignment_filter" class="form-select">
                            <option value="0">All Assignments (Overview)</option>
                            <?php foreach ($assignments as $assignment): ?>
                                <option value="<?= $assignment['assignment_id'] ?>" 
                                        <?= $assignment_filter == $assignment['assignment_id'] ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($assignment['title']) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <button type="submit" class="btn btn-primary">
                            <i class="bx bx-filter"></i> Apply Filter
                        </button>
                    </div>
                </form>
            </div>

            <?php if ($assignment_filter > 0): ?>
                <!-- Individual Assignment Grades -->
                <div class="grades-container">
                    <h3>Student Grades</h3>
                    
                    <?php if (!empty($grades_data)): ?>
                        <table class="grades-table">
                            <thead>
                                <tr>
                                    <th>Student</th>
                                    <th>Email</th>
                                    <th>Submission Date</th>
                                    <th>Grade</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($grades_data as $row): ?>
                                    <?php if ($row['user_id']): // Only show rows with actual students ?>
                                        <tr>
                                            <td>
                                                <strong><?= htmlspecialchars($row['first_name'] . ' ' . $row['last_name']) ?></strong>
                                            </td>
                                            <td><?= htmlspecialchars($row['email']) ?></td>
                                            <td>
                                                <?= $row['submission_date'] ? date('M d, Y g:i A', strtotime($row['submission_date'])) : '-' ?>
                                            </td>
                                            <td>
                                                <?php if ($row['grade'] !== null): ?>
                                                    <span class="grade-display"><?= $row['grade'] ?>%</span>
                                                <?php else: ?>
                                                    <span class="text-muted">-</span>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <span class="status-badge status-<?= strtolower(str_replace(' ', '-', $row['status'])) ?>">
                                                    <?= $row['status'] ?>
                                                </span>
                                            </td>
                                            <td>
                                                <?php if ($row['status'] !== 'Not Submitted'): ?>
                                    <a href="view-submissions.php?id=<?= $row['assignment_id'] ?>" 
                                       class="btn btn-sm btn-outline-primary">
                                        <i class="bx bx-edit"></i> Grade
                                    </a>
                                <?php else: ?>
                                    <span class="text-muted">No submission</span>
                                <?php endif; ?>
                                            </td>
                                        </tr>
                                    <?php endif; ?>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php else: ?>
                        <div class="alert alert-info">
                            <i class="bx bx-info-circle"></i>
                            No student submissions found for this assignment.
                        </div>
                    <?php endif; ?>
                </div>

            <?php else: ?>
                <!-- Assignments Overview -->
                <div class="grades-container">
                    <h3>Assignments Overview</h3>
                    
                    <?php if (!empty($grades_data)): ?>
                        <table class="grades-table">
                            <thead>
                                <tr>
                                    <th>Assignment</th>
                                    <th>Max Points</th>
                                    <th>Submissions</th>
                                    <th>Graded</th>
                                    <th>Average Grade</th>
                                    <th>Grade Range</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($grades_data as $row): ?>
                                    <tr>
                                        <td>
                                            <strong><?= htmlspecialchars($row['assignment_title']) ?></strong>
                                        </td>
                                        <td><?= $row['max_points'] ?? 100 ?> points</td>
                                        <td>
                                            <span class="badge bg-info"><?= $row['total_submissions'] ?></span>
                                        </td>
                                        <td>
                                            <span class="badge bg-success"><?= $row['graded_submissions'] ?></span>
                                        </td>
                                        <td>
                                            <?php if ($row['average_grade'] !== null): ?>
                                                <span class="grade-display"><?= number_format($row['average_grade'], 1) ?>%</span>
                                            <?php else: ?>
                                                <span class="text-muted">-</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if ($row['min_grade'] !== null && $row['max_grade'] !== null): ?>
                                                <?= $row['min_grade'] ?>% - <?= $row['max_grade'] ?>%
                                            <?php else: ?>
                                                <span class="text-muted">-</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <a href="grades.php?assignment_id=<?= $row['assignment_id'] ?>" 
                                               class="btn btn-sm btn-success me-2">
                                                <i class="bx bx-list-ul"></i> View Students
                                            </a>
                                            <a href="view-submissions.php?id=<?= $row['assignment_id'] ?>" 
                                               class="btn btn-sm btn-outline-primary">
                                                <i class="bx bx-edit"></i> Grade
                                            </a>
                                        </td>
                                    </tr>
                                    <?php if ($row['total_submissions'] > 0): ?>
                                        <?php
                                        // Get students for this specific assignment
                                        $student_query = "SELECT 
                                            u.first_name, u.last_name, u.email, 
                                            s.grade, s.submission_date,
                                            CASE 
                                                WHEN s.grade IS NOT NULL THEN 'Graded'
                                                WHEN s.submission_id IS NOT NULL THEN 'Submitted'
                                                ELSE 'Not Submitted'
                                            END as status
                                            FROM assignment_submissions s
                                            JOIN users u ON s.student_id = u.user_id
                                            WHERE s.assignment_id = ?
                                            ORDER BY u.last_name, u.first_name";
                                        
                                        $student_stmt = $conn->prepare($student_query);
                                        $student_stmt->bind_param("i", $row['assignment_id']);
                                        $student_stmt->execute();
                                        $students = $student_stmt->get_result()->fetch_all(MYSQLI_ASSOC);
                                        ?>
                                        
                                        <tr class="student-details-row">
                                            <td colspan="7" class="student-details-cell">
                                                <div class="student-details-container">
                                                    <h6><i class="bx bx-users"></i> Students (<?= count($students) ?>):</h6>
                                                    <div class="students-grid">
                                                        <?php foreach ($students as $student): ?>
                                                            <div class="student-card">
                                                                <div class="student-info">
                                                                    <strong><?= htmlspecialchars($student['first_name'] . ' ' . $student['last_name']) ?></strong>
                                                                    <small class="text-muted d-block"><?= htmlspecialchars($student['email']) ?></small>
                                                                </div>
                                                                <div class="student-grade">
                                                                    <?php if ($student['grade'] !== null): ?>
                                                                        <span class="badge bg-success"><?= $student['grade'] ?>%</span>
                                                                    <?php else: ?>
                                                                        <span class="badge bg-warning">Not Graded</span>
                                                                    <?php endif; ?>
                                                                </div>
                                                                <div class="student-date">
                                                                    <small class="text-muted">
                                                                        <?= $student['submission_date'] ? date('M d, Y', strtotime($student['submission_date'])) : 'No submission' ?>
                                                                    </small>
                                                                </div>
                                                            </div>
                                                        <?php endforeach; ?>
                                                    </div>
                                                </div>
                                            </td>
                                        </tr>
                                    <?php endif; ?>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php else: ?>
                        <div class="alert alert-info">
                            <i class="bx bx-info-circle"></i>
                            No assignments found. <a href="create-assignment.php">Create your first assignment</a>.
                        </div>
                    <?php endif; ?>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<script>
// Auto-submit filter form when assignment is selected
document.getElementById('assignment_filter').addEventListener('change', function() {
    this.form.submit();
});
</script>
