<?php
require_once __DIR__ . '/../auth/functions.php';

// Ensure user is logged in as an admin
requireRole('admin');

$admin_id = $_SESSION['user_id'];

// Check if tables exist first
$tables_exist = true;
$required_tables = ['library_submissions', 'public_library_resources', 'library_categories', 'admin_library_logs'];
foreach ($required_tables as $table) {
    $result = $conn->query("SHOW TABLES LIKE '$table'");
    if (!$result || $result->num_rows == 0) {
        $tables_exist = false;
        break;
    }
}

// Handle approval/rejection actions (only if tables exist)
if ($_POST['action'] ?? false) {
    if (!$tables_exist) {
        $error_message = "Database tables not found. Please run the setup script first.";
    } else {
        $submission_id = (int)$_POST['submission_id'];
        $action = $_POST['action'];
        $notes = trim($_POST['admin_notes'] ?? '');
        
        if (in_array($action, ['approve', 'reject'])) {
            $status = $action === 'approve' ? 'approved' : 'rejected';
            
            // Update submission status
            $stmt = $conn->prepare("UPDATE library_submissions SET status = ?, admin_id = ?, admin_notes = ?, review_date = NOW() WHERE submission_id = ?");
            if ($stmt) {
                $stmt->bind_param("sisi", $status, $admin_id, $notes, $submission_id);
                $stmt->execute();
                $stmt->close();
                
                // If approved, copy to public library
                if ($action === 'approve') {
                    $stmt = $conn->prepare("
                        INSERT INTO public_library_resources (submission_id, teacher_id, title, description, category, tags, file_path, file_name, file_type, file_size)
                        SELECT submission_id, teacher_id, title, description, category, tags, file_path, file_name, file_type, file_size
                        FROM library_submissions WHERE submission_id = ?
                    ");
                    if ($stmt) {
                        $stmt->bind_param("i", $submission_id);
                        $stmt->execute();
                        $stmt->close();
                    }
                }
                
                // Log admin action
                $stmt = $conn->prepare("INSERT INTO admin_library_logs (admin_id, submission_id, action, notes) VALUES (?, ?, ?, ?)");
                if ($stmt) {
                    $stmt->bind_param("iiss", $admin_id, $submission_id, $action === 'approve' ? 'approved' : 'rejected', $notes);
                    $stmt->execute();
                    $stmt->close();
                }
                
                $message = $action === 'approve' ? 'Document approved and published!' : 'Document rejected.';
            } else {
                $error_message = "Database error: " . $conn->error;
            }
        }
    }
}

// Get pending submissions (only if tables exist)
$pending_submissions = [];
if ($tables_exist) {
    $stmt = $conn->prepare("
        SELECT ls.*, u.username as teacher_name, c.category_name
        FROM library_submissions ls
        JOIN users u ON ls.teacher_id = u.id
        LEFT JOIN categories c ON ls.category_id = c.category_id
        WHERE ls.status = 'pending'
        ORDER BY ls.priority DESC, ls.submission_date ASC
    ");
    if ($stmt) {
        $stmt->execute();
        $pending_submissions = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
        $stmt->close();
    }
}

// Get statistics (only if tables exist)
$stats = ['total' => 0, 'pending' => 0, 'approved' => 0, 'rejected' => 0];
if ($tables_exist) {
    $result = $conn->query("SELECT 
        COUNT(*) as total,
        SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
        SUM(CASE WHEN status = 'approved' THEN 1 ELSE 0 END) as approved,
        SUM(CASE WHEN status = 'rejected' THEN 1 ELSE 0 END) as rejected
        FROM library_submissions
    ");
    if ($result) {
        $stats = $result->fetch_assoc();
    }
}
$page_title = 'Library Document Approvals';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard - Panadite Academy</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="../assets/css/admin-dashboard.css">
    <link rel="stylesheet" href="../assets/css/user-cards.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        .admin-header { 
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); 
            color: white; 
            padding: 2rem; 
            border-radius: 16px; 
            margin-bottom: 2rem; 
        }
        .submission-card { 
            background: white; 
            border-radius: 12px; 
            padding: 1.5rem; 
            margin-bottom: 1rem; 
            box-shadow: 0 4px 12px rgba(0,0,0,0.1); 
            border-left: 4px solid #e5e7eb; 
        }
        .submission-card.priority-high { border-left-color: #dc2626; }
        .submission-card.priority-medium { border-left-color: #f59e0b; }
        .submission-card.priority-low { border-left-color: #10b981; }
        .status-badge { 
            padding: 0.25rem 0.75rem; 
            border-radius: 20px; 
            font-size: 0.75rem; 
            font-weight: 600; 
        }
        .status-pending { background: #fef3c7; color: #92400e; }
        .btn-approve { background: #10b981; border: none; color: white; }
        .btn-reject { background: #ef4444; border: none; color: white; }
        .stat-card { 
            background: white; 
            padding: 1.5rem; 
            border-radius: 12px; 
            text-align: center; 
            box-shadow: 0 4px 12px rgba(0,0,0,0.05); 
        }
        .stat-number { font-size: 2rem; font-weight: 700; }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <?php include 'components/sidebar.php'; ?>
    
    <!-- Main Content -->
    <section class="main-content">
        <!-- Navbar -->
        <?php include 'components/navbar.php'; ?>
        
        <!-- Dashboard Content -->
        <div class="dashboard-content">
            <!-- Page Header -->
            <div class="page-header">
                <h1>Library Document Approvals</h1>
                <div>
                    <span class="page-subtitle">Review and approve teacher-submitted educational resources</span>
                </div>
            </div>

            <!-- Admin Header Section -->
            <div class="admin-header">
                <h1><i class="bx bx-shield-check"></i> Library Document Approvals</h1>
                <p class="mb-0">Review and approve teacher-submitted educational resources</p>
            </div>

    <!-- Error Message -->
    <?php if (isset($error_message)): ?>
        <div class="alert alert-danger alert-dismissible fade show">
            <i class="bx bx-error-circle"></i> <?= htmlspecialchars($error_message) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    <?php endif; ?>

    <!-- Success Message -->
    <?php if (isset($message)): ?>
        <div class="alert alert-success alert-dismissible fade show">
            <i class="bx bx-check-circle"></i> <?= htmlspecialchars($message) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    <?php endif; ?>

    <!-- Setup Required Message -->
    <?php if (!$tables_exist): ?>
        <div class="alert alert-warning">
            <h5><i class="bx bx-error-circle"></i> Setup Required</h5>
            <p class="mb-2">The library system database tables have not been created yet. Please run the setup script to initialize the system.</p>
            <a href="../setup_library_tables.php" class="btn btn-warning" target="_blank">
                <i class="bx bx-cog"></i> Run Setup Script
            </a>
        </div>
    <?php endif; ?>

    <!-- Statistics -->
    <div class="row mb-4">
        <div class="col-md-3">
            <div class="stat-card">
                <div class="stat-number text-primary"><?= $stats['total'] ?></div>
                <div class="text-muted">Total Submissions</div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="stat-card">
                <div class="stat-number text-warning"><?= $stats['pending'] ?></div>
                <div class="text-muted">Pending Review</div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="stat-card">
                <div class="stat-number text-success"><?= $stats['approved'] ?></div>
                <div class="text-muted">Approved</div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="stat-card">
                <div class="stat-number text-danger"><?= $stats['rejected'] ?></div>
                <div class="text-muted">Rejected</div>
            </div>
        </div>
    </div>

    <!-- Pending Submissions -->
    <div class="row">
        <div class="col-12">
            <h3 class="mb-3"><i class="bx bx-time-five text-warning"></i> Pending Approvals (<?= count($pending_submissions) ?>)</h3>
            
            <?php if (empty($pending_submissions)): ?>
                <div class="text-center py-5">
                    <i class="bx bx-check-circle text-success" style="font-size: 4rem;"></i>
                    <h5 class="text-muted mt-3">All caught up!</h5>
                    <p class="text-muted">No pending submissions to review.</p>
                </div>
            <?php else: ?>
                <?php foreach ($pending_submissions as $submission): ?>
                    <div class="submission-card priority-<?= $submission['priority'] ?>">
                        <div class="row">
                            <div class="col-md-8">
                                <div class="d-flex align-items-start mb-2">
                                    <div class="me-3">
                                        <div class="p-2 rounded" style="background-color: <?= $submission['color_code'] ?? '#6b7280' ?>20;">
                                            <i class="<?= $submission['icon_class'] ?? 'bx-file' ?>" style="font-size: 1.5rem; color: <?= $submission['color_code'] ?? '#6b7280' ?>;"></i>
                                        </div>
                                    </div>
                                    <div class="flex-grow-1">
                                        <h5 class="mb-1"><?= htmlspecialchars($submission['title']) ?></h5>
                                        <p class="text-muted mb-2"><?= htmlspecialchars($submission['description']) ?></p>
                                        <div class="small text-muted">
                                            <span class="me-3"><i class="bx bx-user"></i> <?= htmlspecialchars($submission['teacher_name']) ?></span>
                                            <span class="me-3"><i class="bx bx-calendar"></i> <?= date('M j, Y', strtotime($submission['submission_date'])) ?></span>
                                            <span class="me-3"><i class="bx bx-category"></i> <?= htmlspecialchars($submission['category']) ?></span>
                                            <span class="me-3"><i class="bx bx-file"></i> <?= strtoupper($submission['file_type']) ?></span>
                                            <span><i class="bx bx-hdd"></i> <?= number_format($submission['file_size'] / 1024, 1) ?> KB</span>
                                        </div>
                                        <?php if (!empty($submission['tags'])): ?>
                                            <div class="mt-2">
                                                <?php foreach (explode(',', $submission['tags']) as $tag): ?>
                                                    <span class="badge bg-light text-dark me-1"><?= htmlspecialchars(trim($tag)) ?></span>
                                                <?php endforeach; ?>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="text-end">
                                    <span class="status-badge status-pending mb-2 d-inline-block">Pending Review</span>
                                    <div class="mb-2">
                                        <span class="badge bg-<?= $submission['priority'] === 'high' ? 'danger' : ($submission['priority'] === 'medium' ? 'warning' : 'info') ?>">
                                            <?= ucfirst($submission['priority']) ?> Priority
                                        </span>
                                    </div>
                                    <div class="btn-group d-block">
                                        <button class="btn btn-approve btn-sm me-2" onclick="showApprovalModal(<?= $submission['submission_id'] ?>, 'approve', '<?= htmlspecialchars($submission['title']) ?>')">
                                            <i class="bx bx-check"></i> Approve
                                        </button>
                                        <button class="btn btn-reject btn-sm" onclick="showApprovalModal(<?= $submission['submission_id'] ?>, 'reject', '<?= htmlspecialchars($submission['title']) ?>')">
                                            <i class="bx bx-x"></i> Reject
                                        </button>
                                    </div>
                                    <div class="mt-2">
                                        <a href="../uploads/library_submissions/<?= basename($submission['file_path']) ?>" class="btn btn-outline-primary btn-sm" target="_blank">
                                            <i class="bx bx-download"></i> Preview
                                        </a>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>
    </div>

<!-- Approval Modal -->
<div class="modal fade" id="approvalModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST">
                <div class="modal-header">
                    <h5 class="modal-title" id="modalTitle">Confirm Action</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <p id="modalMessage"></p>
                    <div class="mb-3">
                        <label class="form-label">Admin Notes (Optional)</label>
                        <textarea name="admin_notes" class="form-control" rows="3" placeholder="Add notes for the teacher..."></textarea>
                    </div>
                    <input type="hidden" name="submission_id" id="submissionId">
                    <input type="hidden" name="action" id="actionType">
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn" id="confirmBtn">Confirm</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
<script>
function showApprovalModal(submissionId, action, title) {
    document.getElementById('submissionId').value = submissionId;
    document.getElementById('actionType').value = action;
    
    const modal = document.getElementById('approvalModal');
    const modalTitle = document.getElementById('modalTitle');
    const modalMessage = document.getElementById('modalMessage');
    const confirmBtn = document.getElementById('confirmBtn');
    
    if (action === 'approve') {
        modalTitle.textContent = 'Approve Document';
        modalMessage.textContent = `Are you sure you want to approve "${title}"? This will make it publicly available in the library.`;
        confirmBtn.className = 'btn btn-success';
        confirmBtn.innerHTML = '<i class="bx bx-check"></i> Approve';
    } else {
        modalTitle.textContent = 'Reject Document';
        modalMessage.textContent = `Are you sure you want to reject "${title}"? The teacher will be notified.`;
        confirmBtn.className = 'btn btn-danger';
        confirmBtn.innerHTML = '<i class="bx bx-x"></i> Reject';
    }
    
    new bootstrap.Modal(modal).show();
}
</script>

        </div>
    </section>

    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>

</body>
</html>
