<?php
// Include the centralized session handling - this MUST be first
require_once __DIR__ . '/../includes/session_start.php';

// Include database connection
require_once __DIR__ . '/../includes/db_connection.php';
require_once __DIR__ . '/../includes/site_config.php';

// Include fast email service for library submission notifications
require_once __DIR__ . '/../config/fast_email_service.php';

// Bridge database connections - Email service needs PDO, but teacher uses MySQLi
if (!isset($pdo)) {
    try {
        // Create PDO connection using the same credentials as MySQLi
        $pdo = new PDO("mysql:host=" . DB_SERVER . ";dbname=" . DB_NAME, DB_USERNAME, DB_PASSWORD);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
        error_log("PDO connection created successfully for email service");
    } catch (Exception $e) {
        error_log("Failed to create PDO connection for email service: " . $e->getMessage());
        $pdo = null;
    }
}

// Check if user is logged in as teacher
if (!isset($_SESSION['user_role']) || $_SESSION['user_role'] !== 'teacher') {
    header("Location: ../auth/login.php?redirect=" . urlencode($_SERVER['REQUEST_URI']));
    exit;
}

$teacher_id = $_SESSION['user_id'];
$teacher_name = $_SESSION['user_name'] ?? 'Teacher';

/**
 * Send library submission notification to teacher
 */
function sendLibrarySubmissionEmail($submission_id) {
    global $conn;
    
    try {
        // Get submission and teacher details
        $stmt = $conn->prepare("
            SELECT ls.title, ls.description, ls.file_name, ls.file_type, ls.file_size, ls.submission_date, ls.teacher_id,
                   u.first_name, u.last_name, u.email, u.user_id,
                   cat.category_name
            FROM library_submissions ls 
            JOIN users u ON ls.teacher_id = u.user_id 
            LEFT JOIN categories cat ON ls.category_id = cat.category_id 
            WHERE ls.submission_id = ?
        ");
        $stmt->bind_param("i", $submission_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $submission = $result->fetch_assoc();
        
        if (!$submission) {
            error_log("Library submission not found for email: $submission_id");
            return false;
        }
        
        $teacher_name = $submission['first_name'] . ' ' . $submission['last_name'];
        $teacher_email = $submission['email'];
        $document_title = $submission['title'];
        $category = $submission['category_name'] ?? 'General';
        $filename = $submission['file_name'];
        $file_size = round($submission['file_size'] / 1024, 1); // Convert to KB
        $submitted_date = date('M j, Y', strtotime($submission['submission_date']));
        
        // Create email content
        $subject = "📚 Document Submitted: \"$document_title\" - Awaiting Approval";
        $message = "
        <div style='max-width:600px; margin:0 auto; font-family:Arial,sans-serif;'>
            <div style='background:#667eea; color:white; padding:20px; text-align:center;'>
                <h1>📚 Document Submitted!</h1>
            </div>
            <div style='padding:20px; background:white;'>
                <h2>Hi " . htmlspecialchars($teacher_name) . "! 🎉</h2>
                <p>Your document <strong>\"" . htmlspecialchars($document_title) . "\"</strong> has been submitted successfully!</p>
                <div style='background:#f8f9fa; padding:15px; border-radius:5px; margin:20px 0;'>
                    <h3>Document Details:</h3>
                    <p><strong>Category:</strong> " . htmlspecialchars($category) . "</p>
                    <p><strong>File Size:</strong> " . $file_size . " KB</p>
                    <p><strong>Submitted:</strong> " . $submitted_date . "</p>
                </div>
                <p style='text-align:center;'>
                    <a href='https://panaditeacademy.com/teacher/library_submissions.php' 
                       style='background:#667eea; color:white; text-decoration:none; padding:15px 25px; border-radius:8px; font-weight:bold; display:inline-block;'>
                        View Submissions
                    </a>
                </p>
            </div>
        </div>";
        
        // Try direct email sending
        if (function_exists('sendEmailNow')) {
            $result = sendEmailNow($teacher_email, $subject, $message);
            if ($result) {
                return true;
            }
        }
        
        // Fallback to native PHP mail
        $headers = "MIME-Version: 1.0" . "\r\n";
        $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
        $headers .= "From: noreply@panaditeacademy.com" . "\r\n";
        
        return mail($teacher_email, $subject, $message, $headers);
        
    } catch (Exception $e) {
        error_log("Error sending library submission email: " . $e->getMessage());
        return false;
    }
}

// Handle deletion request
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_submission'])) {
    $submission_id = (int)$_POST['submission_id'];
    
    // Get submission details for file deletion
    $stmt = $conn->prepare("SELECT file_path, thumbnail_path FROM library_submissions WHERE id = ? AND teacher_id = ?");
    $stmt->bind_param("ii", $submission_id, $teacher_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($submission = $result->fetch_assoc()) {
        // Delete files from server
        if (!empty($submission['file_path']) && file_exists($submission['file_path'])) {
            unlink($submission['file_path']);
        }
        if (!empty($submission['thumbnail_path']) && file_exists($submission['thumbnail_path'])) {
            unlink($submission['thumbnail_path']);
        }
        
        // Delete from database
        $delete_stmt = $conn->prepare("DELETE FROM library_submissions WHERE id = ? AND teacher_id = ?");
        $delete_stmt->bind_param("ii", $submission_id, $teacher_id);
        
        if ($delete_stmt->execute()) {
            $success_message = "Submission deleted successfully!";
        } else {
            $errors[] = "Failed to delete submission: " . $delete_stmt->error;
        }
        $delete_stmt->close();
    } else {
        $errors[] = "Submission not found or you don't have permission to delete it.";
    }
    $stmt->close();
}

// Handle form submission for new library document
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['submit_document'])) {
    // Debug: Log what we received
    error_log("=== FORM SUBMISSION DEBUG ===");
    error_log("POST data: " . print_r($_POST, true));
    error_log("FILES data: " . print_r($_FILES, true));
    
    $title = trim($_POST['title']);
    $description = trim($_POST['description']);
    $category_id = (int)$_POST['category'];
    $custom_category = trim($_POST['custom_category'] ?? '');
    $tags = trim($_POST['tags']);
    $priority = $_POST['priority'] ?? 'medium';
    
    // Handle thumbnail upload (optional)
    $thumbnail_path = null;
    $thumbnail_error = null;
    
    // Validate inputs
    $errors = [];
    if (empty($title)) $errors[] = "Title is required";
    if (empty($description)) $errors[] = "Description is required";
    
    // Handle custom category creation
    if ($category_id === 0 && !empty($custom_category)) {
        // Create new category
        $stmt = $conn->prepare("INSERT INTO categories (category_name, description) VALUES (?, ?)");
        $category_description = "Custom category created by teacher";
        $stmt->bind_param("ss", $custom_category, $category_description);
        if ($stmt->execute()) {
            $category_id = $conn->insert_id;
            $stmt->close();
        } else {
            $errors[] = "Failed to create custom category: " . $stmt->error;
            $stmt->close();
        }
    } elseif ($category_id === 0 && empty($custom_category)) {
        $errors[] = "Category is required or provide a custom category name";
    } elseif (empty($category_id)) {
        $errors[] = "Category is required";
    }
    
    // Handle thumbnail upload (optional)
    if (isset($_FILES['thumbnail']) && $_FILES['thumbnail']['error'] === UPLOAD_ERR_OK) {
        $thumbnail = $_FILES['thumbnail'];
        $allowed_image_types = ['jpg', 'jpeg', 'png', 'gif', 'webp'];
        $thumbnail_extension = strtolower(pathinfo($thumbnail['name'], PATHINFO_EXTENSION));
        
        // Validate thumbnail
        if (!in_array($thumbnail_extension, $allowed_image_types)) {
            $thumbnail_error = "Thumbnail must be an image file (JPG, PNG, GIF, WEBP)";
        } elseif ($thumbnail['size'] > 5 * 1024 * 1024) { // 5MB limit for images
            $thumbnail_error = "Thumbnail file size must be less than 5MB";
        } else {
            // Create upload directory if it doesn't exist
            $upload_dir = __DIR__ . '/../uploads/library_submissions/';
            if (!is_dir($upload_dir)) {
                mkdir($upload_dir, 0755, true);
            }
            
            // Generate unique thumbnail filename
            $thumbnail_name = 'thumb_' . time() . '_' . uniqid() . '.' . $thumbnail_extension;
            $thumbnail_upload_path = $upload_dir . $thumbnail_name;
            
            if (move_uploaded_file($thumbnail['tmp_name'], $thumbnail_upload_path)) {
                $thumbnail_path = 'uploads/library_submissions/' . $thumbnail_name;
                
                // Optional: Resize thumbnail to standard size (300x200) using simple GD functions
                if (extension_loaded('gd')) {
                    resizeImage($thumbnail_upload_path, $thumbnail_upload_path, 300, 200);
                }
            } else {
                $thumbnail_error = "Failed to upload thumbnail";
            }
        }
    }
    
    if ($thumbnail_error) {
        $errors[] = $thumbnail_error;
    }

    // Handle file upload - improved validation
    error_log("Checking document_file: " . (isset($_FILES['document_file']) ? "SET" : "NOT SET"));
    error_log("Document file name: " . ($_FILES['document_file']['name'] ?? "EMPTY"));
    
    if (isset($_FILES['document_file']) && !empty($_FILES['document_file']['name'])) {
        $file = $_FILES['document_file'];
        
        // Check for upload errors
        if ($file['error'] !== UPLOAD_ERR_OK) {
            switch ($file['error']) {
                case UPLOAD_ERR_INI_SIZE:
                case UPLOAD_ERR_FORM_SIZE:
                    $errors[] = "File size too large. Maximum size is 50MB";
                    break;
                case UPLOAD_ERR_PARTIAL:
                    $errors[] = "File was only partially uploaded. Please try again.";
                    break;
                case UPLOAD_ERR_NO_FILE:
                    $errors[] = "Please select a file to upload";
                    break;
                case UPLOAD_ERR_NO_TMP_DIR:
                case UPLOAD_ERR_CANT_WRITE:
                    $errors[] = "Server error during file upload. Please try again.";
                    break;
                default:
                    $errors[] = "Unknown error during file upload. Please try again.";
            }
        } else {
            // File uploaded successfully, now validate it
            $allowed_types = ['pdf', 'doc', 'docx', 'txt', 'ppt', 'pptx', 'xls', 'xlsx', 'epub', 'mobi'];
            $max_size = 50 * 1024 * 1024; // 50MB
            
            $file_extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
            
            if (!in_array($file_extension, $allowed_types)) {
                $errors[] = "File type not allowed. Allowed types: " . implode(', ', $allowed_types);
            }
            
            if ($file['size'] > $max_size) {
                $errors[] = "File size too large. Maximum size is 50MB";
            }
            
            if (empty($errors)) {
                // Create upload directory if it doesn't exist
                $upload_dir = __DIR__ . '/../uploads/library_submissions/';
                if (!is_dir($upload_dir)) {
                    mkdir($upload_dir, 0755, true);
                }
                
                // Generate unique filename
                $file_name = time() . '_' . uniqid() . '.' . $file_extension;
                $file_path = $upload_dir . $file_name;
                
                if (move_uploaded_file($file['tmp_name'], $file_path)) {
                    // Insert into database with better error handling
                    $stmt = $conn->prepare("
                        INSERT INTO library_submissions 
                        (teacher_id, title, description, category_id, tags, file_path, file_name, file_type, file_size, priority, thumbnail_path, submission_date, status) 
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), 'pending')
                    ");
                    
                    if (!$stmt) {
                        $errors[] = "Database prepare error: " . $conn->error;
                        error_log("SQL prepare failed: " . $conn->error);
                    } else {
                        $relative_path = 'uploads/library_submissions/' . $file_name;
                        
                        // Ensure thumbnail_path is not null
                        $thumb_path_safe = $thumbnail_path ?: null;
                        
                        // Debug the values being inserted
                        error_log("=== DATABASE INSERT DEBUG ===");
                        error_log("teacher_id: " . $teacher_id);
                        error_log("title: " . $title);
                        error_log("description: " . $description);
                        error_log("category_id: " . $category_id);
                        error_log("tags: " . $tags);
                        error_log("relative_path: " . $relative_path);
                        error_log("file_name: " . $file['name']);
                        error_log("file_extension: " . $file_extension);
                        error_log("file_size: " . $file['size']);
                        error_log("priority: " . $priority);
                        error_log("thumbnail_path: " . ($thumb_path_safe ?: 'NULL'));
                        
                        $stmt->bind_param("ississssiss", 
                            $teacher_id, $title, $description, $category_id, $tags, 
                            $relative_path, $file['name'], $file_extension, $file['size'], $priority, $thumb_path_safe
                        );
                        
                        if ($stmt->execute()) {
                            $submission_id = $conn->insert_id;
                            
                            // Send library submission notification email - USING EXACT SAME APPROACH AS COURSE CREATION
                            $email_result = sendLibrarySubmissionEmail($submission_id);
                            
                            if ($email_result) {
                                $success_message = "Document submitted successfully! ✉️ Confirmation email sent to you. Your document is awaiting admin approval.";
                            } else {
                                $success_message = "Document submitted successfully, but failed to send confirmation email. Your document is awaiting admin approval.";
                            }
                            error_log("SUCCESS: Submission inserted with ID: " . $submission_id);
                        } else {
                            $errors[] = "Database insert error: " . $stmt->error;
                            error_log("Database insert failed: " . $stmt->error);
                            error_log("SQL state: " . $stmt->sqlstate);
                        }
                        $stmt->close();
                    }
                } else {
                    $errors[] = "Failed to move uploaded file. Please check server permissions.";
                }
            }
        }
    } else {
        $errors[] = "Please select a file to upload";
    }
}

// Get teacher's submissions
$submissions = [];
$stmt = $conn->prepare("
    SELECT ls.*, c.category_name
    FROM library_submissions ls
    LEFT JOIN categories c ON ls.category_id = c.category_id
    WHERE ls.teacher_id = ?
    ORDER BY ls.submission_date DESC
");
$stmt->bind_param("i", $teacher_id);
$stmt->execute();
$result = $stmt->get_result();
while ($row = $result->fetch_assoc()) {
    $submissions[] = $row;
}
$stmt->close();

// Get categories from the main system categories table
$categories = [];
$result = $conn->query("SELECT category_id, category_name FROM categories ORDER BY category_name ASC");
if ($result) {
    while ($row = $result->fetch_assoc()) {
        $categories[] = $row;
    }
}

// Simple image resize function
function resizeImage($source, $destination, $maxWidth, $maxHeight) {
    // Get image info
    $imageInfo = getimagesize($source);
    if (!$imageInfo) return false;
    
    $width = $imageInfo[0];
    $height = $imageInfo[1];
    $type = $imageInfo[2];
    
    // Calculate new dimensions while maintaining aspect ratio
    $ratio = min($maxWidth / $width, $maxHeight / $height);
    $newWidth = intval($width * $ratio);
    $newHeight = intval($height * $ratio);
    
    // Create source image resource
    switch ($type) {
        case IMAGETYPE_JPEG:
            $sourceImage = imagecreatefromjpeg($source);
            break;
        case IMAGETYPE_PNG:
            $sourceImage = imagecreatefrompng($source);
            break;
        case IMAGETYPE_GIF:
            $sourceImage = imagecreatefromgif($source);
            break;
        case IMAGETYPE_WEBP:
            $sourceImage = imagecreatefromwebp($source);
            break;
        default:
            return false;
    }
    
    if (!$sourceImage) return false;
    
    // Create new image resource
    $newImage = imagecreatetruecolor($newWidth, $newHeight);
    
    // Preserve transparency for PNG and GIF
    if ($type == IMAGETYPE_PNG || $type == IMAGETYPE_GIF) {
        imagealphablending($newImage, false);
        imagesavealpha($newImage, true);
        $transparent = imagecolorallocatealpha($newImage, 255, 255, 255, 127);
        imagefilledrectangle($newImage, 0, 0, $newWidth, $newHeight, $transparent);
    }
    
    // Resize the image
    imagecopyresampled($newImage, $sourceImage, 0, 0, 0, 0, $newWidth, $newHeight, $width, $height);
    
    // Save the resized image
    $success = false;
    switch ($type) {
        case IMAGETYPE_JPEG:
            $success = imagejpeg($newImage, $destination, 85);
            break;
        case IMAGETYPE_PNG:
            $success = imagepng($newImage, $destination);
            break;
        case IMAGETYPE_GIF:
            $success = imagegif($newImage, $destination);
            break;
        case IMAGETYPE_WEBP:
            $success = imagewebp($newImage, $destination, 85);
            break;
    }
    
    // Clean up memory
    imagedestroy($sourceImage);
    imagedestroy($newImage);
    
    return $success;
}

// Include the teacher layout
require_once('components/teacher_layout.php');
?>

<style>
/* Library Submissions Styling */
.submission-header {
    background: linear-gradient(135deg, #fffbf0 0%, #fef3e2 100%);
    border-radius: 20px;
    padding: 2rem;
    margin-bottom: 2rem;
    color: #92400e;
    border: 2px solid #fed7aa;
    position: relative;
    overflow: hidden;
}

.submission-header::before {
    content: '';
    position: absolute;
    top: -50%;
    right: -50%;
    width: 100%;
    height: 100%;
    background: url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 100 100"><circle cx="50" cy="50" r="2" fill="rgba(255,255,255,0.1)"/></svg>') repeat;
    animation: float 20s infinite linear;
}

.submission-form {
    background: white;
    border-radius: 16px;
    padding: 2rem;
    box-shadow: 0 10px 30px rgba(0,0,0,0.1);
    margin-bottom: 2rem;
}

.form-group {
    margin-bottom: 1.5rem;
}

.form-label {
    font-weight: 600;
    color: #374151;
    margin-bottom: 0.5rem;
    display: block;
}

.form-control {
    border: 2px solid #e5e7eb;
    border-radius: 8px;
    padding: 0.75rem;
    font-size: 0.95rem;
    transition: border-color 0.3s ease;
    width: 100%;
}

.form-control:focus {
    border-color: #2563eb;
    outline: none;
    box-shadow: 0 0 0 3px rgba(37, 99, 235, 0.1);
}

.file-upload-area {
    border: 2px dashed #d1d5db;
    border-radius: 12px;
    padding: 2rem;
    text-align: center;
    background: #f9fafb;
    transition: all 0.3s ease;
    cursor: pointer;
}

.file-upload-area:hover {
    border-color: #2563eb;
    background: #eff6ff;
}

.file-upload-area.dragover {
    border-color: #2563eb;
    background: #dbeafe;
}

.submission-card {
    background: white;
    border-radius: 12px;
    padding: 1.5rem;
    margin-bottom: 1rem;
    box-shadow: 0 4px 12px rgba(0,0,0,0.05);
    border: 1px solid #e5e7eb;
    transition: all 0.3s ease;
}

.submission-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 20px rgba(0,0,0,0.1);
}

.status-badge {
    padding: 0.25rem 0.75rem;
    border-radius: 20px;
    font-size: 0.75rem;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.status-pending {
    background: #fef3c7;
    color: #92400e;
}

.status-approved {
    background: #d1fae5;
    color: #065f46;
}

.status-rejected {
    background: #fee2e2;
    color: #991b1b;
}

.priority-badge {
    padding: 0.2rem 0.5rem;
    border-radius: 12px;
    font-size: 0.7rem;
    font-weight: 500;
}

.priority-high {
    background: #fef2f2;
    color: #dc2626;
}

.priority-medium {
    background: #fef3c7;
    color: #d97706;
}

.priority-low {
    background: #f0f9ff;
    color: #0369a1;
}

.category-badge {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.5rem 1rem;
    border-radius: 20px;
    font-size: 0.8rem;
    font-weight: 500;
    color: white;
}

.btn-primary {
    background: linear-gradient(135deg, #2563eb, #1d4ed8);
    border: none;
    border-radius: 8px;
    padding: 0.75rem 1.5rem;
    color: white;
    font-weight: 600;
    transition: all 0.3s ease;
}

.btn-primary:hover {
    transform: translateY(-1px);
    box-shadow: 0 4px 12px rgba(37, 99, 235, 0.4);
}

.stats-row {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1rem;
    margin-bottom: 2rem;
}

.stat-card {
    background: white;
    border-radius: 12px;
    padding: 1.5rem;
    text-align: center;
    box-shadow: 0 4px 12px rgba(0,0,0,0.05);
}

.stat-number {
    font-size: 2rem;
    font-weight: 700;
    color: #2563eb;
}

.stat-label {
    color: #6b7280;
    font-size: 0.9rem;
    margin-top: 0.5rem;
}

@keyframes float {
    0% { transform: translateX(-100%); }
    100% { transform: translateX(100%); }
}

/* Tips Sidebar Styling */
.tips-sidebar {
    position: sticky;
    top: 100px;
}

.tips-card {
    background: linear-gradient(135deg, #fffbf0 0%, #fef3e2 100%);
    border: 2px solid #fed7aa;
    border-radius: 16px;
    padding: 1.5rem;
    margin-bottom: 1.5rem;
    box-shadow: 0 4px 15px rgba(251, 191, 36, 0.1);
}

.tips-title {
    font-size: 1.1rem;
    font-weight: 700;
    color: #92400e;
    margin-bottom: 1rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.tips-title i {
    color: #f59e0b;
    font-size: 1.3rem;
}

.tip-item {
    display: flex;
    gap: 0.75rem;
    margin-bottom: 1rem;
    padding-bottom: 1rem;
    border-bottom: 1px solid #fed7aa;
}

.tip-item:last-child {
    margin-bottom: 0;
    padding-bottom: 0;
    border-bottom: none;
}

.tip-icon {
    font-size: 1.2rem;
    flex-shrink: 0;
}

.tip-text {
    flex: 1;
    color: #92400e;
    font-size: 0.9rem;
    line-height: 1.4;
}

.tip-text strong {
    color: #78350f;
}

/* Process Steps Styling */
.process-steps {
    position: relative;
}

.step-item {
    display: flex;
    align-items: center;
    gap: 1rem;
    margin-bottom: 1rem;
    position: relative;
}

.step-item:last-child {
    margin-bottom: 0;
}

.step-item::before {
    content: '';
    position: absolute;
    left: 15px;
    top: 30px;
    width: 2px;
    height: 30px;
    background: #fed7aa;
}

.step-item:last-child::before {
    display: none;
}

.step-number {
    width: 30px;
    height: 30px;
    border-radius: 50%;
    background: linear-gradient(135deg, #f59e0b, #d97706);
    color: white;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: 600;
    font-size: 0.9rem;
    flex-shrink: 0;
    position: relative;
    z-index: 1;
}

.step-text {
    color: #92400e;
    font-size: 0.9rem;
    line-height: 1.4;
}

/* Responsive Design for Sidebar */
@media (max-width: 991px) {
    .tips-sidebar {
        position: static;
        margin-top: 2rem;
    }
    
    .tips-card {
        margin-bottom: 1rem;
    }
}
</style>

<div class="container-fluid">
    <!-- Header Section -->
    <div class="submission-header">
        <div class="row align-items-center">
            <div class="col-md-8">
                <h1 class="mb-2">📚 Library Document Submissions</h1>
                <p class="mb-0 opacity-75">Submit educational resources for admin approval and public sharing</p>
            </div>
            <div class="col-md-4 text-end">
                <i class="bx bx-library" style="font-size: 4rem; opacity: 0.3;"></i>
            </div>
        </div>
    </div>

    <!-- Stats Row -->
    <div class="stats-row">
        <?php
        $total_count = count($submissions);
        $pending_count = count(array_filter($submissions, fn($s) => $s['status'] === 'pending'));
        $approved_count = count(array_filter($submissions, fn($s) => $s['status'] === 'approved'));
        $rejected_count = count(array_filter($submissions, fn($s) => $s['status'] === 'rejected'));
        ?>
        <div class="stat-card">
            <div class="stat-number"><?= $total_count ?></div>
            <div class="stat-label">Total Submissions</div>
        </div>
        <div class="stat-card">
            <div class="stat-number" style="color: #f59e0b;"><?= $pending_count ?></div>
            <div class="stat-label">Pending Review</div>
        </div>
        <div class="stat-card">
            <div class="stat-number" style="color: #10b981;"><?= $approved_count ?></div>
            <div class="stat-label">Approved</div>
        </div>
        <div class="stat-card">
            <div class="stat-number" style="color: #ef4444;"><?= $rejected_count ?></div>
            <div class="stat-label">Rejected</div>
        </div>
    </div>

    <!-- Error/Success Messages -->
    <?php if (isset($errors) && !empty($errors)): ?>
        <div class="alert alert-danger">
            <h6>Please fix the following errors:</h6>
            <ul class="mb-0">
                <?php foreach ($errors as $error): ?>
                    <li><?= htmlspecialchars($error) ?></li>
                <?php endforeach; ?>
            </ul>
        </div>
    <?php endif; ?>

    <?php if (isset($success_message)): ?>
        <div class="alert alert-success">
            <i class="bx bx-check-circle"></i> <?= htmlspecialchars($success_message) ?>
        </div>
    <?php endif; ?>

    <!-- Main Content Row - 2 Column Layout -->
    <div class="row">
        <!-- Left Column: Submission Form -->
        <div class="col-lg-8">
            <div class="submission-form">
        <h3 class="mb-3"><i class="bx bx-plus-circle text-primary"></i> Submit New Document</h3>
        <form method="POST" enctype="multipart/form-data">
            <div class="row">
                <div class="col-md-8">
                    <div class="form-group">
                        <label class="form-label">Document Title *</label>
                        <input type="text" name="title" class="form-control" required 
                               placeholder="Enter a descriptive title for your document">
                    </div>
                </div>
                <div class="col-md-4">
                    <div class="form-group">
                        <label class="form-label">Priority Level</label>
                        <select name="priority" class="form-control">
                            <option value="low">Low Priority</option>
                            <option value="medium" selected>Medium Priority</option>
                            <option value="high">High Priority</option>
                        </select>
                    </div>
                </div>
            </div>

            <div class="form-group">
                <label class="form-label">Description *</label>
                <textarea name="description" class="form-control" rows="4" required
                          placeholder="Provide a detailed description of the document content and its educational value"></textarea>
            </div>

            <div class="row">
                <div class="col-md-6">
                    <div class="form-group">
                        <label class="form-label">Category *</label>
                        <select name="category" class="form-control" required onchange="toggleCustomCategory()">
                            <option value="">Select a category</option>
                            <?php foreach ($categories as $category): ?>
                                <option value="<?= htmlspecialchars($category['category_id']) ?>">
                                    <?= htmlspecialchars($category['category_name']) ?>
                                </option>
                            <?php endforeach; ?>
                            <option value="0">Other (Create Custom Category)</option>
                        </select>
                        
                        <!-- Custom Category Input (Hidden by default) -->
                        <div id="customCategoryDiv" style="display: none; margin-top: 10px;">
                            <input type="text" 
                                   name="custom_category" 
                                   id="customCategoryInput"
                                   class="form-control" 
                                   placeholder="Enter custom category name..."
                                   maxlength="100">
                            <small class="text-muted">This will create a new category in the system</small>
                        </div>
                    </div>
                </div>
                <div class="col-md-6">
                    <div class="form-group">
                        <label class="form-label">Tags (Optional)</label>
                        <input type="text" name="tags" class="form-control" 
                               placeholder="Enter tags separated by commas (e.g., beginner, tutorial, reference)">
                    </div>
                </div>
            </div>

            <div class="form-group">
                <label class="form-label">Document File *</label>
                
                <!-- SIMPLIFIED FILE INPUT - GUARANTEED TO WORK -->
                <div style="border: 2px dashed #d1d5db; border-radius: 8px; padding: 20px; text-align: center; background: #f9fafb;">
                    <div id="simple-file-display">
                        <i class="bx bx-cloud-upload" style="font-size: 2rem; color: #6b7280;"></i>
                        <p class="mt-2 mb-1"><strong>Click below to select your document file</strong></p>
                        <p class="text-muted small mb-3">PDF, DOC, DOCX, TXT, PPT, PPTX, XLS, XLSX, EPUB, MOBI (Max: 50MB)</p>
                    </div>
                    
                    <!-- VISIBLE FILE INPUT - NO DRAG AND DROP COMPLEXITY -->
                    <input type="file" 
                           id="document_file" 
                           name="document_file" 
                           accept=".pdf,.doc,.docx,.txt,.ppt,.pptx,.xls,.xlsx,.epub,.mobi" 
                           required
                           class="form-control"
                           style="display: block; margin: 0 auto; max-width: 400px;"
                           onchange="handleSimpleFileChange(this)">
                           
                    <div id="file-info" style="margin-top: 15px; display: none; padding: 10px; background: rgba(255,255,255,0.1); border-radius: 5px;">
                        <i class="bx bx-file" style="color: #2563eb; font-size: 1.5rem;"></i>
                        <div id="file-details" style="margin-top: 5px;"></div>
                    </div>
                </div>
            </div>

            <div class="form-group">
                <label class="form-label">Thumbnail Image (Optional)</label>
                <div id="thumbnail-upload-area" style="border: 2px dashed #d1d5db; border-radius: 8px; padding: 30px; text-align: center; cursor: pointer; background: #f9fafb; transition: all 0.3s ease;" onclick="document.getElementById('thumbnail').click()">
                    <i class="bx bx-image" style="font-size: 2rem; color: #6b7280;"></i>
                    <p class="mt-2 mb-1"><strong>Click to upload thumbnail</strong> or drag and drop</p>
                    <p class="text-muted small mb-0">JPG, PNG, GIF, WEBP (Max: 5MB) - Will be resized to 300x200px</p>
                </div>
                <input type="file" id="thumbnail" name="thumbnail" accept=".jpg,.jpeg,.png,.gif,.webp" style="display: none;" onchange="handleThumbnailChange(this)">
                <small class="text-muted">
                    <i class="bx bx-info-circle"></i> 
                    Adding a thumbnail makes your document more visually appealing and easier to identify in the library.
                </small>
            </div>

            <button type="submit" name="submit_document" class="btn btn-primary">
                <i class="bx bx-send"></i> Submit for Review
            </button>
            

        </form>
            </div>
        </div>

        <!-- Right Column: Tips & Guidelines -->
        <div class="col-lg-4">
            <div class="tips-sidebar">
                <div class="tips-card">
                    <h4 class="tips-title">
                        <i class="bx bx-lightbulb"></i>
                        Tips for Success
                    </h4>
                    <div class="tips-content">
                        <div class="tip-item">
                            <div class="tip-icon">📝</div>
                            <div class="tip-text">
                                <strong>Clear Titles:</strong> Use descriptive titles that clearly indicate what the document contains.
                            </div>
                        </div>
                        
                        <div class="tip-item">
                            <div class="tip-icon">📁</div>
                            <div class="tip-text">
                                <strong>File Quality:</strong> Ensure your documents are well-formatted and error-free before submission.
                            </div>
                        </div>
                        
                        <div class="tip-item">
                            <div class="tip-icon">🖼️</div>
                            <div class="tip-text">
                                <strong>Add Thumbnails:</strong> Include attractive thumbnails to make your resources more appealing.
                            </div>
                        </div>
                        
                        <div class="tip-item">
                            <div class="tip-icon">🏷️</div>
                            <div class="tip-text">
                                <strong>Use Tags:</strong> Add relevant tags to help students find your resources easily.
                            </div>
                        </div>
                        
                        <div class="tip-item">
                            <div class="tip-icon">⚡</div>
                            <div class="tip-text">
                                <strong>File Size:</strong> Keep files under 50MB for faster uploads and downloads.
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="tips-card">
                    <h4 class="tips-title">
                        <i class="bx bx-info-circle"></i>
                        Submission Process
                    </h4>
                    <div class="process-steps">
                        <div class="step-item">
                            <div class="step-number">1</div>
                            <div class="step-text">Fill out the form with document details</div>
                        </div>
                        <div class="step-item">
                            <div class="step-number">2</div>
                            <div class="step-text">Upload your document file (required)</div>
                        </div>
                        <div class="step-item">
                            <div class="step-number">3</div>
                            <div class="step-text">Add a thumbnail image (optional)</div>
                        </div>
                        <div class="step-item">
                            <div class="step-number">4</div>
                            <div class="step-text">Submit for admin review</div>
                        </div>
                        <div class="step-item">
                            <div class="step-number">5</div>
                            <div class="step-text">Track status in your submissions</div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Submissions history has been moved to publish_to_library.php -->
</div>

<script>
// Simplified file upload handler
function handleSimpleFileChange(input) {
    const fileInfo = document.getElementById('file-info');
    const fileDetails = document.getElementById('file-details');
    
    if (input.files && input.files.length > 0) {
        const file = input.files[0];
        
        // Show file info
        if (fileInfo && fileDetails) {
            fileDetails.innerHTML = `
                <strong>${file.name}</strong><br>
                <small>Size: ${(file.size / 1024 / 1024).toFixed(2)} MB</small><br>
                <small style="color: #10b981;">✅ File ready for upload</small>
            `;
            fileInfo.style.display = 'block';
        }
    } else {
        // Hide file info if no file
        if (fileInfo) {
            fileInfo.style.display = 'none';
        }
    }
}

// Thumbnail upload handler
function handleThumbnailChange(input) {
    const uploadArea = document.getElementById('thumbnail-upload-area');
    
    if (input.files && input.files.length > 0) {
        const file = input.files[0];
        
        // Validate file type
        const allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'];
        if (!allowedTypes.includes(file.type)) {
            alert('Please select a valid image file (JPG, PNG, GIF, WEBP)');
            input.value = '';
            return;
        }
        
        // Validate file size (5MB limit)
        if (file.size > 5 * 1024 * 1024) {
            alert('Thumbnail file size must be less than 5MB');
            input.value = '';
            return;
        }
        
        // Create preview
        const reader = new FileReader();
        reader.onload = function(e) {
            uploadArea.innerHTML = `
                <div style="position: relative;">
                    <img src="${e.target.result}" alt="Thumbnail Preview" 
                         style="max-width: 200px; max-height: 150px; border-radius: 8px; object-fit: cover;">
                    <div style="margin-top: 10px;">
                        <p class="mb-1"><strong>${file.name}</strong></p>
                        <p class="text-muted small mb-0">${(file.size / 1024 / 1024).toFixed(2)} MB</p>
                        <button type="button" class="btn btn-sm btn-outline-secondary mt-2" onclick="clearThumbnail()">
                            <i class='bx bx-x'></i> Remove
                        </button>
                    </div>
                </div>
            `;
        };
        reader.readAsDataURL(file);
    }
}

// Clear thumbnail function
function clearThumbnail() {
    const thumbnailInput = document.getElementById('thumbnail');
    const uploadArea = document.getElementById('thumbnail-upload-area');
    
    thumbnailInput.value = '';
    uploadArea.innerHTML = `
        <i class="bx bx-image" style="font-size: 2rem; color: #6b7280;"></i>
        <p class="mt-2 mb-1"><strong>Click to upload thumbnail</strong> or drag and drop</p>
        <p class="text-muted small mb-0">JPG, PNG, GIF, WEBP (Max: 5MB) - Will be resized to 300x200px</p>
    `;
}

// Delete submission function
function deleteSubmission(submissionId) {
    if (confirm('Are you sure you want to delete this submission? This action cannot be undone.')) {
        // Create a hidden form to submit the deletion request
        const form = document.createElement('form');
        form.method = 'POST';
        form.style.display = 'none';
        
        const deleteInput = document.createElement('input');
        deleteInput.type = 'hidden';
        deleteInput.name = 'delete_submission';
        deleteInput.value = '1';
        
        const idInput = document.createElement('input');
        idInput.type = 'hidden';
        idInput.name = 'submission_id';
        idInput.value = submissionId;
        
        form.appendChild(deleteInput);
        form.appendChild(idInput);
        document.body.appendChild(form);
        form.submit();
    }
}

// Toggle custom category input field
function toggleCustomCategory() {
    const categorySelect = document.querySelector('select[name="category"]');
    const customCategoryDiv = document.getElementById('customCategoryDiv');
    const customCategoryInput = document.getElementById('customCategoryInput');
    
    if (categorySelect.value === '0') {
        // Show custom category input
        customCategoryDiv.style.display = 'block';
        customCategoryInput.required = true;
        customCategoryInput.focus();
    } else {
        // Hide custom category input
        customCategoryDiv.style.display = 'none';
        customCategoryInput.required = false;
        customCategoryInput.value = '';
    }
}

// Thumbnail upload functionality
const thumbnailUploadArea = document.querySelector('.thumbnail-upload-area');
const thumbnailInput = document.getElementById('thumbnail');

if (thumbnailUploadArea && thumbnailInput) {
    // Drag and drop events for thumbnail
    ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
        thumbnailUploadArea.addEventListener(eventName, preventDefaults, false);
    });

    function preventDefaults(e) {
        e.preventDefault();
        e.stopPropagation();
    }

    ['dragenter', 'dragover'].forEach(eventName => {
        thumbnailUploadArea.addEventListener(eventName, highlightThumbnail, false);
    });

    ['dragleave', 'drop'].forEach(eventName => {
        thumbnailUploadArea.addEventListener(eventName, unhighlightThumbnail, false);
    });

    function highlightThumbnail(e) {
        thumbnailUploadArea.style.borderColor = '#2563eb';
        thumbnailUploadArea.style.backgroundColor = '#eff6ff';
    }

    function unhighlightThumbnail(e) {
        thumbnailUploadArea.style.borderColor = '#d1d5db';
        thumbnailUploadArea.style.backgroundColor = '#f9fafb';
    }

    thumbnailUploadArea.addEventListener('drop', handleThumbnailDrop, false);

    function handleThumbnailDrop(e) {
        const dt = e.dataTransfer;
        const files = dt.files;
        
        console.log('🖼️ Thumbnail files dropped:', files);
        
        if (files.length > 0) {
            // CRITICAL FIX: Properly set thumbnail files to input
            const dataTransfer = new DataTransfer();
            dataTransfer.items.add(files[0]);
            thumbnailInput.files = dataTransfer.files;
            
            console.log('🖼️ Thumbnail assigned to input:', thumbnailInput.files);
            updateThumbnailDisplay(files[0]);
            updateDebugInfo(); // Update debug info immediately
        }
    }

    thumbnailInput.addEventListener('change', (e) => {
        if (e.target.files.length > 0) {
            updateThumbnailDisplay(e.target.files[0]);
        }
    });

    function updateThumbnailDisplay(file) {
        // Validate file type
        const allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'];
        if (!allowedTypes.includes(file.type)) {
            alert('Please select a valid image file (JPG, PNG, GIF, WEBP)');
            thumbnailInput.value = '';
            return;
        }

        // Validate file size (5MB)
        if (file.size > 5 * 1024 * 1024) {
            alert('Thumbnail file size must be less than 5MB');
            thumbnailInput.value = '';
            return;
        }

        // Create preview
        const reader = new FileReader();
        reader.onload = function(e) {
            thumbnailUploadArea.innerHTML = `
                <div style="position: relative;">
                    <img src="${e.target.result}" alt="Thumbnail Preview" 
                         style="max-width: 200px; max-height: 150px; border-radius: 8px; object-fit: cover;">
                    <div style="margin-top: 10px;">
                        <p class="mb-1"><strong>${file.name}</strong></p>
                        <p class="text-muted small mb-0">${(file.size / 1024 / 1024).toFixed(2)} MB</p>
                        <button type="button" class="btn btn-sm btn-outline-secondary mt-2" onclick="clearThumbnail()">
                            <i class='bx bx-x'></i> Remove
                        </button>
                    </div>
                </div>
            `;
        };
        reader.readAsDataURL(file);
    }
}

function clearThumbnail() {
    thumbnailInput.value = '';
    thumbnailUploadArea.innerHTML = `
        <i class="bx bx-image" style="font-size: 2rem; color: #6b7280;"></i>
        <p class="mt-2 mb-1"><strong>Click to upload thumbnail</strong> or drag and drop</p>
        <p class="text-muted small mb-0">JPG, PNG, GIF, WEBP (Max: 5MB) - Will be resized to 300x200px</p>
    `;
    updateDebugInfo();
}

// Form validation and debugging
function validateForm() {
    const documentFile = document.getElementById('document_file');
    const thumbnailFile = document.getElementById('thumbnail');
    
    console.log('=== FORM VALIDATION DEBUG ===');
    console.log('Document file input:', documentFile);
    console.log('Document file value:', documentFile ? documentFile.value : 'NULL');
    console.log('Document files:', documentFile ? documentFile.files : 'NULL');
    console.log('Thumbnail file input:', thumbnailFile);
    console.log('Thumbnail file value:', thumbnailFile ? thumbnailFile.value : 'NULL');
    console.log('Thumbnail files:', thumbnailFile ? thumbnailFile.files : 'NULL');
    
    // Update debug display
    updateDebugInfo();
    
    // Check if document file is selected
    if (!documentFile || !documentFile.files || documentFile.files.length === 0) {
        alert('Please select a document file to upload');
        return false;
    }
    
    // Check required fields
    const title = document.querySelector('input[name="title"]');
    const description = document.querySelector('textarea[name="description"]');
    const category = document.querySelector('select[name="category"]');
    
    if (!title || !title.value.trim()) {
        alert('Please enter a document title');
        return false;
    }
    
    if (!description || !description.value.trim()) {
        alert('Please enter a description');
        return false;
    }
    
    if (!category || !category.value) {
        alert('Please select a category');
        return false;
    }
    
    console.log('Form validation passed - submitting...');
    return true;
}

// Update debug info display
function updateDebugInfo() {
    const documentFile = document.getElementById('document_file');
    const thumbnailFile = document.getElementById('thumbnail');
    const docStatus = document.getElementById('doc-status');
    const thumbStatus = document.getElementById('thumb-status');
    
    // Update old debug info
    if (docStatus) {
        if (documentFile && documentFile.files && documentFile.files.length > 0) {
            docStatus.innerHTML = `<span style="color: green;">✅ ${documentFile.files[0].name} (${(documentFile.files[0].size/1024/1024).toFixed(2)} MB)</span>`;
        } else {
            docStatus.innerHTML = '<span style="color: red;">❌ Not selected</span>';
        }
    }
    
    if (thumbStatus) {
        if (thumbnailFile && thumbnailFile.files && thumbnailFile.files.length > 0) {
            thumbStatus.innerHTML = `<span style="color: green;">✅ ${thumbnailFile.files[0].name} (${(thumbnailFile.files[0].size/1024/1024).toFixed(2)} MB)</span>`;
        } else {
            thumbStatus.innerHTML = '<span style="color: gray;">⭕ Optional - not selected</span>';
        }
    }
    
    // Update LIVE debug panel
    updateLiveDebugPanel();
}

// Update the live debug panel with real-time information
function updateLiveDebugPanel() {
    const documentFile = document.getElementById('document_file');
    const thumbnailFile = document.getElementById('thumbnail');
    
    // Update live document status
    const liveDocStatus = document.getElementById('live-doc-status');
    if (liveDocStatus) {
        if (documentFile && documentFile.files && documentFile.files.length > 0) {
            const file = documentFile.files[0];
            liveDocStatus.innerHTML = `<span style="color: #4caf50;">✅ SELECTED: ${file.name} (${(file.size/1024/1024).toFixed(2)} MB)</span>`;
        } else {
            liveDocStatus.innerHTML = '<span style="color: #f44336;">❌ NOT SELECTED</span>';
        }
    }
    
    // Update live thumbnail status
    const liveThumbStatus = document.getElementById('live-thumb-status');
    if (liveThumbStatus) {
        if (thumbnailFile && thumbnailFile.files && thumbnailFile.files.length > 0) {
            const file = thumbnailFile.files[0];
            liveThumbStatus.innerHTML = `<span style="color: #4caf50;">✅ SELECTED: ${file.name} (${(file.size/1024/1024).toFixed(2)} MB)</span>`;
        } else {
            liveThumbStatus.innerHTML = '<span style="color: #ff9800;">⭕ OPTIONAL - Not selected</span>';
        }
    }
    
    // Update form data status
    const liveMethod = document.getElementById('live-method');
    const livePost = document.getElementById('live-post');
    const liveFiles = document.getElementById('live-files');
    
    if (liveMethod) {
        liveMethod.textContent = 'POST (Ready for submission)';
        liveMethod.style.color = '#4caf50';
    }
    
    if (livePost) {
        const formElements = document.querySelectorAll('form input, form textarea, form select');
        let formData = {};
        formElements.forEach(element => {
            if (element.name && element.value) {
                formData[element.name] = element.value.substring(0, 50) + (element.value.length > 50 ? '...' : '');
            }
        });
        livePost.innerHTML = `<span style="color: #4caf50;">${Object.keys(formData).length} fields filled</span>`;
    }
    
    if (liveFiles) {
        let fileCount = 0;
        if (documentFile && documentFile.files && documentFile.files.length > 0) fileCount++;
        if (thumbnailFile && thumbnailFile.files && thumbnailFile.files.length > 0) fileCount++;
        liveFiles.innerHTML = `<span style="color: ${fileCount > 0 ? '#4caf50' : '#f44336'};">${fileCount} files selected</span>`;
    }
}

// Enhanced form validation with detailed logging
function validateForm() {
    console.log('🔍 === DETAILED FORM VALIDATION DEBUG ===');
    
    const form = document.querySelector('form');
    const documentFile = document.getElementById('document_file');
    const thumbnailFile = document.getElementById('thumbnail');
    const formData = new FormData(form);
    
    // Log form element details
    console.log('📋 Form element:', form);
    console.log('📄 Document file input:', documentFile);
    console.log('🖼️ Thumbnail file input:', thumbnailFile);
    console.log('📤 FormData contents:');
    
    for (let [key, value] of formData.entries()) {
        console.log(`  ${key}:`, value);
    }
    
    // Check document file specifically
    console.log('📁 Document file check:');
    console.log('  - Input exists:', !!documentFile);
    console.log('  - Files property:', documentFile ? documentFile.files : 'N/A');
    console.log('  - Files length:', documentFile && documentFile.files ? documentFile.files.length : 0);
    console.log('  - First file:', documentFile && documentFile.files && documentFile.files[0] ? documentFile.files[0] : 'None');
    
    // Update debug display
    updateLiveDebugPanel();
    
    // Perform validation
    if (!documentFile || !documentFile.files || documentFile.files.length === 0) {
        console.log('❌ VALIDATION FAILED: No document file selected');
        alert('🚨 DEBUG: No document file detected!\n\nThe form cannot find a selected document file.\nPlease try selecting the file again.');
        return false;
    }
    
    // Check required fields
    const title = document.querySelector('input[name="title"]');
    const description = document.querySelector('textarea[name="description"]');
    const category = document.querySelector('select[name="category"]');
    
    console.log('📝 Form fields check:');
    console.log('  - Title:', title ? title.value : 'Missing input');
    console.log('  - Description:', description ? description.value : 'Missing input');
    console.log('  - Category:', category ? category.value : 'Missing input');
    
    if (!title || !title.value.trim()) {
        console.log('❌ VALIDATION FAILED: No title');
        alert('🚨 Please enter a document title');
        return false;
    }
    
    if (!description || !description.value.trim()) {
        console.log('❌ VALIDATION FAILED: No description');
        alert('🚨 Please enter a description');
        return false;
    }
    
    if (!category || !category.value) {
        console.log('❌ VALIDATION FAILED: No category');
        alert('🚨 Please select a category');
        return false;
    }
    
    console.log('✅ VALIDATION PASSED - Submitting form...');
    console.log('📤 Final form data being submitted:');
    for (let [key, value] of formData.entries()) {
        if (value instanceof File) {
            console.log(`  ${key}: FILE - ${value.name} (${value.size} bytes)`);
        } else {
            console.log(`  ${key}: ${value}`);
        }
    }
    
    return true;
}

// Update debug info when files change
document.addEventListener('DOMContentLoaded', function() {
    const documentFile = document.getElementById('document_file');
    const thumbnailFile = document.getElementById('thumbnail');
    
    if (documentFile) {
        documentFile.addEventListener('change', updateDebugInfo);
    }
    
    if (thumbnailFile) {
        thumbnailFile.addEventListener('change', updateDebugInfo);
    }
    
    // Initial update
    updateDebugInfo();
});
</script>

<?php require_once('components/footer.php'); ?>
