<?php
// Include the centralized session handling - this MUST be first
require_once __DIR__ . '/../includes/session_start.php';

// Show detailed errors for debugging
ini_set('display_errors', 1);
error_reporting(E_ALL);

// Include database connection
require_once __DIR__ . '/../includes/db_connection.php';
require_once __DIR__ . '/../includes/site_config.php';

// Check if user is logged in as teacher
if (!isset($_SESSION['user_role']) || $_SESSION['user_role'] !== 'teacher') {
    // Allow bypass for development
    $bypass_auth = isset($_GET['bypass_auth']) && $_GET['bypass_auth'] === 'true';
    
    if (!$bypass_auth) {
        header("Location: ../auth/login.php?redirect=" . urlencode($_SERVER['REQUEST_URI']));
        exit;
    }
}

// Get current page for navigation highlighting
$current_page = basename($_SERVER['PHP_SELF']);
// Get teacher ID from session
$teacher_id = $_SESSION['user_id'];

// Include the teacher layout
require_once('components/teacher_layout.php');

// Get selected category or default to 'all'
$current_category = isset($_GET['category']) ? $_GET['category'] : 'all';
$search_term = isset($_GET['search']) ? trim($_GET['search']) : '';

// Function to count materials
function countMaterials($conn, $teacher_id, $category = null) {
    $sql = "SELECT COUNT(*) as count FROM course_materials WHERE teacher_id = ?";
    
    if ($category && $category !== 'all') {
        $sql .= " AND category = ?";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("is", $teacher_id, $category);
    } else {
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("i", $teacher_id);
    }
    
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    return $row['count'];
}

// Function to get materials
function getMaterials($conn, $teacher_id, $category = null, $search = null, $limit = 20, $offset = 0) {
    $sql = "SELECT m.*, c.title AS course_name, c.course_id AS course_code 
            FROM course_materials m
            LEFT JOIN courses c ON m.course_id = c.course_id
            WHERE m.teacher_id = ?";
    
    $params = array($teacher_id);
    $types = "i";
    
    if ($category && $category !== 'all') {
        $sql .= " AND m.category = ?";
        $params[] = $category;
        $types .= "s";
    }
    
    if ($search) {
        $sql .= " AND (m.title LIKE ? OR m.description LIKE ? OR c.title LIKE ?)";
        $searchParam = "%$search%";
        $params[] = $searchParam;
        $params[] = $searchParam;
        $params[] = $searchParam;
        $types .= "sss";
    }
    
    $sql .= " ORDER BY m.date_added DESC LIMIT ? OFFSET ?";
    $params[] = $limit;
    $params[] = $offset;
    $types .= "ii";
    
    $stmt = $conn->prepare($sql);
    
    // Dynamically bind parameters
    $bindParams = array($types);
    for ($i = 0; $i < count($params); $i++) {
        $bindParams[] = &$params[$i];
    }
    call_user_func_array(array($stmt, 'bind_param'), $bindParams);
    
    $stmt->execute();
    $result = $stmt->get_result();
    $materials = array();
    
    while ($row = $result->fetch_assoc()) {
        $materials[] = $row;
    }
    
    return $materials;
}

// Get the total count
$total_materials = countMaterials($conn, $teacher_id, $current_category);

// Pagination
$per_page = 12;
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$offset = ($page - 1) * $per_page;
$total_pages = ceil($total_materials / $per_page);

// Get materials for current page
$materials = getMaterials($conn, $teacher_id, $current_category, $search_term, $per_page, $offset);

// Get categories for filtering
$category_query = "SELECT DISTINCT category FROM course_materials WHERE teacher_id = ? ORDER BY category";
$stmt = $conn->prepare($category_query);
$stmt->bind_param("i", $teacher_id);
$stmt->execute();
$category_result = $stmt->get_result();
$categories = array();

while ($row = $category_result->fetch_assoc()) {
    if (!empty($row['category']) && strtolower($row['category']) !== 'welcome') {
        $categories[] = $row['category'];
    }
}
?>

<!-- Modern Library Page Styling - Panadite Theme -->
<style>
    /* Modern Library Page Styling - Panadite Theme */
    * {
        font-family: 'Inter', 'Poppins', sans-serif !important;
    }

    .container-fluid {
        padding: 0 30px !important;
    }

    /* Page Header */
    h1.h3, .section-title {
        font-size: 28px !important;
        font-weight: 700 !important;
        color: #1f2937 !important;
        letter-spacing: -0.025em !important;
        margin-bottom: 8px !important;
    }

    .text-muted {
        color: #6b7280 !important;
        font-size: 16px !important;
    }

    /* Small Stats Cards */
    .stats-card {
        background: #ffffff !important;
        border-radius: 12px !important;
        padding: 16px !important;
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.08) !important;
        transition: all 0.2s ease !important;
        border: 1px solid #f1f5f9 !important;
        height: 100% !important;
        display: flex !important;
        align-items: center !important;
        gap: 12px !important;
    }

    .stats-card:hover {
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.12) !important;
        transform: translateY(-2px) !important;
    }

    .stats-number {
        font-size: 20px !important;
        font-weight: 700 !important;
        color: #1f2937 !important;
        line-height: 1 !important;
    }

    .stats-label {
        font-size: 12px !important;
        color: #6b7280 !important;
        font-weight: 500 !important;
        text-transform: uppercase !important;
        letter-spacing: 0.5px !important;
    }

    /* Category Tabs */
    .category-tabs {
        display: flex !important;
        overflow-x: auto !important;
        gap: 12px !important;
        padding-bottom: 15px !important;
        margin-bottom: 25px !important;
        -ms-overflow-style: none !important;
        scrollbar-width: none !important;
    }

    .category-tabs::-webkit-scrollbar {
        display: none !important;
    }

    .category-tab {
        background: #ffffff !important;
        border: 1px solid #e5e7eb !important;
        color: #6b7280 !important;
        padding: 10px 20px !important;
        border-radius: 25px !important;
        text-decoration: none !important;
        font-weight: 500 !important;
        font-size: 14px !important;
        white-space: nowrap !important;
        transition: all 0.2s ease !important;
        display: flex !important;
        align-items: center !important;
        gap: 6px !important;
    }

    .category-tab:hover {
        background: #f9fafb !important;
        color: #374151 !important;
        text-decoration: none !important;
        border-color: #d1d5db !important;
    }

    .category-tab.active {
        background: #4361ee !important;
        color: #ffffff !important;
        border-color: #4361ee !important;
        box-shadow: 0 2px 8px rgba(66, 97, 238, 0.2) !important;
    }

    .category-tab i {
        font-size: 16px !important;
    }
    
    /* Material Cards */
    .material-card {
        background: #ffffff !important;
        border: 1px solid #f1f5f9 !important;
        border-radius: 12px !important;
        padding: 20px !important;
        transition: all 0.2s ease !important;
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.08) !important;
        height: 100% !important;
        display: flex !important;
        flex-direction: column !important;
    }
    
    .material-card:hover {
        transform: translateY(-4px) !important;
        box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15) !important;
        border-color: #e5e7eb !important;
    }

    .material-card .card-body {
        flex: 1 !important;
        display: flex !important;
        flex-direction: column !important;
        padding: 0 !important;
    }

    .material-card h6 {
        font-size: 16px !important;
        font-weight: 600 !important;
        color: #1f2937 !important;
        margin-bottom: 8px !important;
        line-height: 1.4 !important;
    }

    .material-card .text-muted {
        font-size: 13px !important;
        color: #6b7280 !important;
        margin-bottom: 12px !important;
    }

    .material-card .badge {
        font-size: 11px !important;
        font-weight: 500 !important;
        padding: 4px 10px !important;
        border-radius: 6px !important;
    }

    /* File type icons */
    .file-type-icon {
        width: 40px !important;
        height: 40px !important;
        border-radius: 8px !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        margin-bottom: 12px !important;
    }

    .file-type-icon i {
        font-size: 20px !important;
    }

    /* Search Form */
    .form-control {
        border: 1px solid #e5e7eb !important;
        border-radius: 8px !important;
        padding: 10px 16px !important;
        font-size: 14px !important;
        transition: all 0.2s ease !important;
    }

    .form-control:focus {
        border-color: #4361ee !important;
        box-shadow: 0 0 0 3px rgba(66, 97, 238, 0.1) !important;
    }

    .btn {
        font-weight: 500 !important;
        padding: 10px 20px !important;
        border-radius: 8px !important;
        transition: all 0.2s ease !important;
        font-size: 14px !important;
    }

    .btn-primary {
        background: #4361ee !important;
        border-color: #4361ee !important;
    }

    .btn-primary:hover {
        background: #3b50e0 !important;
        border-color: #3b50e0 !important;
        transform: translateY(-1px) !important;
    }
    
    .material-card:hover .card-title {
        color: #6a11cb;
    }
    
    .material-card:hover .material-icon {
        transform: scale(1.1) rotate(5deg);
    }
    
    .material-card::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        width: 100%;
        height: 4px;
        background: linear-gradient(90deg, #6a11cb, #8c67f4);
        transform: scaleX(0);
        transform-origin: left;
        transition: transform 0.3s ease;
    }
    
    .material-card:hover::before {
        transform: scaleX(1);
    }
    
    .material-card .card-body {
        padding: 1.75rem;
    }
    
    .material-card .card-title {
        font-size: 1.15rem;
        margin-bottom: 12px;
        font-weight: 700;
        line-height: 1.4;
        transition: color 0.3s ease;
        display: -webkit-box;
        -webkit-line-clamp: 2;
        -webkit-box-orient: vertical;
        overflow: hidden;
    }
    
    .material-card .material-icon {
        width: 60px;
        height: 60px;
        display: flex;
        align-items: center;
        justify-content: center;
        border-radius: 16px;
        margin-bottom: 18px;
        transition: all 0.3s ease;
        box-shadow: 0 8px 15px rgba(0,0,0,0.1);
    }
    
    .material-card .course-badge {
        position: absolute;
        top: 16px;
        right: 16px;
        font-size: 0.8rem;
        padding: 6px 12px;
        border-radius: 20px;
        background-color: rgba(106, 17, 203, 0.08);
        color: #6a11cb;
        font-weight: 600;
        letter-spacing: 0.3px;
        backdrop-filter: blur(5px);
        border: 1px solid rgba(255, 255, 255, 0.1);
        box-shadow: 0 4px 6px rgba(0,0,0,0.05);
    }
    
    .material-card .card-text {
        margin-bottom: 15px;
        color: #6c757d;
        display: -webkit-box;
        -webkit-line-clamp: 2;
        -webkit-box-orient: vertical;
        overflow: hidden;
        font-size: 0.9rem;
    }
    
    .material-card .material-meta {
        font-size: 0.8rem;
        color: #6c757d;
        display: flex;
        justify-content: space-between;
    }
    
    .file-type-badge {
        position: absolute;
        top: 15px;
        left: 15px;
        font-size: 0.7rem;
        padding: 4px 8px;
        border-radius: 4px;
        text-transform: uppercase;
        font-weight: 600;
        letter-spacing: 0.5px;
    }
    
    .file-type-pdf {
        background-color: #fde2e2;
        color: #ef4444;
    }
    
    .file-type-doc {
        background-color: #dbeafe;
        color: #2563eb;
    }
    
    .file-type-ppt {
        background-color: #ffedd5;
        color: #f97316;
    }
    
    .file-type-image {
        background-color: #dcfce7;
        color: #16a34a;
    }
    
    .file-type-audio {
        background-color: #fae8ff;
        color: #c026d3;
    }
    
    .file-type-zip {
        background-color: #e0e7ff;
        color: #4f46e5;
    }
    
    .file-type-other {
        background-color: #f3f4f6;
        color: #6b7280;
    }
    
    /* Empty state styling */
    .empty-state {
        background-color: #ffffff;
        border-radius: 20px;
        padding: 3rem;
        box-shadow: 0 10px 30px rgba(0, 0, 0, 0.05);
        margin: 2rem auto;
        max-width: 600px;
        transition: all 0.3s ease;
    }
    
    .empty-state:hover {
        transform: translateY(-5px);
        box-shadow: 0 15px 40px rgba(106, 17, 203, 0.1);
    }
    
    .empty-state h3 {
        font-weight: 700;
        margin-bottom: 1rem;
        font-size: 1.75rem;
    }
    
    .empty-state p {
        font-size: 1.1rem;
        line-height: 1.6;
        margin-bottom: 1.5rem;
    }
    
    .empty-state .btn {
        font-weight: 600;
        border-radius: 50px;
        padding: 0.75rem 2rem;
        box-shadow: 0 5px 15px rgba(106, 17, 203, 0.2);
        transition: all 0.3s ease;
    }
    
    .empty-state .btn:hover {
        transform: translateY(-3px);
        box-shadow: 0 8px 20px rgba(106, 17, 203, 0.3);
    }
    
    .empty-state .svg-container {
        margin-bottom: 2rem;
        filter: drop-shadow(0 10px 15px rgba(106, 17, 203, 0.15));
    }
    
    .pagination .page-link {
        border-radius: 50%;
        margin: 0 5px;
        width: 40px;
        height: 40px;
        display: flex;
        align-items: center;
        justify-content: center;
        color: #6a11cb;
    }
    
    .pagination .page-item.active .page-link {
        background-color: #6a11cb;
        border-color: #6a11cb;
    }
</style>

        <!-- Small Stats Cards Section -->
        <div class="row mb-4" style="max-width: 25%;">
            <div class="col-md-4 mb-3">
                <div class="stats-card">
                    <div class="d-flex align-items-center">
                        <div class="stats-icon me-3" style="background: rgba(66, 97, 238, 0.1) !important; padding: 12px; border-radius: 8px;">
                            <i class='bx bx-folder' style="font-size: 20px !important; color: #4261ee !important;"></i>
                        </div>
                        <div>
                            <div class="stats-number"><?php echo $total_materials; ?></div>
                            <div class="stats-label">Total Materials</div>
                        </div>
                    </div>
                </div>
            </div>


        </div>

            <!-- Library Management Section -->
            <div class="library-management-section">
                <!-- Section Header with Title -->
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2 class="section-title d-flex align-items-center">
                        <i class='bx bx-library me-2' style="color: var(--primary-color); font-size: 24px;"></i>
                        Course Materials Library
                    </h2>
                    <a href="course-materials.php" class="btn btn-primary rounded-pill">
                        <i class='bx bx-plus me-2' style="font-size: 16px;"></i>
                        Add New Material
                    </a>
                </div>

<div class="container-fluid">
    <!-- Search and Filter Controls -->
    <div class="row mb-4">
        <div class="col-md-6 col-lg-8 mb-3 mb-md-0">
            <div class="category-tabs">
                <a href="manage-course-materials.php" class="category-tab <?= $current_category === 'all' ? 'active' : '' ?>">
                    <i class='bx bx-collection'></i> All Materials
                </a>
                
                <?php 
                // Define category icons
                $category_icons = [
                    'pdf' => 'bxs-file-pdf',
                    'doc' => 'bxs-file-doc',
                    'docx' => 'bxs-file-doc',
                    'ppt' => 'bxs-slideshow',
                    'pptx' => 'bxs-slideshow',
                    'jpg' => 'bxs-image',
                    'jpeg' => 'bxs-image',
                    'png' => 'bxs-image',
                    'gif' => 'bxs-image',
                    'zip' => 'bxs-file-archive',
                    'rar' => 'bxs-file-archive',
                    'other' => 'bxs-file',
                ];
                
                $category_labels = [
                    'pdf' => 'PDF Documents',
                    'doc' => 'Word Documents',
                    'docx' => 'Word Documents',
                    'ppt' => 'Presentations',
                    'pptx' => 'Presentations',
                    'jpg' => 'Images',
                    'jpeg' => 'Images',
                    'png' => 'Images',
                    'gif' => 'Images',
                    'zip' => 'Archives',
                    'rar' => 'Archives',
                ];
                
                foreach ($categories as $category): 
                    $icon = isset($category_icons[$category]) ? $category_icons[$category] : 'bxs-file';
                    $label = isset($category_labels[$category]) ? $category_labels[$category] : ucfirst($category);
                ?>
                <a href="manage-course-materials.php?category=<?= urlencode($category) ?>" class="category-tab <?= $current_category === $category ? 'active' : '' ?>">
                    <i class='bx <?= $icon ?>'></i> <?= htmlspecialchars($label) ?>
                </a>
                <?php endforeach; ?>
                
                <a href="course-materials.php" class="category-tab" style="background: linear-gradient(135deg, #6a11cb 0%, #8c67f4 100%); color: white; font-weight: bold;">
                    <i class="bx bx-plus-circle"></i> Add New
                </a>
            </div>
        </div>
        
        <div class="col-md-6 col-lg-4">
            <form action="manage-course-materials.php" method="get" class="d-flex">
                <?php if ($current_category && $current_category !== 'all'): ?>
                <input type="hidden" name="category" value="<?= htmlspecialchars($current_category) ?>">
                <?php endif; ?>
                
                <div class="input-group">
                    <input type="text" class="form-control" name="search" placeholder="Search materials..." value="<?= htmlspecialchars($search_term) ?>" aria-label="Search materials" oninput="this.form.submit()" autocomplete="off">
                    <button class="btn btn-primary" type="submit">
                        <i class="bx bx-search"></i>
                    </button>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Materials Display -->
    <div class="row g-4">
        <?php if (empty($materials)): ?>
        <div class="col-12">
            <div class="empty-state text-center py-5">
                <div class="svg-container mb-4" style="max-width: 250px; margin: 0 auto;">
                    <!-- Inline SVG for empty folder to prevent 404 errors -->
                    <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512" style="width: 100%; height: auto;">
                        <style>
                            @keyframes floatAnimation {
                                0% { transform: translateY(0); }
                                50% { transform: translateY(-10px); }
                                100% { transform: translateY(0); }
                            }
                            .folder-main { fill: #8c67f4; }
                            .folder-back { fill: #6a11cb; }
                            .folder-shadow { fill: rgba(0,0,0,0.1); }
                            .document { fill: #e2e8f0; stroke: #cbd5e1; stroke-width: 2; }
                            .document-lines { stroke: #94a3b8; stroke-width: 2; }
                            .floating-element { animation: floatAnimation 3s ease-in-out infinite; }
                        </style>
                        <g class="floating-element" style="animation-delay: 0s;">
                            <path class="folder-shadow" d="M40 120v260c0 22.1 17.9 40 40 40h352c22.1 0 40-17.9 40-40V160c0-22.1-17.9-40-40-40H240l-40-40H80c-22.1 0-40 17.9-40 40z" opacity="0.2"/>
                            <path class="folder-back" d="M40 100v260c0 22.1 17.9 40 40 40h352c22.1 0 40-17.9 40-40V140c0-22.1-17.9-40-40-40H240l-40-40H80c-22.1 0-40 17.9-40 40z"/>
                            <path class="folder-main" d="M472 140H40v220c0 22.1 17.9 40 40 40h352c22.1 0 40-17.9 40-40V140z"/>
                        </g>
                        <g class="floating-element" style="animation-delay: 0.5s;">
                            <rect class="document" x="200" y="180" width="120" height="150" rx="8"/>
                            <line class="document-lines" x1="220" y1="210" x2="300" y2="210"/>
                            <line class="document-lines" x1="220" y1="240" x2="300" y2="240"/>
                            <line class="document-lines" x1="220" y1="270" x2="270" y2="270"/>
                        </g>
                    </svg>
                </div>
                <h3 class="mt-3" style="color: #6a11cb;">No materials found</h3>
                <p class="text-muted px-4 mb-4">You haven't added any materials yet or none match your current filter.</p>
                <a href="course-materials.php" class="btn btn-primary px-4 py-2">
                    <i class="bx bx-plus-circle me-2"></i> Upload New Material
                </a>
            </div>
        </div>
        <?php else: ?>
            <?php foreach ($materials as $material): ?>
            <div class="col-md-6 col-lg-4 col-xl-3">
                <div class="card material-card position-relative">
                    <?php 
                    // Determine file type and icon/color
                    $file_extension = pathinfo($material['file_path'], PATHINFO_EXTENSION);
                    $file_type_class = 'file-type-other';
                    $icon_bg_color = '#f3f4f6';
                    $icon_color = '#6b7280';
                    $icon = 'bx bx-file';
                    
                    switch(strtolower($file_extension)) {
                        case 'pdf':
                            $file_type_class = 'file-type-pdf';
                            $icon_bg_color = 'rgba(239, 68, 68, 0.1)';
                            $icon_color = '#ef4444';
                            $icon = 'bx bxs-file-pdf';
                            break;
                        case 'doc':
                        case 'docx':
                            $file_type_class = 'file-type-doc';
                            $icon_bg_color = 'rgba(37, 99, 235, 0.1)';
                            $icon_color = '#2563eb';
                            $icon = 'bx bxs-file-doc';
                            break;
                        case 'xls':
                        case 'xlsx':
                            $icon_bg_color = 'rgba(22, 163, 74, 0.1)';
                            $icon_color = '#16a34a';
                            $icon = 'bx bxs-file-excel';
                            break;
                        case 'ppt':
                        case 'pptx':
                            $file_type_class = 'file-type-ppt';
                            $icon_bg_color = 'rgba(249, 115, 22, 0.1)';
                            $icon_color = '#f97316';
                            $icon = 'bx bxs-file-ppt';
                            break;
                        case 'jpg':
                        case 'jpeg':
                        case 'png':
                        case 'gif':
                            $file_type_class = 'file-type-image';
                            $icon_bg_color = 'rgba(22, 163, 74, 0.1)';
                            $icon_color = '#16a34a';
                            $icon = 'bx bxs-image';
                            break;
                        case 'mp3':
                        case 'wav':
                            $file_type_class = 'file-type-audio';
                            $icon_bg_color = 'rgba(192, 38, 211, 0.1)';
                            $icon_color = '#c026d3';
                            $icon = 'bx bxs-music';
                            break;
                        case 'mp4':
                        case 'mov':
                        case 'avi':
                            $file_type_class = 'file-type-video';
                            $icon_bg_color = 'rgba(147, 51, 234, 0.1)';
                            $icon_color = '#9333ea';
                            $icon = 'bx bxs-video';
                            break;
                    }
                    ?>
                    <span class="file-type-badge <?= $file_type_class ?>"><?= strtoupper($file_extension) ?></span>
                    
                    <?php if (!empty($material['course_code'])): ?>
                    <span class="course-badge"><?= htmlspecialchars($material['course_code']) ?></span>
                    <?php endif; ?>
                    
                    <div class="card-body">
                        <div class="material-icon" style="background-color: <?= $icon_bg_color ?>; color: <?= $icon_color ?>;">
                            <i class="<?= $icon ?>" style="font-size: 24px;"></i>
                        </div>
                        
                        <h5 class="card-title"><?= htmlspecialchars($material['title']) ?></h5>
                        <p class="card-text"><?= htmlspecialchars($material['description']) ?></p>
                        
                        <div class="material-meta mt-auto">
                            <span>
                                <i class="bx bx-calendar me-1"></i>
                                <?= date('M d, Y', strtotime($material['date_added'])) ?>
                            </span>
                            <span>
                                <i class="bx bx-download me-1"></i>
                                <?= $material['download_count'] ?? 0 ?>
                            </span>
                        </div>
                        
                        <div class="d-flex mt-3">
                            <a href="<?= htmlspecialchars($material['file_path']) ?>" class="btn btn-sm btn-outline-primary me-2" target="_blank">
                                <i class="bx bx-show me-1"></i> View
                            </a>
                            <a href="course-materials.php?edit=<?= $material['material_id'] ?>" class="btn btn-sm btn-outline-secondary me-2">
                                <i class="bx bx-edit me-1"></i> Edit
                            </a>
                            <button class="btn btn-sm btn-outline-danger" onclick="confirmDelete(<?= $material['material_id'] ?>)">
                                <i class="bx bx-trash"></i>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
            <?php endforeach; ?>
        <?php endif; ?>
    </div>
    
    <!-- Pagination -->
    <?php if ($total_pages > 1): ?>
    <div class="row mt-5">
        <div class="col-12">
            <nav aria-label="Page navigation">
                <ul class="pagination justify-content-center">
                    <li class="page-item <?= $page <= 1 ? 'disabled' : '' ?>">
                        <a class="page-link" href="?page=<?= $page-1 ?>&category=<?= urlencode($current_category) ?>&search=<?= urlencode($search_term) ?>" aria-label="Previous">
                            <span aria-hidden="true">&laquo;</span>
                        </a>
                    </li>
                    
                    <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                    <li class="page-item <?= $i == $page ? 'active' : '' ?>">
                        <a class="page-link" href="?page=<?= $i ?>&category=<?= urlencode($current_category) ?>&search=<?= urlencode($search_term) ?>">
                            <?= $i ?>
                        </a>
                    </li>
                    <?php endfor; ?>
                    
                    <li class="page-item <?= $page >= $total_pages ? 'disabled' : '' ?>">
                        <a class="page-link" href="?page=<?= $page+1 ?>&category=<?= urlencode($current_category) ?>&search=<?= urlencode($search_term) ?>" aria-label="Next">
                            <span aria-hidden="true">&raquo;</span>
                        </a>
                    </li>
                </ul>
            </nav>
        </div>
    </div>
    <?php endif; ?>
</div>

<!-- Delete Confirmation Modal -->
<div class="modal fade" id="deleteModal" tabindex="-1" aria-labelledby="deleteModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="deleteModalLabel">Confirm Deletion</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                Are you sure you want to delete this material? This action cannot be undone.
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <form id="deleteForm" action="course-materials.php" method="post">
                    <input type="hidden" name="action" value="delete">
                    <input type="hidden" name="material_id" id="deleteMaterialId">
                    <button type="submit" class="btn btn-danger">Delete</button>
                </form>
            </div>
        </div>
    </div>
</div>

<script>
    function confirmDelete(materialId) {
        document.getElementById('deleteMaterialId').value = materialId;
        const deleteModal = new bootstrap.Modal(document.getElementById('deleteModal'));
        deleteModal.show();
    }
</script>

        </div>
    </div>
</div>

<style>
/* Targeted fixes for manage-course-materials page only */
body {
    overflow-x: hidden;
}

.page-content {
    padding: 20px 30px;
    max-width: 100%;
    overflow-x: hidden;
}

.container-fluid {
    padding-left: 15px;
    padding-right: 15px;
}

.row {
    margin-left: 0;
    margin-right: 0;
}
</style>

<?php require_once('components/teacher_layout_end.php'); ?>
