<?php
/**
 * Teacher Notifications - Panadite Academy
 * Real-time notification system with database integration
 */

// Set current page for navigation highlighting
$current_page = 'notifications.php';

// Include the centralized teacher layout (handles all authentication, session, DB connection)
require_once __DIR__ . '/components/teacher_layout.php';

// Include notification service
require_once __DIR__ . '/../includes/NotificationService.php';

// Get teacher ID from session
$teacher_id = $_SESSION['user_id'] ?? null;

// Session fallback
if (!$teacher_id || !is_numeric($teacher_id)) {
    $teacher_id = 1; // Default to Teacher 1 where sample data exists
    error_log("Notifications: Session user_id not set, defaulting to Teacher 1");
}

// Initialize notification service
$notificationService = new NotificationService($conn);

// Handle AJAX requests for marking notifications as read
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    header('Content-Type: application/json');
    
    if ($_POST['action'] === 'mark_read' && isset($_POST['notification_id'])) {
        $notification_id = intval($_POST['notification_id']);
        
        // Mark notification as read using notification service
        $success = $notificationService->markAsRead($notification_id, $teacher_id);
        echo json_encode(['success' => $success]);
        exit;
    }
    
    if ($_POST['action'] === 'mark_all_read') {
        // Mark all notifications as read for this teacher
        $success = $notificationService->markAllAsRead($teacher_id);
        echo json_encode(['success' => $success]);
        exit;
    }
    
    if ($_POST['action'] === 'get_new_notifications') {
        // Real-time polling endpoint - get latest notifications
        $notifications = $notificationService->getTeacherNotifications($teacher_id, 10);
        $unread_count = $notificationService->getUnreadCount($teacher_id);
        
        echo json_encode([
            'success' => true,
            'notifications' => $notifications,
            'unread_count' => $unread_count
        ]);
        exit;
    }
}

// Get real notifications from database
$notifications = $notificationService->getTeacherNotifications($teacher_id, 20);
$unread_count = $notificationService->getUnreadCount($teacher_id);

// Convert database notifications to display format
$formatted_notifications = [];
foreach ($notifications as $notification) {
    // Map notification types to icons and colors
    $type_config = [
        'student_enrollment' => ['icon' => 'bx-user-plus', 'color' => 'success', 'action_url' => 'courses.php'],
        'course_completion' => ['icon' => 'bx-badge-check', 'color' => 'primary', 'action_url' => 'analytics.php'],
        'withdrawal_update' => ['icon' => 'bx-money', 'color' => 'warning', 'action_url' => 'withdraw-earnings.php'],
        'referral_reward' => ['icon' => 'bx-group', 'color' => 'success', 'action_url' => 'withdraw-earnings.php'],
        'course_creation' => ['icon' => 'bx-book-add', 'color' => 'primary', 'action_url' => 'courses.php'],
        'video_upload' => ['icon' => 'bx-video', 'color' => 'info', 'action_url' => 'courses.php'],
        'enrollment' => ['icon' => 'bx-user-plus', 'color' => 'success', 'action_url' => 'courses.php'],
        'referral' => ['icon' => 'bx-group', 'color' => 'success', 'action_url' => 'withdraw-earnings.php']
    ];
    
    $config = $type_config[$notification['type']] ?? ['icon' => 'bx-info-circle', 'color' => 'info', 'action_url' => null];
    
    $formatted_notifications[] = [
        'id' => $notification['notification_id'],
        'type' => $notification['type'],
        'icon' => $config['icon'],
        'color' => $config['color'],
        'title' => $notification['title'],
        'message' => $notification['content'],
        'time' => $notification['time_ago'],
        'is_read' => (bool)$notification['is_read'],
        'action_url' => $config['action_url'],
        'created_at' => $notification['created_at']
    ];
}

$notifications = $formatted_notifications;
?>

<style>
    /* Clean Notifications Interface */
    .notifications-container {
        background: #ffffff;
        border-radius: 12px;
        border: 1px solid #e5e7eb;
        box-shadow: 0 1px 3px rgba(0, 0, 0, 0.05);
        overflow: hidden;
        margin: 0 20px;
    }

    .page-header {
        padding: 1.5rem 2rem;
        border-bottom: 1px solid #f3f4f6;
        background: #ffffff;
    }

    .page-header h2 {
        margin: 0;
        font-weight: 600;
        font-size: 1.5rem;
        color: #111827;
        display: flex;
        align-items: center;
        gap: 10px;
    }

    .page-header h2 i {
        font-size: 1.25rem;
        color: #6b7280;
    }

    .notifications-stats {
        margin-top: 0.75rem;
        display: flex;
        align-items: center;
        gap: 16px;
    }

    .stat-item {
        display: flex;
        align-items: center;
        gap: 6px;
        font-size: 0.8rem;
        color: #6b7280;
    }

    .stat-item i {
        font-size: 0.875rem;
    }

    .notifications-controls {
        background: #f9fafb;
        padding: 1rem 2rem;
        border-bottom: 1px solid #e5e7eb;
        display: flex;
        align-items: center;
        justify-content: space-between;
    }

    .notifications-filter {
        display: flex;
        gap: 8px;
        align-items: center;
    }

    .filter-btn {
        padding: 6px 12px;
        border: 1px solid #d1d5db;
        background: white;
        border-radius: 6px;
        font-size: 0.8125rem;
        color: #4b5563;
        cursor: pointer;
        transition: all 0.2s ease;
        font-weight: 500;
    }

    .filter-btn:hover {
        border-color: #9ca3af;
        background: #f9fafb;
    }

    .filter-btn.active {
        background: #3b82f6;
        color: white;
        border-color: #3b82f6;
    }

    .mark-all-read-btn {
        background: #10b981;
        color: white;
        border: none;
        padding: 6px 16px;
        border-radius: 6px;
        font-size: 0.8125rem;
        font-weight: 500;
        cursor: pointer;
        transition: all 0.2s ease;
        display: flex;
        align-items: center;
        gap: 6px;
    }

    .mark-all-read-btn:hover {
        background: #059669;
    }

    .notifications-list {
        max-height: 600px;
        overflow-y: auto;
    }

    .notification-item {
        padding: 1rem 1.5rem;
        border-bottom: 1px solid #f3f4f6;
        display: flex;
        align-items: flex-start;
        gap: 12px;
        transition: all 0.2s ease;
        cursor: pointer;
        position: relative;
    }

    .notification-item:hover {
        background: #f9fafb;
    }

    .notification-item.unread {
        background: rgba(59, 130, 246, 0.02);
        border-left: 3px solid #3b82f6;
    }

    .notification-icon {
        width: 36px;
        height: 36px;
        border-radius: 8px;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 16px;
        color: white;
        flex-shrink: 0;
    }

    .notification-icon.success {
        background: linear-gradient(135deg, #10b981, #059669);
    }

    .notification-icon.primary {
        background: linear-gradient(135deg, #3b82f6, #2563eb);
    }

    .notification-icon.info {
        background: linear-gradient(135deg, #0ea5e9, #0284c7);
    }

    .notification-icon.warning {
        background: linear-gradient(135deg, #f59e0b, #d97706);
    }

    .notification-icon.danger {
        background: linear-gradient(135deg, #ef4444, #dc2626);
    }

    .notification-content {
        flex: 1;
        min-width: 0;
    }

    .notification-title {
        font-weight: 600;
        color: #111827;
        margin-bottom: 2px;
        font-size: 0.875rem;
    }

    .notification-message {
        color: #6b7280;
        font-size: 0.8125rem;
        line-height: 1.4;
        margin-bottom: 6px;
    }

    .notification-meta {
        display: flex;
        align-items: center;
        justify-content: space-between;
        margin-top: 6px;
    }

    .notification-time {
        color: #9ca3af;
        font-size: 0.75rem;
        display: flex;
        align-items: center;
        gap: 3px;
    }

    .notification-time i {
        font-size: 0.75rem;
    }

    .notification-actions {
        display: flex;
        gap: 6px;
    }

    .action-btn {
        padding: 3px 8px;
        border: 1px solid #d1d5db;
        background: white;
        border-radius: 4px;
        font-size: 0.6875rem;
        color: #6b7280;
        cursor: pointer;
        transition: all 0.2s ease;
    }

    .action-btn:hover {
        background: #f9fafb;
        border-color: #9ca3af;
    }

    .unread-indicator {
        width: 6px;
        height: 6px;
        background: #3b82f6;
        border-radius: 50%;
        flex-shrink: 0;
        margin-top: 4px;
    }

    .empty-state {
        text-align: center;
        padding: 4rem 2rem;
        color: #64748b;
    }

    .empty-state i {
        font-size: 4rem;
        color: #cbd5e1;
        margin-bottom: 1rem;
    }

    /* Responsive Design */
    @media (max-width: 768px) {
        .notifications-header {
            padding: 1.5rem;
        }
        
        .notifications-controls {
            flex-direction: column;
            gap: 16px;
            align-items: stretch;
        }
        
        .notifications-filter {
            flex-wrap: wrap;
        }
        
        .notification-item {
            padding: 1rem;
        }
        
        .notification-icon {
            width: 40px;
            height: 40px;
            font-size: 18px;
        }
    }
</style>

<!-- Teacher Notifications Interface -->
<div class="container-fluid py-4">
    <div class="row">
        <div class="col-12">
            <div class="notifications-container">
                <!-- Clean Header -->
                <div class="page-header">
                    <h2>
                        <i class='bx bx-bell'></i>
                        Notifications
                    </h2>
                    <div class="notifications-stats">
                        <div class="stat-item">
                            <i class='bx bx-bell-ring'></i>
                            <span><?= $unread_count ?> unread</span>
                        </div>
                        <div class="stat-item">
                            <i class='bx bx-list-ul'></i>
                            <span><?= count($notifications) ?> total</span>
                        </div>
                    </div>
                </div>

                <!-- Controls -->
                <div class="notifications-controls">
                    <div class="notifications-filter">
                        <button class="filter-btn active" data-filter="all">All</button>
                        <button class="filter-btn" data-filter="unread">Unread</button>
                    </div>
                    
                    <?php if ($unread_count > 0): ?>
                    <button class="mark-all-read-btn" onclick="markAllAsRead()">
                        <i class='bx bx-check-double'></i>
                        Mark All as Read
                    </button>
                    <?php endif; ?>
                </div>

                <!-- Notifications List -->
                <div class="notifications-list" id="notifications-list">
                    <?php if (empty($notifications)): ?>
                        <div class="empty-state">
                            <i class='bx bx-bell-off'></i>
                            <h5>No notifications yet</h5>
                            <p>When you have new notifications, they'll appear here.</p>
                        </div>
                    <?php else: ?>
                        <?php foreach ($notifications as $notification): ?>
                            <div class="notification-item <?= !$notification['is_read'] ? 'unread' : '' ?>" 
                                 data-notification-id="<?= $notification['id'] ?>"
                                 data-type="<?= $notification['type'] ?>"
                                 onclick="handleNotificationClick(<?= $notification['id'] ?>, '<?= $notification['action_url'] ?? '' ?>')">
                                
                                <div class="notification-icon <?= $notification['color'] ?>">
                                    <i class='bx <?= $notification['icon'] ?>'></i>
                                </div>
                                
                                <div class="notification-content">
                                    <div class="notification-title"><?= htmlspecialchars($notification['title']) ?></div>
                                    <div class="notification-message"><?= htmlspecialchars($notification['message']) ?></div>
                                    
                                    <div class="notification-meta">
                                        <div class="notification-time">
                                            <i class='bx bx-time-five'></i>
                                            <?= $notification['time'] ?>
                                        </div>
                                        
                                        <div class="notification-actions">
                                            <?php if (!$notification['is_read']): ?>
                                                <button class="action-btn" onclick="markAsRead(<?= $notification['id'] ?>); event.stopPropagation();">
                                                    Mark as read
                                                </button>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                                
                                <?php if (!$notification['is_read']): ?>
                                    <div class="unread-indicator"></div>
                                <?php endif; ?>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Filter functionality
    const filterBtns = document.querySelectorAll('.filter-btn');
    const notificationItems = document.querySelectorAll('.notification-item');
    
    filterBtns.forEach(btn => {
        btn.addEventListener('click', function() {
            // Update active filter
            filterBtns.forEach(b => b.classList.remove('active'));
            this.classList.add('active');
            
            const filter = this.getAttribute('data-filter');
            
            // Filter notifications
            notificationItems.forEach(item => {
                const type = item.getAttribute('data-type');
                const isUnread = item.classList.contains('unread');
                
                let shouldShow = false;
                
                if (filter === 'all') {
                    shouldShow = true;
                } else if (filter === 'unread') {
                    shouldShow = isUnread;
                } else {
                    shouldShow = type === filter;
                }
                
                item.style.display = shouldShow ? 'flex' : 'none';
            });
        });
    });
    
    // Real-time notification polling
    let pollInterval;
    
    function startRealTimePolling() {
        // Poll every 15 seconds for new notifications
        pollInterval = setInterval(checkForNewNotifications, 15000);
        console.log('🔔 Real-time notification polling started');
    }
    
    function checkForNewNotifications() {
        fetch('notifications.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: 'action=get_new_notifications'
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                updateNavbarBadge(data.unread_count);
            }
        })
        .catch(error => {
            console.error('Error polling notifications:', error);
        });
    }
    
    function updateNavbarBadge(unreadCount) {
        // Update notification badge in navbar if it exists
        const navbarBadge = document.querySelector('.nav-icon-btn .notification-badge');
        if (navbarBadge) {
            navbarBadge.textContent = unreadCount;
            navbarBadge.style.display = unreadCount > 0 ? 'inline-block' : 'none';
        }
    }
    
    // Page visibility API - pause polling when tab is not active
    document.addEventListener('visibilitychange', function() {
        if (document.hidden) {
            // Page is hidden, reduce polling frequency
            clearInterval(pollInterval);
            pollInterval = setInterval(checkForNewNotifications, 60000); // Every 60 seconds
        } else {
            // Page is visible, increase polling frequency
            clearInterval(pollInterval);
            pollInterval = setInterval(checkForNewNotifications, 15000); // Every 15 seconds
            // Check immediately when page becomes visible
            checkForNewNotifications();
        }
    });
    
    // Start real-time polling
    startRealTimePolling();
});

function handleNotificationClick(notificationId, actionUrl) {
    // Mark as read when clicked
    if (!document.querySelector(`[data-notification-id="${notificationId}"]`).classList.contains('read')) {
        markAsRead(notificationId);
    }
    
    // Navigate to action URL if provided
    if (actionUrl && actionUrl.trim() !== '') {
        window.location.href = actionUrl;
    }
}

function markAsRead(notificationId) {
    fetch('notifications.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: `action=mark_read&notification_id=${notificationId}`
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            const item = document.querySelector(`[data-notification-id="${notificationId}"]`);
            if (item) {
                item.classList.remove('unread');
                const indicator = item.querySelector('.unread-indicator');
                if (indicator) {
                    indicator.remove();
                }
                const markBtn = item.querySelector('.action-btn');
                if (markBtn) {
                    markBtn.remove();
                }
            }
            
            // Update unread count in header
            updateUnreadCount();
        }
    })
    .catch(error => {
        console.error('Error marking notification as read:', error);
    });
}

function markAllAsRead() {
    fetch('notifications.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: 'action=mark_all_read'
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            // Remove unread styling from all notifications
            document.querySelectorAll('.notification-item.unread').forEach(item => {
                item.classList.remove('unread');
                const indicator = item.querySelector('.unread-indicator');
                if (indicator) {
                    indicator.remove();
                }
                const markBtn = item.querySelector('.action-btn');
                if (markBtn) {
                    markBtn.remove();
                }
            });
            
            // Hide mark all as read button
            const markAllBtn = document.querySelector('.mark-all-read-btn');
            if (markAllBtn) {
                markAllBtn.style.display = 'none';
            }
            
            // Update unread count
            updateUnreadCount();
        }
    })
    .catch(error => {
        console.error('Error marking all notifications as read:', error);
    });
}

function updateUnreadCount() {
    const unreadItems = document.querySelectorAll('.notification-item.unread');
    const countElement = document.querySelector('.notifications-stats .stat-item span');
    if (countElement) {
        countElement.textContent = `${unreadItems.length} unread`;
    }
}
</script>

<?php
// Include the footer component
require_once 'components/footer.php';
?>
