<?php
// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Set upload limits at runtime
ini_set('upload_max_filesize', '20M');
ini_set('post_max_size', '40M');
ini_set('memory_limit', '128M');
ini_set('max_execution_time', '300');
ini_set('max_input_time', '300');

// Use centralized session management instead of direct session_start
require_once "../includes/session_start.php";
require_once "../config/db_connect.php";
require_once "../auth/functions.php";

// Include fast email service for course creation notifications
require_once __DIR__ . '/../config/fast_email_service.php';

// Bridge database connections - Email service needs PDO, but teacher uses MySQLi
if (!isset($pdo)) {
    try {
        // Create PDO connection using the same credentials as MySQLi
        $pdo = new PDO("mysql:host=" . DB_SERVER . ";dbname=" . DB_NAME, DB_USERNAME, DB_PASSWORD);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
        error_log("PDO connection created successfully for email service");
    } catch (Exception $e) {
        error_log("Failed to create PDO connection for email service: " . $e->getMessage());
        $pdo = null;
    }
}

/**
 * Send course creation notification to teacher
 */
function sendCourseCreationEmail($course_id) {
    global $conn;
    
    try {
        // Get course and teacher details
        $stmt = $conn->prepare("
            SELECT c.title, c.description, c.price, c.created_at, 
                   u.first_name, u.last_name, u.email, u.user_id,
                   cat.category_name
            FROM courses c 
            JOIN users u ON c.teacher_id = u.user_id 
            LEFT JOIN categories cat ON c.category_id = cat.category_id 
            WHERE c.course_id = ?
        ");
        $stmt->bind_param("i", $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $course = $result->fetch_assoc();
        
        if (!$course) {
            error_log("Course not found for creation email: $course_id");
            return false;
        }
        
        $teacher_name = $course['first_name'] . ' ' . $course['last_name'];
        $teacher_email = $course['email'];
        $course_title = $course['title'];
        $category = $course['category_name'] ?? 'General';
        $price = floatval($course['price']);
        $created_date = date('M j, Y', strtotime($course['created_at']));
        
        // Create email content
        $subject = "🎓 Course Created: \"$course_title\" - Awaiting Approval";
        $message = "
        <div style='max-width:600px; margin:0 auto; font-family:Arial,sans-serif;'>
            <div style='background:#667eea; color:white; padding:20px; text-align:center;'>
                <h1>🎓 Course Created Successfully!</h1>
            </div>
            <div style='padding:20px; background:white;'>
                <h2>Hi " . htmlspecialchars($teacher_name) . "! 🎉</h2>
                <p>Your course has been successfully created and is awaiting admin approval.</p>
                <div style='background:#f8f9fa; padding:15px; border-radius:5px; margin:20px 0;'>
                    <h3>Course Details:</h3>
                    <p><strong>Title:</strong> " . htmlspecialchars($course_title) . "</p>
                    <p><strong>Category:</strong> " . htmlspecialchars($category) . "</p>
                    <p><strong>Price:</strong> R" . number_format($price, 2) . "</p>
                    <p><strong>Created:</strong> " . $created_date . "</p>
                </div>
                <div style='background:#fff3cd; padding:15px; border-radius:5px; margin:20px 0;'>
                    <h4>⏱️ What's Next?</h4>
                    <p>Your course will be reviewed by our admin team within 1-2 business days.</p>
                    <p>You'll receive an email notification when your course is approved and goes live.</p>
                </div>
                <p style='text-align:center;'>
                    <a href='https://panaditeacademy.com/teacher/courses.php' 
                       style='background:#667eea; color:white; text-decoration:none; padding:15px 25px; border-radius:8px; font-weight:bold; display:inline-block;'>
                        View Dashboard
                    </a>
                </p>
            </div>
        </div>";
        
        // Try direct email sending
        if (function_exists('sendEmailNow')) {
            $result = sendEmailNow($teacher_email, $subject, $message);
            if ($result) {
                return true;
            }
        }
        
        // Fallback to native PHP mail
        $headers = "MIME-Version: 1.0" . "\r\n";
        $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
        $headers .= "From: noreply@panaditeacademy.com" . "\r\n";
        
        return mail($teacher_email, $subject, $message, $headers);
        
    } catch (Exception $e) {
        error_log("Error sending course creation email: " . $e->getMessage());
        return false;
    }
}

// Helper function to translate file upload error codes into readable messages
function getFileUploadErrorMessage($errorCode) {
    switch ($errorCode) {
        case UPLOAD_ERR_INI_SIZE:
            return "The uploaded file exceeds the upload_max_filesize directive in php.ini (Error: $errorCode)";
        case UPLOAD_ERR_FORM_SIZE:
            return "The uploaded file exceeds the MAX_FILE_SIZE directive in the HTML form (Error: $errorCode)";
        case UPLOAD_ERR_PARTIAL:
            return "The uploaded file was only partially uploaded (Error: $errorCode)";
        case UPLOAD_ERR_NO_FILE:
            return "No file was uploaded (Error: $errorCode)";
        case UPLOAD_ERR_NO_TMP_DIR:
            return "Missing a temporary folder (Error: $errorCode)";
        case UPLOAD_ERR_CANT_WRITE:
            return "Failed to write file to disk (Error: $errorCode)";
        case UPLOAD_ERR_EXTENSION:
            return "A PHP extension stopped the file upload (Error: $errorCode)";
        default:
            return "Unknown upload error (Error: $errorCode)";
    }
}

// Create debug log file
function logDebug($message, $data = null) {
    $logFile = __DIR__ . '/course_creation_debug.log';
    $timestamp = date('Y-m-d H:i:s');
    $logMessage = "[$timestamp] $message";
    
    if ($data !== null) {
        $logMessage .= " - " . print_r($data, true);
    }
    
    file_put_contents($logFile, $logMessage . "\n", FILE_APPEND);
}

logDebug("------ New Form Submission ------");

// Set JSON content type header
header('Content-Type: application/json');

// Check if user is logged in and is a teacher
if (!isset($_SESSION['user_id']) || $_SESSION['user_role'] !== 'teacher') {
    http_response_code(401);
    $response = ['success' => false, 'message' => 'Unauthorized access', 'errors' => ['You must be logged in as a teacher to perform this action.']];
    logDebug("Auth error", $response);
    echo json_encode($response);
    exit;
}

$teacher_id = $_SESSION['user_id'];
$response = ['success' => false, 'message' => '', 'errors' => []];

// Check teacher identity verification status
try {
    $identity_check_stmt = $conn->prepare("SELECT identity_verification_status FROM users WHERE user_id = ? AND user_role = 'teacher'");
    $identity_check_stmt->bind_param("i", $teacher_id);
    $identity_check_stmt->execute();
    $identity_result = $identity_check_stmt->get_result();
    $teacher_identity = $identity_result->fetch_assoc();
    
    if (!$teacher_identity || $teacher_identity['identity_verification_status'] !== 'verified') {
        http_response_code(403);
        $verification_status = $teacher_identity['identity_verification_status'] ?? 'not_submitted';
        
        $status_messages = [
            'not_submitted' => 'You must upload and verify your identity documents before creating courses. Please visit your profile page to upload required documents.',
            'pending' => 'Your identity verification is currently pending admin review. You cannot create courses until your identity is verified.',
            'rejected' => 'Your identity verification was rejected. Please visit your profile page to upload new documents for review.'
        ];
        
        $response = [
            'success' => false, 
            'message' => 'Identity Verification Required', 
            'errors' => [$status_messages[$verification_status]],
            'verification_status' => $verification_status
        ];
        logDebug("Identity verification failed for teacher ID: $teacher_id", $response);
        echo json_encode($response);
        exit;
    }
    
    logDebug("Identity verification passed for teacher ID: $teacher_id");
} catch (Exception $e) {
    http_response_code(500);
    $response = [
        'success' => false, 
        'message' => 'Verification Check Failed', 
        'errors' => ['Unable to verify your identity status. Please try again later.']
    ];
    logDebug("Identity verification check error: " . $e->getMessage());
    echo json_encode($response);
    exit;
}

// Validate form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Validate required fields
    $required_fields = [
        'title' => 'Course Title',
        'description' => 'Course Description',
        'category' => 'Category',
        'level' => 'Difficulty Level',
        'price' => 'Price',
        'duration' => 'Duration',
        'assignment_weight' => 'Assignment Weight',
        'quiz_weight' => 'Quiz Weight'
    ];

    $errors = [];
    $data = [];

    // Check for required fields with special handling for numeric fields that can be 0
    foreach ($required_fields as $field => $label) {
        $value = trim($_POST[$field] ?? '');
        
        // Special handling for assignment_weight and quiz_weight - they can be "0"
        if ($field === 'assignment_weight' || $field === 'quiz_weight') {
            if ($value === '' || $value === null) {
                $errors[] = "$label is required.";
            } else {
                $data[$field] = $value;
            }
        } else {
            // Standard validation for other fields
            if (empty($value)) {
                $errors[] = "$label is required.";
            } else {
                $data[$field] = $value;
            }
        }
    }

    // Validate price is numeric and positive
    if (!empty($data['price']) && (!is_numeric($data['price']) || $data['price'] < 0)) {
        $errors[] = "Price must be a positive number.";
    }

    // Validate duration is a positive integer
    if (!empty($data['duration']) && (!is_numeric($data['duration']) || $data['duration'] < 1)) {
        $errors[] = "Duration must be a positive number.";
    }

    // Validate grade weighting
    if (!empty($data['assignment_weight']) && (!is_numeric($data['assignment_weight']) || $data['assignment_weight'] < 0 || $data['assignment_weight'] > 100)) {
        $errors[] = "Assignment weight must be a number between 0 and 100.";
    }

    if (!empty($data['quiz_weight']) && (!is_numeric($data['quiz_weight']) || $data['quiz_weight'] < 0 || $data['quiz_weight'] > 100)) {
        $errors[] = "Quiz weight must be a number between 0 and 100.";
    }

    // Validate that assignment weight + quiz weight equals 100%
    if (!empty($data['assignment_weight']) && !empty($data['quiz_weight'])) {
        $total_weight = floatval($data['assignment_weight']) + floatval($data['quiz_weight']);
        if (abs($total_weight - 100) >= 0.01) {
            $errors[] = "Assignment weight and quiz weight must add up to exactly 100%. Current total: " . number_format($total_weight, 2) . "%";
        }
    }

    // Handle file upload for thumbnail - using base64 encoding to bypass PHP limits
    $thumbnail_path = '';
    
    // First check for base64 encoded image data (our preferred method)
    if (isset($_POST['thumbnail_base64']) && !empty($_POST['thumbnail_base64'])) {
        logDebug("Processing base64 thumbnail data");
        
        // Get filename information for extension
        $filename = isset($_POST['thumbnail_filename']) ? $_POST['thumbnail_filename'] : 'thumbnail_' . time() . '.jpg';
        $upload_dir = "../uploads/thumbnails/";
        
        // Create directory if it doesn't exist
        if (!file_exists($upload_dir)) {
            if (!mkdir($upload_dir, 0777, true)) {
                logDebug("Failed to create upload directory: $upload_dir");
                $errors[] = "Failed to create upload directory. Please contact support.";
            }
        }
        
        // Process the base64 string
        $base64_string = $_POST['thumbnail_base64'];
        $base64_parts = explode(',', $base64_string);
        
        // If we have a proper base64 encoded image with header
        if (count($base64_parts) > 1) {
            $image_data = base64_decode($base64_parts[1]);
            
            if ($image_data === false) {
                logDebug("Failed to decode base64 image data");
                $errors[] = "Invalid image data. Please select another image.";
            } else {
                // Get file extension from original filename or from base64 header
                $file_extension = pathinfo($filename, PATHINFO_EXTENSION);
                if (empty($file_extension)) {
                    // Try to get extension from mime type in base64 header
                    if (preg_match('/^data:image\/(jpeg|jpg|png);base64,/', $base64_string, $matches)) {
                        $file_extension = $matches[1];
                    } else {
                        $file_extension = 'jpg'; // Default to jpg if we can't determine
                    }
                }
                
                // Generate a unique filename
                $file_name = 'course_' . time() . '_' . uniqid() . '.' . $file_extension;
                $target_path = $upload_dir . $file_name;
                
                // Save the file
                if (file_put_contents($target_path, $image_data)) {
                    logDebug("Successfully saved base64 image to: $target_path");
                    $thumbnail_path = '../uploads/thumbnails/' . $file_name;
                } else {
                    logDebug("Failed to save base64 image to: $target_path");
                    $errors[] = "Failed to save thumbnail. Please try again.";
                }
            }
        } else {
            logDebug("Invalid base64 data format");
            $errors[] = "Invalid image data format. Please select another image.";
        }
    }
    // Fallback to regular file upload method (likely to fail due to PHP limits)
    else if (isset($_FILES['thumbnail'])) {
        // Log detailed file information
        logDebug("Detailed thumbnail info", array(
            'name' => $_FILES['thumbnail']['name'],
            'type' => $_FILES['thumbnail']['type'],
            'size' => $_FILES['thumbnail']['size'],
            'error_code' => $_FILES['thumbnail']['error'],
            'error_message' => getFileUploadErrorMessage($_FILES['thumbnail']['error'])
        ));
        
        // Only try to process if we have a valid file
        if ($_FILES['thumbnail']['error'] === UPLOAD_ERR_OK) {
        $allowed_types = ['image/jpeg', 'image/png', 'image/jpg'];
        $file_info = finfo_open(FILEINFO_MIME_TYPE);
        $mime_type = finfo_file($file_info, $_FILES['thumbnail']['tmp_name']);
        
        if (!in_array($mime_type, $allowed_types)) {
            $errors[] = "Only JPG, JPEG, and PNG files are allowed for thumbnails.";
        } else {
            $upload_dir = "../uploads/thumbnails/";
            if (!file_exists($upload_dir)) {
                mkdir($upload_dir, 0777, true);
            }
            
            $file_extension = pathinfo($_FILES['thumbnail']['name'], PATHINFO_EXTENSION);
            $file_name = 'course_' . time() . '_' . uniqid() . '.' . $file_extension;
            $target_path = $upload_dir . $file_name;
            
            if (move_uploaded_file($_FILES['thumbnail']['tmp_name'], $target_path)) {
                // Store URL-friendly path for web access instead of server file path
                $thumbnail_path = '../uploads/thumbnails/' . $file_name;
            } else {
                $errors[] = "Failed to upload thumbnail. Please try again.";
            }
        }
        } else {
            // More detailed error message based on the error code
            $errorCode = $_FILES['thumbnail']['error'];
            $errors[] = "Course thumbnail error: " . getFileUploadErrorMessage($errorCode);
        }
    } else {
        $errors[] = "Course thumbnail is required.";
    }

    // If no validation errors, proceed with database insertion
    if (empty($errors)) {
        try {
            logDebug("Starting database transaction");
            logDebug("Form data", $data);
            logDebug("Thumbnail path", $thumbnail_path);
            
            $conn->begin_transaction();
            
            // Check database connection
            if ($conn->connect_error) {
                throw new Exception("Connection failed: " . $conn->connect_error);
            }
            
            // Verify the courses table structure and get column names
            $tableCheck = $conn->query("DESCRIBE courses");
            if (!$tableCheck) {
                throw new Exception("Courses table does not exist or is not accessible");
            }
            
            // Get all column names from the table
            $columns = [];
            $result = $conn->query('DESCRIBE courses');
            while ($row = $result->fetch_assoc()) {
                $columns[$row['Field']] = $row['Type'];
            }
            logDebug("Table columns", $columns);
            
            // Get category_id from the form
            // The form now submits the category_id directly as a number
            $categoryId = 1; // Default to category ID 1 if not valid
            
            if (!empty($data['category']) && is_numeric($data['category'])) {
                $categoryId = (int)$data['category'];
                logDebug("Using category_id from form: {$categoryId}");
                
                // Verify the category exists in the database
                $checkCatStmt = $conn->prepare("SELECT category_id FROM categories WHERE category_id = ? LIMIT 1");
                if ($checkCatStmt) {
                    $checkCatStmt->bind_param("i", $categoryId);
                    $checkCatStmt->execute();
                    $checkCatResult = $checkCatStmt->get_result();
                    if ($checkCatResult->num_rows == 0) {
                        // Category doesn't exist, fall back to ID 1
                        logDebug("Category ID {$categoryId} not found in database, falling back to ID 1");
                        $categoryId = 1;
                    }
                    $checkCatStmt->close();
                }
            } else {
                logDebug("Invalid category value '{$data['category']}', using default ID 1");
            }
            
            // Dynamically build SQL query based on actual table columns
            $fieldNames = [];
            $placeholders = [];
            $bindTypes = '';
            $bindValues = [];
            
            // Always include teacher_id
            $fieldNames[] = 'teacher_id';
            $placeholders[] = '?';
            $bindTypes .= 'i';  // Integer
            $bindValues[] = $teacher_id;
            
            // Map form fields to database columns (only include columns that exist in the table)
            $columnMapping = [
                'title' => ['field' => 'title', 'type' => 's', 'value' => $data['title'] ?? ''],
                'description' => ['field' => 'description', 'type' => 's', 'value' => $data['description'] ?? ''],
                'category_id' => ['field' => 'category_id', 'type' => 'i', 'value' => $categoryId], // Always use int for category_id
                'level' => ['field' => 'level', 'type' => 's', 'value' => $data['level'] ?? ''],
                'price' => ['field' => 'price', 'type' => 'd', 'value' => $data['price'] ?? 0],
                'duration' => ['field' => 'duration', 'type' => 'i', 'value' => (int)($data['duration'] ?? 0)],
                'thumbnail' => ['field' => 'thumbnail', 'type' => 's', 'value' => $thumbnail_path],
                'what_you_will_learn' => ['field' => 'what_you_will_learn', 'type' => 's', 'value' => $data['outcomes'] ?? ''],
                'requirements' => ['field' => 'requirements', 'type' => 's', 'value' => $data['requirements'] ?? ''],
                'target_audience' => ['field' => 'target_audience', 'type' => 's', 'value' => $data['target_audience'] ?? ''],
                'assignment_weight' => ['field' => 'assignment_weight', 'type' => 'd', 'value' => floatval($data['assignment_weight'] ?? 0)],
                'quiz_weight' => ['field' => 'quiz_weight', 'type' => 'd', 'value' => floatval($data['quiz_weight'] ?? 100)],
                'course_passing_grade' => ['field' => 'course_passing_grade', 'type' => 'd', 'value' => floatval($data['course_passing_grade'] ?? 60)],
                'is_published' => ['field' => 'is_published', 'type' => 'i', 'value' => ($data['status'] ?? 'draft') === 'published' ? 1 : 0]
            ];
            
            // Remove any string-type category columns to avoid SQL errors
            foreach(['category', 'course_category', 'cat_id'] as $catField) {
                if (array_key_exists($catField, $columnMapping) && $catField !== 'category_id') {
                    unset($columnMapping[$catField]);
                }
            }
            
            foreach ($columnMapping as $dbField => $mapping) {
                if (array_key_exists($dbField, $columns)) {
                    $fieldNames[] = $dbField;
                    $placeholders[] = '?';
                    $bindTypes .= $mapping['type'];
                    $bindValues[] = $mapping['value'];
                    logDebug("Adding field: {$dbField} with value: {$mapping['value']}");
                }
            }
            
            // Build the SQL query
            $fieldsList = implode(', ', $fieldNames);
            $placeholdersList = implode(', ', $placeholders);
            $sql = "INSERT INTO courses ({$fieldsList}) VALUES ({$placeholdersList})";
            
            logDebug("Dynamic SQL Query - {$sql}");
            logDebug("Bind types - {$bindTypes}");
            logDebug("Bind values", $bindValues);
            
            $stmt = $conn->prepare($sql);
            
            if (!$stmt) {
                throw new Exception("Prepare statement failed: " . $conn->error);
            }
            
            // Bind parameters dynamically
            if (!empty($bindValues)) {
                // Create a reference array for bind_param
                $bindParams = [];
                $bindParams[] = &$bindTypes;
                
                for ($i = 0; $i < count($bindValues); $i++) {
                    $bindParams[] = &$bindValues[$i];
                }
                
                logDebug("Binding parameters count: " . count($bindParams));
                
                try {
                    // Call bind_param with dynamic parameters using call_user_func_array
                    call_user_func_array([$stmt, 'bind_param'], $bindParams);
                } catch (Exception $e) {
                    logDebug("Bind param error: " . $e->getMessage());
                    logDebug("Bind types: " . $bindTypes);
                    logDebug("Bind values count: " . count($bindValues));
                    throw new Exception("Error binding parameters: " . $e->getMessage());
                }
            } else {
                throw new Exception("No parameters to bind");
            }
            
            logDebug("Executing statement");
            if ($stmt->execute()) {
                $course_id = $conn->insert_id;
                logDebug("Course inserted successfully with ID: " . $course_id);
                
                // Handle learning objectives if provided
                if (!empty($_POST['outcomes'])) {
                    logDebug("Processing learning objectives from 'outcomes' field");
                    $objectives = array_filter(array_map('trim', explode("\n", $_POST['outcomes'])));
                    if (!empty($objectives)) {
                        logDebug("Learning objectives", $objectives);
                        
                        try {
                            // Check if course_learning_objectives table exists
                            $tableExists = false;
                            $tableResult = $conn->query("SHOW TABLES LIKE 'course_learning_objectives'");
                            if ($tableResult) {
                                $tableExists = ($tableResult->num_rows > 0);
                            }
                            
                            if (!$tableExists) {
                                logDebug("INFO: course_learning_objectives table does not exist. Objectives will not be saved.");
                                // Skip saving objectives but don't fail the course creation
                            } else {
                                $stmt = $conn->prepare("INSERT INTO course_learning_objectives (course_id, objective) VALUES (?, ?)");
                                if (!$stmt) {
                                    logDebug("WARNING: Failed to prepare learning objectives statement: " . $conn->error);
                                } else {
                                    foreach ($objectives as $objective) {
                                        $stmt->bind_param('is', $course_id, $objective);
                                        if (!$stmt->execute()) {
                                            logDebug("WARNING: Failed to insert objective: " . $stmt->error);
                                        }
                                    }
                                }
                            }
                        } catch (Exception $e) {
                            // Log the error but continue - don't fail the whole course creation
                            logDebug("WARNING: Could not process learning objectives: " . $e->getMessage());
                        }
                    }
                }
                
                $conn->commit();
                logDebug("Transaction committed successfully");
                
                // Send course creation notification email - USING EXACT SAME APPROACH AS WITHDRAWAL
                $email_result = sendCourseCreationEmail($course_id);
                
                // Trigger course creation notification
                try {
                    require_once __DIR__ . '/../includes/NotificationService.php';
                    $notificationService = new NotificationService($conn);
                    $notificationResult = $notificationService->notifyTeacherOfCourseCreation($teacher_id, $course_id);
                    logDebug("Course creation notification result", $notificationResult ? 'SUCCESS' : 'FAILED');
                } catch (Exception $e) {
                    logDebug("Notification error (non-critical): " . $e->getMessage());
                    // Don't fail course creation if notification fails
                }
                
                // Check if this is an AJAX request or a regular form submission
                $isAjax = !empty($_SERVER['HTTP_X_REQUESTED_WITH']) && 
                    strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest';
                
                // Prepare the success data with email confirmation
                $response['success'] = true;
                if ($email_result) {
                    $response['message'] = 'Course created successfully! ✉️ Confirmation email sent to you. Your course is awaiting admin approval.';
                } else {
                    $response['message'] = 'Course created successfully, but failed to send confirmation email. Your course is awaiting admin approval.';
                }
                $response['course_id'] = $course_id;
                logDebug("Success response", $response);
                
                if ($isAjax) {
                    // For AJAX requests, return JSON
                    echo json_encode($response);
                } else {
                    // For regular form submissions, redirect to the course page
                    if ($email_result) {
                        $_SESSION['flash_message'] = 'Course created successfully! ✉️ Confirmation email sent to you. Your course is awaiting admin approval.';
                    } else {
                        $_SESSION['flash_message'] = 'Course created successfully, but failed to send confirmation email. Your course is awaiting admin approval.';
                    }
                    $_SESSION['flash_type'] = 'success';
                    
                    // Redirect to the courses listing page with success message
                    header("Location: courses.php?new_course_id=" . $course_id);
                }
                exit;
            } else {
                throw new Exception("Failed to create course: " . $conn->error);
            }
        } catch (Exception $e) {
            $conn->rollback();
            logDebug("EXCEPTION: " . $e->getMessage());
            logDebug("Exception trace: " . $e->getTraceAsString());
            
            // Delete uploaded thumbnail if any
            if (!empty($thumbnail_path) && file_exists($thumbnail_path)) {
                @unlink($thumbnail_path);
                logDebug("Deleted uploaded thumbnail: " . $thumbnail_path);
            }
            
            $errors[] = "An error occurred while creating the course: " . $e->getMessage();
            error_log("Course Creation Error: " . $e->getMessage());
            
            // More detailed error information
            if ($conn->error) {
                logDebug("Database error: " . $conn->error);
                $errors[] = "Database error: " . $conn->error;
            }
            
            http_response_code(500);
            $response['errors'] = $errors;
            $response['debug_info'] = $e->getMessage();
            
            logDebug("Error response", $response);
            echo json_encode($response);
            exit;
        }
    }
    
    // If we get here, there were errors
    http_response_code(400);
    $response['success'] = false;
    $response['errors'] = $errors;
    logDebug("Validation errors", $errors);
    logDebug("POST data", $_POST);
    logDebug("FILES data", isset($_FILES) ? $_FILES : "No files");
    echo json_encode($response);
    exit;
} else {
    // Not a POST request
    http_response_code(405);
    $methodResponse = [
        'success' => false, 
        'message' => 'Method not allowed', 
        'errors' => ['Invalid request method: ' . $_SERVER['REQUEST_METHOD']]
    ];
    logDebug("Method not allowed", $methodResponse);
    echo json_encode($methodResponse);
    exit;
}
