<?php
// Start session to access session variables
session_start();

// Configure PHP for large file uploads
ini_set('upload_max_filesize', '1000M'); // Must include 'M' for megabytes
ini_set('post_max_size', '1100M');       // Should be larger than upload_max_filesize
ini_set('memory_limit', '2048M');        // 2GB for large file processing
ini_set('max_execution_time', 600);     // 10 minutes
ini_set('max_input_time', 600);         // 10 minutes

// Force the correct file size settings using alternative method
@ini_set('upload_max_filesize', '1000M');
@ini_set('post_max_size', '1100M');
@ini_set('max_file_uploads', '20');

// Debug: Log the actual PHP settings after trying to set them
error_log("PHP settings after ini_set:");
error_log("upload_max_filesize: " . ini_get('upload_max_filesize'));
error_log("post_max_size: " . ini_get('post_max_size'));
error_log("memory_limit: " . ini_get('memory_limit'));
error_log("max_execution_time: " . ini_get('max_execution_time'));
error_log("max_input_time: " . ini_get('max_input_time'));
error_log("max_file_uploads: " . ini_get('max_file_uploads'));

// Convert PHP memory limit formats to bytes for proper comparison
function parse_size($size) {
    $unit = preg_replace('/[^bkmgtpezy]/i', '', $size); // Remove the non-unit characters from the size.
    $size = preg_replace('/[^0-9\.]/', '', $size); // Remove the non-numeric characters from the size.
    if ($unit) {
        // Find the position of the unit in the ordered string which is the power of magnitude to multiply a kilobyte by.
        return round($size * pow(1024, stripos('bkmgtpezy', $unit[0])));
    } else {
        return round($size);
    }
}

// Log the active php.ini file for debugging
error_log("Loaded PHP config file: " . php_ini_loaded_file());
error_log("Scanned PHP config files: " . implode(", ", php_ini_scanned_files() ?: ["None"]));


// Setup error handling for debugging
ini_set('display_errors', 1); // Show errors for debugging
error_reporting(E_ALL);       // Report all error types

// Include necessary files
if (!defined('CONFIG_INCLUDED')) {
    require_once '../includes/config.php';
}

// Debug mode (set to false in production)
$debug_mode = true;

// Initialize variables
$upload_error = false;
$upload_success = false;
$insert_success = false;
$error = '';
$thumbnail_path = null;

// Function to log detailed error info
function log_upload_error($message, $details = null) {
    global $debug_mode;
    
    error_log("UPLOAD ERROR: $message");
    
    if ($debug_mode && $details) {
        error_log("DEBUG DETAILS: " . print_r($details, true));
    }
}

// Function to translate PHP file upload error codes to human-readable messages
function getFileErrorMessage($errorCode) {
    switch ($errorCode) {
        case UPLOAD_ERR_INI_SIZE:
            return 'The uploaded file exceeds the upload_max_filesize directive in php.ini';
        case UPLOAD_ERR_FORM_SIZE:
            return 'The uploaded file exceeds the MAX_FILE_SIZE directive that was specified in the HTML form';
        case UPLOAD_ERR_PARTIAL:
            return 'The uploaded file was only partially uploaded';
        case UPLOAD_ERR_NO_FILE:
            return 'No file was uploaded';
        case UPLOAD_ERR_NO_TMP_DIR:
            return 'Missing a temporary folder';
        case UPLOAD_ERR_CANT_WRITE:
            return 'Failed to write file to disk';
        case UPLOAD_ERR_EXTENSION:
            return 'A PHP extension stopped the file upload';
        default:
            return 'Unknown upload error';
    }
}

// Check if user is authenticated and is a teacher
if (!isset($_SESSION['user_id']) || $_SESSION['user_role'] !== 'teacher') {
    log_upload_error('Unauthorized access attempt', $_SESSION);
    header('Location: ../login.php');
    exit();
}

// Get teacher ID from session (critical security measure)
$teacher_id = (int)$_SESSION['user_id']; // Force integer to prevent SQL injectionuration
error_log("PHP Upload settings check:");
error_log("upload_max_filesize: " . ini_get('upload_max_filesize'));
error_log("post_max_size: " . ini_get('post_max_size'));
error_log("max_file_uploads: " . ini_get('max_file_uploads'));
error_log("max_input_time: " . ini_get('max_input_time'));
error_log("memory_limit: " . ini_get('memory_limit'));

// Log critical information at the start
error_log("====== VIDEO UPLOAD PROCESS STARTED ======");
error_log("SESSION ID: " . session_id());
error_log("USER ID IN SESSION: " . (isset($_SESSION['user_id']) ? $_SESSION['user_id'] : "NOT SET"));

// Database connection
include "../includes/config.php";

// Check if user is logged in and is a teacher
if (!isset($_SESSION['logged_in']) || $_SESSION['user_role'] !== 'teacher') {
    error_log("UPLOAD ERROR: User not logged in or not a teacher. Role: " . (isset($_SESSION['user_role']) ? $_SESSION['user_role'] : "not set"));
    $_SESSION['error_message'] = "You must be logged in as a teacher to upload videos.";
    header("Location: login.php");
    exit;
}

// Double check that we have a valid teacher ID from session
if (!isset($_SESSION['user_id']) || empty($_SESSION['user_id'])) {
    error_log("CRITICAL ERROR: No user_id in session despite being logged in");
    $_SESSION['error_message'] = "Session error. Please log out and log in again.";
    header("Location: login.php");
    exit;
}

// Initialize variables
$error = "";
$upload_error = false;
$upload_success = false;
$video_id = 0;
$insert_success = false;
$teacher_id = (int)$_SESSION['user_id'];
$thumbnail_path = null;

// Process form submission
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // Get form data
    $title = trim($_POST['title'] ?? '');
    $description = trim($_POST['description'] ?? '');
    $course_id = isset($_POST['course_id']) ? (int)$_POST['course_id'] : 0;
    $video_duration = isset($_POST['videoDuration']) ? trim($_POST['videoDuration']) : (isset($_POST['duration']) ? trim($_POST['duration']) : '00:00');
    $duration_seconds = isset($_POST['duration_seconds']) ? (int)$_POST['duration_seconds'] : 0;
    
    // Debug log the form data and FILES array
    error_log("Form data received - title: $title, course_id: $course_id, teacher_id: $teacher_id, duration: $video_duration");
    error_log("FILES array contents: " . print_r($_FILES, true));
    
    // Debug the request method and content type
    error_log("Request method: {$_SERVER['REQUEST_METHOD']}");
    if(isset($_SERVER['CONTENT_TYPE'])) {
        error_log("Content type: {$_SERVER['CONTENT_TYPE']}");
    }
    
    // Validate required fields
    if (empty($title)) {
        $error = "Video title is required";
        $upload_error = true;
    } elseif ($course_id <= 0) {
        $error = "Please select a valid course";
        $upload_error = true;
    } elseif (!isset($_FILES['video']) && !isset($_POST['base64data'])) {
        error_log("ERROR: No video file found in _FILES array or base64data");
        $error = "Please upload a video file";
        $upload_error = true;
    } elseif (isset($_FILES['video']) && $_FILES['video']['error'] != UPLOAD_ERR_OK) {
        error_log("ERROR: Video file upload error: " . $_FILES['video']['error']);
        $error = "Video upload failed with error code: " . $_FILES['video']['error'];
        $upload_error = true;
    } elseif (isset($_FILES['video']) && $_FILES['video']['size'] <= 0) {
        error_log("ERROR: Video file size is 0 or negative");
        $error = "Video file appears to be empty";
        $upload_error = true;
    } elseif (isset($_POST['base64data']) && empty($_POST['base64data'])) {
        error_log("ERROR: Base64 data is empty");
        $error = "Video file appears to be empty";
        $upload_error = true;
    } else {
        // Initialize variables
        $video_name = '';
        $video_tmp = '';
        $video_size = 0;
        $video_type = '';
        $video_error = UPLOAD_ERR_NO_FILE; // Default to error, set to OK on success
        $upload_method = 'unknown';
        
        // Process either regular file upload or base64 data
        if (isset($_FILES['video']) && $_FILES['video']['error'] === UPLOAD_ERR_OK) {
            // Standard file upload method
            $video_file = $_FILES['video'];
            $video_name = $video_file['name'];
            $video_tmp = $video_file['tmp_name'];
            $video_size = $video_file['size'];
            $video_type = $video_file['type'];
            $video_error = $video_file['error'];
            
            $upload_method = 'standard';
            error_log("Using standard file upload method");
        }
        elseif (isset($_POST['base64data']) && !empty($_POST['base64data'])) {
            // Base64 upload method
            error_log("Using base64 upload method");
            error_log("POST data keys: " . implode(', ', array_keys($_POST)));
            
            // Get file details from POST data
            $video_name = isset($_POST['fileName']) ? trim($_POST['fileName']) : 'video_'.time().'.mp4';
            $video_type = isset($_POST['fileType']) ? trim($_POST['fileType']) : 'video/mp4';
            
            error_log("Base64 processing - video_name: $video_name, video_type: $video_type");
            
            // Log base64 string length and presence
            $base64_string = $_POST['base64data'];
            error_log("Base64 data received: " . (empty($base64_string) ? "empty" : "length: " . strlen($base64_string) . " bytes"));
            
            // Remove potential header in base64 string (like "data:video/mp4;base64,")
            if (strpos($base64_string, ',') !== false) {
                $parts = explode(',', $base64_string);
                error_log("Base64 header found: " . $parts[0]);
                $base64_string = $parts[1];
            }
            
            try {
                // Create temporary file
                $video_tmp = tempnam(sys_get_temp_dir(), 'video_upload_');
                error_log("Temporary file created at: $video_tmp");
                
                // Decode base64 data
                $video_data = base64_decode($base64_string);
                if ($video_data === false) {
                    error_log("ERROR: Failed to decode base64 data");
                    throw new Exception("Failed to decode base64 data");
                }
                
                $video_size = strlen($video_data);
                error_log("Decoded data size: $video_size bytes");
                
                // Write decoded data to temp file
                if (file_put_contents($video_tmp, $video_data)) {
                    $video_error = UPLOAD_ERR_OK;
                    $upload_method = 'base64';
                    error_log("Base64 data decoded and written to temp file: $video_tmp (size: $video_size bytes)");
                } else {
                    error_log("ERROR: Failed to write decoded data to temp file");
                    throw new Exception("Failed to write decoded data to temporary file");
                }
                
                // Clean up memory
                unset($video_data);
                unset($base64_string);
                
            } catch (Exception $e) {
                $error = "Error processing video data: " . $e->getMessage();
                $upload_error = true;
                $video_error = UPLOAD_ERR_CANT_WRITE; // Set specific error code
                error_log("Exception while processing base64 data: " . $e->getMessage());
            }
        } else {
            $error = "No valid upload method found (neither standard nor base64)"; 
            $upload_error = true;
            error_log("ERROR: No valid upload method found. FILES: " . print_r($_FILES, true) . ", base64data present: " . (isset($_POST['base64data']) ? "yes" : "no"));
        }
        
        // Log all information about the uploaded file
        error_log("File upload details: name=$video_name, size=$video_size, type=$video_type, error=$video_error, method=$upload_method");
        error_log("Upload error flag: " . ($upload_error ? 'true' : 'false'));
        
        // Check for upload errors first
        if ($video_error !== UPLOAD_ERR_OK) {
            error_log("Video error detected: $video_error (" . getFileErrorMessage($video_error) . ")");
            $error = "Failed to upload video file";
            $upload_error = true;
            error_log("UPLOAD ERROR: $error");
            
            // Add detailed debugging information
            $error_details = array(
                'error_details' => $error,
                'php_version' => phpversion(),
                'upload_max_filesize' => ini_get('upload_max_filesize'),
                'post_max_size' => ini_get('post_max_size'),
                'memory_limit' => ini_get('memory_limit'),
                'file_error_code' => $video_error,
                'error_time' => date('Y-m-d H:i:s')
            );
            
            $_SESSION['upload_debug'] = $error_details;
        }
        
        // Check file type
        $allowed_types = ['video/mp4', 'video/webm', 'video/ogg', 'video/quicktime'];
        if (!in_array($video_type, $allowed_types) && !strpos($video_type, 'video/') === 0) {
            $error = "Please upload a valid video file";
            $upload_error = true;
        } 
        // Check file size (limit to 1000MB = 1GB)
        elseif ($video_size > 1000 * 1024 * 1024) {
            error_log("File size check failed: $video_size bytes (" . round($video_size / 1024 / 1024, 2) . " MB) > 1000MB limit");
            $error = "Video file size should not exceed 1000MB";
            $upload_error = true;
        } 
        else {
            // Create upload directories if they don't exist
            $upload_dir = "../uploads/videos/";
            $thumbnails_dir = "../uploads/thumbnails/";
            
            if (!file_exists($upload_dir)) {
                mkdir($upload_dir, 0755, true);
            }
            
            if (!file_exists($thumbnails_dir)) {
                mkdir($thumbnails_dir, 0755, true);
            }
            
            // Generate unique filenames
            $video_filename = 'video_' . uniqid() . '.' . pathinfo($video_name, PATHINFO_EXTENSION);
            $video_path = $upload_dir . $video_filename;
            
            // Upload video file
            if (move_uploaded_file($video_tmp, $video_path)) {
                // Video uploaded successfully
                error_log("Video uploaded successfully to $video_path");
                $upload_success = true;
                $db_video_url = "uploads/videos/" . $video_filename;
                
                // Handle thumbnail if provided
                if (isset($_FILES['thumbnail']) && $_FILES['thumbnail']['error'] == UPLOAD_ERR_OK) {
                    $thumbnail_file = $_FILES['thumbnail'];
                    $thumbnail_tmp = $thumbnail_file['tmp_name'];
                    $thumbnail_name = $thumbnail_file['name'];
                    
                    $thumbnail_filename = 'thumb_' . uniqid() . '.' . pathinfo($thumbnail_name, PATHINFO_EXTENSION);
                    $thumbnail_path = $thumbnails_dir . $thumbnail_filename;
                    
                    if (move_uploaded_file($thumbnail_tmp, $thumbnail_path)) {
                        // Thumbnail uploaded successfully
                        error_log("Thumbnail uploaded successfully to $thumbnail_path");
                        $thumbnail_path = "uploads/thumbnails/" . $thumbnail_filename;
                    } else {
                        error_log("Thumbnail upload failed");
                        $thumbnail_path = null;
                    }
                }
                
                // Insert into database using prepared statement (more secure)
                try {
                    // Always use session teacher_id for security
                    $teacher_id = (int)$_SESSION['user_id'];
                    $course_id = (int)$course_id;
                    
                    // Use prepared statement to prevent SQL injection
                    $stmt = $conn->prepare("INSERT INTO videos (title, video_url, course_id, teacher_id, description, duration, thumbnail) 
                                          VALUES (?, ?, ?, ?, ?, ?, ?)");
                    
                    if (!$stmt) {
                        throw new Exception("Database prepare error: " . $conn->error);
                    }
                    
                    // Bind parameters securely
                    $stmt->bind_param('ssiisss', 
                        $title,
                        $db_video_url,
                        $course_id, 
                        $teacher_id, 
                        $description, 
                        $duration, 
                        $thumbnail_path
                    );
                    
                    error_log("Executing prepared statement for video insertion. Teacher ID: $teacher_id, Course ID: $course_id");
                    
                    if ($stmt->execute()) {
                        $insert_success = true;
                        $video_id = $conn->insert_id;
                        error_log("Video record inserted successfully with ID: $video_id");
                        error_log("Inserted video for teacher_id: $teacher_id, course_id: $course_id");
                        
                        // Verify insertion and teacher_id
                        $check_sql = "SELECT * FROM videos WHERE video_id = $video_id";
                        $check_result = $conn->query($check_sql);
                        
                        if ($check_result && $check_result->num_rows > 0) {
                            $video_data = $check_result->fetch_assoc();
                            error_log("Verified video data: " . print_r($video_data, true));
                            
                            // Fix teacher_id if needed
                            if ((int)$video_data['teacher_id'] !== $teacher_id) {
                                error_log("Teacher ID mismatch found. Fixing now.");
                                $fix_sql = "UPDATE videos SET teacher_id = $teacher_id WHERE video_id = $video_id";
                                if ($conn->query($fix_sql)) {
                                    error_log("Teacher ID fixed successfully");
                                } else {
                                    error_log("Failed to fix teacher ID: " . $conn->error);
                                }
                            }
                        }
                    } else {
                        error_log("Database insertion failed: " . $conn->error);
                        $error = "Failed to save video information: " . $conn->error;
                        $upload_error = true;
                    }
                } catch (Exception $e) {
                    error_log("Exception during database insertion: " . $e->getMessage());
                    $error = "Database error: " . $e->getMessage();
                    $upload_error = true;
                }
            } else {
                error_log("Failed to move uploaded video file");
                $error = "Failed to upload video file";
                $upload_error = true;
            }
        }
    }
    
    // Log detailed error information for debugging
    $error_details = [];
    if ($upload_error) {
        $error_details['error'] = $error;
        $error_details['upload_error_code'] = $_FILES['video']['error'] ?? 'No file uploaded';
        $error_details['upload_error_message'] = getFileErrorMessage($_FILES['video']['error'] ?? UPLOAD_ERR_NO_FILE);
        $error_details['php_version'] = phpversion();
        $error_details['upload_max_filesize'] = ini_get('upload_max_filesize');
        $error_details['post_max_size'] = ini_get('post_max_size');
        $error_details['memory_limit'] = ini_get('memory_limit');
        $error_details['max_execution_time'] = ini_get('max_execution_time');
        $error_details['file_uploads'] = ini_get('file_uploads');
    }
    
    // Prepare detailed response for the user
    if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
        // For AJAX requests (Legacy support - we now prefer direct form submission)
        header('Content-Type: application/json');
        
        if ($upload_error) {
            // Return detailed error information
            $response = [
                'success' => false,
                'message' => $error,
            ];
            
            // Add debugging information if in debug mode
            if ($debug_mode) {
                $response['debug'] = [
                    'php_version' => phpversion(),
                    'upload_max_filesize' => ini_get('upload_max_filesize'),
                    'post_max_size' => ini_get('post_max_size'),
                    'memory_limit' => ini_get('memory_limit'),
                    'max_execution_time' => ini_get('max_execution_time'),
                    'file_uploads' => ini_get('file_uploads'),
                    'upload_error_code' => $_FILES['video']['error'] ?? 'No file uploaded',
                ];
            }
            
            echo json_encode($response);
        } else {
            // Return success with video details
            echo json_encode([
                'success' => true,
                'message' => 'Video uploaded successfully!',
                'video_id' => $video_id ?? null,
                'redirect' => 'videos.php?success=1'
            ]);
        }
    } else {
        // For direct form submissions (our preferred method)
        if ($upload_error) {
            // Store error in session for display
            $_SESSION['error_message'] = $error;
            
            // Add detailed debug info if in debug mode
            if ($debug_mode) {
                $_SESSION['upload_debug'] = [
                    'error_details' => $error,
                    'php_version' => phpversion(),
                    'upload_max_filesize' => ini_get('upload_max_filesize'),
                    'post_max_size' => ini_get('post_max_size'),
                    'memory_limit' => ini_get('memory_limit'),
                    'file_error_code' => $_FILES['video']['error'] ?? 'No file uploaded',
                    'error_time' => date('Y-m-d H:i:s'),
                ];
            }
            
            // Redirect back to upload form with error flag using complete URL with correct port
            $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https://' : 'http://';
            $host = $_SERVER['HTTP_HOST']; // This includes the port
            header('Location: '.$protocol.$host.'/teacher/upload-video.php?error=1');
        } else {
            // Store success message in session
            $_SESSION['success_message'] = 'Video uploaded successfully! You can now manage it in your videos list.';
            
            // Redirect to videos listing page with correct port
            $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https://' : 'http://';
            $host = $_SERVER['HTTP_HOST']; // This includes the port
            header('Location: '.$protocol.$host.'/teacher/videos.php?success=1');
        }
    }
}
?>
