<?php
// Use centralized secure session handling
require_once '../includes/session_start.php';

// Configure PHP for processing (not upload limits, we're using base64)
ini_set('memory_limit', '2048M');
ini_set('max_execution_time', 0);
ini_set('display_errors', 1); // Enable errors for debugging
error_reporting(E_ALL);

// Ensure clean JSON output
ob_start();
header('Content-Type: application/json');

// Debug info array
$debug_info = [];

// Include database connection - use the same one as other pages
require_once '../config/db_connect.php';
require_once '../includes/functions.php';
require_once '../auth/functions.php';

// Function to send JSON response with debug info
function sendResponse($success, $message, $data = []) {
    global $debug_info;
    
    // Clear any output buffer to ensure clean JSON
    if (ob_get_length()) {
        ob_clean();
    }
    
    echo json_encode([
        'success' => $success,
        'message' => $message,
        'data' => $data,
        'debug' => $debug_info
    ]);
    exit();
}

function addDebug($key, $value) {
    global $debug_info;
    $debug_info[$key] = $value;
}

// Check authentication
if (!isset($_SESSION['user_id']) || $_SESSION['user_role'] !== 'teacher') {
    sendResponse(false, 'Unauthorized access');
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendResponse(false, 'Invalid request method');
}

$teacher_id = (int)$_SESSION['user_id'];

try {
    addDebug('teacher_id', $teacher_id);
    addDebug('post_keys', array_keys($_POST));
    
    // Get form data
    $title = trim($_POST['title'] ?? '');
    $course_id = (int)($_POST['course_id'] ?? 0);
    $description = trim($_POST['description'] ?? '');
    $module_order = (int)($_POST['module_order'] ?? 1);
    $price = (float)($_POST['price'] ?? 0);
    $category_name = trim($_POST['category_name'] ?? '');
    $is_published = (int)($_POST['is_published'] ?? 1);
    $is_free = (int)($_POST['is_free'] ?? 0);
    $duration = (float)($_POST['duration'] ?? 0);
    
    // Check for base64 video data
    $base64_data = $_POST['video_base64'] ?? '';
    $video_filename = $_POST['video_filename'] ?? '';
    
    addDebug('has_base64_data', !empty($base64_data));
    addDebug('base64_length', strlen($base64_data));
    addDebug('video_filename', $video_filename);

    // Basic validation
    if (empty($title)) {
        sendResponse(false, 'Video title is required');
    }

    if ($course_id <= 0) {
        sendResponse(false, 'Please select a valid course');
    }
    
    if (empty($base64_data)) {
        sendResponse(false, 'No video data received');
    }

    // Verify course ownership
    $course_check = $conn->prepare("SELECT course_id FROM courses WHERE course_id = ? AND teacher_id = ?");
    $course_check->bind_param("ii", $course_id, $teacher_id);
    $course_check->execute();
    if ($course_check->get_result()->num_rows === 0) {
        sendResponse(false, 'Course not found or access denied');
    }
    $course_check->close();

    // Process base64 video data
    addDebug('processing_base64', true);
    
    // Remove data URI header if present
    if (strpos($base64_data, ',') !== false) {
        $parts = explode(',', $base64_data);
        $base64_data = $parts[1];
        addDebug('removed_data_uri_header', true);
    }
    
    // Decode base64 data
    $video_binary = base64_decode($base64_data);
    if ($video_binary === false) {
        addDebug('base64_decode_failed', true);
        sendResponse(false, 'Invalid video data');
    }
    
    $video_size = strlen($video_binary);
    addDebug('decoded_video_size', $video_size);
    
    if ($video_size === 0) {
        sendResponse(false, 'Video data is empty');
    }

    // Create upload directories
    $uploads_dir = '../uploads';
    $videos_dir = $uploads_dir . '/videos';
    $thumbnails_dir = $uploads_dir . '/thumbnails';

    if (!file_exists($uploads_dir)) mkdir($uploads_dir, 0755, true);
    if (!file_exists($videos_dir)) mkdir($videos_dir, 0755, true);
    if (!file_exists($thumbnails_dir)) mkdir($thumbnails_dir, 0755, true);

    // Generate unique filename
    $file_extension = $video_filename ? pathinfo($video_filename, PATHINFO_EXTENSION) : 'mp4';
    $unique_filename = 'video_' . uniqid() . '_' . time() . '.' . $file_extension;
    $video_path = $videos_dir . '/' . $unique_filename;
    
    addDebug('target_video_path', $video_path);

    // Write video file
    $bytes_written = file_put_contents($video_path, $video_binary);
    if ($bytes_written === false) {
        addDebug('file_write_failed', true);
        sendResponse(false, 'Failed to save video file');
    }
    
    addDebug('bytes_written', $bytes_written);
    
    // Verify file was written correctly
    if (!file_exists($video_path) || filesize($video_path) !== $video_size) {
        addDebug('file_verification_failed', true);
        sendResponse(false, 'Video file verification failed');
    }

    // Handle thumbnail if provided
    $thumbnail_path = null;
    if (isset($_POST['thumbnail_base64']) && !empty($_POST['thumbnail_base64'])) {
        $thumbnail_base64 = $_POST['thumbnail_base64'];
        $thumbnail_filename = $_POST['thumbnail_filename'] ?? 'thumbnail.jpg';
        
        // Remove data URI header
        if (strpos($thumbnail_base64, ',') !== false) {
            $parts = explode(',', $thumbnail_base64);
            $thumbnail_base64 = $parts[1];
        }
        
        $thumbnail_binary = base64_decode($thumbnail_base64);
        if ($thumbnail_binary !== false) {
            $thumb_extension = pathinfo($thumbnail_filename, PATHINFO_EXTENSION);
            $unique_thumb_filename = 'thumb_' . uniqid() . '_' . time() . '.' . $thumb_extension;
            $thumbnail_full_path = $thumbnails_dir . '/' . $unique_thumb_filename;
            
            if (file_put_contents($thumbnail_full_path, $thumbnail_binary)) {
                $thumbnail_path = 'uploads/thumbnails/' . $unique_thumb_filename;
                addDebug('thumbnail_saved', true);
            }
        }
    }

    // Format duration
    $duration_formatted = '';
    if ($duration > 0) {
        $hours = (int)floor($duration / 3600);
        $minutes = (int)floor(($duration % 3600) / 60);
        $seconds = (int)floor($duration % 60);
        
        if ($hours > 0) {
            $duration_formatted = sprintf('%d:%02d:%02d', $hours, $minutes, $seconds);
        } else {
            $duration_formatted = sprintf('%d:%02d', $minutes, $seconds);
        }
    }

    // Insert into database
    $video_url = 'uploads/videos/' . $unique_filename;
    
    $sql = "INSERT INTO videos (title, video_url, course_id, teacher_id, description, duration, thumbnail, price, category_name, is_free, is_published) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
    
    $stmt = $conn->prepare($sql);
    if (!$stmt) {
        addDebug('db_prepare_failed', $conn->error);
        sendResponse(false, 'Database error occurred');
    }

    $stmt->bind_param(
        'ssiiissdsii',
        $title,
        $video_url,
        $course_id,
        $teacher_id,
        $description,
        $duration_formatted,
        $thumbnail_path,
        $price,
        $category_name,
        $is_free,
        $is_published
    );

    if (!$stmt->execute()) {
        addDebug('db_insert_failed', $stmt->error);
        // Clean up uploaded files
        if (file_exists($video_path)) unlink($video_path);
        if ($thumbnail_path && file_exists('../' . $thumbnail_path)) unlink('../' . $thumbnail_path);
        sendResponse(false, 'Failed to save video information');
    }

    $video_id = $conn->insert_id;
    $stmt->close();

    addDebug('video_id', $video_id);
    addDebug('success', true);

    sendResponse(true, 'Video uploaded successfully!', [
        'video_id' => $video_id,
        'video_url' => $video_url,
        'thumbnail' => $thumbnail_path,
        'title' => $title,
        'duration' => $duration_formatted,
        'file_size' => $video_size
    ]);

} catch (Exception $e) {
    addDebug('exception', $e->getMessage());
    sendResponse(false, 'An error occurred: ' . $e->getMessage());
}
?>
