<?php
session_start();

// Configure PHP for unlimited uploads
ini_set('upload_max_filesize', '0');
ini_set('post_max_size', '0');
ini_set('memory_limit', '-1');
ini_set('max_execution_time', 0);
ini_set('max_input_time', -1);

// Enable error reporting
ini_set('display_errors', 1);
error_reporting(E_ALL);

// Set content type for JSON response
header('Content-Type: application/json');

// Include database connection
require_once '../includes/config.php';

// Global debug info collector
$debug_info = [];

// Function to send JSON response with debug info
function sendResponse($success, $message, $data = []) {
    global $debug_info;
    echo json_encode([
        'success' => $success,
        'message' => $message,
        'data' => $data,
        'debug' => $debug_info
    ]);
    exit();
}

// Helper function to add debug info
function addDebug($key, $value) {
    global $debug_info;
    $debug_info[$key] = $value;
}

// Check authentication
if (!isset($_SESSION['user_id']) || $_SESSION['user_role'] !== 'teacher') {
    sendResponse(false, 'Unauthorized access');
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendResponse(false, 'Invalid request method');
}

$teacher_id = (int)$_SESSION['user_id'];

try {
    error_log("UNLIMITED UPLOAD STARTED for teacher $teacher_id");
    error_log("POST keys: " . implode(', ', array_keys($_POST)));
    error_log("FILES keys: " . implode(', ', array_keys($_FILES)));
    
    // Get form data
    $title = trim($_POST['title'] ?? '');
    $course_id = (int)($_POST['course_id'] ?? 0);
    $description = trim($_POST['description'] ?? '');
    $module_order = (int)($_POST['module_order'] ?? 1);
    $price = (float)($_POST['price'] ?? 0);
    $category_name = trim($_POST['category_name'] ?? '');
    $is_published = (int)($_POST['is_published'] ?? 1);
    $is_free = (int)($_POST['is_free'] ?? 0);
    $duration = (float)($_POST['duration'] ?? 0);

    // Basic validation
    if (empty($title)) {
        sendResponse(false, 'Video title is required');
    }

    if ($course_id <= 0) {
        sendResponse(false, 'Please select a valid course');
    }

    // Verify course ownership
    $course_check = $conn->prepare("SELECT course_id FROM courses WHERE course_id = ? AND teacher_id = ?");
    $course_check->bind_param("ii", $course_id, $teacher_id);
    $course_check->execute();
    if ($course_check->get_result()->num_rows === 0) {
        sendResponse(false, 'Course not found or access denied');
    }
    $course_check->close();

    // Handle video upload - IGNORE upload errors and process anyway
    $video_uploaded = false;
    $video_path = null;
    $video_filename = null;
    
    if (isset($_FILES['video'])) {
        addDebug('files_video_exists', true);
        addDebug('video_name', $_FILES['video']['name']);
        addDebug('video_size', $_FILES['video']['size']);
        addDebug('video_error', $_FILES['video']['error']);
        addDebug('video_type', $_FILES['video']['type']);
        addDebug('video_tmp_name', $_FILES['video']['tmp_name']);
        
        error_log("=== DETAILED FILE DEBUG INFO ===");
        error_log("Video file info: name={$_FILES['video']['name']}, size={$_FILES['video']['size']}, error={$_FILES['video']['error']}");
        error_log("Video file type: {$_FILES['video']['type']}");
        error_log("Video tmp_name: {$_FILES['video']['tmp_name']}");
        
        // Get file info regardless of error status
        $video_name = $_FILES['video']['name'];
        $video_tmp = $_FILES['video']['tmp_name'];
        $video_size = $_FILES['video']['size'];
        $video_type = $_FILES['video']['type'];
        
        addDebug('temp_file_path', $video_tmp);
        addDebug('temp_file_exists', file_exists($video_tmp));
        if (file_exists($video_tmp)) {
            addDebug('temp_file_size_on_disk', filesize($video_tmp));
            addDebug('temp_file_readable', is_readable($video_tmp));
            addDebug('temp_file_permissions', substr(sprintf('%o', fileperms($video_tmp)), -4));
        }
        addDebug('is_uploaded_file', is_uploaded_file($video_tmp));
        
        error_log("Temp file path: $video_tmp");
        error_log("Temp file exists: " . (file_exists($video_tmp) ? 'YES' : 'NO'));
        if (file_exists($video_tmp)) {
            error_log("Temp file size on disk: " . filesize($video_tmp));
            error_log("Temp file is readable: " . (is_readable($video_tmp) ? 'YES' : 'NO'));
            error_log("Temp file permissions: " . substr(sprintf('%o', fileperms($video_tmp)), -4));
        }
        error_log("Is uploaded file: " . (is_uploaded_file($video_tmp) ? 'YES' : 'NO'));
        
        // Create upload directories
        $uploads_dir = '../uploads';
        $videos_dir = $uploads_dir . '/videos';
        $thumbnails_dir = $uploads_dir . '/thumbnails';
        
        error_log("Creating directories...");
        if (!file_exists($uploads_dir)) {
            $result = mkdir($uploads_dir, 0755, true);
            error_log("Created uploads dir: " . ($result ? 'SUCCESS' : 'FAILED'));
        }
        if (!file_exists($videos_dir)) {
            $result = mkdir($videos_dir, 0755, true);
            error_log("Created videos dir: " . ($result ? 'SUCCESS' : 'FAILED'));
        }
        if (!file_exists($thumbnails_dir)) {
            $result = mkdir($thumbnails_dir, 0755, true);
            error_log("Created thumbnails dir: " . ($result ? 'SUCCESS' : 'FAILED'));
        }
        
        error_log("Directory checks:");
        error_log("Uploads dir exists: " . (file_exists($uploads_dir) ? 'YES' : 'NO'));
        error_log("Uploads dir writable: " . (is_writable($uploads_dir) ? 'YES' : 'NO'));
        error_log("Videos dir exists: " . (file_exists($videos_dir) ? 'YES' : 'NO'));
        error_log("Videos dir writable: " . (is_writable($videos_dir) ? 'YES' : 'NO'));

        // Generate filename
        $file_extension = pathinfo($video_name, PATHINFO_EXTENSION);
        $video_filename = 'video_' . uniqid() . '_' . time() . '.' . $file_extension;
        $video_path = $videos_dir . '/' . $video_filename;
        error_log("Target file path: $video_path");

        // Try multiple upload methods
        if (file_exists($video_tmp)) {
            if (is_uploaded_file($video_tmp)) {
                addDebug('method_1_attempted', true);
                error_log("Method 1: Using move_uploaded_file");
                if (move_uploaded_file($video_tmp, $video_path)) {
                    $video_uploaded = true;
                    addDebug('method_1_success', true);
                    error_log("SUCCESS: Video moved to $video_path");
                } else {
                    addDebug('method_1_failed', true);
                    addDebug('method_1_error', error_get_last()['message'] ?? 'no error');
                    error_log("FAILED: move_uploaded_file failed");
                    error_log("Last error: " . error_get_last()['message'] ?? 'no error');
                }
            }
            
            if (!$video_uploaded) {
                error_log("Method 2: Using copy function");
                if (copy($video_tmp, $video_path)) {
                    $video_uploaded = true;
                    error_log("SUCCESS: Video copied to $video_path");
                } else {
                    error_log("FAILED: copy failed");
                    error_log("Last error: " . error_get_last()['message'] ?? 'no error');
                }
            }
            
            if (!$video_uploaded) {
                error_log("Method 3: Using file_get_contents/file_put_contents");
                $content = file_get_contents($video_tmp);
                if ($content !== false) {
                    if (file_put_contents($video_path, $content)) {
                        $video_uploaded = true;
                        error_log("SUCCESS: Video written to $video_path using file_put_contents");
                    } else {
                        error_log("FAILED: file_put_contents failed");
                    }
                } else {
                    error_log("FAILED: file_get_contents failed");
                }
            }
        } else {
            error_log("CRITICAL: Temp file does not exist: $video_tmp");
        }
        
        // Final verification
        if ($video_uploaded && file_exists($video_path)) {
            $final_size = filesize($video_path);
            error_log("FINAL CHECK: Uploaded file size: $final_size bytes");
            if ($final_size == 0) {
                error_log("ERROR: Uploaded file is empty!");
                unlink($video_path);
                $video_uploaded = false;
            }
        }
        
        addDebug('video_uploaded_final', $video_uploaded);
        error_log("=== END FILE DEBUG INFO ===");
    } else {
        addDebug('files_video_exists', false);
        error_log("CRITICAL: No video file in _FILES array");
    }

    if (!$video_uploaded) {
        addDebug('upload_failed_reason', 'Video processing failed after all attempts');
        sendResponse(false, 'Failed to process video file. Please check debug info.');
    }

    // Handle thumbnail upload (optional)
    $thumbnail_path = null;
    if (isset($_FILES['thumbnail']) && $_FILES['thumbnail']['error'] === UPLOAD_ERR_OK) {
        $thumbnail_file = $_FILES['thumbnail'];
        $thumbnail_name = $thumbnail_file['name'];
        $thumbnail_tmp = $thumbnail_file['tmp_name'];

        $thumb_extension = pathinfo($thumbnail_name, PATHINFO_EXTENSION);
        $thumbnail_filename = 'thumb_' . uniqid() . '_' . time() . '.' . $thumb_extension;
        $thumbnail_full_path = $thumbnails_dir . '/' . $thumbnail_filename;

        if (move_uploaded_file($thumbnail_tmp, $thumbnail_full_path)) {
            $thumbnail_path = 'uploads/thumbnails/' . $thumbnail_filename;
        }
    }

    // Format duration
    $duration_formatted = '';
    if ($duration > 0) {
        $hours = floor($duration / 3600);
        $minutes = floor(($duration % 3600) / 60);
        $seconds = floor($duration % 60);
        
        if ($hours > 0) {
            $duration_formatted = sprintf('%d:%02d:%02d', $hours, $minutes, $seconds);
        } else {
            $duration_formatted = sprintf('%d:%02d', $minutes, $seconds);
        }
    }

    // Insert into database
    $video_url = 'uploads/videos/' . $video_filename;
    
    $sql = "INSERT INTO videos (title, video_url, course_id, teacher_id, description, duration, thumbnail, price, category_name, is_free, is_published, module_order) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
    
    $stmt = $conn->prepare($sql);
    if (!$stmt) {
        error_log('Failed to prepare statement: ' . $conn->error);
        sendResponse(false, 'Database error occurred');
    }

    $stmt->bind_param(
        'ssiiissdssii',
        $title,
        $video_url,
        $course_id,
        $teacher_id,
        $description,
        $duration_formatted,
        $thumbnail_path,
        $price,
        $category_name,
        $is_free,
        $is_published,
        $module_order
    );

    if (!$stmt->execute()) {
        error_log('Failed to insert video record: ' . $stmt->error);
        // Clean up uploaded files
        if (file_exists($video_path)) unlink($video_path);
        if ($thumbnail_path && file_exists('../' . $thumbnail_path)) unlink('../' . $thumbnail_path);
        sendResponse(false, 'Failed to save video information');
    }

    $video_id = $conn->insert_id;
    $stmt->close();

    error_log("VIDEO UPLOAD SUCCESS: Video ID $video_id uploaded by teacher $teacher_id");

    sendResponse(true, 'Video uploaded successfully!', [
        'video_id' => $video_id,
        'video_url' => $video_url,
        'thumbnail' => $thumbnail_path,
        'title' => $title,
        'duration' => $duration_formatted
    ]);

} catch (Exception $e) {
    error_log('Exception occurred: ' . $e->getMessage());
    sendResponse(false, 'An error occurred: ' . $e->getMessage());
}
?>
