<?php
// Include the centralized session handling - this MUST be first
require_once __DIR__ . '/../includes/session_start.php';

// Show detailed errors for debugging
ini_set('display_errors', 1);
error_reporting(E_ALL);

// Include database connection
require_once __DIR__ . '/../includes/db_connection.php';
require_once __DIR__ . '/../includes/site_config.php';

// Check if user is logged in as teacher
if (!isset($_SESSION['user_role']) || $_SESSION['user_role'] !== 'teacher') {
    // Allow bypass for development
    $bypass_auth = isset($_GET['bypass_auth']) && $_GET['bypass_auth'] === 'true';
    
    if (!$bypass_auth) {
        header("Location: ../auth/login.php?redirect=" . urlencode($_SERVER['REQUEST_URI']));
        exit;
    }
}

// Get current page for navigation highlighting
$current_page = basename($_SERVER['PHP_SELF']);
// Get teacher ID from session
$teacher_id = $_SESSION['user_id'];

// Include the teacher layout
require_once('components/teacher_layout.php');

// Get teacher ID
$teacher_id = $_SESSION['user_id'];

// Enable error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Set page title
$page_title = "Published to Library";

// Check if library_submissions table exists, create it if it doesn't
$check_submissions_table = "SHOW TABLES LIKE 'library_submissions'";
$submissions_table_exists = $conn->query($check_submissions_table)->num_rows > 0;

if (!$submissions_table_exists) {
    // Create library_submissions table
    $create_submissions_table = "
        CREATE TABLE library_submissions (
            id INT AUTO_INCREMENT PRIMARY KEY,
            teacher_id INT NOT NULL,
            title VARCHAR(255) NOT NULL,
            description TEXT,
            category_id INT,
            tags TEXT,
            file_path VARCHAR(255),
            file_name VARCHAR(255),
            file_type VARCHAR(50),
            file_size INT,
            priority ENUM('low', 'medium', 'high') DEFAULT 'medium',
            thumbnail_path VARCHAR(255),
            submission_date DATETIME DEFAULT CURRENT_TIMESTAMP,
            status ENUM('pending', 'approved', 'rejected') DEFAULT 'pending',
            admin_notes TEXT,
            FOREIGN KEY (teacher_id) REFERENCES users(user_id) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
    ";
    
    if (!$conn->query($create_submissions_table)) {
        die("Error creating library_submissions table: " . $conn->error);
    }
}

// Get teacher's library submissions with error handling
$submissions = [];
$stmt = $conn->prepare("
    SELECT ls.*, c.category_name
    FROM library_submissions ls
    LEFT JOIN categories c ON ls.category_id = c.category_id
    WHERE ls.teacher_id = ?
    ORDER BY ls.submission_date DESC
");

if ($stmt) {
    $stmt->bind_param("i", $teacher_id);
    $stmt->execute();
    $result = $stmt->get_result();
    while ($row = $result->fetch_assoc()) {
        $submissions[] = $row;
    }
    $stmt->close();
} else {
    // If query fails (table doesn't exist), submissions array stays empty
    error_log("Library submissions query failed: " . $conn->error);
}

// Get categories for the upload form
$categories = [];
$result = $conn->query("SELECT category_id, category_name FROM categories ORDER BY category_name ASC");
if ($result) {
    while ($row = $result->fetch_assoc()) {
        $categories[] = $row;
    }
} else {
    // If categories table doesn't exist, create some default categories
    error_log("Categories query failed: " . $conn->error);
}

// Process form submission
$success_message = "";
$error_message = "";

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Check if it's a document submission request
    if (isset($_POST['action']) && $_POST['action'] === 'publish' && isset($_POST['material_id'])) {
        $material_id = intval($_POST['material_id']);
        
        // First, get the material details
        $material_query = $conn->prepare("
            SELECT * FROM course_materials 
            WHERE material_id = ? AND teacher_id = ?
        ");
        $material_query->bind_param("ii", $material_id, $teacher_id);
        $material_query->execute();
        $material_result = $material_query->get_result();
        
        if ($material_result->num_rows > 0) {
            $material = $material_result->fetch_assoc();
            
            // Check if this material already exists in the library
            // Use the same column name we detected above
            $check_query = $conn->prepare("
                SELECT resource_id FROM library_resources 
                WHERE $source_material_column = ?
            ");
            $check_query->bind_param("i", $material_id);
            $check_query->execute();
            $check_result = $check_query->get_result();
            
            if ($check_result->num_rows > 0) {
                // Update the existing library entry
                $resource = $check_result->fetch_assoc();
                $resource_id = $resource['resource_id'];
                
                $update_sql = "
                    UPDATE library_resources SET
                    title = ?,
                    description = ?,
                    type = ?,
                    format = ?,
                    file_path = ?,
                    author = (SELECT CONCAT(first_name, ' ', last_name) FROM users WHERE user_id = ?),
                    created_at = NOW(),
                    is_featured = 1
                    WHERE resource_id = ?
                ";
                
                $update_query = $conn->prepare($update_sql);
                
                if (!$update_query) {
                    $error_message = "Database error: Failed to prepare update statement. " . $conn->error;
                    echo "<div class='alert alert-danger'>$error_message</div>";
                } else {
                    // Determine the format based on file extension
                    $file_format = pathinfo($material['file_path'], PATHINFO_EXTENSION);
                    
                    $update_query->bind_param(
                        "sssssii", 
                        $material['title'], 
                        $material['description'], 
                        $material['category'], 
                        $file_format,
                        $material['file_path'],
                        $teacher_id,
                        $resource_id
                    );
                
                    if ($update_query->execute()) {
                        $success_message = "Material '{$material['title']}' has been updated in the library.";
                    } else {
                        $error_message = "Failed to update material in library: " . $conn->error;
                    }
                }
            } else {
                // Get all column names from library_resources table to ensure we're using existing columns
                $columns_query = "SHOW COLUMNS FROM library_resources";
                $columns_result = $conn->query($columns_query);
                $columns = [];
                while ($col = $columns_result->fetch_assoc()) {
                    $columns[] = $col['Field'];
                }
                
                // Check which columns exist (using actual column names)
                $downloads_exists = in_array('downloads', $columns);
                $is_featured_exists = in_array('is_featured', $columns);
                $tags_exists = in_array('tags', $columns);
                
                // Build the column list and values dynamically based on what exists
                $column_list = "title, description, type, format, file_path, author, date_added, $source_material_column";
                $values_list = "?, ?, ?, ?, ?, (SELECT CONCAT(first_name, ' ', last_name) FROM users WHERE user_id = ?), NOW(), ?";
                $param_types = "sssssii";
                
                // Add downloads if it exists (set to 0 initially)
                if ($downloads_exists) {
                    $column_list .= ", downloads";
                    $values_list .= ", 0";
                }
                
                // Add is_featured if it exists (set to 1 to mark as published/featured)
                if ($is_featured_exists) {
                    $column_list .= ", is_featured";
                    $values_list .= ", 1";
                }
                
                // Add tags if it exists (set to empty for now)
                if ($tags_exists) {
                    $column_list .= ", tags";
                    $values_list .= ", ''";
                }
                
                // Prepare the insert query
                $insert_query = $conn->prepare("
                    INSERT INTO library_resources (
                        $column_list
                    ) VALUES (
                        $values_list
                    )
                ");
                
                // Determine the format based on file extension
                $file_format = pathinfo($material['file_path'], PATHINFO_EXTENSION);
                
                // Create array of parameters
                $params = [
                    $material['title'], 
                    $material['description'], 
                    $material['category'], 
                    $file_format,
                    $material['file_path'],
                    $teacher_id,
                    $material_id
                ];
                
                // No additional parameters needed for downloads, is_featured, or tags
                // as they are set to fixed values in the SQL
                
                // Create dynamic bind_param call
                $bind_params = [$param_types];
                foreach ($params as &$param) {
                    $bind_params[] = &$param;
                }
                
                // Call bind_param with the dynamic parameters
                call_user_func_array([$insert_query, 'bind_param'], $bind_params);
                
                if ($insert_query->execute()) {
                    $success_message = "Material '{$material['title']}' has been published to the library.";
                } else {
                    $error_message = "Failed to publish material to library: " . $conn->error;
                }
            }
        } else {
            $error_message = "Material not found or you don't have permission to publish it.";
        }
    } elseif (isset($_POST['action']) && $_POST['action'] === 'unpublish' && isset($_POST['material_id'])) {
        $material_id = intval($_POST['material_id']);
        
        // Check if is_published column exists
        $check_published_column = "SHOW COLUMNS FROM library_resources LIKE 'is_published'";
        $has_published_column = $conn->query($check_published_column)->num_rows > 0;
        
        if ($has_published_column) {
            // Use is_published field if it exists
            $unpublish_query = $conn->prepare("
                UPDATE library_resources 
                SET is_published = 0 
                WHERE $source_material_column = ?
            ");
            $unpublish_query->bind_param("i", $material_id);
            
            if ($unpublish_query->execute()) {
                $success_message = "Material has been removed from the library.";
            } else {
                $error_message = "Failed to remove material from library: " . $conn->error;
            }
        } else {
            // Delete the record if is_published doesn't exist
            $delete_query = $conn->prepare("
                DELETE FROM library_resources 
                WHERE $source_material_column = ?
            ");
            $delete_query->bind_param("i", $material_id);
            
            if ($delete_query->execute()) {
                $success_message = "Material has been removed from the library.";
            } else {
                $error_message = "Failed to remove material from library: " . $conn->error;
            }
        }
    }
}

// Library resources table has already been checked/created at the top of the file

// Create library resource categories table if needed
$check_categories_query = "SHOW TABLES LIKE 'library_resource_categories'";
$check_categories_result = $conn->query($check_categories_query);
$categories_table_exists = ($check_categories_result->num_rows > 0);

if (!$categories_table_exists) {
    $create_categories_sql = "
        CREATE TABLE `library_resource_categories` (
            `category_id` int(11) NOT NULL AUTO_INCREMENT,
            `name` varchar(100) NOT NULL,
            `icon` varchar(100) DEFAULT 'bx-folder',
            `display_order` int(11) DEFAULT 0,
            PRIMARY KEY (`category_id`),
            UNIQUE KEY `name` (`name`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
    ";
    
    if ($conn->query($create_categories_sql) === TRUE) {
        // Insert default categories
        $default_categories = [
            ['Document', 'bx-file'],
            ['Video', 'bx-video'],
            ['Audio', 'bx-music'],
            ['Image', 'bx-image'],
            ['Presentation', 'bx-slideshow'],
            ['Spreadsheet', 'bx-table'],
            ['Archive', 'bx-archive'],
            ['Other', 'bx-category']
        ];
        
        $insert_category_sql = "INSERT INTO `library_resource_categories` (`name`, `icon`, `display_order`) VALUES (?, ?, ?)";
        $insert_stmt = $conn->prepare($insert_category_sql);
        
        $order = 0;
        foreach ($default_categories as $category) {
            $insert_stmt->bind_param("ssi", $category[0], $category[1], $order);
            $insert_stmt->execute();
            $order++;
        }
        
        $success_message .= " Library categories created successfully.";
    } else {
        $error_message .= " Error creating library categories table: " . $conn->error;
    }
}

// Materials query removed since we're now showing submissions history instead

// Include the centralized teacher layout
require_once __DIR__ . '/components/teacher_layout.php';
?>

<style>
    .library-materials {
        margin-bottom: 30px;
    }
    
    /* Material Cards */
    .material-card {
        background: #ffffff !important;
        border: 1px solid #f1f5f9 !important;
        border-radius: 12px !important;
        padding: 0 !important;
        transition: all 0.2s ease !important;
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.08) !important;
        height: 100% !important;
        overflow: hidden !important;
    }
    
    .material-card:hover {
        transform: translateY(-4px) !important;
        box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15) !important;
        border-color: #e5e7eb !important;
    }
    
    .material-header {
        padding: 20px !important;
        border-bottom: 1px solid #f1f5f9 !important;
    }
    
    .material-title {
        margin: 0 !important;
        font-weight: 600 !important;
        font-size: 16px !important;
        color: #1f2937 !important;
        line-height: 1.4 !important;
    }
    
    .material-badge {
        padding: 4px 12px !important;
        border-radius: 6px !important;
        font-size: 11px !important;
        font-weight: 500 !important;
        text-transform: uppercase !important;
        letter-spacing: 0.5px !important;
    }
    
    .badge-published {
        background-color: rgba(34, 197, 94, 0.1) !important;
        color: #16a34a !important;
    }
    
    .badge-unpublished {
        background-color: rgba(107, 114, 128, 0.1) !important;
        color: #6b7280 !important;
    }
    
    .material-body {
        padding: 20px !important;
        flex: 1 !important;
    }
    
    .material-footer {
        padding: 20px !important;
        border-top: 1px solid #f1f5f9 !important;
        display: flex !important;
        justify-content: space-between !important;
        align-items: center !important;
    }
    
    .material-info {
        display: flex;
        align-items: center;
    }
    
    .material-info span {
        margin-right: 15px;
        font-size: 0.85rem;
        color: #6c757d;
    }
    
    .material-info i {
        margin-right: 5px;
    }
    
    .material-actions .btn {
        margin-left: 5px;
    }
    
    .file-type-icon {
        width: 40px !important;
        height: 40px !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        border-radius: 8px !important;
        margin-right: 12px !important;
        font-size: 18px !important;
        flex-shrink: 0 !important;
    }
    
    .file-type-document {
        background-color: #e3f2fd;
        color: #0d6efd;
    }
    
    .file-type-video {
        background-color: #fff8e1;
        color: #ffa000;
    }
    
    .file-type-audio {
        background-color: #e8f5e9;
        color: #2e7d32;
    }
    
    .file-type-image {
        background-color: #f3e5f5;
        color: #9c27b0;
    }
    
    .file-type-other {
        background-color: #f5f5f5;
        color: #607d8b;
    }
    
    .material-meta {
        display: flex;
        align-items: center;
    }
    
    /* Modern Publish to Library Page Styling - Panadite Theme */
    * {
        font-family: 'Inter', 'Poppins', sans-serif !important;
    }

    .container-fluid {
        padding: 0 30px !important;
    }

    /* Page Header */
    h1.h3, .section-title {
        font-size: 28px !important;
        font-weight: 700 !important;
        color: #1f2937 !important;
        letter-spacing: -0.025em !important;
        margin-bottom: 8px !important;
    }

    .text-muted {
        color: #6b7280 !important;
        font-size: 16px !important;
    }

    /* Clean Header Section */
    .page-header {
        margin-bottom: 32px !important;
    }

    .page-description {
        font-size: 16px !important;
        color: #6b7280 !important;
        line-height: 1.6 !important;
        margin-top: 8px !important;
    }
    
    /* Modern Form Elements */
    .card {
        background: #ffffff !important;
        border: 1px solid #f1f5f9 !important;
        border-radius: 12px !important;
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.08) !important;
    }

    .card-header {
        background: #ffffff !important;
        border-bottom: 1px solid #f1f5f9 !important;
        padding: 20px !important;
    }

    .card-header h5 {
        font-size: 18px !important;
        font-weight: 600 !important;
        color: #1f2937 !important;
        margin: 0 !important;
    }

    .card-body {
        padding: 20px !important;
    }

    .btn {
        font-weight: 500 !important;
        padding: 8px 16px !important;
        border-radius: 6px !important;
        transition: all 0.2s ease !important;
        font-size: 13px !important;
        border: 1px solid transparent !important;
    }

    .btn-primary {
        background: #4361ee !important;
        border-color: #4361ee !important;
        color: #ffffff !important;
    }

    .btn-primary:hover {
        background: #3b50e0 !important;
        border-color: #3b50e0 !important;
        transform: translateY(-1px) !important;
    }

    .btn-outline-danger {
        border-color: #ef4444 !important;
        color: #ef4444 !important;
        background: transparent !important;
    }

    .btn-outline-danger:hover {
        background: #ef4444 !important;
        border-color: #ef4444 !important;
        color: #ffffff !important;
    }

    .alert {
        border-radius: 8px !important;
        border: none !important;
        padding: 16px 20px !important;
        font-size: 14px !important;
    }

    .alert-success {
        background: rgba(34, 197, 94, 0.1) !important;
        color: #16a34a !important;
    }

    .alert-danger {
        background: rgba(239, 68, 68, 0.1) !important;
        color: #dc2626 !important;
    }

    .alert-info {
        background: rgba(66, 97, 238, 0.1) !important;
        color: #4361ee !important;
    }

    /* Submission Card Styles */
    .submission-card {
        background: white;
        border-radius: 12px;
        padding: 1.5rem;
        margin-bottom: 1rem;
        box-shadow: 0 4px 12px rgba(0,0,0,0.05);
        border: 1px solid #e5e7eb;
        transition: all 0.3s ease;
    }

    .submission-card:hover {
        transform: translateY(-2px);
        box-shadow: 0 8px 20px rgba(0,0,0,0.1);
    }

    .status-badge {
        padding: 0.25rem 0.75rem;
        border-radius: 20px;
        font-size: 0.75rem;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }

    .status-pending {
        background: #fef3c7;
        color: #92400e;
    }

    .status-approved {
        background: #d1fae5;
        color: #065f46;
    }

    .status-rejected {
        background: #fee2e2;
        color: #991b1b;
    }

    .priority-badge {
        padding: 0.2rem 0.5rem;
        border-radius: 12px;
        font-size: 0.7rem;
        font-weight: 500;
    }

    .priority-high {
        background: #fef2f2;
        color: #dc2626;
    }

    .priority-medium {
        background: #fef3c7;
        color: #d97706;
    }

    .priority-low {
        background: #f0f9ff;
        color: #0369a1;
    }

    .category-badge {
        display: inline-flex;
        align-items: center;
        gap: 0.5rem;
        padding: 0.5rem 1rem;
        border-radius: 20px;
        font-size: 0.8rem;
        font-weight: 500;
        color: white;
    }
</style>

        <!-- Clean Page Header -->
        <div class="page-header d-flex justify-content-between align-items-center">
            <div>
                <h1 class="section-title d-flex align-items-center">
                    <i class='bx bx-library me-3' style="color: var(--primary-color); font-size: 32px;"></i>
                    Published to Library
                </h1>
                <p class="page-description">
                    Manage your library submissions and track their approval status. Submit new documents to the institutional library for review.
                </p>
            </div>
            <div>
                <a href="library_submissions.php" class="btn btn-primary">
                    <i class='bx bx-plus-circle'></i> Upload New Document
                </a>
            </div>
        </div>
    
    <?php if (!empty($success_message)): ?>
    <div class="alert alert-success alert-dismissible fade show" role="alert">
        <?= $success_message ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
    <?php endif; ?>
    
    <?php if (!empty($error_message)): ?>
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
        <?= $error_message ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
    <?php endif; ?>
    
    <!-- Submissions History -->
    <div class="row">
        <div class="col-12">
            <h3 class="mb-3"><i class="bx bx-history text-primary"></i> Your Submissions History</h3>
            
            <?php if (empty($submissions)): ?>
                <div class="text-center py-5">
                    <i class="bx bx-folder-open" style="font-size: 4rem; color: #d1d5db;"></i>
                    <h5 class="text-muted mt-3">No submissions yet</h5>
                    <p class="text-muted">Submit your first document to get started!</p>
                    <a href="library_submissions.php" class="btn btn-primary mt-3">
                        <i class='bx bx-plus-circle'></i> Upload First Document
                    </a>
                </div>
            <?php else: ?>
                <?php foreach ($submissions as $submission): ?>
                    <div class="submission-card">
                        <div class="row align-items-center">
                            <!-- Thumbnail Column -->
                            <div class="col-md-2">
                                <div class="d-flex align-items-center">
                                    <!-- Thumbnail Display -->
                                    <div class="me-3">
                                        <?php if (!empty($submission['thumbnail_path'])): ?>
                                            <img src="../<?= $submission['thumbnail_path'] ?>" 
                                                 alt="Thumbnail" 
                                                 style="width: 60px; height: 40px; object-fit: cover; border-radius: 4px; border: 1px solid #e5e7eb;"
                                                 onerror="this.parentNode.innerHTML='<div class=&quot;category-badge&quot; style=&quot;background-color: <?= $submission['color_code'] ?? '#6b7280' ?>; width: 60px; height: 40px; display: flex; align-items: center; justify-content: center;&quot;><i class=&quot;bx bx-file-blank&quot; style=&quot;font-size: 1.5rem;&quot;></i></div>'">
                                        <?php else: ?>
                                            <div class="category-badge" style="background-color: <?= $submission['color_code'] ?? '#6b7280' ?>; width: 60px; height: 40px; display: flex; align-items: center; justify-content: center;">
                                                <i class="<?= $submission['icon_class'] ?? 'bx-file' ?>" style="font-size: 1.5rem;"></i>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Content Column -->
                            <div class="col-md-5">
                                <h6 class="mb-1"><?= htmlspecialchars($submission['title']) ?></h6>
                                <p class="text-muted small mb-1"><?= htmlspecialchars(substr($submission['description'], 0, 100)) ?>...</p>
                                <small class="text-muted">
                                    <i class="bx bx-calendar"></i> <?= date('M j, Y', strtotime($submission['submission_date'])) ?>
                                    | <i class="bx bx-file"></i> <?= strtoupper($submission['file_type']) ?>
                                    | <i class="bx bx-hdd"></i> <?= number_format($submission['file_size'] / 1024, 1) ?> KB
                                </small>
                            </div>
                            
                            <!-- Priority Column -->
                            <div class="col-md-2">
                                <span class="priority-badge priority-<?= $submission['priority'] ?>">
                                    <?= ucfirst($submission['priority']) ?> Priority
                                </span>
                            </div>
                            
                            <!-- Status Column -->
                            <div class="col-md-2">
                                <span class="status-badge status-<?= $submission['status'] ?>">
                                    <?= ucfirst($submission['status']) ?>
                                </span>
                            </div>
                            
                            <!-- Actions Column -->
                            <div class="col-md-1 text-end">
                                <div class="dropdown">
                                    <button class="btn btn-sm btn-outline-secondary" type="button" data-bs-toggle="dropdown">
                                        <i class="bx bx-dots-vertical-rounded"></i>
                                    </button>
                                    <ul class="dropdown-menu dropdown-menu-end">
                                        <li>
                                            <a class="dropdown-item" href="../<?= $submission['file_path'] ?>" download>
                                                <i class="bx bx-download me-2"></i> Download
                                            </a>
                                        </li>
                                        <li><hr class="dropdown-divider"></li>
                                        <li>
                                            <a class="dropdown-item text-danger" href="#" onclick="deleteSubmission(<?= $submission['id'] ?>)">
                                                <i class="bx bx-trash me-2"></i> Delete
                                            </a>
                                        </li>
                                    </ul>
                                </div>
                            </div>
                        </div>
                        
                        <?php if (!empty($submission['admin_notes'])): ?>
                            <div class="row mt-2">
                                <div class="col-12">
                                    <div class="alert alert-info small mb-0">
                                        <strong>Admin Notes:</strong> <?= htmlspecialchars($submission['admin_notes']) ?>
                                    </div>
                                </div>
                            </div>
                        <?php endif; ?>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>
    </div>
</div>

<script>
    document.addEventListener('DOMContentLoaded', function() {
        // Enable tooltips
        var tooltips = document.querySelectorAll('[data-bs-toggle="tooltip"]');
        tooltips.forEach(function(tooltip) {
            new bootstrap.Tooltip(tooltip);
        });
        
        // Automatically dismiss alerts after 5 seconds
        setTimeout(function() {
            document.querySelectorAll('.alert').forEach(function(alert) {
                var bsAlert = new bootstrap.Alert(alert);
                bsAlert.close();
            });
        }, 5000);
    });

    // Delete submission function
    function deleteSubmission(submissionId) {
        if (confirm('Are you sure you want to delete this submission? This action cannot be undone.')) {
            // Create form and submit
            const form = document.createElement('form');
            form.method = 'POST';
            form.style.display = 'none';
            
            const actionInput = document.createElement('input');
            actionInput.type = 'hidden';
            actionInput.name = 'action';
            actionInput.value = 'delete_submission';
            
            const idInput = document.createElement('input');
            idInput.type = 'hidden';
            idInput.name = 'submission_id';
            idInput.value = submissionId;
            
            form.appendChild(actionInput);
            form.appendChild(idInput);
            document.body.appendChild(form);
            form.submit();
        }
    }
</script>

<style>
/* Targeted fixes for publish_to_library page only */
.page-content {
    padding: 20px 30px;
}

.container-fluid {
    padding-left: 15px;
    padding-right: 15px;
}
</style>

<?php require_once __DIR__ . '/components/teacher_layout_end.php'; ?>
