<?php
// Include the centralized teacher layout
require_once __DIR__ . '/components/teacher_layout.php';
require_once '../auth/functions.php';
requireRole('teacher');
require_once '../config/db_connect.php';

// Include ID encryption utility
require_once "../includes/id_encryption.php";

// Get current teacher ID
$teacher_id = $_SESSION['user_id'];

// Handle quiz actions (delete, publish/unpublish)
$message = '';
$error = '';

if (isset($_POST['action']) && isset($_POST['quiz_id'])) {
    $quiz_id = intval($_POST['quiz_id']);
    $action = $_POST['action'];
    
    // Verify quiz belongs to teacher
    $verify_stmt = $pdo->prepare("SELECT quiz_id FROM quizzes WHERE quiz_id = ? AND teacher_id = ?");
    $verify_stmt->execute([$quiz_id, $teacher_id]);
    
    if ($verify_stmt->rowCount() > 0) {
        switch ($action) {
            case 'delete':
                // Check for submissions and force delete flag
                $sub_check = $pdo->prepare("SELECT COUNT(*) FROM quiz_attempts WHERE quiz_id = ?");
                $sub_check->execute([$quiz_id]);
                $sub_count = $sub_check->fetchColumn();
                
                $force_delete = isset($_POST['force_delete']) && $_POST['force_delete'] == '1';
                
                // For testing purposes, always allow cascade deletion
                $pdo->beginTransaction();
                try {
                    // Delete quiz attempts (student submissions and scores)
                    if ($sub_count > 0) {
                        $pdo->prepare("DELETE FROM quiz_attempts WHERE quiz_id = ?")->execute([$quiz_id]);
                    }
                    
                    // Delete quiz question options
                    $pdo->prepare("DELETE FROM quiz_question_options WHERE question_id IN (SELECT question_id FROM quiz_questions WHERE quiz_id = ?)")->execute([$quiz_id]);
                    
                    // Delete quiz questions
                    $pdo->prepare("DELETE FROM quiz_questions WHERE quiz_id = ?")->execute([$quiz_id]);
                    
                    // Delete the quiz itself
                    $pdo->prepare("DELETE FROM quizzes WHERE quiz_id = ?")->execute([$quiz_id]);
                    
                    $pdo->commit();
                    
                    if ($sub_count > 0) {
                        $message = "Quiz deleted successfully along with {$sub_count} student attempt(s) and all related data.";
                    } else {
                        $message = "Quiz deleted successfully.";
                    }
                } catch (Exception $e) {
                    $pdo->rollBack();
                    error_log("Error deleting quiz: " . $e->getMessage());
                    $error = "Error deleting quiz. Please try again.";
                }
                break;
                
            case 'toggle_publish':
                $toggle_stmt = $pdo->prepare("UPDATE quizzes SET is_published = 1 - is_published WHERE quiz_id = ?");
                if ($toggle_stmt->execute([$quiz_id])) {
                    $message = "Quiz status updated successfully.";
                } else {
                    $error = "Error updating quiz status.";
                }
                break;
        }
    } else {
        $error = "Unauthorized action.";
    }
}

// Get teacher's courses
$courses_stmt = $pdo->prepare("SELECT course_id, title FROM courses WHERE teacher_id = ? ORDER BY title");
$courses_stmt->execute([$teacher_id]);
$courses = $courses_stmt->fetchAll(PDO::FETCH_ASSOC);

// Get course filter - handle both encrypted and legacy IDs
$course_filter = null;
if (isset($_GET['course_id'])) {
    if (is_numeric($_GET['course_id'])) {
        // Legacy integer ID (for backward compatibility)
        $course_filter = (int)$_GET['course_id'];
    } else {
        // Try to decrypt encrypted ID
        $course_filter = IdEncryption::decrypt($_GET['course_id']);
        if ($course_filter === false) {
            $course_filter = null; // Invalid encrypted ID, ignore filter
        }
    }
}

// Get quizzes with statistics
$quiz_query = "
    SELECT 
        q.quiz_id,
        q.course_id,
        q.title,
        q.description,
        q.time_limit,
        q.passing_score,
        q.is_published,
        q.created_at,
        c.title as course_title,
        COUNT(DISTINCT qq.question_id) as question_count,
        COUNT(DISTINCT qa.attempt_id) as submission_count
    FROM quizzes q
    JOIN courses c ON q.course_id = c.course_id
    LEFT JOIN quiz_questions qq ON q.quiz_id = qq.quiz_id
    LEFT JOIN quiz_attempts qa ON q.quiz_id = qa.quiz_id
    WHERE q.teacher_id = ?
";

$params = [$teacher_id];
if ($course_filter) {
    $quiz_query .= " AND q.course_id = ?";
    $params[] = $course_filter;
}

$quiz_query .= " GROUP BY q.quiz_id ORDER BY q.created_at DESC";

$quiz_stmt = $pdo->prepare($quiz_query);
$quiz_stmt->execute($params);
$quizzes = $quiz_stmt->fetchAll(PDO::FETCH_ASSOC);

// Get statistics
$total_quizzes = count($quizzes);
$published_quizzes = count(array_filter($quizzes, function($q) { return $q['is_published']; }));
$draft_quizzes = $total_quizzes - $published_quizzes;
$total_submissions = array_sum(array_column($quizzes, 'submission_count'));

$page_title = "Quiz Management";
?>

<!-- Additional CSS for quiz management -->
    
    <style>
        :root {
            --panadite-primary: #2563eb;
            --panadite-secondary: #1d4ed8;
            --panadite-light: #dbeafe;
            --panadite-dark: #1e40af;
            --panadite-accent: #f59e0b;
            --text-primary: #1f2937;
            --text-secondary: #6b7280;
            --border-color: #e5e7eb;
            --success-color: #10b981;
            --warning-color: #f59e0b;
            --error-color: #ef4444;
            --bg-light: #f8fafc;
        }

        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }

        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, sans-serif;
            background: linear-gradient(135deg, var(--bg-light) 0%, #e2e8f0 100%);
            color: var(--text-primary);
            line-height: 1.6;
            min-height: 100vh;
        }

        /* Header */
        .page-header {
            background: linear-gradient(135deg, var(--panadite-primary) 0%, var(--panadite-secondary) 100%);
            color: white;
            padding: 2rem 0;
            margin-bottom: 2rem;
            box-shadow: 0 4px 20px rgba(37, 99, 235, 0.15);
        }

        .header-content {
            max-width: 1200px;
            margin: 0 auto;
            padding: 0 1rem;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .page-title {
            font-size: 2.5rem;
            font-weight: 700;
            margin-bottom: 0.5rem;
            text-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }

        .page-subtitle {
            font-size: 1.1rem;
            opacity: 0.9;
            font-weight: 400;
        }

        /* Container */
        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 0 1rem 2rem;
        }

        /* Statistics Cards */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }

        .stat-card {
            background: white;
            border-radius: 16px;
            padding: 1.5rem;
            box-shadow: 0 4px 20px rgba(0,0,0,0.08);
            border: 2px solid transparent;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }

        .stat-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 4px;
            background: linear-gradient(90deg, var(--panadite-primary), var(--panadite-secondary));
        }

        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 8px 30px rgba(37, 99, 235, 0.15);
        }

        .stat-icon {
            width: 50px;
            height: 50px;
            background: linear-gradient(135deg, var(--panadite-light) 0%, rgba(37, 99, 235, 0.1) 100%);
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.5rem;
            color: var(--panadite-primary);
            margin-bottom: 1rem;
        }

        .stat-number {
            font-size: 2rem;
            font-weight: 700;
            color: var(--panadite-primary);
            margin-bottom: 0.25rem;
        }

        .stat-label {
            font-size: 0.9rem;
            color: var(--text-secondary);
            font-weight: 500;
        }

        /* Quiz Cards */
        .quiz-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
            gap: 1.5rem;
        }

        .quiz-card {
            background: white;
            border-radius: 20px;
            overflow: hidden;
            box-shadow: 0 6px 25px rgba(0,0,0,0.08);
            transition: all 0.3s ease;
            border: 2px solid transparent;
        }

        .quiz-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 12px 40px rgba(37, 99, 235, 0.15);
            border-color: var(--panadite-light);
        }

        .quiz-header {
            padding: 1.5rem;
            background: linear-gradient(135deg, var(--bg-light) 0%, #e2e8f0 50%);
            border-bottom: 1px solid var(--border-color);
        }

        .quiz-title {
            font-size: 1.3rem;
            font-weight: 600;
            color: var(--text-primary);
            margin-bottom: 0.5rem;
        }

        .quiz-course {
            font-size: 0.9rem;
            color: var(--text-secondary);
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .course-badge {
            background: var(--panadite-light);
            color: var(--panadite-primary);
            padding: 0.25rem 0.75rem;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 600;
        }

        .quiz-content {
            padding: 1.5rem;
        }

        .quiz-description {
            color: var(--text-secondary);
            font-size: 0.95rem;
            margin-bottom: 1rem;
            line-height: 1.5;
            display: -webkit-box;
            -webkit-line-clamp: 2;
            -webkit-box-orient: vertical;
            overflow: hidden;
        }

        .quiz-meta {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 1rem;
            margin-bottom: 1rem;
        }

        .meta-item {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            font-size: 0.9rem;
            color: var(--text-secondary);
        }

        .meta-icon {
            color: var(--panadite-primary);
        }

        /* Status Badges */
        .status-badge {
            padding: 0.5rem 1rem;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            display: inline-flex;
            align-items: center;
            gap: 0.3rem;
        }

        .status-published {
            background: #d1fae5;
            color: #059669;
        }

        .status-draft {
            background: #f3f4f6;
            color: #6b7280;
        }

        /* Action Buttons */
        .quiz-actions {
            display: flex;
            gap: 0.5rem;
            flex-wrap: wrap;
        }

        .action-btn {
            padding: 0.5rem 1rem;
            border: none;
            border-radius: 8px;
            font-weight: 500;
            font-size: 0.85rem;
            cursor: pointer;
            transition: all 0.3s ease;
            text-decoration: none;
            display: flex;
            align-items: center;
            gap: 0.4rem;
        }

        .btn-primary {
            background: var(--panadite-primary);
            color: white;
        }

        .btn-primary:hover {
            background: var(--panadite-secondary);
        }

        .btn-success {
            background: var(--success-color);
            color: white;
        }

        .btn-success:hover {
            background: #059669;
            color: white;
            transform: translateY(-1px);
        }

        .btn-warning {
            background: var(--warning-color);
            color: white;
        }

        .btn-warning:hover {
            background: #d97706;
            color: white;
            transform: translateY(-1px);
        }

        .btn-danger {
            background: var(--error-color);
            color: white;
        }

        .btn-danger:hover {
            background: #dc2626;
            color: white;
            transform: translateY(-1px);
        }

        .btn-info {
            background: #0ea5e9;
            color: white;
        }

        .btn-info:hover {
            background: #0284c7;
            color: white;
            transform: translateY(-1px);
        }

        .btn-outline {
            background: transparent;
            color: var(--panadite-primary);
            border: 1px solid var(--panadite-primary);
        }

        .btn-outline:hover {
            background: var(--panadite-primary);
            color: white;
        }

        /* Controls */
        .controls-section {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 2rem;
            flex-wrap: wrap;
            gap: 1rem;
        }

        .filter-controls {
            display: flex;
            gap: 1rem;
            align-items: center;
        }

        .form-select {
            border: 2px solid var(--border-color);
            border-radius: 8px;
            padding: 0.5rem 1rem;
            font-size: 0.9rem;
            min-width: 200px;
        }

        .form-select:focus {
            outline: none;
            border-color: var(--panadite-primary);
            box-shadow: 0 0 0 3px rgba(37, 99, 235, 0.1);
        }

        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 4rem 2rem;
            color: var(--text-secondary);
        }

        .empty-icon {
            font-size: 4rem;
            color: var(--border-color);
            margin-bottom: 1rem;
        }

        .empty-title {
            font-size: 1.5rem;
            font-weight: 600;
            margin-bottom: 0.5rem;
            color: var(--text-primary);
        }

        /* Alerts */
        .alert {
            padding: 1rem 1.5rem;
            border-radius: 8px;
            margin-bottom: 1.5rem;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }

        .alert-success {
            background: #d1fae5;
            color: #059669;
            border: 1px solid #a7f3d0;
        }

        .alert-danger {
            background: #fee2e2;
            color: #dc2626;
            border: 1px solid #fca5a5;
        }

        /* Mobile Responsive */
        @media (max-width: 768px) {
            .page-title {
                font-size: 2rem;
            }
            
            .quiz-grid {
                grid-template-columns: 1fr;
            }
            
            .stats-grid {
                grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            }
            
            .controls-section {
                flex-direction: column;
                align-items: stretch;
            }
        }
    </style>
</head>
<body>
    <?php include_once 'components/sidebar.php'; ?>
    
    <div class="page-header">
        <div class="header-content">
            <h1 class="page-title">
                <i class="bx bx-help-circle"></i>
                Quiz Management
            </h1>
            <p class="page-subtitle">Create, manage and monitor your course quizzes</p>
        </div>
    </div>

    <div class="container">
        <!-- Messages -->
        <?php if ($message): ?>
            <div class="alert alert-success">
                <i class="bx bx-check-circle"></i>
                <?= htmlspecialchars($message) ?>
            </div>
        <?php endif; ?>

        <?php if ($error): ?>
            <div class="alert alert-danger">
                <i class="bx bx-error-circle"></i>
                <?= htmlspecialchars($error) ?>
            </div>
        <?php endif; ?>

        <!-- Statistics -->
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="bx bx-collection"></i>
                </div>
                <div class="stat-number"><?= $total_quizzes ?></div>
                <div class="stat-label">Total Quizzes</div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="bx bx-show"></i>
                </div>
                <div class="stat-number"><?= $published_quizzes ?></div>
                <div class="stat-label">Published</div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="bx bx-edit"></i>
                </div>
                <div class="stat-number"><?= $draft_quizzes ?></div>
                <div class="stat-label">Drafts</div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="bx bx-user-check"></i>
                </div>
                <div class="stat-number"><?= $total_submissions ?></div>
                <div class="stat-label">Total Submissions</div>
            </div>
        </div>

        <!-- Controls -->
        <div class="controls-section">
            <div class="filter-controls">
                <select class="form-select" id="courseFilter" onchange="filterByCourse()">
                    <option value="">All Courses</option>
                    <?php foreach ($courses as $course): ?>
                        <option value="<?= $course['course_id'] ?>" <?= $course_filter == $course['course_id'] ? 'selected' : '' ?>>
                            <?= htmlspecialchars($course['title']) ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="d-flex gap-2">
                <a href="quiz_submissions.php" class="action-btn btn-info" style="font-size: 1rem; padding: 0.75rem 1.5rem;">
                    <i class="bx bx-clipboard-check"></i>
                    View Submissions
                </a>
                <a href="create_quiz.php" class="action-btn btn-primary" style="font-size: 1rem; padding: 0.75rem 1.5rem;">
                    <i class="bx bx-plus"></i>
                    Create New Quiz
                </a>
            </div>
        </div>

        <!-- Quiz Grid -->
        <?php if (empty($quizzes)): ?>
            <div class="empty-state">
                <div class="empty-icon">
                    <i class="bx bx-question-mark"></i>
                </div>
                <h3 class="empty-title">No Quizzes Found</h3>
                <p>You haven't created any quizzes yet. Click "Create New Quiz" to get started.</p>
                <a href="create_quiz.php" class="action-btn btn-primary" style="margin-top: 1rem;">
                    <i class="bx bx-plus"></i>
                    Create Your First Quiz
                </a>
            </div>
        <?php else: ?>
            <div class="quiz-grid">
                <?php foreach ($quizzes as $quiz): ?>
                    <div class="quiz-card" data-course-id="<?= $quiz['course_id'] ?>">
                        <div class="quiz-header">
                            <h3 class="quiz-title"><?= htmlspecialchars($quiz['title']) ?></h3>
                            <div class="quiz-course">
                                <i class="bx bx-book"></i>
                                <span class="course-badge"><?= htmlspecialchars($quiz['course_title']) ?></span>
                            </div>
                        </div>
                        
                        <div class="quiz-content">
                            <p class="quiz-description"><?= htmlspecialchars($quiz['description'] ?: 'No description provided') ?></p>
                            
                            <div class="quiz-meta">
                                <div class="meta-item">
                                    <i class="meta-icon bx bx-time"></i>
                                    <span><?= $quiz['time_limit'] ?> minutes</span>
                                </div>
                                <div class="meta-item">
                                    <i class="meta-icon bx bx-target-lock"></i>
                                    <span><?= $quiz['passing_score'] ?>% to pass</span>
                                </div>
                                <div class="meta-item">
                                    <i class="meta-icon bx bx-help-circle"></i>
                                    <span><?= $quiz['question_count'] ?> questions</span>
                                </div>
                                <div class="meta-item">
                                    <i class="meta-icon bx bx-user-check"></i>
                                    <span><?= $quiz['submission_count'] ?> submissions</span>
                                </div>
                            </div>
                            
                            <div class="status-badge status-<?= $quiz['is_published'] ? 'published' : 'draft' ?>">
                                <i class="bx <?= $quiz['is_published'] ? 'bx-show' : 'bx-hide' ?>"></i>
                                <?= $quiz['is_published'] ? 'Published' : 'Draft' ?>
                            </div>
                            
                            <div class="quiz-actions" style="margin-top: 1rem;">
                                <a href="edit_quiz.php?id=<?= $quiz['quiz_id'] ?>" class="action-btn btn-primary">
                                    <i class="bx bx-edit"></i>
                                    Edit
                                </a>
                                
                                <button class="action-btn <?= $quiz['is_published'] ? 'btn-warning' : 'btn-success' ?>" onclick="togglePublish(<?= $quiz['quiz_id'] ?>, <?= $quiz['is_published'] ? 'true' : 'false' ?>)">
                                    <i class="bx <?= $quiz['is_published'] ? 'bx-hide' : 'bx-show' ?>"></i>
                                    <?= $quiz['is_published'] ? 'Unpublish' : 'Publish' ?>
                                </button>
                                
                                <a href="quiz_submissions.php?id=<?= $quiz['quiz_id'] ?>" class="action-btn btn-outline">
                                    <i class="bx bx-bar-chart"></i>
                                    Results
                                </a>
                                
                                <button class="action-btn btn-danger" onclick="deleteQuiz(<?= $quiz['quiz_id'] ?>, '<?= htmlspecialchars($quiz['title']) ?>', <?= $quiz['submission_count'] ?>)">
                                    <i class="bx bx-trash"></i>
                                    Delete
                                </button>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Quiz action handlers - Enhanced for cascade deletion
        function deleteQuiz(quizId, quizTitle, submissionCount) {
            // Enhanced confirmation message for testing purposes
            const confirmMessage = `⚠️ DELETE QUIZ: "${quizTitle}"\n\n` +
                `This will permanently delete:\n` +
                `• The quiz\n` +
                `• ALL student attempts (if any: ${submissionCount})\n` +
                `• ALL grades and scores (if any)\n` +
                `• ALL quiz questions and answers\n\n` +
                `This action CANNOT be undone!\n\n` +
                `Are you sure you want to continue?`;
            
            if (confirm(confirmMessage)) {
                // Show loading state
                const deleteBtn = event.target.closest('button');
                const originalHTML = deleteBtn.innerHTML;
                deleteBtn.innerHTML = '<i class="bx bx-loader-alt bx-spin"></i> Deleting...';
                deleteBtn.disabled = true;
                
                const form = document.createElement('form');
                form.method = 'POST';
                form.innerHTML = `
                    <input type="hidden" name="action" value="delete">
                    <input type="hidden" name="quiz_id" value="${quizId}">
                    <input type="hidden" name="force_delete" value="1">
                `;
                document.body.appendChild(form);
                form.submit();
            }
        }

        function togglePublish(quizId, currentStatus) {
            const actionText = currentStatus ? 'unpublish' : 'publish';
            
            if (confirm(`Are you sure you want to ${actionText} this quiz?`)) {
                // Show loading state
                const publishBtn = event.target.closest('button');
                const originalHTML = publishBtn.innerHTML;
                publishBtn.innerHTML = '<i class="bx bx-loader-alt bx-spin"></i> Processing...';
                publishBtn.disabled = true;
                
                const form = document.createElement('form');
                form.method = 'POST';
                form.innerHTML = `
                    <input type="hidden" name="action" value="toggle_publish">
                    <input type="hidden" name="quiz_id" value="${quizId}">
                `;
                document.body.appendChild(form);
                form.submit();
            }
        }

        // Course filter functionality - Main function called by dropdown
        function filterByCourse() {
            const selectedCourse = document.getElementById('courseFilter').value;
            const quizCards = document.querySelectorAll('.quiz-card');
            
            console.log('Filtering by course:', selectedCourse); // Debug log
            
            quizCards.forEach(card => {
                const cardCourseId = card.dataset.courseId;
                console.log('Card course ID:', cardCourseId, 'Selected:', selectedCourse); // Debug log
                
                if (selectedCourse === '' || cardCourseId === selectedCourse) {
                    card.style.display = 'block';
                    card.style.opacity = '1';
                } else {
                    card.style.display = 'none';
                    card.style.opacity = '0';
                }
            });
        }

        // Also keep event listener for consistency
        document.getElementById('courseFilter').addEventListener('change', function() {
            filterByCourse();
        });

        // Animate cards on load
        document.addEventListener('DOMContentLoaded', function() {
            const cards = document.querySelectorAll('.quiz-card, .stat-card');
            cards.forEach((card, index) => {
                card.style.opacity = '0';
                card.style.transform = 'translateY(20px)';
                
                setTimeout(() => {
                    card.style.transition = 'all 0.6s ease';
                    card.style.opacity = '1';
                    card.style.transform = 'translateY(0)';
                }, index * 100);
            });
        });
    </script>

<style>
/* Targeted fixes for quizzes page only */
.page-content {
    padding: 20px 30px;
}

.container-fluid, .container {
    padding-left: 15px;
    padding-right: 15px;
}

.row {
    margin-left: 0;
    margin-right: 0;
}
</style>
</body>
</html>
