<?php
/**
 * Teacher Referral Commission Dashboard - Panadite Academy
 * Track earnings from student referrals
 */

// Set current page for navigation highlighting
$current_page = 'referrals.php';

// Include the centralized teacher layout (handles all authentication, session, DB connection)
require_once __DIR__ . '/components/teacher_layout.php';

require_once '../includes/referral_processor.php';

$teacher_id = $_SESSION['user_id'];

// Get user information
$user_query = "SELECT first_name, last_name, email FROM users WHERE user_id = ?";
$stmt = $conn->prepare($user_query);
$stmt->bind_param("i", $teacher_id);
$stmt->execute();
$user_data = $stmt->get_result()->fetch_assoc();

$teacher_name = trim($user_data['first_name'] . ' ' . $user_data['last_name']);

// Try to use secure referral system, fallback to legacy if it fails
$referral_code = null;

try {
    // Include secure referral generator
    require_once '../includes/secure_referral_generator.php';
    
    // Generate secure referral code (new format) or get existing one
    $referral_code = SecureReferralGenerator::getOrCreateSecureCode($teacher_id, $conn);
} catch (Exception $e) {
    // Log the error but continue with legacy system
    error_log("Secure referral system error for teacher: " . $e->getMessage());
}

// Fallback to legacy format if secure generation fails
if (!$referral_code) {
    $referral_code = strtoupper('T' . substr($user_data['first_name'], 0, 2) . substr($user_data['last_name'], 0, 2) . $teacher_id . rand(10, 99));
}

// Initialize referral processor
$referral_processor = new ReferralProcessor($conn);

// Get teacher referral statistics
$stats_query = "
    SELECT 
        COUNT(DISTINCT r.id) as total_referrals,
        COUNT(DISTINCT CASE WHEN r.status = 'completed' THEN r.id END) as successful_referrals,
        COUNT(DISTINCT CASE WHEN r.status = 'pending' THEN r.id END) as pending_referrals,
        COUNT(DISTINCT CASE WHEN r.status = 'registered' THEN r.id END) as registered_referrals,
        COALESCE(SUM(trc.commission_amount), 0) as total_commissions,
        COALESCE(SUM(CASE WHEN trc.status = 'pending' THEN trc.commission_amount END), 0) as pending_commissions,
        COALESCE(SUM(CASE WHEN trc.status = 'approved' THEN trc.commission_amount END), 0) as approved_commissions,
        COALESCE(SUM(CASE WHEN trc.status = 'paid' THEN trc.commission_amount END), 0) as paid_commissions
    FROM referrals r
    LEFT JOIN teacher_referral_commissions trc ON r.id = trc.referral_id
    WHERE r.referrer_id = ? AND r.referrer_type = 'teacher'
";
$stmt = $conn->prepare($stats_query);
$stmt->bind_param("i", $teacher_id);
$stmt->execute();
$stats = $stmt->get_result()->fetch_assoc();

// Get commission details
$commissions = $referral_processor->getTeacherCommissions($teacher_id);

// Get recent referrals with enhanced data
$recent_query = "
    SELECT 
        r.*,
        u.first_name,
        u.last_name,
        u.email,
        trc.commission_amount,
        trc.status as commission_status,
        trc.created_at as commission_date,
        c.title as course_title,
        trc.purchase_amount
    FROM referrals r
    LEFT JOIN users u ON r.referred_user_id = u.user_id
    LEFT JOIN teacher_referral_commissions trc ON r.id = trc.referral_id
    LEFT JOIN courses c ON trc.course_id = c.course_id
    WHERE r.referrer_id = ? AND r.referrer_type = 'teacher'
    ORDER BY r.created_at DESC
    LIMIT 15
";
$stmt = $conn->prepare($recent_query);
$stmt->bind_param("i", $teacher_id);
$stmt->execute();
$recent_referrals = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
?>

<style>
    :root {
        --primary-blue: #4285f4;
        --danger-red: #ea4335;
        --success-green: #34a853;
        --warning-orange: #fbbc04;
        --purple: #8b5cf6;
        --text-dark: #1f2937;
        --text-muted: #6b7280;
        --border-light: #e5e7eb;
        --bg-light: #f9fafb;
        --white: #ffffff;
    }

    .referral-header {
        background: linear-gradient(135deg, var(--purple) 0%, #7c3aed 100%);
        color: white;
        padding: 3rem 2rem;
        border-radius: 16px;
        margin-bottom: 2rem;
        text-align: center;
        box-shadow: 0 10px 30px rgba(139, 92, 246, 0.3);
    }

    .referral-header h1 {
        font-size: 2.5rem;
        margin-bottom: 0.5rem;
        font-weight: 700;
    }

    .stats-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
        gap: 1.5rem;
        margin-bottom: 2rem;
    }

    .stat-card {
        background: white;
        padding: 2rem;
        border-radius: 1rem;
        box-shadow: 0 4px 6px rgba(0, 0, 0, 0.05);
        border: 1px solid #e5e7eb;
        text-align: center;
        transition: transform 0.2s;
    }

    .stat-card:hover {
        transform: translateY(-2px);
    }

    .stat-icon {
        width: 60px;
        height: 60px;
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        margin: 0 auto 1rem;
        font-size: 1.5rem;
        color: white;
    }

    .stat-icon.referrals { background: linear-gradient(135deg, #3b82f6, #1d4ed8); }
    .stat-icon.successful { background: linear-gradient(135deg, #10b981, #059669); }
    .stat-icon.commissions { background: linear-gradient(135deg, #f59e0b, #d97706); }
    .stat-icon.paid { background: linear-gradient(135deg, #8b5cf6, #7c3aed); }

    .stat-number {
        font-size: 2rem;
        font-weight: bold;
        color: #1f2937;
        margin-bottom: 0.5rem;
    }

    .stat-label {
        color: #6b7280;
        font-size: 0.9rem;
    }

    .commission-section {
        background: white;
        padding: 2rem;
        border-radius: 1rem;
        box-shadow: 0 4px 6px rgba(0, 0, 0, 0.05);
        margin-bottom: 2rem;
        border: 1px solid #e5e7eb;
    }

    .commission-table {
        width: 100%;
        border-collapse: collapse;
        margin-top: 1rem;
    }

    .commission-table th,
    .commission-table td {
        padding: 1rem;
        text-align: left;
        border-bottom: 1px solid #e5e7eb;
    }

    .commission-table th {
        background: #f9fafb;
        font-weight: 600;
        color: #374151;
    }

    .commission-status {
        padding: 0.25rem 0.75rem;
        border-radius: 12px;
        font-size: 0.8rem;
        font-weight: 600;
    }

    .status-pending { background: #fef3c7; color: #92400e; }
    .status-approved { background: #d1fae5; color: #065f46; }
    .status-paid { background: #dbeafe; color: #1e40af; }

    .referral-code-section {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        padding: 2rem;
        border-radius: 1rem;
        margin-bottom: 2rem;
        text-align: center;
    }

    .code-display {
        background: rgba(255, 255, 255, 0.2);
        padding: 1rem 2rem;
        border-radius: 8px;
        font-size: 1.5rem;
        font-weight: bold;
        letter-spacing: 2px;
        margin: 1rem 0;
        display: inline-block;
    }

    .copy-button {
        background: rgba(255, 255, 255, 0.2);
        color: white;
        border: 2px solid rgba(255, 255, 255, 0.3);
        padding: 0.75rem 1.5rem;
        border-radius: 8px;
        cursor: pointer;
        font-weight: 600;
        transition: all 0.3s;
        margin: 0 0.5rem;
    }

    .copy-button:hover {
        background: rgba(255, 255, 255, 0.3);
    }

    .commission-info {
        background: #f0f9ff;
        border: 1px solid #bae6fd;
        padding: 1.5rem;
        border-radius: 8px;
        margin-bottom: 2rem;
    }

    .commission-info h4 {
        color: #0369a1;
        margin-bottom: 1rem;
    }

    .commission-info ul {
        color: #1e40af;
        line-height: 1.6;
    }
</style>

<!-- Teacher Referral Dashboard -->
<div class="container-fluid">
    <!-- Page Header -->
    <div class="page-header">
        <h1>Teacher Referral Program</h1>
        <div>
            <span class="badge bg-primary">Commission Rate: <?= $referral_processor->settings['teacher_commission_percentage'] ?? '5' ?>%</span>
        </div>
    </div>

    <!-- Header -->
    <div class="referral-header">
        <h1>💼 Teacher Referral Program</h1>
        <p>Earn commissions by referring students to Panadite Academy</p>
    </div>

    <!-- Commission Info -->
    <div class="commission-info">
        <h4>💡 How Teacher Referrals Work</h4>
        <ul>
            <li>Share your referral code with potential students</li>
            <li>When they register and make their first purchase, you earn commission</li>
            <li>You continue earning from ALL their future course purchases</li>
            <li>Commission rates are set by the admin (currently <?= $referral_processor->settings['teacher_commission_percentage'] ?? '5' ?>%)</li>
            <li>Note: Teacher-to-teacher referrals do not earn commissions</li>
        </ul>
    </div>

    <!-- Stats Grid -->
    <div class="stats-grid">
        <div class="stat-card">
            <div class="stat-icon referrals">
                <i class='bx bx-user-plus'></i>
            </div>
            <div class="stat-number"><?= $stats['total_referrals'] ?></div>
            <div class="stat-label">Total Referrals</div>
        </div>
        
        <div class="stat-card">
            <div class="stat-icon successful">
                <i class='bx bx-check-circle'></i>
            </div>
            <div class="stat-number"><?= $stats['successful_referrals'] ?></div>
            <div class="stat-label">Successful Referrals</div>
        </div>
        
        <div class="stat-card">
            <div class="stat-icon commissions">
                <i class='bx bx-money'></i>
            </div>
            <div class="stat-number">R<?= number_format($stats['total_commissions'], 2) ?></div>
            <div class="stat-label">Total Commissions</div>
        </div>

        <div class="stat-card">
            <div class="stat-icon paid">
                <i class='bx bx-wallet'></i>
            </div>
            <div class="stat-number">R<?= number_format($stats['paid_commissions'], 2) ?></div>
            <div class="stat-label">Paid Out</div>
        </div>
    </div>

    <!-- Referral Code Section -->
    <div class="referral-code-section">
        <h3>🔗 Your Teacher Referral Code</h3>
        <p>Share this code with students to start earning commissions</p>
        <div class="code-display" id="referralCode"><?= $referral_code ?></div>
        <div>
            <button class="copy-button" onclick="copyCode()">
                <i class='bx bx-copy'></i> Copy Code
            </button>
            <button class="copy-button" onclick="copyLink()">
                <i class='bx bx-link'></i> Copy Link
            </button>
        </div>
    </div>

    <!-- Commission History -->
    <div class="commission-section">
        <h3>💰 Commission History</h3>
        
        <?php if (empty($commissions)): ?>
            <div style="text-align: center; padding: 2rem; color: #6b7280;">
                <i class='bx bx-money' style="font-size: 3rem; opacity: 0.5;"></i>
                <p>No commissions earned yet. Start referring students to begin earning!</p>
            </div>
        <?php else: ?>
            <table class="commission-table">
                <thead>
                    <tr>
                        <th>Student</th>
                        <th>Course</th>
                        <th>Purchase Amount</th>
                        <th>Commission</th>
                        <th>Status</th>
                        <th>Date</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($commissions as $commission): ?>
                    <tr>
                        <td>
                            <?php 
                            $student_query = "SELECT first_name, last_name FROM users WHERE user_id = ?";
                            $stmt = $conn->prepare($student_query);
                            $stmt->bind_param("i", $commission['referred_student_id']);
                            $stmt->execute();
                            $student = $stmt->get_result()->fetch_assoc();
                            echo htmlspecialchars($student['first_name'] . ' ' . $student['last_name']);
                            ?>
                        </td>
                        <td>
                            <?php 
                            $course_query = "SELECT title FROM courses WHERE course_id = ?";
                            $stmt = $conn->prepare($course_query);
                            $stmt->bind_param("i", $commission['course_id']);
                            $stmt->execute();
                            $course = $stmt->get_result()->fetch_assoc();
                            echo htmlspecialchars($course['title']);
                            ?>
                        </td>
                        <td>R<?= number_format($commission['purchase_amount'], 2) ?></td>
                        <td>R<?= number_format($commission['commission_amount'], 2) ?></td>
                        <td>
                            <span class="commission-status status-<?= $commission['status'] === 'pending' ? 'approved' : $commission['status'] ?>">
                                <?= $commission['status'] === 'pending' ? 'Active' : ucfirst($commission['status']) ?>
                            </span>
                        </td>
                        <td><?= date('M j, Y', strtotime($commission['created_at'])) ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>
    </div>
</div>

<script>
function copyCode() {
    const code = document.getElementById('referralCode').textContent;
    navigator.clipboard.writeText(code).then(() => {
        showToast('Referral code copied to clipboard!', 'success');
    });
}

function copyLink() {
    const code = document.getElementById('referralCode').textContent;
    const link = `${window.location.origin}/auth/register.php?ref=${code}`;
    navigator.clipboard.writeText(link).then(() => {
        showToast('Referral link copied to clipboard!', 'success');
    });
}

function showToast(message, type) {
    // Simple toast notification
    const toast = document.createElement('div');
    toast.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        background: ${type === 'success' ? '#10b981' : '#ef4444'};
        color: white;
        padding: 1rem 1.5rem;
        border-radius: 8px;
        z-index: 1000;
        box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
    `;
    toast.textContent = message;
    document.body.appendChild(toast);
    
    setTimeout(() => {
        toast.remove();
    }, 3000);
}
</script>

<style>
/* Targeted fixes for referrals page only */
.page-content {
    padding: 20px 30px;
}

.container-fluid {
    padding-left: 15px;
    padding-right: 15px;
}

.row {
    margin-left: 0;
    margin-right: 0;
}
</style>

<?php
// Include the footer component
require_once 'components/footer.php';
?>
