<?php
// Set current page for active menu highlighting
$current_page = 'request-withdrawal.php';

// Include the centralized teacher layout (handles all authentication, session, DB connection)
require_once __DIR__ . '/components/teacher_layout.php';

// Include auth functions - SAME as admin courses (line 2)
require_once '../auth/functions.php';

// Include email service for withdrawal notifications - SAME as admin courses (line 3)
require_once '../config/fast_email_service.php';

// Bridge database connections - FIXED: Ensure DB constants are available first
if (!isset($pdo)) {
    try {
        // Create PDO connection using the same credentials as MySQLi
        $pdo = new PDO("mysql:host=" . DB_SERVER . ";dbname=" . DB_NAME, DB_USERNAME, DB_PASSWORD);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
        error_log("PDO connection created successfully for email service");
    } catch (Exception $e) {
        error_log("Failed to create PDO connection for email service: " . $e->getMessage());
        $pdo = null;
    }
}

// Get teacher ID from session (same pattern as other teacher pages)
$teacher_id = $_SESSION['user_id'] ?? null;

// Session fallback - same pattern as dashboard
if (!$teacher_id || !is_numeric($teacher_id)) {
    $teacher_id = 1; // Default fallback
    error_log("Withdrawal: Session user_id not set, using fallback");
}

$success_message = '';
$error_message = '';

// Handle withdrawal request submission
if ($_POST && isset($_POST['submit_withdrawal'])) {
    $amount = floatval($_POST['amount']);
    $bank_name = $_POST['bank_name'];
    $account_holder = $_POST['account_holder'];
    $account_number = $_POST['account_number'];
    $branch_code = $_POST['branch_code'];
    $account_type = $_POST['account_type'];
    $additional_notes = $_POST['additional_notes'] ?? '';
    
    // Validate amount
    if ($amount < 500 || $amount > 8725) {
        $error_message = "Invalid withdrawal amount. Must be between R500.00 and R8,725.00";
    } else {
        // Insert withdrawal request - ORIGINAL working system (simple & proven)
        $sql = "INSERT INTO withdrawal_requests (teacher_id, amount, bank_name, account_holder, account_number, branch_code, account_type, notes, status, requested_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'pending', NOW())";
        $stmt = $conn->prepare($sql);
        
        if ($stmt === false) {
            // Prepare failed - likely table doesn't exist or SQL syntax error
            error_log("SQL Prepare Error: " . $conn->error);
            $error_message = "Database error. Please contact support. Error: " . $conn->error;
        } else {
            // Prepare succeeded, now bind parameters (original format)
            if ($stmt->bind_param("idsssiss", $teacher_id, $amount, $bank_name, $account_holder, $account_number, $branch_code, $account_type, $additional_notes)) {
                if ($stmt->execute()) {
                    // Send immediate notification - EXACT same pattern as admin courses (lines 158-192)
                    $email_sent = sendWithdrawalRequestEmail($teacher_id, $amount, $bank_name, $account_holder, $account_number, $branch_code);
                    
                    if ($email_sent) {
                        $success_message = "Withdrawal request submitted successfully! ✉️ Confirmation email sent to you.";
                    } else {
                        $success_message = "Withdrawal request submitted successfully, but failed to send notification email.";
                        error_log("Failed to send withdrawal notification email for teacher_id: $teacher_id");
                    }
                } else {
                    error_log("SQL Execute Error: " . $stmt->error);
                    $error_message = "Error submitting withdrawal request: " . $stmt->error;
                }
            } else {
                error_log("SQL Bind Error: " . $stmt->error);
                $error_message = "Error preparing withdrawal request: " . $stmt->error;
            }
            $stmt->close();
        }
    }
}

// Get teacher's available balance - EXACT SAME calculation as dashboard.php
$total_earnings = 0;
$total_withdrawn = 0;
$available_balance = 0;

try {
    // Step 1: Get total earnings (same as dashboard page)
    $earnings_query = "SELECT SUM(amount) as total_earnings, COUNT(*) as transaction_count
                      FROM teacher_earnings 
                      WHERE teacher_id = ?";
    $stmt = $conn->prepare($earnings_query);
    if ($stmt) {
        $stmt->bind_param("i", $teacher_id);
        $stmt->execute();
        $earnings_result = $stmt->get_result();
        if ($earnings_result) {
            $row = $earnings_result->fetch_assoc();
            $total_earnings = $row['total_earnings'] ?? 0;
        }
        $stmt->close();
    }
    
    // Step 2: Get withdrawn amounts (EXACT same query as dashboard page)
    $withdrawn_query = "SELECT SUM(amount) as total_withdrawn 
                       FROM withdrawal_requests 
                       WHERE teacher_id = ? AND status IN ('approved', 'completed')";
    $stmt = $conn->prepare($withdrawn_query);
    if ($stmt) {
        $stmt->bind_param("i", $teacher_id);
        $stmt->execute();
        $withdrawn_result = $stmt->get_result()->fetch_assoc();
        $total_withdrawn = $withdrawn_result['total_withdrawn'] ?? 0;
        $stmt->close();
    }
    
    // Step 3: Get referral commissions (EXACT same query as dashboard)
    $total_referral_commissions = 0;
    $referral_query = "SELECT SUM(commission_amount) as total_referral_commissions
                      FROM teacher_referral_commissions 
                      WHERE teacher_id = ?";
    $stmt = $conn->prepare($referral_query);
    if ($stmt) {
        $stmt->bind_param("i", $teacher_id);
        $stmt->execute();
        $referral_result = $stmt->get_result();
        if ($referral_result) {
            $row = $referral_result->fetch_assoc();
            $total_referral_commissions = $row['total_referral_commissions'] ?? 0;
        }
        $stmt->close();
    }
    
    // Step 4: Calculate available balance including referral commissions (EXACT same as dashboard)
    $available_balance = ($total_earnings + $total_referral_commissions) - $total_withdrawn;
    
} catch (Exception $e) {
    error_log("Error calculating available balance in withdrawal page: " . $e->getMessage());
    $available_balance = 0; // Fallback to 0 on error
}

// Get withdrawal history
$history_query = "SELECT * FROM withdrawal_requests WHERE teacher_id = ? ORDER BY requested_at DESC LIMIT 10";
$stmt = $conn->prepare($history_query);
$stmt->bind_param("i", $teacher_id);
$stmt->execute();
$withdrawal_history = $stmt->get_result();

// ORIGINAL working email function - EXACT same pattern as admin courses
function sendWithdrawalRequestEmail($teacher_id, $amount, $bank_name, $account_holder, $account_number, $branch_code) {
    global $conn;
    
    try {
        // Get teacher details - EXACT same pattern as admin courses (lines 29-42)
        $stmt = $conn->prepare("
            SELECT u.first_name, u.last_name, u.email, u.user_id
            FROM users u 
            WHERE u.user_id = ?
        ");
        $stmt->bind_param("i", $teacher_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $teacher = $result->fetch_assoc();
        
        if (!$teacher) {
            error_log("Teacher not found for withdrawal email: $teacher_id");
            return false;
        }
        
        $teacher_name = $teacher['first_name'] . ' ' . $teacher['last_name'];
        $teacher_email = $teacher['email'];
        
        // Create SIMPLE notification email - like admin courses + security notice
        $subject = "💰 Withdrawal Request Submitted - Panadite Academy";
        $message = "
        <div style='max-width:600px; margin:0 auto; font-family:Arial,sans-serif;'>
            <div style='background:#28a745; color:white; padding:20px; text-align:center;'>
                <h1>💰 Withdrawal Request Submitted</h1>
            </div>
            <div style='padding:20px; background:white;'>
                <h2>Hello " . htmlspecialchars($teacher_name) . "! 👋</h2>
                <p>Your withdrawal request has been submitted successfully and is now under review.</p>
                
                <div style='background:#f8f9fa; padding:15px; border-radius:5px; margin:20px 0;'>
                    <h3>Withdrawal Details:</h3>
                    <p><strong>Amount:</strong> R" . number_format($amount, 2) . "</p>
                    <p><strong>Bank:</strong> " . htmlspecialchars($bank_name) . "</p>
                    <p><strong>Account Holder:</strong> " . htmlspecialchars($account_holder) . "</p>
                    <p><strong>Account Number:</strong> ****" . substr($account_number, -4) . "</p>
                    <p><strong>Branch Code:</strong> " . htmlspecialchars($branch_code) . "</p>
                </div>
                
                <div style='background:#f8d7da; border:1px solid #f5c6cb; padding:15px; border-radius:5px; margin:20px 0;'>
                    <p><strong>⚠️ Security Notice:</strong></p>
                    <p>If you did <strong>NOT</strong> make this withdrawal request, please contact our admin immediately at <strong>admin@panaditeacademy.com</strong> or through the support system.</p>
                </div>
                
                <p style='text-align:center;'>
                    <a href='https://panaditeacademy.com/teacher/request-withdrawal.php' 
                       style='background:#667eea; color:white; text-decoration:none; padding:15px 25px; border-radius:8px; font-weight:bold; display:inline-block;'>
                        View Withdrawal Requests
                    </a>
                </p>
            </div>
        </div>";
        
        // Try direct email sending - EXACT same as admin courses (lines 82-87)
        if (function_exists('sendEmailNow')) {
            $result = sendEmailNow($teacher_email, $subject, $message);
            if ($result) {
                return true;
            }
        }
        
        // Fallback to native PHP mail - EXACT same as admin courses (lines 89-94)
        $headers = "MIME-Version: 1.0" . "\r\n";
        $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
        $headers .= "From: noreply@panaditeacademy.com" . "\r\n";
        
        return mail($teacher_email, $subject, $message, $headers);
        
    } catch (Exception $e) {
        error_log("Error sending withdrawal email: " . $e->getMessage());
        return false;
    }
}
?>

<style>
.withdrawal-container {
    max-width: 1200px;
    margin: 0 auto;
    padding: 20px;
}

.balance-card {
    background: #f5f5f5; /* White smoke background */
    color: #374151; /* Dark gray text for readability */
    padding: 30px;
    border-radius: 15px;
    margin-bottom: 30px;
    text-align: center;
    box-shadow: 0 8px 25px rgba(0, 0, 0, 0.1);
    border: 1px solid #e5e7eb;
}

.balance-amount {
    font-size: 3rem;
    font-weight: 700;
    margin: 10px 0;
    text-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.bank-selection {
    background: white;
    padding: 30px;
    border-radius: 15px;
    box-shadow: 0 4px 15px rgba(0,0,0,0.08);
    margin-bottom: 30px;
}

.bank-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 20px;
    margin-top: 20px;
}

.bank-card {
    background: linear-gradient(135deg, #2563eb 0%, #1e40af 100%); /* Blue background */
    border: 2px solid #1d4ed8;
    border-radius: 12px;
    padding: 20px;
    text-align: center;
    cursor: pointer;
    transition: all 0.3s ease;
    position: relative;
    overflow: hidden;
    color: white; /* White text for blue background */
}

.bank-card:hover {
    border-color: #1d4ed8;
    transform: translateY(-5px);
    box-shadow: 0 12px 30px rgba(37, 99, 235, 0.4);
    background: linear-gradient(135deg, #1d4ed8 0%, #1e3a8a 100%); /* Darker blue on hover */
}

.bank-logo {
    width: 80px;
    height: 60px;
    object-fit: contain;
    margin: 0 auto 15px;
    display: block;
}

.bank-name {
    font-weight: 600;
    color: white; /* White text for blue background */
    font-size: 16px;
    margin: 0;
    text-shadow: 0 1px 2px rgba(0,0,0,0.1); /* Subtle text shadow for readability */
}

.withdrawal-modal {
    display: none;
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.7);
    z-index: 1000;
    justify-content: center;
    align-items: flex-start;
    padding: 20px;
    box-sizing: border-box;
    overflow-y: auto;
}

.modal-content {
    background: rgba(255, 255, 255, 0.95);
    backdrop-filter: blur(15px);
    border-radius: 16px;
    padding: 30px;
    width: 90%;
    max-width: 700px;
    max-height: 85vh;
    overflow-y: auto;
    border: 1px solid rgba(245, 158, 11, 0.3);
    box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.15), 0 10px 10px -5px rgba(0, 0, 0, 0.08);
    margin: 20px;
}

.modal-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 25px;
    padding-bottom: 15px;
    border-bottom: 2px solid rgba(245, 158, 11, 0.2);
}

.modal-title {
    font-size: 24px;
    font-weight: 700;
    color: #1e40af;
    margin: 0;
}

.close-modal {
    background: none;
    border: none;
    font-size: 28px;
    cursor: pointer;
    color: #6b7280;
    padding: 0;
    width: 30px;
    height: 30px;
    display: flex;
    align-items: center;
    justify-content: center;
}

.form-row {
    display: flex;
    flex-wrap: wrap;
    margin: 0 -10px;
    margin-bottom: 15px;
}

.form-group {
    margin-bottom: 20px;
    padding: 0 10px;
}

.col-6 {
    flex: 0 0 50%;
    max-width: 50%;
}

.col-12 {
    flex: 0 0 100%;
    max-width: 100%;
}

.form-label {
    display: block;
    color: #374151;
    font-weight: 600;
    margin-bottom: 6px;
    font-size: 14px;
}

.form-input {
    width: 100%;
    padding: 12px 15px;
    border: 2px solid #e5e7eb;
    border-radius: 8px;
    font-size: 16px;
    transition: border-color 0.3s ease;
    background: rgba(255, 255, 255, 0.8);
    box-sizing: border-box;
}

.form-input:focus {
    outline: none;
    border-color: #f59e0b;
    box-shadow: 0 0 0 3px rgba(245, 158, 11, 0.1);
}

.form-select {
    width: 100%;
    padding: 12px 15px;
    border: 2px solid #e5e7eb;
    border-radius: 8px;
    font-size: 16px;
    background: rgba(255, 255, 255, 0.9);
    cursor: pointer;
    box-sizing: border-box;
}

.submit-btn {
    background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
    color: white;
    border: none;
    padding: 15px 30px;
    border-radius: 8px;
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    width: 100%;
    transition: all 0.3s ease;
}

.submit-btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(245, 158, 11, 0.3);
}

.history-section {
    background: white;
    padding: 30px;
    border-radius: 15px;
    box-shadow: 0 4px 15px rgba(0,0,0,0.08);
}

.history-table {
    width: 100%;
    border-collapse: collapse;
    margin-top: 20px;
}

.history-table th {
    background: #f8fafc;
    padding: 15px;
    text-align: left;
    font-weight: 600;
    color: #374151;
    border-bottom: 2px solid #e5e7eb;
}

.history-table td {
    padding: 15px;
    border-bottom: 1px solid #e5e7eb;
    color: #6b7280;
}

.status-badge {
    display: inline-block;
    padding: 4px 12px;
    border-radius: 20px;
    font-size: 12px;
    font-weight: 600;
    text-transform: uppercase;
}

.status-pending {
    background: #fef3c7;
    color: #92400e;
}

.status-completed {
    background: #d1fae5;
    color: #065f46;
}

.status-rejected {
    background: #fee2e2;
    color: #991b1b;
}

.alert {
    padding: 15px 20px;
    border-radius: 8px;
    margin-bottom: 20px;
    font-weight: 500;
}

.alert-success {
    background: #d1fae5;
    color: #065f46;
    border: 1px solid #a7f3d0;
}

.alert-error {
    background: #fee2e2;
    color: #991b1b;
    border: 1px solid #fecaca;
}

@media (max-width: 768px) {
    .bank-grid {
        grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
        gap: 15px;
    }
    
    .balance-amount {
        font-size: 2.5rem;
    }
    
    .modal-content {
        margin: 20px;
        padding: 20px;
    }
}
</style>

<div class="withdrawal-container">
    <!-- Page Header -->
    <div style="margin-bottom: 30px;">
        <h1 style="color: #1e40af; font-size: 2.5rem; font-weight: 700; margin: 0;">Request Withdrawal</h1>
        <p style="color: #6b7280; font-size: 1.1rem; margin: 5px 0 0 0;">Withdraw your earnings to your bank account</p>
    </div>

    <!-- Success/Error Messages -->
    <?php if ($success_message): ?>
        <div class="alert alert-success"><?php echo $success_message; ?></div>
    <?php endif; ?>
    
    <?php if ($error_message): ?>
        <div class="alert alert-error"><?php echo $error_message; ?></div>
    <?php endif; ?>

    <!-- Available Balance Card -->
    <div class="balance-card">
        <h2 style="margin: 0; font-size: 1.2rem; opacity: 0.9;">Available Balance</h2>
        <div class="balance-amount">R<?php echo number_format($available_balance, 2); ?></div>
        <p style="margin: 0; opacity: 0.8;">Ready for withdrawal</p>
    </div>

    <!-- Bank Selection -->
    <div class="bank-selection">
        <h3 style="color: #1e40af; font-size: 1.5rem; margin: 0 0 10px 0; font-weight: 700;">Select Your Bank</h3>
        <p style="color: #6b7280; margin: 0;">Choose your bank to proceed with the withdrawal request</p>
        
        <div class="bank-grid">
            <div class="bank-card" onclick="openWithdrawalModal('Standard Bank')">
                <img src="../banks/standard_bank.png" alt="Standard Bank" class="bank-logo">
                <h4 class="bank-name">Standard Bank</h4>
            </div>
            
            <div class="bank-card" onclick="openWithdrawalModal('FNB')">
                <img src="../banks/FNB.svg" alt="FNB" class="bank-logo">
                <h4 class="bank-name">FNB</h4>
            </div>
            
            <div class="bank-card" onclick="openWithdrawalModal('ABSA')">
                <img src="../banks/absa.png" alt="ABSA" class="bank-logo">
                <h4 class="bank-name">ABSA</h4>
            </div>
            
            <div class="bank-card" onclick="openWithdrawalModal('Nedbank')">
                <img src="../banks/Nedbank.svg" alt="Nedbank" class="bank-logo">
                <h4 class="bank-name">Nedbank</h4>
            </div>
            
            <div class="bank-card" onclick="openWithdrawalModal('Capitec Bank')">
                <img src="../banks/capitec.svg" alt="Capitec Bank" class="bank-logo">
                <h4 class="bank-name">Capitec Bank</h4>
            </div>
            
            <div class="bank-card" onclick="openWithdrawalModal('Discovery Bank')">
                <img src="../banks/discovery_bank.svg" alt="Discovery Bank" class="bank-logo">
                <h4 class="bank-name">Discovery Bank</h4>
            </div>
            
            <div class="bank-card" onclick="openWithdrawalModal('TymeBank')">
                <img src="../banks/tymebank.png" alt="TymeBank" class="bank-logo">
                <h4 class="bank-name">TymeBank</h4>
            </div>
            
            <div class="bank-card" onclick="openWithdrawalModal('African Bank')">
                <img src="../banks/african-bank.png" alt="African Bank" class="bank-logo">
                <h4 class="bank-name">African Bank</h4>
            </div>
            
            <div class="bank-card" onclick="openWithdrawalModal('Old Mutual Bank')">
                <img src="../banks/old_mutual_bank.svg" alt="Old Mutual Bank" class="bank-logo">
                <h4 class="bank-name">Old Mutual Bank</h4>
            </div>
        </div>
    </div>

    <!-- Withdrawal History -->
    <div class="history-section">
        <h3 style="color: #1e40af; font-size: 1.5rem; margin: 0 0 10px 0; font-weight: 700;">Withdrawal History</h3>
        <p style="color: #6b7280; margin: 0;">Your recent withdrawal requests and their status</p>
        
        <?php if ($withdrawal_history->num_rows > 0): ?>
            <table class="history-table">
                <thead>
                    <tr>
                        <th>Date</th>
                        <th>Amount</th>
                        <th>Bank</th>
                        <th>Status</th>
                    </tr>
                </thead>
                <tbody>
                    <?php while ($row = $withdrawal_history->fetch_assoc()): ?>
                        <tr>
                            <td><?php echo date('M j, Y', strtotime($row['requested_at'])); ?></td>
                            <td>R<?php echo number_format($row['amount'], 2); ?></td>
                            <td><?php echo htmlspecialchars($row['bank_name']); ?></td>
                            <td>
                                <span class="status-badge status-<?php echo $row['status']; ?>">
                                    <?php echo ucfirst($row['status']); ?>
                                </span>
                            </td>
                        </tr>
                    <?php endwhile; ?>
                </tbody>
            </table>
        <?php else: ?>
            <p style="color: #6b7280; text-align: center; padding: 40px; background: #f9fafb; border-radius: 8px; margin-top: 20px;">
                No withdrawal requests yet. Make your first request above!
            </p>
        <?php endif; ?>
    </div>
</div>

<!-- Withdrawal Modal -->
<div id="withdrawalModal" class="withdrawal-modal">
    <div class="modal-content">
        <div class="modal-header">
            <div style="display: flex; align-items: center; gap: 15px;">
                <img id="selectedBankLogo" src="" alt="" style="width: 48px; height: 48px; object-fit: contain; display: none;">
                <h3 class="modal-title">Withdrawal Request</h3>
            </div>
            <button type="button" class="close-modal" onclick="closeWithdrawalModal()">&times;</button>
        </div>
        
        <form method="POST" id="withdrawalForm">
            <input type="hidden" name="submit_withdrawal" value="1">
            
            <!-- Full width fields -->
            <div class="form-row">
                <div class="form-group col-12">
                    <label class="form-label">Selected Bank</label>
                    <input type="text" name="bank_name" id="bankNameInput" class="form-input" readonly>
                </div>
            </div>
            
            <div class="form-row">
                <div class="form-group col-12">
                    <label class="form-label">Withdrawal Amount (R)</label>
                    <input type="number" name="amount" class="form-input" min="500" max="<?php echo $available_balance; ?>" step="0.01" required placeholder="Enter amount">
                    <small style="color: #6b7280; font-size: 12px;">Minimum: R500.00 | Maximum: R<?php echo number_format($available_balance, 2); ?></small>
                </div>
            </div>
            
            <!-- Two column layout for account details -->
            <div class="form-row">
                <div class="form-group col-6">
                    <label class="form-label">Account Holder Name</label>
                    <input type="text" name="account_holder" class="form-input" required placeholder="Full name as on bank account">
                </div>
                
                <div class="form-group col-6">
                    <label class="form-label">Account Number</label>
                    <input type="text" name="account_number" class="form-input" required placeholder="Your bank account number">
                </div>
            </div>
            
            <div class="form-row">
                <div class="form-group col-6">
                    <label class="form-label">Branch Code</label>
                    <input type="text" name="branch_code" class="form-input" required placeholder="6-digit branch code">
                </div>
                
                <div class="form-group col-6">
                    <label class="form-label">Account Type</label>
                    <select name="account_type" class="form-select" required>
                        <option value="">Select account type</option>
                        <option value="savings">Savings Account</option>
                        <option value="current">Current/Cheque Account</option>
                        <option value="transmission">Transmission Account</option>
                    </select>
                </div>
            </div>
            
            <div class="form-row">
                <div class="form-group col-12">
                    <label class="form-label">Additional Notes (Optional)</label>
                    <textarea name="additional_notes" class="form-input" rows="3" placeholder="Any additional information..."></textarea>
                </div>
            </div>
            
            <div class="form-row">
                <div class="col-12">
                    <button type="submit" class="submit-btn">
                        <i class="bx bx-paper-plane"></i> Submit Withdrawal Request
                    </button>
                </div>
            </div>
        </form>
    </div>
</div>

<script>
function openWithdrawalModal(bankName) {
    // Set bank name
    document.getElementById('bankNameInput').value = bankName;
    
    // Set bank logo
    const bankLogo = document.getElementById('selectedBankLogo');
    const bankLogos = {
        'Standard Bank': '../banks/standard_bank.png',
        'FNB': '../banks/FNB.svg',
        'ABSA': '../banks/absa.png',
        'Nedbank': '../banks/Nedbank.svg',
        'Capitec Bank': '../banks/capitec.svg',
        'Discovery Bank': '../banks/discovery_bank.svg',
        'TymeBank': '../banks/tymebank.png',
        'African Bank': '../banks/african-bank.png',
        'Old Mutual Bank': '../banks/old_mutual_bank.svg'
    };
    
    if (bankLogos[bankName]) {
        bankLogo.src = bankLogos[bankName];
        bankLogo.alt = bankName;
        bankLogo.style.display = 'block';
    }
    
    // Show modal
    document.getElementById('withdrawalModal').style.display = 'flex';
    document.body.style.overflow = 'hidden';
}

function closeWithdrawalModal() {
    document.getElementById('withdrawalModal').style.display = 'none';
    document.body.style.overflow = 'auto';
    document.getElementById('withdrawalForm').reset();
}

// Close modal when clicking outside
document.getElementById('withdrawalModal').addEventListener('click', function(e) {
    if (e.target === this) {
        closeWithdrawalModal();
    }
});

// Escape key to close modal
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        closeWithdrawalModal();
    }
});
</script>

<?php
// Include the centralized teacher layout end (closes HTML structure)
require_once __DIR__ . '/components/teacher_layout_end.php';
?>