<?php
/**
 * Session Fix for Teacher Dashboard
 * This script runs automatically on page load through the browser
 * MODIFIED: Added redirect loop prevention and only performs minimal checks
 */

// Ensure we have a valid session
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Only execute redirects if this script is being accessed directly
// This prevents the include in teacher_layout.php from causing redirects
if (basename($_SERVER['SCRIPT_NAME']) === 'session_fix.php') {
    // Get redirect target (default to dashboard)
    $redirect_to = isset($_GET['to']) ? $_GET['to'] : 'dashboard.php';
    
    // Check if session needs a reset
    $needs_reset = isset($_GET['reset']) && $_GET['reset'] === 'true';
    
    if ($needs_reset) {
        // Clear session and cookies
        $_SESSION = array();
        if (ini_get("session.use_cookies")) {
            $params = session_get_cookie_params();
            setcookie(session_name(), '', time() - 42000,
                $params["path"], $params["domain"],
                $params["secure"], $params["httponly"]
            );
        }
        session_destroy();
        
        // Go to login page
        header("Location: ../auth/login.php");
        exit;
    }
    
    // If valid session with teacher role, redirect to dashboard
    if (isset($_SESSION['user_role']) && $_SESSION['user_role'] === 'teacher') {
        header("Location: $redirect_to");
        exit;
    }
    
    // Otherwise redirect to login
    header("Location: ../auth/login.php?redirect=teacher/" . urlencode($redirect_to));
    exit;
}

// This is a JavaScript fix that will be included on teacher pages
// to help recover from authentication issues
$js_fix = <<<EOT
<script>
// Session recovery helper
function checkSessionStatus() {
    // If we've been on this page for more than 3 seconds with no content,
    // it might indicate an authentication issue
    setTimeout(function() {
        // Check if the main content container is empty or very small
        var contentArea = document.querySelector('.content-area') || 
                          document.querySelector('.content-wrapper') || 
                          document.querySelector('main');
        
        if (!contentArea || contentArea.offsetHeight < 100) {
            console.warn("Possible session issue detected - redirecting to recovery page");
            window.location.href = "session_fix.php?reset=true";
        }
    }, 3000);
}

// Execute on page load
document.addEventListener('DOMContentLoaded', checkSessionStatus);
</script>
EOT;

// Output the JavaScript fix
echo $js_fix;
?>
