<?php
// Include the centralized session handling
require_once __DIR__ . '/../includes/session_start.php';

// Enable error reporting during development
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Register a fatal error handler for capturing and displaying errors during development
function students_fatal_error_handler() {
    $error = error_get_last();
    if ($error !== NULL && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
        echo '<div class="alert alert-danger">';
        echo '<h4>Fatal Error Occurred:</h4>';
        echo '<p>' . $error['message'] . '</p>';
        echo '<p>File: ' . $error['file'] . ' on line ' . $error['line'] . '</p>';
        echo '</div>';
    }
}
register_shutdown_function('students_fatal_error_handler');

// Include database and functions after session is started
require_once __DIR__ . '/../includes/db_connection.php';
require_once __DIR__ . '/../includes/functions.php';

// Check if user is logged in and is a teacher
if (!isset($_SESSION['user_id']) || $_SESSION['user_role'] !== 'teacher') {
    header("Location: ../login.php");
    exit;
}

// Get teacher data with robust fallback
function students_get_teacher($conn, $teacher_id) {
    // For emergency/test account (ID: 999)
    if ($teacher_id == 999) {
        return [
            'user_id' => 999,
            'username' => 'emergency_teacher',
            'first_name' => 'Emergency',
            'last_name' => 'Teacher',
            'email' => 'emergency@example.com',
            'profile_image' => null,
            'bio' => 'Emergency teacher account',
            'expertise' => 'Emergency Teaching',
            'created_at' => date('Y-m-d H:i:s')
        ];
    }
    
    // Check if we have a valid database connection
    if (!$conn || $conn->connect_error) {
        error_log("Database connection failed in students.php");
        return [
            'user_id' => $teacher_id,
            'username' => 'db_error_user',
            'first_name' => 'Database', 
            'last_name' => 'Error',
            'email' => 'error@example.com',
            'profile_image' => null,
            'bio' => 'Could not connect to database',
            'expertise' => 'Error Handling',
            'created_at' => date('Y-m-d H:i:s')
        ];
    }
    
    try {
        $stmt = $conn->prepare("SELECT u.user_id, u.username, u.first_name, u.last_name, u.email, 
                                 u.created_at, tp.profile_image, tp.bio, tp.expertise 
                               FROM users u 
                               LEFT JOIN teacher_profiles tp ON u.user_id = tp.teacher_id 
                               WHERE u.user_id = ? AND u.user_role = 'teacher'");
        
        if ($stmt === false) {
            throw new Exception("Failed to prepare statement: " . $conn->error);
        }
        
        $stmt->bind_param("i", $teacher_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows > 0) {
            return $result->fetch_assoc();
        } else {
            error_log("Teacher with ID {$teacher_id} not found");
            // Return fallback data
            return [
                'user_id' => $teacher_id,
                'username' => 'unknown',
                'first_name' => 'Unknown',
                'last_name' => 'Teacher',
                'email' => 'unknown@example.com',
                'profile_image' => null,
                'bio' => 'No data available',
                'expertise' => 'Unknown',
                'created_at' => date('Y-m-d H:i:s')
            ];
        }
    } catch (Exception $e) {
        error_log("Error retrieving teacher: " . $e->getMessage());
        return [
            'user_id' => $teacher_id,
            'username' => 'error_user',
            'first_name' => 'Error',
            'last_name' => 'Retrieving',
            'email' => 'error@example.com',
            'profile_image' => null,
            'bio' => 'Error retrieving teacher data',
            'expertise' => 'Error Handling',
            'created_at' => date('Y-m-d H:i:s')
        ];
    }
}

$teacher_id = $_SESSION['user_id'];
$teacher = students_get_teacher($conn, $teacher_id);

// Functions for robust student data handling
function robust_count_teacher_students($conn, $teacher_id) {
    // Check if we have a valid connection
    if (!$conn || $conn->connect_error) {
        error_log("Database connection failed in robust_count_teacher_students");
        return 0;
    }
    
    try {
        // Check if teacher_student_assignments table exists
        $tableCheck = $conn->query("SHOW TABLES LIKE 'teacher_student_assignments'");
        if ($tableCheck === false || $tableCheck->num_rows === 0) {
            error_log("teacher_student_assignments table does not exist");
            return 0;
        }
        
        $stmt = $conn->prepare("SELECT COUNT(*) as count FROM teacher_student_assignments WHERE teacher_id = ? AND status = 'active'");
        if ($stmt === false) {
            error_log("Failed to prepare statement in robust_count_teacher_students: " . $conn->error);
            return 0;
        }
        
        $stmt->bind_param("i", $teacher_id);
        if (!$stmt->execute()) {
            error_log("Failed to execute statement in robust_count_teacher_students: " . $stmt->error);
            return 0;
        }
        
        $result = $stmt->get_result();
        if ($result && $row = $result->fetch_assoc()) {
            return $row['count'];
        } else {
            return 0;
        }
    } catch (Exception $e) {
        error_log("Error in robust_count_teacher_students: " . $e->getMessage());
        return 0;
    }
}

function robust_get_new_students_this_month($conn, $teacher_id) {
    // Check if we have a valid connection
    if (!$conn || $conn->connect_error) {
        error_log("Database connection failed in robust_get_new_students_this_month");
        return 0;
    }
    
    $currentMonthStart = date('Y-m-01');
    
    try {
        // Check if teacher_student_assignments table exists
        $tableCheck = $conn->query("SHOW TABLES LIKE 'teacher_student_assignments'");
        if ($tableCheck === false || $tableCheck->num_rows === 0) {
            error_log("teacher_student_assignments table does not exist");
            return 0;
        }
        
        $stmt = $conn->prepare("SELECT COUNT(*) as count FROM teacher_student_assignments 
                             WHERE teacher_id = ? 
                             AND status = 'active' 
                             AND assignment_date >= ?");
        if ($stmt === false) {
            error_log("Failed to prepare statement in robust_get_new_students_this_month: " . $conn->error);
            return 0;
        }
        
        $stmt->bind_param("is", $teacher_id, $currentMonthStart);
        if (!$stmt->execute()) {
            error_log("Failed to execute statement in robust_get_new_students_this_month: " . $stmt->error);
            return 0;
        }
        
        $result = $stmt->get_result();
        if ($result && $row = $result->fetch_assoc()) {
            return $row['count'];
        } else {
            return 0;
        }
    } catch (Exception $e) {
        error_log("Error in robust_get_new_students_this_month: " . $e->getMessage());
        return 0;
    }
}

function robust_get_avg_completion_rate($conn, $teacher_id) {
    // Check if we have a valid connection
    if (!$conn || $conn->connect_error) {
        error_log("Database connection failed in robust_get_avg_completion_rate");
        return 0;
    }
    
    try {
        // Check if enrollments and courses tables exist
        $enrollmentsCheck = $conn->query("SHOW TABLES LIKE 'enrollments'");
        $coursesCheck = $conn->query("SHOW TABLES LIKE 'courses'");
        
        if ($enrollmentsCheck === false || $enrollmentsCheck->num_rows === 0 || 
            $coursesCheck === false || $coursesCheck->num_rows === 0) {
            error_log("enrollments or courses table does not exist");
            return 0;
        }
        
        // Check which column exists in enrollments table
        $columnCheck = $conn->query("SHOW COLUMNS FROM enrollments LIKE 'completion_percentage'");
        if ($columnCheck === false || $columnCheck->num_rows === 0) {
            error_log("completion_percentage column does not exist in enrollments");
            return 0;
        }
        
        $stmt = $conn->prepare("SELECT AVG(e.completion_percentage) as avg_completion 
                             FROM enrollments e 
                             INNER JOIN courses c ON e.course_id = c.course_id 
                             WHERE c.teacher_id = ?");
        if ($stmt === false) {
            error_log("Failed to prepare statement in robust_get_avg_completion_rate: " . $conn->error);
            return 0;
        }
        
        $stmt->bind_param("i", $teacher_id);
        if (!$stmt->execute()) {
            error_log("Failed to execute statement in robust_get_avg_completion_rate: " . $stmt->error);
            return 0;
        }
        
        $result = $stmt->get_result();
        if ($result && $row = $result->fetch_assoc()) {
            return $row['avg_completion'] ?: 0;
        } else {
            return 0;
        }
    } catch (Exception $e) {
        error_log("Error in robust_get_avg_completion_rate: " . $e->getMessage());
        return 0;
    }
}

function get_teacher_students($conn, $teacher_id) {
    // Default empty array
    $students = [];
    
    // Check if we have a valid connection
    if (!$conn || $conn->connect_error) {
        error_log("Database connection failed in get_teacher_students");
        return $students;
    }
    
    try {
        // Check if necessary tables exist
        $usersCheck = $conn->query("SHOW TABLES LIKE 'users'");
        $tsaCheck = $conn->query("SHOW TABLES LIKE 'teacher_student_assignments'");
        
        if ($usersCheck === false || $usersCheck->num_rows === 0 || 
            $tsaCheck === false || $tsaCheck->num_rows === 0) {
            error_log("users or teacher_student_assignments table does not exist");
            return $students;
        }
        
        // Prepare SQL to get students assigned to this teacher
        $stmt = $conn->prepare("SELECT u.user_id, u.username, u.first_name, u.last_name, u.email, 
                                 u.created_at, sp.bio 
                                FROM users u 
                                INNER JOIN teacher_student_assignments tsa ON u.user_id = tsa.student_id 
                                LEFT JOIN student_profiles sp ON u.user_id = sp.student_id 
                                WHERE tsa.teacher_id = ? AND tsa.status = 'active'
                                ORDER BY u.first_name, u.last_name");
        
        if ($stmt === false) {
            error_log("Failed to prepare statement in get_teacher_students: " . $conn->error);
            return $students;
        }
        
        $stmt->bind_param('i', $teacher_id);
        if (!$stmt->execute()) {
            error_log("Failed to execute statement in get_teacher_students: " . $stmt->error);
            return $students;
        }
        
        $result = $stmt->get_result();
        
        while ($row = $result->fetch_assoc()) {
            // Calculate courses this student is enrolled in that are taught by this teacher
            $courses = get_student_courses($conn, $row['user_id'], $teacher_id);
            
            // Add student with courses to the array
            $row['courses'] = $courses;
            $students[] = $row;
        }
        
        return $students;
    } catch (Exception $e) {
        error_log("Error in get_teacher_students: " . $e->getMessage());
        return $students;
    }
}

function get_student_courses($conn, $student_id, $teacher_id) {
    $courses = [];
    
    // Check if we have a valid connection
    if (!$conn || $conn->connect_error) {
        error_log("Database connection failed in get_student_courses");
        return $courses;
    }
    
    try {
        // Check if necessary tables exist
        $coursesCheck = $conn->query("SHOW TABLES LIKE 'courses'");
        $enrollmentsCheck = $conn->query("SHOW TABLES LIKE 'enrollments'");
        
        if ($coursesCheck === false || $coursesCheck->num_rows === 0 || 
            $enrollmentsCheck === false || $enrollmentsCheck->num_rows === 0) {
            error_log("courses or enrollments table does not exist");
            return $courses;
        }
        
        // Check which ID column exists in enrollments
        $studentIdCheck = $conn->query("SHOW COLUMNS FROM enrollments LIKE 'student_id'");
        
        if ($studentIdCheck !== false && $studentIdCheck->num_rows > 0) {
            $stmt = $conn->prepare("SELECT c.course_id, c.title, c.thumbnail 
                                 FROM courses c 
                                 INNER JOIN enrollments e ON c.course_id = e.course_id 
                                 WHERE e.student_id = ? AND c.teacher_id = ?");
        } else {
            // Use user_id column instead
            $stmt = $conn->prepare("SELECT c.course_id, c.title, c.thumbnail 
                                 FROM courses c 
                                 INNER JOIN enrollments e ON c.course_id = e.course_id 
                                 WHERE e.user_id = ? AND c.teacher_id = ?");
        }
        
        if ($stmt === false) {
            error_log("Failed to prepare statement in get_student_courses: " . $conn->error);
            return $courses;
        }
        
        $stmt->bind_param('ii', $student_id, $teacher_id);
        if (!$stmt->execute()) {
            error_log("Failed to execute statement in get_student_courses: " . $stmt->error);
            return $courses;
        }
        
        $result = $stmt->get_result();
        
        while ($course = $result->fetch_assoc()) {
            $courses[] = $course;
        }
        
        return $courses;
    } catch (Exception $e) {
        error_log("Error in get_student_courses: " . $e->getMessage());
        return $courses;
    }
}

// Get student data with robust approach
$students = get_teacher_students($conn, $teacher_id);
$totalStudents = robust_count_teacher_students($conn, $teacher_id);
$newStudentsThisMonth = robust_get_new_students_this_month($conn, $teacher_id);
$avgCompletionRate = robust_get_avg_completion_rate($conn, $teacher_id);
$avgRating = 4.5; // Default placeholder rating since there's no table for it

// Set a default error message variable (will be empty if no errors)
$error_message = '';

// Log any connection errors to the PHP error log
if (!$conn || $conn->connect_error) {
    $error_message = "Database connection failed. Please try again later.";
    error_log("Database connection failed in students.php");
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Students - Panadite Academy</title>
    
    <!-- Favicon -->
    <link rel="shortcut icon" href="../assets/images/favicon.ico" type="image/x-icon">
    
    <!-- CSS Libraries -->
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/toastify-js/src/toastify.min.css">
    <link rel="stylesheet" href="https://cdn.datatables.net/1.13.6/css/dataTables.bootstrap5.min.css">
    
    <style>
        :root {
            --primary: #276BE4;
            --primary-light: #E5EDFD;
            --secondary: #6C757D;
            --success: #28A745;
            --danger: #DC3545;
            --warning: #FFC107;
            --info: #17A2B8;
            --light: #F8F9FA;
            --dark: #343A40;
            --sidebar-width: 250px;
        }
        
        body {
            font-family: 'Poppins', sans-serif;
            background-color: #f5f7fb;
            color: #333;
            display: flex;
            min-height: 100vh;
            margin: 0;
        }
        
        /* Dashboard container */
        .dashboard-container {
            display: flex;
            width: 100%;
        }
        
        /* Sidebar styles */
        .sidebar {
            width: var(--sidebar-width);
            background-color: whitesmoke;
            height: 100vh;
            position: fixed;
            left: 0;
            top: 0;
            overflow-y: auto;
            transition: all 0.3s ease;
            z-index: 1000;
            box-shadow: 0 0 15px rgba(0, 0, 0, 0.05);
        }
        
        /* Sidebar header */
        .sidebar-header {
            padding: 20px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-bottom: 1px solid rgba(0, 0, 0, 0.05);
        }
        
        .sidebar-brand {
            color: var(--primary);
            text-decoration: none;
            font-weight: 700;
            font-size: 1.2rem;
            display: flex;
            align-items: center;
            transition: all 0.2s ease;
        }
        
        .sidebar-brand:hover {
            opacity: 0.9;
        }
        
        .sidebar-brand img {
            height: 30px;
            margin-right: 10px;
        }
        
        /* Sidebar navigation */
        .sidebar-nav {
            padding: 0;
            list-style: none;
            margin: 0;
        }
        
        .sidebar-heading {
            font-size: 0.75rem;
            font-weight: 600;
            color: #666;
            padding: 20px 20px 8px;
            margin-bottom: 0;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .nav-item {
            margin-bottom: 5px;
        }
        
        .nav-link {
            display: flex;
            align-items: center;
            padding: 12px 20px;
            color: #495057;
            text-decoration: none;
            border-radius: 5px;
            margin: 0 10px;
            transition: all 0.2s ease;
        }
        
        .nav-link i {
            margin-right: 10px;
            font-size: 1.1rem;
            width: 20px;
            text-align: center;
            color: #555;
        }
        
        .nav-link:hover {
            background-color: rgba(39, 107, 228, 0.1);
            color: var(--primary);
        }
        
        .nav-link:hover i {
            color: var(--primary);
        }
        
        .nav-link.active {
            background-color: var(--primary-light);
            color: var(--primary);
            font-weight: 500;
            box-shadow: 0 2px 5px rgba(39, 107, 228, 0.1);
        }
        
        .nav-link.active i {
            color: var(--primary);
        }
        
        .nav-badge {
            background-color: var(--primary);
            color: white;
            font-size: 0.7rem;
            padding: 2px 6px;
            border-radius: 10px;
            margin-left: 10px;
        }
        
        /* Content wrapper */
        .content-wrapper {
            margin-left: var(--sidebar-width);
            width: calc(100% - var(--sidebar-width));
            max-width: 100%;
            transition: all 0.3s ease;
            padding: 20px;
        }
        
        /* Dashboard navbar */
        .dashboard-navbar {
            background-color: #fff;
            box-shadow: 0 2px 15px rgba(0, 0, 0, 0.05);
            border-radius: 10px;
            padding: 15px 20px;
            margin-bottom: 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        /* Student card */
        .student-card {
            transition: all 0.3s ease;
            border: none;
            box-shadow: 0 2px 15px rgba(0, 0, 0, 0.05);
        }
        
        .student-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 5px 20px rgba(0, 0, 0, 0.1);
        }
        
        .student-avatar {
            width: 60px;
            height: 60px;
            border-radius: 50%;
            background-color: var(--primary-light);
            display: flex;
            align-items: center;
            justify-content: center;
            color: var(--primary);
            font-size: 24px;
            font-weight: 700;
            margin-right: 15px;
        }
        
        .course-badge {
            font-size: 0.7rem;
            padding: 4px 8px;
            border-radius: 15px;
            margin-right: 5px;
            margin-bottom: 5px;
            display: inline-block;
            background-color: var(--primary-light);
            color: var(--primary);
        }
        
        /* DataTables customization */
        .dataTables_wrapper .dataTables_length, 
        .dataTables_wrapper .dataTables_filter {
            margin-bottom: 15px;
        }
        
        .dataTables_wrapper .dataTables_length select {
            padding: 5px 10px;
            border-radius: 5px;
            border: 1px solid #dee2e6;
        }
        
        .dataTables_wrapper .dataTables_filter input {
            padding: 5px 10px;
            border-radius: 5px;
            border: 1px solid #dee2e6;
            margin-left: 5px;
        }
        
        .dataTables_wrapper .dataTables_info,
        .dataTables_wrapper .dataTables_paginate {
            margin-top: 15px;
        }
        
        .dataTables_wrapper .dataTables_paginate .paginate_button {
            padding: 5px 10px;
            margin: 0 2px;
            border-radius: 5px;
        }
        
        .dataTables_wrapper .dataTables_paginate .paginate_button.current {
            background-color: var(--primary) !important;
            border-color: var(--primary) !important;
            color: white !important;
        }
        
        .dataTables_wrapper .dataTables_paginate .paginate_button:hover {
            background-color: var(--primary-light) !important;
            border-color: var(--primary-light) !important;
            color: var(--primary) !important;
        }
        
        .table-action-btn {
            width: 32px;
            height: 32px;
            border-radius: 50%;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            transition: all 0.2s ease;
            margin-right: 5px;
        }
        
        .status-active {
            background-color: rgba(40, 167, 69, 0.1);
            color: var(--success);
            padding: 3px 10px;
            border-radius: 20px;
            font-size: 0.75rem;
        }
        
        .status-inactive {
            background-color: rgba(108, 117, 125, 0.1);
            color: var(--secondary);
            padding: 3px 10px;
            border-radius: 20px;
            font-size: 0.75rem;
        }
             @media (max-width: 991px) {
            .sidebar {
                transform: translateX(-100%);
                z-index: 1050;
            }
            
            .content-wrapper {
                margin-left: 0;
                width: 100%;
                max-width: 100%;
            }
            
            .sidebar.show {
                transform: translateX(0);
            }
        }
    </style>
</head>
<body>
    <div class="dashboard-container">
        <!-- Sidebar -->
        <aside class="sidebar">
            <div class="sidebar-header">
                <a href="../index.php" class="sidebar-brand">
                    <img src="../assets/images/logo.jpeg" alt="Panadite Academy" onerror="this.src='../assets/images/favicon.ico';">
                    <span>Panadite Academy</span>
                </a>
            </div>
            
            <ul class="sidebar-nav">
                <li class="sidebar-heading">MAIN</li>
                <li class="nav-item">
                    <a href="dashboard.php" class="nav-link">
                        <i class="fas fa-home"></i>
                        <span>Dashboard</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="courses.php" class="nav-link">
                        <i class="fas fa-book"></i>
                        <span>Courses</span>
                        <?php 
                        // Course count badge
                        $course_count = 0;
                        try {
                            // Check if courses table exists
                            $course_check = $conn->query("SHOW TABLES LIKE 'courses'");
                            if ($course_check->num_rows > 0) {
                                $course_count_query = $conn->prepare("SELECT COUNT(*) as count FROM courses WHERE teacher_id = ?");
                                $course_count_query->bind_param('i', $teacher_id);
                                $course_count_query->execute();
                                $count_result = $course_count_query->get_result()->fetch_assoc();
                                $course_count = $count_result['count'];
                            }
                        } catch (Exception $e) {
                            error_log("Error getting course count: " . $e->getMessage());
                        }
                        ?>
                        <span class="badge bg-primary rounded-pill"><?php echo $course_count; ?></span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="videos.php" class="nav-link">
                        <i class="fas fa-video"></i>
                        <span>Videos</span>
                        <?php 
                        // Video count badge
                        $video_count = 0;
                        try {
                            // Check if videos table exists
                            $video_check = $conn->query("SHOW TABLES LIKE 'videos'");
                            if ($video_check->num_rows > 0) {
                                $video_count_query = $conn->prepare("SELECT COUNT(*) as count FROM videos WHERE teacher_id = ?");
                                $video_count_query->bind_param('i', $teacher_id);
                                $video_count_query->execute();
                                $count_result = $video_count_query->get_result()->fetch_assoc();
                                $video_count = $count_result['count'];
                            }
                        } catch (Exception $e) {
                            error_log("Error getting video count: " . $e->getMessage());
                        }
                        ?>
                        <span class="badge bg-primary rounded-pill"><?php echo $video_count; ?></span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="students.php" class="nav-link active">
                        <i class="fas fa-users"></i>
                        <span>Students</span>
                        <span class="badge bg-primary rounded-pill"><?php echo $totalStudents; ?></span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="upload-video.php" class="nav-link">
                        <i class="fas fa-upload"></i>
                        <span>Upload Video</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="create-course.php" class="nav-link">
                        <i class="fas fa-plus-circle"></i>
                        <span>Create Course</span>
                    </a>
                </li>
                
                <li class="sidebar-heading">TOOLS</li>
                
                <li class="nav-item">
                    <a href="analytics.php" class="nav-link">
                        <i class="fas fa-chart-line"></i>
                        <span>Analytics</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="quizzes.php" class="nav-link">
                        <i class="fas fa-question-circle"></i>
                        <span>Quizzes</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="assignments.php" class="nav-link">
                        <i class="fas fa-tasks"></i>
                        <span>Assignments</span>
                    </a>
                </li>
                
                <li class="sidebar-heading">ACCOUNT</li>
                
                <li class="nav-item">
                    <a href="profile.php" class="nav-link">
                        <i class="fas fa-user-circle"></i>
                        <span>Profile</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="settings.php" class="nav-link">
                        <i class="fas fa-cog"></i>
                        <span>Settings</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="../logout_handler.php" class="nav-link">
                        <i class="fas fa-sign-out-alt"></i>
                        <span>Logout</span>
                    </a>
                </li>
            </ul>
        </aside>
        
        <!-- Main Content Wrapper -->
        <div class="content-wrapper">
            <!-- Navbar -->
            <nav class="dashboard-navbar">
                <div class="d-flex align-items-center">
                    <button id="sidebarToggle" class="btn btn-sm btn-light me-2">
                        <i class="fas fa-bars"></i>
                    </button>
                    <h4 class="mb-0">Students</h4>
                </div>
                
                <div class="d-flex align-items-center">
                    <!-- Search -->
                    <div class="me-3 d-none d-md-block">
                        <form class="d-flex">
                            <input type="text" class="form-control form-control-sm" placeholder="Search...">
                        </form>
                    </div>
                    
                    <!-- Notifications -->
                    <div class="dropdown me-3">
                        <button class="btn btn-light position-relative" data-bs-toggle="dropdown">
                            <i class="fas fa-bell"></i>
                            <span class="position-absolute top-0 start-100 translate-middle badge rounded-pill bg-danger">
                                3
                            </span>
                        </button>
                        <div class="dropdown-menu dropdown-menu-end">
                            <div class="p-2 border-bottom">
                                <h6 class="mb-0">Notifications</h6>
                            </div>
                            <div class="p-2 border-bottom">
                                <div class="d-flex">
                                    <div class="flex-shrink-0 me-2">
                                        <div class="rounded-circle bg-light p-2">
                                            <i class="fas fa-user-graduate text-primary"></i>
                                        </div>
                                    </div>
                                    <div>
                                        <p class="mb-0">New student enrolled</p>
                                        <small class="text-muted">2 hours ago</small>
                                    </div>
                                </div>
                            </div>
                            <div class="p-2 text-center">
                                <a href="#" class="text-decoration-none">View all notifications</a>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Profile -->
                    <div class="dropdown">
                        <button class="btn btn-light d-flex align-items-center" data-bs-toggle="dropdown">
                            <img src="<?php echo !empty($teacher['profile_picture']) ? '../uploads/profile/' . htmlspecialchars($teacher['profile_picture']) : '../assets/images/avatar-placeholder.jpg'; ?>" 
                                 class="rounded-circle me-2" height="32" width="32" alt="<?php echo htmlspecialchars($teacher['first_name']); ?>">
                            <span class="d-none d-md-inline"><?php echo htmlspecialchars($teacher['first_name']); ?></span>
                        </button>
                        <div class="dropdown-menu dropdown-menu-end">
                            <a href="profile.php" class="dropdown-item">
                                <i class="fas fa-user me-2"></i> Profile
                            </a>
                            <a href="settings.php" class="dropdown-item">
                                <i class="fas fa-cog me-2"></i> Settings
                            </a>
                            <div class="dropdown-divider"></div>
                            <a href="../logout.php" class="dropdown-item">
                                <i class="fas fa-sign-out-alt me-2"></i> Logout
                            </a>
                        </div>
                    </div>
                </div>
            </nav>

            <!-- Main Content -->
            <div class="container-fluid p-0">
                <!-- Page Header -->
                <div class="row mb-4">
                    <div class="col-12">
                        <div class="bg-white p-4 rounded shadow-sm">
                            <div class="d-flex justify-content-between align-items-center mb-4">
                                <h5 class="mb-0">Student Management</h5>
                                <div>
                                    <?php if (!empty($students)): ?>
                                    <button class="btn btn-outline-primary btn-sm me-2" id="filterBtn">
                                        <i class="fas fa-filter me-1"></i> Filter
                                    </button>
                                    <button class="btn btn-primary btn-sm" data-bs-toggle="modal" data-bs-target="#emailAllModal">
                                        <i class="fas fa-envelope me-1"></i> Email All
                                    </button>
                                    <?php endif; ?>
                                </div>
                            </div>
                            
                            <div class="row mb-4">
                                <div class="col-md-3 mb-3 mb-md-0">
                                    <div class="card h-100 border-0 shadow-sm">
                                        <div class="card-body d-flex align-items-center">
                                            <div class="stat-icon bg-primary-light rounded me-3">
                                                <i class="fas fa-users text-primary"></i>
                                            </div>
                                            <div>
                                                <h3 class="mb-0"><?php echo $totalStudents; ?></h3>
                                                <p class="text-muted mb-0">Total Students</p>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="col-md-3 mb-3 mb-md-0">
                                    <div class="card h-100 border-0 shadow-sm">
                                        <div class="card-body d-flex align-items-center">
                                            <div class="stat-icon bg-success-light rounded me-3">
                                                <i class="fas fa-user-plus text-success"></i>
                                            </div>
                                            <div>
                                                <h3 class="mb-0"><?php echo $newStudentsThisMonth; ?></h3>
                                                <p class="text-muted mb-0">New This Month</p>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="col-md-3 mb-3 mb-md-0">
                                    <div class="card h-100 border-0 shadow-sm">
                                        <div class="card-body d-flex align-items-center">
                                            <div class="stat-icon bg-warning-light rounded me-3">
                                                <i class="fas fa-chart-line text-warning"></i>
                                            </div>
                                            <div>
                                                <h3 class="mb-0"><?php echo round($avgCompletionRate); ?>%</h3>
                                                <p class="text-muted mb-0">Completion Rate</p>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="col-md-3">
                                    <div class="card h-100 border-0 shadow-sm">
                                        <div class="card-body d-flex align-items-center">
                                            <div class="stat-icon bg-info-light rounded me-3">
                                                <i class="fas fa-star text-info"></i>
                                            </div>
                                            <div>
                                                <h3 class="mb-0"><?php echo number_format($avgRating, 1); ?>/5</h3>
                                                <p class="text-muted mb-0">Average Rating</p>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <?php if (empty($students) && !isset($error_message)): ?>
                            <div class="alert alert-info">
                                <i class="fas fa-info-circle me-2"></i>
                                No students have been assigned to you yet. Admin will assign students to your account.
                            </div>
                            <?php elseif (isset($error_message)): ?>
                            <div class="alert alert-danger">
                                <i class="fas fa-exclamation-circle me-2"></i>
                                <?php echo $error_message; ?>
                            </div>
                            <?php endif; ?>
                            
                            <!-- Students Table -->
                            <?php if (!empty($students)): ?>
                            <div class="table-responsive mt-4">
                                <table id="studentsTable" class="table table-hover">
                                    <thead>
                                        <tr>
                                            <th>ID</th>
                                            <th>Student</th>
                                            <th>Email</th>
                                            <th>Enrolled Courses</th>
                                            <th>Joined Date</th>
                                            <th>Status</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($students as $student): 
                                            // Format date
                                            $joined_date = new DateTime($student['created_at']);
                                            $formatted_date = $joined_date->format('M j, Y');
                                            
                                            // Generate student ID with prefix
                                            $student_display_id = 'STU' . str_pad($student['user_id'], 3, '0', STR_PAD_LEFT);
                                        ?>
                                        <tr>
                                            <td><?php echo $student_display_id; ?></td>
                                            <td>
                                                <div class="d-flex align-items-center">
                                                    <div class="student-avatar">
                                                        <div class="avatar-placeholder rounded-circle d-flex align-items-center justify-content-center bg-primary-light text-primary">
                                                            <?php echo strtoupper(substr($student['first_name'], 0, 1) . substr($student['last_name'], 0, 1)); ?>
                                                        </div>
                                                    </div>
                                                    <div class="ms-3">
                                                        <div class="fw-bold"><?php echo htmlspecialchars($student['first_name'] . ' ' . $student['last_name']); ?></div>
                                                        <div class="text-muted small">@<?php echo htmlspecialchars($student['username']); ?></div>
                                                    </div>
                                                </div>
                                            </td>
                                            <td><?php echo htmlspecialchars($student['email']); ?></td>
                                            <td>
                                                <div class="course-badges">
                                                    <?php if (empty($student['courses'])): ?>
                                                    <span class="text-muted">No courses enrolled</span>
                                                    <?php else: 
                                                        foreach ($student['courses'] as $index => $course): 
                                                            if ($index < 3): // Show only first 3 courses
                                                    ?>
                                                    <span class="badge bg-primary course-badge"><?php echo htmlspecialchars($course['title']); ?></span>
                                                    <?php 
                                                            endif;
                                                        endforeach; 
                                                        
                                                        // Show count for additional courses
                                                        if (count($student['courses']) > 3): 
                                                    ?>
                                                    <span class="badge bg-secondary course-badge">+<?php echo count($student['courses']) - 3; ?> more</span>
                                                    <?php endif; ?>
                                                    <?php endif; ?>
                                                </div>
                                            </td>
                                            <td><?php echo $formatted_date; ?></td>
                                            <td>
                                                <span class="badge bg-success">Active</span>
                                            </td>
                                            <td class="text-end">
                                                <a href="mailto:<?php echo htmlspecialchars($student['email']); ?>" class="btn btn-light table-action-btn" title="Send Email">
                                                    <i class="fas fa-envelope text-primary"></i>
                                                </a>
                                                <a href="view-student.php?id=<?php echo $student['user_id']; ?>" class="btn btn-light table-action-btn" title="View Student">
                                                    <i class="fas fa-eye text-info"></i>
                                                </a>
                                                <div class="dropdown d-inline-block">
                                                    <button class="btn btn-light table-action-btn" data-bs-toggle="dropdown">
                                                        <i class="fas fa-ellipsis-v"></i>
                                                    </button>
                                                    <div class="dropdown-menu dropdown-menu-end">
                                                        <a href="view-student.php?id=<?php echo $student['user_id']; ?>" class="dropdown-item">View Details</a>
                                                        <a href="javascript:void(0)" class="dropdown-item send-message" data-student-id="<?php echo $student['user_id']; ?>" data-student-name="<?php echo htmlspecialchars($student['first_name'] . ' ' . $student['last_name']); ?>">Send Message</a>
                                                        <a href="javascript:void(0)" class="dropdown-item text-danger toggle-status" data-student-id="<?php echo $student['user_id']; ?>" data-status="inactive">Deactivate Student</a>
                                                    </div>
                                                </div>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div><!-- End content-wrapper -->
    </div><!-- End dashboard-container -->
    
    <!-- Email All Modal -->
    <div class="modal fade" id="emailAllModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Email All Students</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <form>
                        <div class="mb-3">
                            <label for="emailSubject" class="form-label">Subject</label>
                            <input type="text" class="form-control" id="emailSubject" placeholder="Email subject">
                        </div>
                        <div class="mb-3">
                            <label for="emailBody" class="form-label">Message</label>
                            <textarea class="form-control" id="emailBody" rows="5" placeholder="Type your message here..."></textarea>
                        </div>
                        <div class="form-check mb-3">
                            <input class="form-check-input" type="checkbox" id="sendCopy">
                            <label class="form-check-label" for="sendCopy">
                                Send a copy to myself
                            </label>
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-primary">Send Email</button>
                </div>
            </div>
        </div>
    </div>
    
    <!-- JavaScript Libraries -->
    <script src="../assets/vendor/jquery/jquery.min.js"></script>
    <script src="../assets/vendor/bootstrap/js/bootstrap.bundle.min.js"></script>
    <script src="../assets/vendor/jquery-easing/jquery.easing.min.js"></script>
    <script src="../assets/vendor/datatables/jquery.dataTables.min.js"></script>
    <script src="../assets/vendor/datatables/dataTables.bootstrap4.min.js"></script>
    <!-- Toastify JS -->
    <link rel="stylesheet" type="text/css" href="https://cdn.jsdelivr.net/npm/toastify-js/src/toastify.min.css">
    <script type="text/javascript" src="https://cdn.jsdelivr.net/npm/toastify-js"></script>
    
    <script>
        // Sidebar toggle functionality
        document.getElementById('sidebarToggle').addEventListener('click', function() {
            document.querySelector('.sidebar').classList.toggle('show');
        });
        
        // Initialize DataTables and form validation
        $(document).ready(function() {
            // Initialize DataTables for student list
            $('#studentsTable').DataTable({
                responsive: true,
                lengthMenu: [[5, 10, 25, 50, -1], [5, 10, 25, 50, "All"]],
                order: [[0, 'asc']]
            });
            
            // Form validation for email modal
            $('#emailAllForm').submit(function(e) {
                let valid = true;
                const subject = $('#emailSubject').val().trim();
                const body = $('#emailBody').val().trim();
                
                if (subject === '') {
                    Toastify({
                        text: "Please enter an email subject",
                        duration: 3000,
                        close: true,
                        gravity: "top",
                        position: "center",
                        backgroundColor: "#e74a3b",
                    }).showToast();
                    $('#emailSubject').addClass('is-invalid');
                    valid = false;
                } else {
                    $('#emailSubject').removeClass('is-invalid');
                }
                
                if (body === '') {
                    Toastify({
                        text: "Please enter an email message",
                        duration: 3000,
                        close: true,
                        gravity: "top",
                        position: "center", 
                        backgroundColor: "#e74a3b",
                    }).showToast();
                    $('#emailBody').addClass('is-invalid');
                    valid = false;
                } else {
                    $('#emailBody').removeClass('is-invalid');
                }
                
                if (!valid) {
                    e.preventDefault();
                    return false;
                }
                
                // Show loading state
                $('#sendEmailBtn').html('<i class="fas fa-spinner fa-spin"></i> Sending...').prop('disabled', true);
                
                // Here you would normally submit the form, but for demo we'll just show a success message
                setTimeout(function() {
                    Toastify({
                        text: "Email sent successfully to all students",
                        duration: 3000,
                        close: true,
                        gravity: "top",
                        position: "center",
                        backgroundColor: "#1cc88a",
                    }).showToast();
                    
                    $('#emailAllModal').modal('hide');
                    $('#emailSubject').val('');
                    $('#emailBody').val('');
                    $('#sendEmailBtn').html('Send Email').prop('disabled', false);
                }, 1500);
                
                e.preventDefault();
                return false;
            });
            
            // Action button handlers with confirmations
            $('.send-message').click(function() {
                const studentId = $(this).data('student-id');
                const studentName = $(this).data('student-name');
                
                Toastify({
                    text: `Opening message form for ${studentName}...`,
                    duration: 3000,
                    close: true,
                    gravity: "top",
                    position: "center",
                    backgroundColor: "#4e73df",
                }).showToast();
            });
            
            $('.toggle-status').click(function() {
                const studentId = $(this).data('student-id');
                const action = $(this).data('status') === 'active' ? 'activate' : 'deactivate';
                
                if (confirm(`Are you sure you want to ${action} this student?`)) {
                    Toastify({
                        text: `Student status updated successfully`,
                        duration: 3000,
                        close: true,
                        gravity: "top",
                        position: "center",
                        backgroundColor: "#1cc88a",
                    }).showToast();
                }
            });
            
            // Handle network errors
            $(document).ajaxError(function(event, jqxhr, settings, thrownError) {
                console.error("AJAX error:", thrownError);
                Toastify({
                    text: "A network error occurred. Please try again later.",
                    duration: 5000,
                    close: true,
                    gravity: "top",
                    position: "center",
                    backgroundColor: "#e74a3b",
                }).showToast();
            });
        });
    </script>
</body>
</html>
