<?php
require_once __DIR__ . '/../auth/functions.php';
require_once __DIR__ . '/../config/fast_email_service.php';

// Ensure user is logged in as an admin
requireRole('admin');

// Bridge database connections - Email service needs PDO, but admin uses MySQLi
if (!isset($pdo)) {
    try {
        // Create PDO connection using the same credentials as MySQLi
        $pdo = new PDO("mysql:host=" . DB_SERVER . ";dbname=" . DB_NAME, DB_USERNAME, DB_PASSWORD);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
        error_log("PDO connection created successfully for email service");
    } catch (Exception $e) {
        error_log("Failed to create PDO connection for email service: " . $e->getMessage());
        $pdo = null;
    }
}

/**
 * Send teacher identity verification approval email notification
 */
function sendTeacherApprovalEmailNotification($teacher_id) {
    global $conn;
    
    try {
        // Get teacher details
        $stmt = $conn->prepare("
            SELECT u.first_name, u.last_name, u.email, u.user_id, u.created_at,
                   u.identity_uploaded_at, u.identity_verified_at
            FROM users u 
            WHERE u.user_id = ? AND u.user_role = 'teacher'
        ");
        $stmt->bind_param("i", $teacher_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $teacher = $result->fetch_assoc();
        
        if (!$teacher) {
            error_log("Teacher not found for verification approval email: $teacher_id");
            return false;
        }
        
        $teacher_name = $teacher['first_name'] . ' ' . $teacher['last_name'];
        $teacher_email = $teacher['email'];
        $verified_date = date('M j, Y', strtotime($teacher['identity_verified_at']));
        $joined_date = date('M j, Y', strtotime($teacher['created_at']));
        
        // Create email content
        $subject = "🎉 Identity Verified: You can now create courses on Panadite Academy!";
        $message = "
        <div style='max-width:600px; margin:0 auto; font-family:Arial,sans-serif;'>
            <div style='background:#28a745; color:white; padding:20px; text-align:center;'>
                <h1>🛡️ Identity Verified!</h1>
            </div>
            <div style='padding:20px; background:white;'>
                <h2>Congratulations, " . htmlspecialchars($teacher_name) . "! 🎉</h2>
                <p>Your identity verification has been <strong>approved</strong> and you can now create courses and upload videos!</p>
                <div style='background:#f8f9fa; padding:15px; border-radius:5px; margin:20px 0;'>
                    <h3>Verification Details:</h3>
                    <p><strong>Verified On:</strong> " . $verified_date . "</p>
                    <p><strong>Member Since:</strong> " . $joined_date . "</p>
                    <p><strong>Status:</strong> <span style='color:#28a745; font-weight:bold;'>✅ Verified Teacher</span></p>
                </div>
                <p><strong>What you can do now:</strong></p>
                <ul>
                    <li>✅ Create and publish courses</li>
                    <li>✅ Upload educational videos</li>
                    <li>✅ Earn revenue from course sales</li>
                    <li>✅ Access full teacher dashboard</li>
                </ul>
                <p style='text-align:center;'>
                    <a href='https://panaditeacademy.com/teacher/create-course.php' 
                       style='background:#667eea; color:white; text-decoration:none; padding:15px 25px; border-radius:8px; font-weight:bold; display:inline-block; margin-right:10px;'>
                        Create Your First Course
                    </a>
                    <a href='https://panaditeacademy.com/teacher/dashboard.php' 
                       style='background:#28a745; color:white; text-decoration:none; padding:15px 25px; border-radius:8px; font-weight:bold; display:inline-block;'>
                        View Dashboard
                    </a>
                </p>
            </div>
        </div>";
        
        // Try direct email sending
        if (function_exists('sendEmailNow')) {
            $result = sendEmailNow($teacher_email, $subject, $message);
            if ($result) {
                return true;
            }
        }
        
        // Fallback to native PHP mail
        $headers = "MIME-Version: 1.0" . "\r\n";
        $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
        $headers .= "From: noreply@panaditeacademy.com" . "\r\n";
        
        return mail($teacher_email, $subject, $message, $headers);
        
    } catch (Exception $e) {
        error_log("Error sending teacher approval email: " . $e->getMessage());
        return false;
    }
}

/**
 * Send teacher identity verification rejection email notification
 */
function sendTeacherRejectionEmailNotification($teacher_id, $rejection_reason) {
    global $conn;
    
    try {
        // Get teacher details
        $stmt = $conn->prepare("
            SELECT u.first_name, u.last_name, u.email, u.user_id, u.created_at,
                   u.identity_uploaded_at
            FROM users u 
            WHERE u.user_id = ? AND u.user_role = 'teacher'
        ");
        $stmt->bind_param("i", $teacher_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $teacher = $result->fetch_assoc();
        
        if (!$teacher) {
            error_log("Teacher not found for verification rejection email: $teacher_id");
            return false;
        }
        
        $teacher_name = $teacher['first_name'] . ' ' . $teacher['last_name'];
        $teacher_email = $teacher['email'];
        $uploaded_date = $teacher['identity_uploaded_at'] ? date('M j, Y', strtotime($teacher['identity_uploaded_at'])) : 'recently';
        
        // Create email content
        $subject = "⚠️ Identity Verification Update - Action Required";
        $message = "
        <div style='max-width:600px; margin:0 auto; font-family:Arial,sans-serif;'>
            <div style='background:#dc3545; color:white; padding:20px; text-align:center;'>
                <h1>🔍 Identity Verification Update</h1>
            </div>
            <div style='padding:20px; background:white;'>
                <h2>Hello " . htmlspecialchars($teacher_name) . ",</h2>
                <p>We've reviewed your identity verification document submitted on <strong>" . $uploaded_date . "</strong>.</p>
                <div style='background:#f8d7da; color:#721c24; padding:15px; border-radius:5px; margin:20px 0; border-left:4px solid #dc3545;'>
                    <h3>⚠️ Verification Not Approved</h3>
                    <p><strong>Reason:</strong> " . htmlspecialchars($rejection_reason) . "</p>
                </div>
                <p><strong>What happens next:</strong></p>
                <ul>
                    <li>📋 Please upload a new identity document</li>
                    <li>✅ Ensure the document is clear and legible</li>
                    <li>🆔 Use a valid government-issued ID (passport, driver's license, national ID)</li>
                    <li>⏰ Make sure the document is not expired</li>
                </ul>
                <div style='background:#fff3cd; color:#856404; padding:15px; border-radius:5px; margin:20px 0; border-left:4px solid #ffc107;'>
                    <p><strong>💡 Tips for successful verification:</strong></p>
                    <ul>
                        <li>Take a clear, well-lit photo</li>
                        <li>Ensure all text is readable</li>
                        <li>Include the full document in the frame</li>
                        <li>Use PNG, JPG, or PDF format</li>
                    </ul>
                </div>
                <p style='text-align:center;'>
                    <a href='https://panaditeacademy.com/teacher/profile.php#identity-verification' 
                       style='background:#dc3545; color:white; text-decoration:none; padding:15px 25px; border-radius:8px; font-weight:bold; display:inline-block;'>
                        Upload New Document
                    </a>
                </p>
            </div>
        </div>";
        
        // Try direct email sending
        if (function_exists('sendEmailNow')) {
            $result = sendEmailNow($teacher_email, $subject, $message);
            if ($result) {
                return true;
            }
        }
        
        // Fallback to native PHP mail
        $headers = "MIME-Version: 1.0" . "\r\n";
        $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
        $headers .= "From: noreply@panaditeacademy.com" . "\r\n";
        
        return mail($teacher_email, $subject, $message, $headers);
        
    } catch (Exception $e) {
        error_log("Error sending teacher rejection email: " . $e->getMessage());
        return false;
    }
}

$message = '';

// Handle verification action (approve/reject)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    $teacher_id = (int)$_POST['teacher_id'];
    $action = $_POST['action'];
    $admin_id = $_SESSION['user_id'];
    
    if ($action === 'verify') {
        // Approve identity verification
        $update_sql = "UPDATE users SET 
                      identity_verification_status = 'verified',
                      identity_verified_at = NOW(),
                      identity_verified_by = ?,
                      identity_rejection_reason = NULL
                      WHERE user_id = ? AND user_role = 'teacher'";
        
        $stmt = $conn->prepare($update_sql);
        $stmt->bind_param("ii", $admin_id, $teacher_id);
        
        if ($stmt->execute()) {
            // Log the verification action
            $log_sql = "INSERT INTO identity_verification_logs (teacher_id, action, previous_status, new_status, admin_id) 
                       VALUES (?, 'verified', 'pending', 'verified', ?)";
            $log_stmt = $conn->prepare($log_sql);
            $log_stmt->bind_param("ii", $teacher_id, $admin_id);
            $log_stmt->execute();
            
            // Send approval email notification
            global $debug_info;
            $email_sent = sendTeacherApprovalEmailNotification($teacher_id);
            
            // Build debug output for display
            $debug_output = '';
            if (isset($debug_info) && is_array($debug_info)) {
                $debug_output .= '<div class="mt-3"><h6>🔍 Email Debug Information:</h6>';
                
                // Show steps
                if (isset($debug_info['steps']) && is_array($debug_info['steps'])) {
                    $debug_output .= '<div><strong>Steps:</strong><ul>';
                    foreach ($debug_info['steps'] as $step) {
                        $debug_output .= '<li>' . htmlspecialchars($step) . '</li>';
                    }
                    $debug_output .= '</ul></div>';
                }
                
                // Show final result
                if (isset($debug_info['final_method'])) {
                    $debug_output .= '<div><strong>Email Method Used:</strong> ' . htmlspecialchars($debug_info['final_method']) . '</div>';
                }
                
                $debug_output .= '<div><strong>Final Result:</strong> ' . ($debug_info['success'] ? '✅ Success' : '❌ Failed') . '</div>';
                $debug_output .= '</div>';
            }
            
            if ($email_sent) {
                $message = '<div class="alert alert-success">Teacher identity verified successfully! ✉️ Verification email sent to teacher.' . $debug_output . '</div>';
            } else {
                $message = '<div class="alert alert-warning">Teacher identity verified successfully, but failed to send notification email to teacher.' . $debug_output . '</div>';
            }
        } else {
            $message = '<div class="alert alert-danger">Error verifying teacher: ' . $stmt->error . '</div>';
        }
    } elseif ($action === 'reject') {
        // Reject identity verification
        $reason = trim($_POST['rejection_reason']);
        
        if (!empty($reason)) {
            $update_sql = "UPDATE users SET 
                          identity_verification_status = 'rejected',
                          identity_verified_at = NULL,
                          identity_verified_by = NULL,
                          identity_rejection_reason = ?
                          WHERE user_id = ? AND user_role = 'teacher'";
            
            $stmt = $conn->prepare($update_sql);
            $stmt->bind_param("si", $reason, $teacher_id);
            
            if ($stmt->execute()) {
                // Log the rejection action
                $log_sql = "INSERT INTO identity_verification_logs (teacher_id, action, previous_status, new_status, admin_id, reason) 
                           VALUES (?, 'rejected', 'pending', 'rejected', ?, ?)";
                $log_stmt = $conn->prepare($log_sql);
                $log_stmt->bind_param("iis", $teacher_id, $admin_id, $reason);
                $log_stmt->execute();
                
                // Send rejection email notification
                global $debug_info;
                $email_sent = sendTeacherRejectionEmailNotification($teacher_id, $reason);
                
                // Build debug output for display
                $debug_output = '';
                if (isset($debug_info) && is_array($debug_info)) {
                    $debug_output .= '<div class="mt-3"><h6>🔍 Email Debug Information:</h6>';
                    
                    // Show steps
                    if (isset($debug_info['steps']) && is_array($debug_info['steps'])) {
                        $debug_output .= '<div><strong>Steps:</strong><ul>';
                        foreach ($debug_info['steps'] as $step) {
                            $debug_output .= '<li>' . htmlspecialchars($step) . '</li>';
                        }
                        $debug_output .= '</ul></div>';
                    }
                    
                    // Show final result
                    if (isset($debug_info['final_method'])) {
                        $debug_output .= '<div><strong>Email Method Used:</strong> ' . htmlspecialchars($debug_info['final_method']) . '</div>';
                    }
                    
                    $debug_output .= '<div><strong>Final Result:</strong> ' . ($debug_info['success'] ? '✅ Success' : '❌ Failed') . '</div>';
                    $debug_output .= '</div>';
                }
                
                if ($email_sent) {
                    $message = '<div class="alert alert-warning">Teacher identity verification rejected. ✉️ Rejection email sent to teacher.' . $debug_output . '</div>';
                } else {
                    $message = '<div class="alert alert-danger">Teacher identity verification rejected, but failed to send notification email to teacher.' . $debug_output . '</div>';
                }
            } else {
                $message = '<div class="alert alert-danger">Error rejecting verification: ' . $stmt->error . '</div>';
            }
        } else {
            $message = '<div class="alert alert-danger">Please provide a reason for rejection.</div>';
        }
    }
}

// Get teachers pending verification
$pending_query = "SELECT u.user_id, u.first_name, u.last_name, u.email, u.identity_document_path, 
                         u.identity_uploaded_at, u.identity_verification_status 
                  FROM users u 
                  WHERE u.user_role = 'teacher' 
                  AND u.identity_verification_status IN ('pending', 'rejected')
                  AND u.identity_document_path IS NOT NULL
                  ORDER BY u.identity_uploaded_at DESC";

$pending_result = $conn->query($pending_query);

// Get verified teachers
$verified_query = "SELECT u.user_id, u.first_name, u.last_name, u.email, 
                          u.identity_verified_at, ua.first_name as admin_first_name, ua.last_name as admin_last_name
                   FROM users u 
                   LEFT JOIN users ua ON u.identity_verified_by = ua.user_id
                   WHERE u.user_role = 'teacher' 
                   AND u.identity_verification_status = 'verified'
                   ORDER BY u.identity_verified_at DESC
                   LIMIT 20";

$verified_result = $conn->query($verified_query);

// Get verification statistics
$stats_query = "SELECT 
    COUNT(CASE WHEN identity_verification_status = 'pending' THEN 1 END) as pending_count,
    COUNT(CASE WHEN identity_verification_status = 'verified' THEN 1 END) as verified_count,
    COUNT(CASE WHEN identity_verification_status = 'rejected' THEN 1 END) as rejected_count,
    COUNT(*) as total_teachers
    FROM users WHERE user_role = 'teacher'";

$stats_result = $conn->query($stats_query);
$stats = $stats_result->fetch_assoc();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Teacher Identity Verification - Panadite Academy Admin</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        .verification-card { border-radius: 12px; border: none; box-shadow: 0 4px 6px rgba(0,0,0,0.1); }
        .badge-status { font-size: 0.85em; }
        .document-preview { max-width: 200px; border-radius: 8px; }
        .stats-card { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <?php include 'components/sidebar.php'; ?>
    
    <!-- Main Content -->
    <section class="main-content">
        <!-- Navbar -->
        <?php include 'components/navbar.php'; ?>
        
        <!-- Dashboard Content -->
        <div class="dashboard-content">
            <!-- Page Header -->
            <div class="page-header">
                <h1><i class='bx bx-shield-alt-2 me-2'></i>Teacher Identity Verification</h1>
                <div>
                    <a href="dashboard.php" class="btn btn-sm btn-outline-primary">
                        <i class='bx bx-arrow-left'></i> Back to Dashboard
                    </a>
                </div>
            </div>
        <?= $message ?>
        
        <!-- Statistics Cards -->
        <div class="row mb-4">
            <div class="col-md-3">
                <div class="card stats-card">
                    <div class="card-body text-center">
                        <h3 class="mb-1"><?= $stats['pending_count'] ?></h3>
                        <p class="mb-0"><i class="bi bi-clock me-1"></i>Pending Review</p>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card bg-success text-white">
                    <div class="card-body text-center">
                        <h3 class="mb-1"><?= $stats['verified_count'] ?></h3>
                        <p class="mb-0"><i class="bi bi-check-circle me-1"></i>Verified</p>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card bg-danger text-white">
                    <div class="card-body text-center">
                        <h3 class="mb-1"><?= $stats['rejected_count'] ?></h3>
                        <p class="mb-0"><i class="bi bi-x-circle me-1"></i>Rejected</p>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card bg-info text-white">
                    <div class="card-body text-center">
                        <h3 class="mb-1"><?= $stats['total_teachers'] ?></h3>
                        <p class="mb-0"><i class="bi bi-people me-1"></i>Total Teachers</p>
                    </div>
                </div>
            </div>
        </div>

        <div class="row">
            <!-- Pending Verifications -->
            <div class="col-lg-8">
                <div class="card verification-card">
                    <div class="card-header bg-warning text-dark">
                        <h5 class="mb-0">
                            <i class="bi bi-clock-history me-2"></i>
                            Pending Identity Verifications (<?= $stats['pending_count'] ?>)
                        </h5>
                    </div>
                    <div class="card-body p-0">
                        <?php if ($pending_result && $pending_result->num_rows > 0): ?>
                            <div class="table-responsive">
                                <table class="table table-hover mb-0">
                                    <thead class="table-light">
                                        <tr>
                                            <th>Teacher</th>
                                            <th>Email</th>
                                            <th>Status</th>
                                            <th>Uploaded</th>
                                            <th>Document</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php while ($teacher = $pending_result->fetch_assoc()): ?>
                                            <tr id="teacher-<?= $teacher['user_id'] ?>">
                                                <td>
                                                    <div class="fw-bold"><?= htmlspecialchars($teacher['first_name'] . ' ' . $teacher['last_name']) ?></div>
                                                    <small class="text-muted">ID: <?= $teacher['user_id'] ?></small>
                                                </td>
                                                <td><?= htmlspecialchars($teacher['email']) ?></td>
                                                <td>
                                                    <?php 
                                                    $badge_class = $teacher['identity_verification_status'] === 'rejected' ? 'bg-danger' : 'bg-warning';
                                                    $status_text = ucfirst($teacher['identity_verification_status']);
                                                    ?>
                                                    <span class="badge <?= $badge_class ?> badge-status"><?= $status_text ?></span>
                                                </td>
                                                <td>
                                                    <small><?= date('M j, Y', strtotime($teacher['identity_uploaded_at'])) ?></small>
                                                </td>
                                                <td>
                                                    <?php if (!empty($teacher['identity_document_path'])): ?>
                                                        <a href="../<?= htmlspecialchars($teacher['identity_document_path']) ?>" 
                                                           target="_blank" class="btn btn-sm btn-outline-primary">
                                                            <i class="bi bi-file-earmark-text me-1"></i>View Document
                                                        </a>
                                                    <?php else: ?>
                                                        <span class="text-muted">No document</span>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <div class="btn-group" role="group">
                                                        <button type="button" class="btn btn-sm btn-success" 
                                                                onclick="approveTeacher(<?= $teacher['user_id'] ?>)">
                                                            <i class="bi bi-check-lg me-1"></i>Approve
                                                        </button>
                                                        <button type="button" class="btn btn-sm btn-danger" 
                                                                onclick="showRejectModal(<?= $teacher['user_id'] ?>, '<?= htmlspecialchars($teacher['first_name'] . ' ' . $teacher['last_name']) ?>')">
                                                            <i class="bi bi-x-lg me-1"></i>Reject
                                                        </button>
                                                    </div>
                                                </td>
                                            </tr>
                                        <?php endwhile; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php else: ?>
                            <div class="text-center py-5">
                                <i class="bi bi-check-circle text-success" style="font-size: 3rem;"></i>
                                <h4 class="mt-3">All caught up!</h4>
                                <p class="text-muted">No pending identity verifications at this time.</p>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <!-- Recently Verified -->
            <div class="col-lg-4">
                <div class="card verification-card">
                    <div class="card-header bg-success text-white">
                        <h6 class="mb-0">
                            <i class="bi bi-check-circle me-2"></i>
                            Recently Verified
                        </h6>
                    </div>
                    <div class="card-body">
                        <?php if ($verified_result && $verified_result->num_rows > 0): ?>
                            <?php while ($teacher = $verified_result->fetch_assoc()): ?>
                                <div class="d-flex align-items-center mb-3">
                                    <div class="flex-grow-1">
                                        <div class="fw-bold small"><?= htmlspecialchars($teacher['first_name'] . ' ' . $teacher['last_name']) ?></div>
                                        <div class="text-muted small">
                                            Verified <?= date('M j', strtotime($teacher['identity_verified_at'])) ?>
                                            <?php if ($teacher['admin_first_name']): ?>
                                                by <?= htmlspecialchars($teacher['admin_first_name']) ?>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    <i class="bi bi-check-circle-fill text-success"></i>
                                </div>
                            <?php endwhile; ?>
                        <?php else: ?>
                            <div class="text-center text-muted">
                                <i class="bi bi-info-circle me-1"></i>
                                No recent verifications
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Rejection Modal -->
    <div class="modal fade" id="rejectModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <form method="post" id="rejectForm">
                    <div class="modal-header bg-danger text-white">
                        <h5 class="modal-title">
                            <i class="bi bi-exclamation-triangle me-2"></i>
                            Reject Identity Verification
                        </h5>
                        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <input type="hidden" name="teacher_id" id="reject_teacher_id">
                        <input type="hidden" name="action" value="reject">
                        
                        <p>You are about to reject the identity verification for <strong id="teacher_name"></strong>.</p>
                        
                        <div class="mb-3">
                            <label for="rejection_reason" class="form-label">
                                <strong>Reason for Rejection</strong> <span class="text-danger">*</span>
                            </label>
                            <textarea class="form-control" id="rejection_reason" name="rejection_reason" 
                                    rows="3" required placeholder="Please provide a clear reason for rejection..."></textarea>
                            <div class="form-text">
                                Be specific about why the document was rejected so the teacher can resubmit correctly.
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-danger">
                            <i class="bi bi-x-circle me-1"></i>Reject Verification
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function approveTeacher(teacherId) {
            if (confirm('Are you sure you want to approve this teacher\'s identity verification?')) {
                const form = document.createElement('form');
                form.method = 'POST';
                form.innerHTML = `
                    <input type="hidden" name="teacher_id" value="${teacherId}">
                    <input type="hidden" name="action" value="verify">
                `;
                document.body.appendChild(form);
                form.submit();
            }
        }

        function showRejectModal(teacherId, teacherName) {
            document.getElementById('reject_teacher_id').value = teacherId;
            document.getElementById('teacher_name').textContent = teacherName;
            document.getElementById('rejection_reason').value = '';
            
            const modal = new bootstrap.Modal(document.getElementById('rejectModal'));
            modal.show();
        }
    </script>
        </div> <!-- Close dashboard-content -->
    </section> <!-- Close main-content -->
</body>
</html>
