<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Video API Integration Test</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    
    <style>
        .test-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 2rem 0;
        }
        
        .platform-card {
            transition: all 0.3s ease;
            cursor: pointer;
        }
        
        .platform-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 8px 25px rgba(0,0,0,0.1);
        }
        
        .test-result {
            min-height: 100px;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .youtube-card { border-left: 4px solid #ff0000; }
        .tiktok-card { border-left: 4px solid #000000; }
        .vimeo-card { border-left: 4px solid #1ab7ea; }
        
        .status-loading { color: #007bff; }
        .status-success { color: #28a745; }
        .status-error { color: #dc3545; }
        
        .api-info {
            background: #f8f9fa;
            border: 1px solid #e9ecef;
            border-radius: 8px;
            padding: 1rem;
            margin: 1rem 0;
        }
    </style>
</head>
<body>

    <!-- Header -->
    <div class="test-header">
        <div class="container">
            <h1><i class="fas fa-flask me-3"></i>Video API Integration Test</h1>
            <p class="lead mb-0">Testing automatic duration detection for YouTube, TikTok, and Vimeo</p>
        </div>
    </div>

    <div class="container mt-4">
        
        <!-- Integration Status -->
        <div class="row mb-4">
            <div class="col-12">
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">
                            <i class="fas fa-cogs me-2 text-primary"></i>API Integration Status
                        </h5>
                    </div>
                    <div class="card-body">
                        <div id="integrationStatus" class="d-flex align-items-center">
                            <div class="spinner-border spinner-border-sm text-primary me-2" role="status"></div>
                            <span>Loading API integration...</span>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Test Platforms -->
        <div class="row">
            
            <!-- YouTube Test -->
            <div class="col-md-4 mb-4">
                <div class="card platform-card youtube-card h-100">
                    <div class="card-header bg-danger text-white">
                        <h6 class="mb-0">
                            <i class="fab fa-youtube me-2"></i>YouTube API Test
                        </h6>
                    </div>
                    <div class="card-body">
                        <div class="api-info">
                            <strong>Method:</strong> YouTube IFrame API<br>
                            <strong>Authentication:</strong> None required<br>
                            <strong>Features:</strong> Duration, Title, Thumbnail
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Test YouTube URL:</label>
                            <input type="url" class="form-control" id="youtubeUrl" 
                                   value="https://www.youtube.com/watch?v=vpAJ0s5S2t0"
                                   placeholder="Enter YouTube URL">
                        </div>
                        
                        <button class="btn btn-danger w-100 mb-3" onclick="testYouTube()">
                            <i class="fas fa-play me-2"></i>Test YouTube Detection
                        </button>
                        
                        <div id="youtubeResult" class="test-result bg-light rounded">
                            <span class="text-muted">Click test to check API</span>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- TikTok Test -->
            <div class="col-md-4 mb-4">
                <div class="card platform-card tiktok-card h-100">
                    <div class="card-header bg-dark text-white">
                        <h6 class="mb-0">
                            <i class="fab fa-tiktok me-2"></i>TikTok API Test
                        </h6>
                    </div>
                    <div class="card-body">
                        <div class="api-info">
                            <strong>Method:</strong> PostMessage API<br>
                            <strong>Authentication:</strong> None required<br>
                            <strong>Features:</strong> Duration, Current Time Events
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Test TikTok URL:</label>
                            <input type="url" class="form-control" id="tiktokUrl" 
                                   value="https://www.tiktok.com/@webdevtips/video/7234567890123456789"
                                   placeholder="Enter TikTok URL">
                        </div>
                        
                        <button class="btn btn-dark w-100 mb-3" onclick="testTikTok()">
                            <i class="fas fa-play me-2"></i>Test TikTok Detection
                        </button>
                        
                        <div id="tiktokResult" class="test-result bg-light rounded">
                            <span class="text-muted">Click test to check API</span>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Vimeo Test -->
            <div class="col-md-4 mb-4">
                <div class="card platform-card vimeo-card h-100">
                    <div class="card-header bg-info text-white">
                        <h6 class="mb-0">
                            <i class="fab fa-vimeo-v me-2"></i>Vimeo API Test
                        </h6>
                    </div>
                    <div class="card-body">
                        <div class="api-info">
                            <strong>Method:</strong> oEmbed API<br>
                            <strong>Authentication:</strong> None required<br>
                            <strong>Features:</strong> Duration, Title, Thumbnail
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Test Vimeo URL:</label>
                            <input type="url" class="form-control" id="vimeoUrl" 
                                   value="https://vimeo.com/76979871"
                                   placeholder="Enter Vimeo URL">
                        </div>
                        
                        <button class="btn btn-info w-100 mb-3" onclick="testVimeo()">
                            <i class="fas fa-play me-2"></i>Test Vimeo Detection
                        </button>
                        
                        <div id="vimeoResult" class="test-result bg-light rounded">
                            <span class="text-muted">Click test to check API</span>
                        </div>
                    </div>
                </div>
            </div>
            
        </div>
        
        <!-- Test All Button -->
        <div class="row mb-4">
            <div class="col-12 text-center">
                <button class="btn btn-success btn-lg px-5" onclick="testAllPlatforms()">
                    <i class="fas fa-rocket me-2"></i>Test All Platforms
                </button>
            </div>
        </div>
        
        <!-- Console Log -->
        <div class="row">
            <div class="col-12">
                <div class="card">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h6 class="mb-0">
                            <i class="fas fa-terminal me-2"></i>Console Log
                        </h6>
                        <button class="btn btn-sm btn-outline-secondary" onclick="clearLog()">
                            <i class="fas fa-trash me-1"></i>Clear
                        </button>
                    </div>
                    <div class="card-body">
                        <div id="consoleLog" class="bg-dark text-light p-3 rounded" style="height: 200px; overflow-y: auto; font-family: monospace; font-size: 12px;">
                            <div class="text-success">🎬 Video API Integration Test Console</div>
                            <div class="text-info">Ready to test duration detection APIs...</div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
    </div>

    <!-- Scripts -->
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="../js/video-api-integration.js"></script>

    <script>
    // Console logging function
    function logToConsole(message, type = 'info') {
        const colors = {
            info: 'text-info',
            success: 'text-success', 
            error: 'text-danger',
            warning: 'text-warning'
        };
        
        const timestamp = new Date().toLocaleTimeString();
        const logEntry = `<div class="${colors[type]}">[${timestamp}] ${message}</div>`;
        
        const consoleLog = document.getElementById('consoleLog');
        consoleLog.innerHTML += logEntry;
        consoleLog.scrollTop = consoleLog.scrollHeight;
        
        // Also log to browser console
        console.log(`[${timestamp}] ${message}`);
    }
    
    function clearLog() {
        document.getElementById('consoleLog').innerHTML = `
            <div class="text-success">🎬 Video API Integration Test Console</div>
            <div class="text-info">Ready to test duration detection APIs...</div>
        `;
    }
    
    // Check integration status
    function checkIntegrationStatus() {
        const statusDiv = document.getElementById('integrationStatus');
        
        if (typeof window.VideoAPIIntegration !== 'undefined') {
            statusDiv.innerHTML = `
                <i class="fas fa-check-circle text-success me-2"></i>
                <span class="text-success">API Integration Loaded Successfully</span>
            `;
            logToConsole('✅ VideoAPIIntegration loaded successfully', 'success');
        } else {
            statusDiv.innerHTML = `
                <i class="fas fa-exclamation-triangle text-danger me-2"></i>
                <span class="text-danger">API Integration Failed to Load</span>
            `;
            logToConsole('❌ VideoAPIIntegration failed to load', 'error');
        }
    }
    
    // Test YouTube
    function testYouTube() {
        const url = document.getElementById('youtubeUrl').value;
        const resultDiv = document.getElementById('youtubeResult');
        
        logToConsole(`🔍 Testing YouTube: ${url}`);
        
        resultDiv.innerHTML = `
            <div class="status-loading">
                <div class="spinner-border spinner-border-sm me-2" role="status"></div>
                Testing YouTube API...
            </div>
        `;
        
        if (window.VideoAPIIntegration) {
            window.VideoAPIIntegration.getVideoDuration(
                url,
                function(result) {
                    logToConsole(`✅ YouTube Success: ${result.formattedDuration} via ${result.method}`, 'success');
                    resultDiv.innerHTML = `
                        <div class="status-success text-center">
                            <i class="fas fa-check-circle fa-2x mb-2"></i><br>
                            <strong>${result.formattedDuration}</strong><br>
                            <small>${result.method}</small>
                        </div>
                    `;
                },
                function(error) {
                    logToConsole(`❌ YouTube Error: ${error}`, 'error');
                    resultDiv.innerHTML = `
                        <div class="status-error text-center">
                            <i class="fas fa-times-circle fa-2x mb-2"></i><br>
                            <strong>Failed</strong><br>
                            <small>${error}</small>
                        </div>
                    `;
                }
            );
        } else {
            logToConsole('❌ VideoAPIIntegration not available', 'error');
        }
    }
    
    // Test TikTok
    function testTikTok() {
        const url = document.getElementById('tiktokUrl').value;
        const resultDiv = document.getElementById('tiktokResult');
        
        logToConsole(`🔍 Testing TikTok: ${url}`);
        
        resultDiv.innerHTML = `
            <div class="status-loading">
                <div class="spinner-border spinner-border-sm me-2" role="status"></div>
                Testing TikTok API...
            </div>
        `;
        
        if (window.VideoAPIIntegration) {
            window.VideoAPIIntegration.getVideoDuration(
                url,
                function(result) {
                    logToConsole(`✅ TikTok Success: ${result.formattedDuration} via ${result.method}`, 'success');
                    resultDiv.innerHTML = `
                        <div class="status-success text-center">
                            <i class="fas fa-check-circle fa-2x mb-2"></i><br>
                            <strong>${result.formattedDuration}</strong><br>
                            <small>${result.method}</small>
                        </div>
                    `;
                },
                function(error) {
                    logToConsole(`❌ TikTok Error: ${error}`, 'error');
                    resultDiv.innerHTML = `
                        <div class="status-error text-center">
                            <i class="fas fa-times-circle fa-2x mb-2"></i><br>
                            <strong>Failed</strong><br>
                            <small>${error}</small>
                        </div>
                    `;
                }
            );
        } else {
            logToConsole('❌ VideoAPIIntegration not available', 'error');
        }
    }
    
    // Test Vimeo
    function testVimeo() {
        const url = document.getElementById('vimeoUrl').value;
        const resultDiv = document.getElementById('vimeoResult');
        
        logToConsole(`🔍 Testing Vimeo: ${url}`);
        
        resultDiv.innerHTML = `
            <div class="status-loading">
                <div class="spinner-border spinner-border-sm me-2" role="status"></div>
                Testing Vimeo API...
            </div>
        `;
        
        if (window.VideoAPIIntegration) {
            window.VideoAPIIntegration.getVideoDuration(
                url,
                function(result) {
                    logToConsole(`✅ Vimeo Success: ${result.formattedDuration} via ${result.method}`, 'success');
                    resultDiv.innerHTML = `
                        <div class="status-success text-center">
                            <i class="fas fa-check-circle fa-2x mb-2"></i><br>
                            <strong>${result.formattedDuration}</strong><br>
                            <small>${result.method}</small>
                        </div>
                    `;
                },
                function(error) {
                    logToConsole(`❌ Vimeo Error: ${error}`, 'error');
                    resultDiv.innerHTML = `
                        <div class="status-error text-center">
                            <i class="fas fa-times-circle fa-2x mb-2"></i><br>
                            <strong>Failed</strong><br>
                            <small>${error}</small>
                        </div>
                    `;
                }
            );
        } else {
            logToConsole('❌ VideoAPIIntegration not available', 'error');
        }
    }
    
    // Test all platforms
    function testAllPlatforms() {
        logToConsole('🚀 Testing all platforms...', 'info');
        testYouTube();
        setTimeout(() => testTikTok(), 1000);
        setTimeout(() => testVimeo(), 2000);
    }
    
    // Initialize when page loads
    document.addEventListener('DOMContentLoaded', function() {
        logToConsole('🎬 Test page loaded', 'info');
        
        // Check integration status after a short delay
        setTimeout(checkIntegrationStatus, 500);
    });
    </script>

</body>
</html>
