<?php
/**
 * Debug Script for Teacher Feedback Files System
 * This script helps diagnose issues with file uploads and viewing
 */

// Include authentication and database
require_once __DIR__ . '/components/teacher_layout.php';

echo "<h1>🔧 Teacher Feedback Files Debug</h1>";

// Check directory structure
$upload_dir = __DIR__ . '/../uploads/teacher_feedback/';
echo "<h3>📁 Directory Structure Check:</h3>";
echo "<p><strong>Upload Directory:</strong> " . $upload_dir . "</p>";
echo "<p><strong>Directory Exists:</strong> " . (is_dir($upload_dir) ? '✅ YES' : '❌ NO') . "</p>";

if (!is_dir($upload_dir)) {
    echo "<p>🔧 <strong>Creating directory...</strong></p>";
    if (mkdir($upload_dir, 0755, true)) {
        echo "<p>✅ Directory created successfully!</p>";
    } else {
        echo "<p>❌ Failed to create directory</p>";
    }
}

echo "<p><strong>Directory Writable:</strong> " . (is_writable($upload_dir) ? '✅ YES' : '❌ NO') . "</p>";

// List files in directory
if (is_dir($upload_dir)) {
    $files = scandir($upload_dir);
    $files = array_filter($files, function($file) { return !in_array($file, ['.', '..']); });
    
    echo "<h3>📋 Files in Directory (" . count($files) . "):</h3>";
    if (empty($files)) {
        echo "<p>📭 No files found in directory</p>";
    } else {
        echo "<ul>";
        foreach ($files as $file) {
            $file_path = $upload_dir . $file;
            $file_size = filesize($file_path);
            echo "<li><strong>$file</strong> (" . number_format($file_size / 1024, 2) . " KB)</li>";
        }
        echo "</ul>";
    }
}

// Check database records
echo "<h3>🗄️ Database Records Check:</h3>";
$db_query = "SELECT submission_id, student_id, assignment_id, teacher_feedback_files 
             FROM assignment_submissions 
             WHERE teacher_feedback_files IS NOT NULL AND teacher_feedback_files != ''
             LIMIT 10";

$result = $conn->query($db_query);
if ($result && $result->num_rows > 0) {
    echo "<p>✅ Found " . $result->num_rows . " submissions with teacher feedback files:</p>";
    echo "<table border='1' cellpadding='5' cellspacing='0' style='border-collapse: collapse;'>";
    echo "<tr><th>Submission ID</th><th>Student ID</th><th>Assignment ID</th><th>Files Info</th></tr>";
    
    while ($row = $result->fetch_assoc()) {
        echo "<tr>";
        echo "<td>" . $row['submission_id'] . "</td>";
        echo "<td>" . $row['student_id'] . "</td>";
        echo "<td>" . $row['assignment_id'] . "</td>";
        echo "<td>";
        
        $files = json_decode($row['teacher_feedback_files'], true);
        if ($files && is_array($files)) {
            foreach ($files as $index => $file) {
                $file_path = $upload_dir . ($file['file_path'] ?? 'unknown');
                $exists = file_exists($file_path);
                echo "<strong>" . ($file['original_name'] ?? 'Unknown') . "</strong><br>";
                echo "Path: " . ($file['file_path'] ?? 'Unknown') . "<br>";
                echo "Exists: " . ($exists ? '✅' : '❌') . "<br>";
                if ($exists) {
                    echo "Size: " . number_format(filesize($file_path) / 1024, 2) . " KB<br>";
                }
                echo "<br>";
            }
        } else {
            echo "❌ Invalid JSON data";
        }
        
        echo "</td>";
        echo "</tr>";
    }
    echo "</table>";
} else {
    echo "<p>📭 No submissions with teacher feedback files found</p>";
}

// Test file upload form
echo "<h3>🧪 Test File Upload:</h3>";
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['test_file'])) {
    $test_file = $_FILES['test_file'];
    if ($test_file['error'] === UPLOAD_ERR_OK) {
        $test_filename = 'test_' . time() . '_' . $test_file['name'];
        $test_path = $upload_dir . $test_filename;
        
        if (move_uploaded_file($test_file['tmp_name'], $test_path)) {
            echo "<p>✅ <strong>Test upload successful!</strong></p>";
            echo "<p>File: $test_filename</p>";
            echo "<p>Size: " . number_format($test_file['size'] / 1024, 2) . " KB</p>";
            echo "<p>Test Image: <img src='../uploads/teacher_feedback/$test_filename' style='max-width: 200px; max-height: 200px;'></p>";
        } else {
            echo "<p>❌ Failed to move uploaded file</p>";
        }
    } else {
        echo "<p>❌ Upload error: " . $test_file['error'] . "</p>";
    }
}

echo "<form method='POST' enctype='multipart/form-data'>";
echo "<p>Upload a test image to verify the system:</p>";
echo "<input type='file' name='test_file' accept='image/*' required>";
echo "<button type='submit'>🧪 Test Upload</button>";
echo "</form>";

echo "<hr>";
echo "<p><strong>Instructions:</strong></p>";
echo "<ol>";
echo "<li>Check if the directory exists and is writable</li>";
echo "<li>Verify database records match actual files</li>";
echo "<li>Test upload functionality</li>";
echo "<li>If issues persist, check server error logs</li>";
echo "</ol>";

$conn->close();
?>

<style>
body { font-family: Arial, sans-serif; margin: 20px; line-height: 1.6; }
table { border-collapse: collapse; width: 100%; margin: 10px 0; }
th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
th { background-color: #f2f2f2; }
h1, h3 { color: #333; }
</style>
